/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



#include <stdlib.h>

#include "gettext.h"
#include "cdw_thread.h" /* run_command() */
#include "cdw_config.h"
#include "cdw_processwin.h"
#include "cdw_widgets.h"
#include "cdw_string.h"
#include "cdw_dvd_rw_mediainfo.h"
#include "cdw_debug.h"
#include "cdw_drive.h"

extern cdw_config_t global_config;      /* variable holding cdw configuration */


/**
   \brief Get basic info about disc currently in drive

   Run dvd+rw-mediainfo to obtain some information about disc that is
   currently in drive. The information is written to log file and will
   be processed by regexp code.

   \param task - variable describing current task

   \return CDW_MEM_ERROR on concat() errors
   \return CDW_OK otherwise
*/
cdw_rv_t cdw_dvd_rw_mediainfo_run_task(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->media_info.tool.fullpath != (char *) NULL, "ERROR: tool fullpath is NULL\n");
	const char *drive = cdw_drive_get_drive_fullpath();
	char *command = cdw_string_concat(task->media_info.tool.fullpath, " ", drive, (char *) NULL);
	if (command == (char *) NULL) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("An error occurred when attempting to read DVD info."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_MEM_ERROR;
	}

	/* 2TRANS: this is message displayed in process window */
	cdw_processwin_display_sub_info(_("Getting disc information"));
	run_command(command, task);
	free(command);
	command = (char *) NULL;

	return CDW_OK;
}





/* dvd_rw_mediainfo pipe regexp has set values of some fields of "disc" data
   structure, and this function interprets these values; typically this
   function should be called by "disc" resolver function which doesn't need
   to be bothered with *how* dvd_rw_mediainfo sees disc, but is interested
   *what* dvd_rw_mediainfo sees (i.e. sees that "disc is empty" or "disc is
   not appendable anymore") */
cdw_rv_t cdw_dvd_rw_mediainfo_set_disc_states(cdw_disc_t *disc)
{
	if (disc->type == CDW_DVD_ROM) {
		/* dvd+rw-mediainfo doesn't print disc status for
		   DVD-ROM discs, let's do some "manual" intervention */
		disc->dvd_rw_mediainfo_info.disc_status = 'c';
	}

	if (disc->dvd_rw_mediainfo_info.disc_status == 'b') {
		cdw_vdm ("INFO: disc status=\"blank\", setting disc state to empty=true and writable=true\n");
		disc->state_empty = CDW_TRUE;
		disc->state_writable = CDW_TRUE;

	} else if (disc->dvd_rw_mediainfo_info.disc_status == 'a') {
		cdw_vdm ("INFO: disc status=\"appendable\", setting disc state to empty=false and wriable=true\n");
		disc->state_empty = CDW_FALSE;
		disc->state_writable = CDW_TRUE;

	} else if (disc->dvd_rw_mediainfo_info.disc_status == 'c') {
		cdw_vdm ("INFO: disc status=\"complete\", setting disc state to empty=false and writable=false\n");
		disc->state_empty = CDW_FALSE;
		disc->state_writable = CDW_FALSE;

	} else if (disc->dvd_rw_mediainfo_info.disc_status == 'o') {
		cdw_vdm ("WARNING: disc status=\"other\", setting state to empty=unknown and writable=unknown\n");
		disc->state_empty = CDW_UNKNOWN;
		disc->state_writable = CDW_UNKNOWN;
	} else {
		cdw_vdm ("ERROR: incorrect value in disc_status field: '%c'\n",
			 disc->dvd_rw_mediainfo_info.disc_status);
		/* setting safe values just in case if caller ignores
		   returned error value */
		disc->state_empty = CDW_UNKNOWN;
		disc->state_writable = CDW_UNKNOWN;

		return CDW_GEN_ERROR;
	}

	if (disc->type == CDW_DVD_R
	    || disc->type == CDW_DVD_R_SEQ
	    || disc->type == CDW_DVD_R_RES
	    || disc->type == CDW_DVD_RP
	    || disc->type == CDW_DVD_RW
	    || disc->type == CDW_DVD_RW_SEQ
	    || disc->type == CDW_DVD_ROM) {

		if (disc->dvd_rw_mediainfo_info.read_capacity == 0) {
			if (disc->state_empty != CDW_TRUE) {
				cdw_vdm ("WARNING: read capacity == 0, but state_empty was not set to \"true\"\n");
			}
			cdw_vdm ("INFO: read capacity == 0, setting disc state_empty to \"true\"\n");
			disc->state_empty = CDW_TRUE;

		} else if (disc->dvd_rw_mediainfo_info.read_capacity > 0) {
			if (disc->state_empty != CDW_FALSE) {
				cdw_vdm ("WARNING: read capacity > 0, but state_empty was not set to \"no\"\n");
			}
			cdw_vdm ("INFO: read capacity > 0, setting disc state_empty to \"false\"\n");
			disc->state_empty = CDW_FALSE;
		} else {
			cdw_vdm ("WARNING: read capacity < 0, setting disc state_empty to \"unknown\"\n");
			disc->state_empty = CDW_UNKNOWN;
		}

		return CDW_OK;

	} else if (disc->type == CDW_DVD_RW_RES
		   || disc->type == CDW_DVD_RWP) {

		cdw_vdm ("INFO: disc type = \"%s\", setting disc state_appendable to \"yes\"\n",
			 disc->type_label);
		disc->state_writable = CDW_TRUE;

		if (disc->dvd_rw_mediainfo_info.read_capacity == 0) {
			cdw_vdm ("INFO: read capacity == 0, setting disc state_empty to \"yes\"\n");
			disc->state_empty = CDW_TRUE;
		} else {
			cdw_vdm ("INFO: read capacity != 0, setting disc state_empty to \"unknown\"\n");
			disc->state_empty = CDW_UNKNOWN;
		}

		return CDW_OK;

	} else if (disc->type == CDW_DVD_RP_DL
		   && global_config.support_dvd_rp_dl) {
		/* probably this branch should be merged with some
		   other branch */

		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: unknown disc type %d\n", disc->type);
		return CDW_GEN_ERROR;
	}
}

