/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdbool.h>
#include <unistd.h>

#include "cdw_disc.h"
#include "cdw_task.h"
#include "cdw_file.h"
#include "cdw_file_picker.h"
#include "cdw_file_manager.h"
#include "gettext.h"
#include "cdw_create_image.h"
#include "cdw_widgets.h"
#include "cdw_utils.h"
#include "cdw_fs.h"
#include "cdw_config.h"
#include "cdw_config_ui.h"
#include "cdw_debug.h"
#include "cdw_mkisofs.h"
#include "cdw_main_window.h"
#include "cdw_processwin.h"

extern cdw_config_t global_config;      /* variable holding cdw configuration */

//static cdw_rv_t cdw_create_image_check_boot_image(void);
static cdw_rv_t cdw_create_image_get_target_data_image_path(void);


/**
   \brief Check preconditions for creating image and invoke process creating image

   Check if there are any files selected by user, do any other
   necessary checks and then call run_command_*() to create ISO
   image from the files.

   \return CDW_NO if some of preconditions were not met
   \return CDW_GEN_ERROR on errors
   \return value returned by cdw_mkisofs_run_task()
*/
cdw_rv_t cdw_create_image(void)
{
	CDW_TASK_CREATE_TASK(task_create_image, CDW_TASK_CREATE_IMAGE);

	/* 1: check if we have proper tool to do it */
	cdw_rv_t crv = cdw_task_select_tools_for_task(&task_create_image, (cdw_disc_t *) NULL);
	if (crv != CDW_OK) {
		/* cdw_ext_tools_select_tool() displays error
		   dialog if a tool can't be found */
		return CDW_NO;
	}
#if 0
	/* 2: check if user specified boot image, and if so,
	   if the boot image exists */
	crv = cdw_create_image_check_boot_image();
	if (crv != CDW_OK) {
		return CDW_NO;
	}
#endif
	/* 3: create graftpoints file for mkisofs; first checks if there
	   are any files that will be used as source data */
	crv = cdw_file_manager_create_graftpoints_file();
	if (crv != CDW_OK) {
		return CDW_NO;
	}

	/* 4: */
	crv = cdw_create_image_get_target_data_image_path();
	if (crv != CDW_OK) {
		cdw_vdm ("INFO: cancelled at getting target image path\n");
		cdw_file_manager_delete_graftpoints_file();
		return CDW_NO;
	}

	/* 5: ask user for volume name (depends on checkbox in configuration) */
	crv = cdw_config_ui_conditional_volume_label_dialog();
	if (crv != CDW_OK) {
		cdw_vdm ("INFO: cancelled at creating volume label\n");
		cdw_file_manager_delete_graftpoints_file();
		/* e.g. "Escape" key */
		return CDW_NO;
	}

	/* 2TRANS: this is title of dialog window */
	cdw_processwin_create(_("Create image"),
			      /* 2TRANS: this is message in dialog
				 window - creating iso image is in
				 progress */
			      _("Creating iso image..."), true);

	/* 6: go! */
	cdw_disc_t *current_disc = cdw_disc_get();
	crv = cdw_mkisofs_run_task(&task_create_image, current_disc);

	/* 2TRANS: this is message in dialog window:
	   operation finished with unknown result */
	cdw_processwin_destroy(_("Finished"), true);

	cdw_file_manager_delete_graftpoints_file();

	/* this function sets task.success according to task->tool_status,
	   and resets task->tool_status; displays error messages in case
	   of (some) problems reported by mkisofs */
	cdw_task_check_tool_status(&task_create_image);

	if (crv == CDW_OK) {
		/* 2TRANS: this is title of dialog window, window shows
		   messages from program writing selected files to iso
		   image (creating iso image from selected files) */
		after_event(_("\"Create image\" log"), 1);
	} else {
		cdw_vdm ("ERROR: failed to create image\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      probably some malloc() call failed when
				      program was preparing call of mkisofs;
				      'command' is command-line command. 'Image'
				      means iso image file. */
				   _("System error occurred when preparing command. Image not created."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
	}

	return crv;
}




#if 0
cdw_rv_t cdw_create_image_check_boot_image(void)
{
	if (strlen(global_config.boot_image_path)) {
		int rv = cdw_fs_check_existing_path(global_config.boot_image_path, R_OK, CDW_FS_FILE);
		if (rv != 0) {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Boot image error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Boot image file doesn't exist or has wrong permissions! Check \"Boot image\" option in Configuration."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_NO;
		}
	}
	return CDW_OK;
}
#endif




cdw_rv_t cdw_create_image_get_target_data_image_path(void)
{
	/* 2TRANS: this is title of dialog window */
	cdw_rv_t crv = cdw_fs_ui_file_picker(_("Path to iso image"),
					     /* 2TRANS: this is message in dialog window;
					     below it there is an input field where user can
					     enter path to ISO image file */
					     _("Please enter FULL path to new iso image file:"),
					     &(global_config.iso_image_full_path),
					     CDW_FS_FILE, R_OK | W_OK, CDW_FS_NEW | CDW_FS_EXISTING);

	cdw_vcrv ("cdw_fs_ui_file_picker", crv);
	if (crv == CDW_OK) {
		cdw_vdm ("INFO: file picker returns \"%s\"\n", global_config.iso_image_full_path);
	} else if (crv == CDW_CANCEL) {
		cdw_vdm ("INFO: pressed escape in file picker, fullpath is \"%s\"\n", global_config.iso_image_full_path);
	} else {
		cdw_vdm ("INFO: file picker returns CDW_GEN_ERROR, fullpath is \"%s\"\n", global_config.iso_image_full_path);
	}
	cdw_main_ui_main_window_wrefresh();

	return crv;
}


