/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <errno.h>
/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "gettext.h"
#include "cdw_thread.h" /* run_command() */
#include "cdw_widgets.h"
#include "cdw_config.h"
#include "cdw_config_window.h"
#include "cdw_string.h"
#include "cdw_utils.h"
#include "cdw_processwin.h"
#include "cdw_mkisofs.h"
#include "cdw_cdrecord.h"
#include "cdw_cdrecord_options.h"
#include "cdw_iso9660.h"
#include "cdw_logging.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_erase_disc.h"
#include "cdw_drive.h"
#include "cdw_fs.h"

/**
  you may want to take a look at this page:

  \li http://www.troubleshooters.com/linux/coasterless.htm
*/

/**
\verbatim
                         DVD-R              DVD+R              DVD-RW Seq.        DVD-RW Res.     DVD+RW      DVD+R DL

start appendable                                                                                  OK

create single

continue appendable                                                                               NO (1)

write final                                                                                       NO (1)

write image (first                                                                                OK (2)
session, appendable)

write image (first                                                                                NO (3)
session, non-appen.)

Notes:
1. since "cdrecord -msinfo" returns "0,0" for non-empty DVD+RW, it is
   impossible (using msinfo information) to create command for appending
   second (and following) session, or write closing session
2. writing to DVD+RW always creates appendable disc
3. you just can't close DVD+RW disc, so this combination is invalid

\endverbatim

\verbatim
output of "cdrecord -toc dev=<dev file>" for empty (virgin) CD-R:
"
    Capacity  Blklen/Sparesz.  Format-type  Type
    16763814             2048         0x00  No Media Present or Unknown Capacity
    [...]
cdrecord: Cannot read TOC header
cdrecord: Cannot read TOC/PMA
"


output of "cdrecord -toc dev=<dev file>" for empty (erased) CD-RW:
"
    Capacity  Blklen/Sparesz.  Format-type  Type
      359849             2048         0x00  Unformated or Blank Media
           0         14064598         0x10  Reserved (0)
           0         14064598         0x11  Reserved (0)
           0         14064598         0x12  Reserved (0)
            [...]
cdrecord: Cannot read TOC header
cdrecord: Cannot read TOC/PMA
"


output of "cdrecord -toc dev=<dev file>" for non-empty, appendable CD-RW (first session, size of written data was 49507 sectors):
"
    Capacity  Blklen/Sparesz.  Format-type  Type
       49509             2048         0x00  Formatted Media
           0         14064598         0x10  Reserved (0)
           0         14064598         0x11  Reserved (0)
           0         14064598         0x12  Reserved (0)
first: 1 last 1
track:   1 lba:         0 (        0) 00:02:00 adr: 1 control: 4 mode: 2
track:lout lba:     49509 (   198036) 11:02:09 adr: 1 control: 4 mode: -1
"


output of "cdrecord -toc dev=<dev file>" for non-empty, appendable CD-RW (second session, size of written data was 23530 sectors):
"
    Capacity  Blklen/Sparesz.  Format-type  Type
       84441             2048         0x00  Formatted Media
           0         14064598         0x10  Reserved (0)
           0         14064598         0x11  Reserved (0)
           0         14064598         0x12  Reserved (0)
first: 1 last 2
track:   1 lba:         0 (        0) 00:02:00 adr: 1 control: 4 mode: 2
track:   2 lba:     60909 (   243636) 13:34:09 adr: 1 control: 4 mode: 2
track:lout lba:     84441 (   337764) 18:47:66 adr: 1 control: 4 mode: -1
"


output of "cdrecord -toc dev=<dev file>" for non-empty, closed CD-RW (third session, size of written data was 22781 sectors):
"
    Capacity  Blklen/Sparesz.  Format-type  Type
      114124             2048         0x00  Formatted Media
           0         14064598         0x10  Reserved (0)
           0         14064598         0x11  Reserved (0)
           0         14064598         0x12  Reserved (0)
first: 1 last 3
track:   1 lba:         0 (        0) 00:02:00 adr: 1 control: 4 mode: 2
track:   2 lba:     60909 (   243636) 13:34:09 adr: 1 control: 4 mode: 2
track:   3 lba:     91341 (   365364) 20:19:66 adr: 1 control: 4 mode: 2
track:lout lba:    114124 (   456496) 25:23:49 adr: 1 control: 4 mode: -1
"

\endverbatim
*/


static cdw_rv_t cdw_cdrecord_write_from_image(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_cdrecord_write_from_files(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_cdrecord_erase_disc(cdw_task_t *task, cdw_disc_t *disc);
static cdw_rv_t cdw_cdrecord_get_meta(cdw_task_t *task, cdw_disc_t *disc);

static char *cdw_cdrecord_create_command_for_writing(cdw_task_t *task, cdw_disc_t *disc);
static void cdw_cdrecord_set_disc_mode_arg(cdw_task_t *task, char *disc_mode);
static bool cdw_cdrecord_add_multisession_info(cdw_task_t *task);

static void cdw_cdrecord_internal_error_dialog(cdw_id_t id);

#define TSIZE_LEN 20



/* used in only one function: cdw_rv_t run_command_cdrecord_write_direct(void);
   I know that it is an awful solution, but it has to stay for now */
#define CDRECORD_WRITE_DATA_FREE_RESOURCES \
	/* ANSI C allows free(NULL); */\
	free(final_command); \
	final_command = NULL; \
	\
	free(mkisofs_command); \
	mkisofs_command = NULL; \
	\
	free(tsize_command); \
	tsize_command = NULL; \
	\
	free(cdrecord_command); \
	cdrecord_command = NULL; \




cdw_rv_t cdw_cdrecord_run_task(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_rv_t crv = CDW_OK;
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD, "ERROR: incorrect tool id %lld for task \"burn from files\"\n", task->burn.tool.id);
		crv = cdw_cdrecord_write_from_files(task, disc);
	} else if (task->id == CDW_TASK_BURN_FROM_IMAGE) {
		cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD, "ERROR: incorrect tool id %lld for task \"burn from image\"\n", task->burn.tool.id);
		crv = cdw_cdrecord_write_from_image(task, disc);
	} else if (task->id == CDW_TASK_MEDIA_INFO) {
		cdw_assert (task->media_info.tool.id == CDW_TOOL_CDRECORD, "ERROR: incorrect tool id %lld for task \"media info\"\n", task->media_info.tool.id);
		crv = cdw_cdrecord_get_meta(task, disc);
	} else if (task->id == CDW_TASK_ERASE_DISC) {
		cdw_assert (task->erase.tool.id == CDW_TOOL_CDRECORD, "ERROR: incorrect tool id %lld for task \"erase\"\n", task->erase.tool.id);
		crv = cdw_cdrecord_erase_disc(task, disc);
	} else {
		cdw_assert (0, "ERROR: incorrect task->id %lld\n", task->id);
	}
	return crv;
}





/**
   \brief Write selected iso image to optical disc

   Create command line string (program name + arguments) for writing iso
   image to optical disc. Run the command, show a window with process
   information to the user.

   Valid file image path should be set in options.
   Function does not check validity of the path.

   \pre \p task->source is set to OBJ_ISO_IMAGE.
   \pre \p task->write_session_mode is set to proper value

   \param task - variable describing current task

   \return CDW_OK on success
   \return CDW_GEN_ERROR on errors
 */
cdw_rv_t cdw_cdrecord_write_from_image(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT,
		    "ERROR: session mode is not set properly\n");
	cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT,
		    "ERROR: disc mode is not set properly\n");
	cdw_assert (task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task id %lld\n", task->id);

	/* the function will examine 'taks' to check if data
	   should be read from ISO file */
	char *command = cdw_cdrecord_create_command_for_writing(task, disc);
	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create command for burning from image\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window:
				      'image file' refers to 'ISO image file' */
				   _("Failed to prepare burning image file to disc"),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	cdw_vdm ("INFO: final command for writing image to optical disc:\n");
	cdw_vdm ("INFO: \"%s\"\n", command);

	int rv = run_command(command, task);
	free(command);
	command = (char *) NULL;

	if (rv == 0) {
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}





/**
   \brief Write files selected by user to optical disc

   Create command line string (program name + arguments) for writing files
   to optical disc. Run the command (using run_command()), show a window with
   process information to the user.

   The command line string (when passed to shell) will do the following:
   Run program (i.e. mkisofs) that will create iso image and write the image
   to its stdout. Run another program (i.e. cdrecord), that will read iso
   filesystem (via normal shell pipe '|') from its stdin and write it to CD
   disk.

   In previous versions cdw created symlinks to all files that should be
   written to iso image, and told program creating iso image to follow these
   symlinks; current method is to write list of files to tmp file and feed
   this file to program creating iso image (that is how k3b does it, I
   think). Look for graft-points in mkisofs man page for details.

   \param task - variable describing current task

   \return CDW_OK on success
   \return CDW_ERROR on failure
 */
cdw_rv_t cdw_cdrecord_write_from_files(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT,
		    "ERROR: session mode is not set properly\n");
	cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT,
		    "ERROR: disc mode is not set properly\n");
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES,
		    "ERROR: incorrect task id %lld\n", task->id);

	char *mkisofs_command = (char *) NULL;
	char *tsize_command = (char *) NULL;
	char *cdrecord_command = (char *) NULL;
	char *final_command = (char *) NULL;

	/* prepare command creating iso image;
	   second argument is true - append path to mkisofs at beginning of string;
	   third argument is write_to_stdout = true: mkisofs will write to stdout */
        mkisofs_command = cdw_mkisofs_create_command(task, disc);
	if (mkisofs_command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to prepare mkisofs command string\n");

		cdw_cdrecord_internal_error_dialog(1);
		return CDW_ERROR;
	}
	cdw_vdm ("INFO: mkisofs command for creating ISO file system:\n");
	cdw_vdm ("INFO: \"%s\"\n", mkisofs_command);

	/* prepare command that will compute value of tsize - shell variable
	   that will be used by program writing to disk - read
	   cdw_cdrecord_create_command_for_writing() function for explanation */
	tsize_command = cdw_string_concat("tsize=`", mkisofs_command, " --print-size -quiet`", (char *) NULL);
	if (tsize_command == CDW_OK) {
		cdw_cdrecord_internal_error_dialog(2);
		CDRECORD_WRITE_DATA_FREE_RESOURCES;
		cdw_vdm ("ERROR: failed to prepare tsize command string\n");
		return CDW_ERROR;
	}
	cdw_vdm ("INFO: mkisofs command for calculating tsize value:\n");
	cdw_vdm ("INFO: \"%s\"\n", tsize_command);

	/* prepare command reading iso image from stdin and
	   writing it to CD disk; the function will examine "task"
	   to check that data should be read by mkisofs from files on hard
	   drive and streamed to cdrecord */
	cdrecord_command = cdw_cdrecord_create_command_for_writing(task, disc);
	if (cdrecord_command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to prepare cdrecord command string\n");

		cdw_cdrecord_internal_error_dialog(3);
		CDRECORD_WRITE_DATA_FREE_RESOURCES;

		return CDW_ERROR;
	}
	cdw_vdm ("INFO: cdrecord command for burning data incoming from stdin:\n");
	cdw_vdm ("INFO: \"%s\"\n", cdrecord_command);

	/* 'write direct' command in final shape is created here */
	final_command = cdw_string_concat(tsize_command, "; ", mkisofs_command, " | ", cdrecord_command, (char *) NULL);
	if (final_command == (char *) NULL) {
		cdw_vdm ("failed to prepare final command string\n");

		cdw_cdrecord_internal_error_dialog(4);
		CDRECORD_WRITE_DATA_FREE_RESOURCES;

		return CDW_ERROR;
	}
	cdw_vdm ("INFO: final command for writing files to optical disc:\n");
	cdw_vdm ("INFO: \"%s\"\n", final_command);

	/* 2TRANS: this is message printed in log file */
	cdw_logging_write(_("\n\nPreparing to write to optical disc directly from files.\n"));
	/* 2TRANS: this is message printed in log file; "%s" is command
	   (program name + options) passed to shell */
	cdw_logging_write(_("Command for calculating tsize: \"%s\"\n"), tsize_command);
	/* 2TRANS: this is message printed in log file; "%s" is command
	   (program name + options) passed to shell */
	cdw_logging_write(_("Command for mkisofs: \"%s\"\n"), mkisofs_command);
	/* 2TRANS: this is message printed in log file; "%s" is command
	   (program name + options) passed to shell */
	cdw_logging_write(_("Command for cdrecord: \"%s\"\n"), cdrecord_command);
	cdw_logging_write("\n\n");
	/* no need to print "final_command" to log - this is done in run_command() */

	int rv = run_command(final_command, task);

	CDRECORD_WRITE_DATA_FREE_RESOURCES;
	if (rv == 0) {
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}







/**
   \brief Erase optical disk using 'all' or 'fast' mode

   This function does the following:
   \li Prepares blanking command
   \li Calls run_command() to blank disc
   \li If blanking could not be performed, user is informed about this

   Media should be checked for "erasability" before calling this function.
   Erasing method (fast/all) should be selected before calling this function.

   \param task - variable describing current task

   \return CDW_OK on success (probably)
   \return CDW_NO on failure
   \return CDW_ERROR on errors
 */
cdw_rv_t cdw_cdrecord_erase_disc(cdw_task_t *task, cdw_disc_t *disc)
{
	/* using config.scsi (X:Y:Z) is optional and advised only if
	   config.cdrw_device (/dev/xxx) does not work; if /dev/xxx works fine,
	   then config.scsi should be left empty */
	bool use_scsi = cdw_config_has_scsi_device();
	const char *device = use_scsi ? cdw_config_get_scsi_drive() : cdw_drive_get_drive_fullpath();

	task->erase.erase_time = 0;

	char speed_string[4 + 1];
	snprintf(speed_string, 4 + 1, "%lld", task->erase.speed_id);

	cdw_assert (task->erase.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
	char *command = cdw_string_concat(task->erase.tool.label, " -v ",
					  " dev=", device,
					  " speed=", speed_string,
					  " blank=", task->erase.erase_mode == CDW_ERASE_MODE_FAST ? "fast " : "all ",
					  (char *) NULL);

	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create command for erasing disc\n");
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("An error occurred when attempting to erase disc."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		return CDW_ERROR;
	}

	cdw_vdm ("INFO: cdrecord command for erasing disc:\n");
	cdw_vdm ("INFO: \"%s\"\n", command);

	cdw_vdm ("command = \"%s\"\n", command);

	int rv = run_command(command, task);

	free(command);
	command = (char *) NULL;

	if (rv != 0) {
		return CDW_ERROR;
	}

	/* it might occur during blanking that 'this media does not
	   support blanking, ignoring', and disc->type_erasable was
	   set to 'MEDIA_ERASABLE_NO' in run_command(); let's inform user
	   about this sad fact */
	if (disc->type_erasable == CDW_FALSE) {
		if (task->erase.erase_mode == CDW_ERASE_MODE_FAST) {
			if (disc->type == CDW_DVD_RWP) {
				/* wodim (version 1.1.10) can't blank DVD+RW
				   discs on my machine */
				/* 2TRANS: this is title of dialog window:
				   erasing refers to erasing of CD-RW*/
				cdw_buttons_dialog(_("Erasing error"),
						   /* 2TRANS: this is message
						      in dialog window */
						   _("Can't erase the disc. Try selecting different implementation of cdrecord in Configuration -> Tools or switch to dvd+rw-tools (if available)."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			} else {
				/* first attempt of erase was using blank = fast,
				   cdrecord may suggest using blank=all */

				/* 2TRANS: this is title of dialog window:
				   erasing refers to erasing of CD-RW*/
				cdw_buttons_dialog(_("Erasing error"),
						   /* 2TRANS: this is message
						      in dialog window */
						   _("This disc cannot be erased. Please try erasing with option \'Erase all content\'."),
						   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			}
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Erasing error"),
					   /* 2TRANS: this is message
					      in dialog window */
					   _("This disc cannot be erased."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		}
		return CDW_CANCEL;
	} else {
		if (task->erase.erase_time == 0) {
			return CDW_NO;
		} else {
			return CDW_OK;
		}
	}
}





/**
   \brief Get basic info about disc currently in drive

   Run cdrecord with one of three specified argument: "-atip", "-prcap",
   "-msinfo" to get meta information about disc in drive. Function selects
   one of these three arguments based on task->id value.

   A processwin is created to indicate that some activity is in progress.

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return CDW_ERROR on errors
   \return CDW_OK on success
*/
cdw_rv_t cdw_cdrecord_get_meta(cdw_task_t *task, cdw_disc_t *disc)
{
	/* using config.scsi (X:Y:Z) is optional and advised only if
	   config.cdrw_device (/dev/xxx) does not work; if /dev/xxx works fine,
	   then config.scsi should be left empty */
	bool use_scsi = cdw_config_has_scsi_device();
	const char *device = use_scsi ? cdw_config_get_scsi_drive() : cdw_drive_get_drive_fullpath();

	cdw_assert (task->media_info.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
	char *command = (char *) NULL;
	bool success = true;

	/* 2TRANS: this is title of dialog window */
	char *error_dialog_title = _("Error");
	/* 2TRANS: this is message in dialog window */
	char *error_dialog_message = _("An error occurred when attempting to read some meta information from disc.");


// #define CDW_CDRECORD_DISABLE_ATIP

#ifdef CDW_CDRECORD_DISABLE_ATIP
	bool w = cdw_ext_tools_is_cdrecord_wodim();
	if (disc->type == CDW_DVD_R && !w) {
		/* sometimes cdrecord may have biiiig problems reading DVD-R
		   atip data (when it happened on my machine it took very
		   much time), so it may be wise to sometimes disable getting
		   atip data for DVD-R */
	} else {
#else
	{
#endif
		command = cdw_string_concat(task->media_info.tool.label, " -vv -atip "
					    " dev=", device, (char *) NULL);
		if (command == (char *) NULL) {
			cdw_buttons_dialog(error_dialog_title, error_dialog_message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			success = false;
		} else {
			cdw_vdm ("INFO: cdrecord ATIP command:\n");
			cdw_vdm ("INFO: \"%s\"\n", command);
			/* 2TRANS: this is message printed into log file */
			cdw_logging_write(_("cdrecord command for extracting ATIP information: \"%s\"\n"), command);
			/* 2TRANS: this is message in process window,
			   it is about getting some meta information from
			   optical disc */
			cdw_processwin_display_sub_info(_("Getting ATIP information"));
			int rv = run_command(command, task);
			free(command);
			command = (char *) NULL;
			cdw_task_check_tool_status(task);
			if (!task->tool_status.ok || rv != 0) {
				cdw_vdm ("ERROR: failed to get ATIP information\n");
				success = false;
			}
		}
	}

	if (success) {
		command = cdw_string_concat(task->media_info.tool.label, " -vvvv -prcap ",
					    " dev=", device, (char *) NULL);
		if (command == (char *) NULL) {
			cdw_buttons_dialog(error_dialog_title, error_dialog_message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			success = false;
		} else {
			cdw_vdm ("INFO: cdrecord PRCAP command:\n");
			cdw_vdm ("INFO: \"%s\"\n", command);
			/* 2TRANS: this is message printed into log file */
			cdw_logging_write(_("cdrecord command for extracting PRCAP information: \"%s\"\n"), command);
			/* 2TRANS: this is message in process window,
			   it is about getting some meta information from
			   optical disc */
			cdw_processwin_display_sub_info(_("Getting PRCAP information"));
			int rv = run_command(command, task);
			free(command);
			command = (char *) NULL;
			cdw_task_check_tool_status(task);
			if (!task->tool_status.ok || rv != 0) {
				cdw_vdm ("ERROR: failed to get PRCAP information\n");
				success = false;
			}
		}
	}

	/* check state_empty: getting TOC of empty DVD may take looooong time */
	if (success && (disc->state_empty != CDW_TRUE)) {
		command = cdw_string_concat(task->media_info.tool.label, " -toc -vv ",
					    " dev=", device, (char *) NULL);
		if (command == (char *) NULL) {
			cdw_buttons_dialog(error_dialog_title, error_dialog_message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			success = false;
		} else {
			cdw_vdm ("INFO: cdrecord TOC command:\n");
			cdw_vdm ("INFO: \"%s\"\n", command);
			/* 2TRANS: this is message printed into log file */
			cdw_logging_write(_("cdrecord command for extracting TOC information: \"%s\"\n"), command);
			/* 2TRANS: this is message in process window,
			   it is about getting some meta information from
			   optical disc */
			cdw_processwin_display_sub_info(_("Getting TOC information"));
			int rv = run_command(command, task);
			free(command);
			command = (char *) NULL;
			cdw_task_check_tool_status(task);
			if (!task->tool_status.ok || rv != 0) {
				cdw_vdm ("ERROR: failed to get TOC information\n");
				success = false;
			}
		}
	}

	if (success) {
		command = cdw_string_concat(task->media_info.tool.label, " -msinfo ",
					    " dev=", device, (char *) NULL);
		if (command == (char *) NULL) {
			cdw_buttons_dialog(error_dialog_title, error_dialog_message, CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			success = false;
		} else {
			cdw_vdm ("INFO: cdrecord MSINFO command:\n");
			cdw_vdm ("INFO: \"%s\"\n", command);
			/* 2TRANS: this is message printed into log file */
			cdw_logging_write(_("cdrecord command for extracting MSINFO information: \"%s\"\n"), command);
			/* 2TRANS: this is message in process window,
			   it is about getting some meta information from
			   optical disc */
			cdw_processwin_display_sub_info(_("Getting MSINFO information"));
			int rv = run_command(command, task);
			free(command);
			command = (char *) NULL;
			cdw_task_check_tool_status(task);
			if (!task->tool_status.ok || rv != 0) {
				cdw_vdm ("ERROR: failed to get MSINFO information\n");
				success = false;
			}
		}
	}

	if (success) {
		return CDW_OK;
	} else {
		if (task->tool_status.cdrecord & CDW_TOOL_STATUS_CDRECORD_NO_DISK_WRONG_DISK) {
			/* there may be some information in "disc"
			   variable, but it is either incomplete,
			   or wrong; or both; */
			cdw_disc_reset(disc);
		}

		cdw_vdm ("ERROR: failed to correctly get meta information with cdrecord; consult error informations above\n");
		return CDW_ERROR;
	}
}





/**
   \brief Prepare command line string for writing data with CDRECORD

   Go through all supported CDRECORD options and if given option is enabled,
   put its value into one string. Put path to CDRECORD at the beginning of
   the string. This creates a string that can be passed to run_command().

   \pre \p task->source is set to either OBJ_DATA_FILES or OBJ_ISO_IMAGE.
   These two values identify two valid sources of data that can be
   written to CD by cdrecord.

   \pre \p task->write_session_mode is set to proper value

   \p command pointer should not be malloced() by caller, however it should
   be freed() by caller

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return command string on success
   \return NULL on error
*/
char *cdw_cdrecord_create_command_for_writing(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->burn.session_mode != CDW_SESSION_MODE_INIT,
		    "you forgot to set session mode\n");
	cdw_assert (task->burn.disc_mode != CDW_DISC_MODE_INIT,
		    "you forgot to set disc mode\n");
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_BURN_FROM_IMAGE,
		    "ERROR: incorrect task->id %lld\n", task->id);
	cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD, "ERROR: incorrect tool id %lld\n", task->burn.tool.id);

	/* cdrecord needs to know size of files in two cases: when writing
	   in dao mode and when writing image that is piped to cdrecord (writing
	   files directly form hard drive); but let's append it in all cases */

	char tsize_value[TSIZE_LEN + 1];
	memset(tsize_value, '\0', TSIZE_LEN + 1);
	/* when REALLY we have to append this information? answer: let's do this always */
	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		strcpy(tsize_value, "${tsize}");
	} else { /* task->id == CDW_TASK_BURN_FROM_IMAGE */
		long long int fs_size = cdw_fs_get_file_size(task->burn.iso9660_file_fullpath);
		snprintf(tsize_value, TSIZE_LEN + 1, "%lld", fs_size / 2048);
		cdw_vdm ("INFO: tsize_value string = \"%s\", strlen(tsize_value) = %zd\n", tsize_value, strlen(tsize_value));
	}

	bool add_multi = cdw_cdrecord_add_multisession_info(task);

	/* cdrecord will read data to burn either from its stdin (streamed
	   from mkisofs) or from files from hard disc */
	bool read_from_stdin = task->id == CDW_TASK_BURN_FROM_FILES ? true : false;

	/* using config.scsi (X:Y:Z) is optional and advised only if
	   config.cdrw_device (/dev/xxx) does not work; if /dev/xxx works fine,
	   then config.scsi should be left empty */
	bool use_scsi = cdw_config_has_scsi_device();
	const char *device = use_scsi ? cdw_config_get_scsi_drive() : cdw_drive_get_drive_fullpath();

	char disc_mode[10];
	cdw_cdrecord_set_disc_mode_arg(task, disc_mode);
	cdw_vdm ("disc mode = \"%s\"\n", disc_mode);


	/* e.g. " padsize=63s -pad ", note spaces a beginning and end;
	   value of padsize is no longer than PADSIZE_FIELD_LEN_MAX
	   chars (not including ending \0); last +1 below is for ending \0 */
	size_t pad_len_max = 9 + CDW_CDRECORD_OPTIONS_PADSIZE_FIELD_LEN_MAX + 7;
	char pad_string[pad_len_max + 1];
	if (task->burn.cdrecord.pad_size > 0) {
		snprintf(pad_string, pad_len_max + 1,
			 " padsize=%ds %s ", task->burn.cdrecord.pad_size, task->burn.cdrecord.pad ? "-pad" : " ");
	} else {
		snprintf(pad_string, pad_len_max + 1, " %s ", task->burn.cdrecord.pad ? "-pad" : " ");
	}
	cdw_vdm ("pad string is \"%s\", %zd / %zd chars\n", pad_string, strlen(pad_string), pad_len_max);


	size_t speed_len_max = 7 + 3 + 1; /* " speed=" + "999" + " " */
	char speed_string[speed_len_max + 1];
	snprintf(speed_string, speed_len_max + 1, " speed=%lld ", task->burn.speed_id);
	cdw_vdm ("speed string is \"%s\", %zd / %zd chars\n", speed_string, strlen(speed_string), speed_len_max);

	char track_format[10];
	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		/* "-data": CD-ROM Mode 1 sector format, 2048 bytes per sector */
		/* "-xa": CD-ROM Mode 2 form 1 sector format, 2048 bytes per sector */
		strcpy(track_format,
		       task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE
		       ? " -data " : " -xa ");
	} else { /* dvd, deliberately not using any track format specifier */
		strcpy(track_format, " ");
	}

	cdw_assert (task->burn.tool.label != (char *) NULL, "ERROR: tool fullpath is NULL\n");
	char *command = cdw_string_concat(task->burn.tool.label,
					  " -v ",
					  " dev=", device,
					  speed_string, /* surrounded by spaces */

					  /* dao, tao, etc. */
					  strlen(disc_mode) ? disc_mode : " ",

					  task->burn.dummy ? " -dummy " : " ",
					  task->burn.cdrecord.burnproof ? " driveropts=burnfree " : " ",

					  /* cdrecord man page says: "Values below
					     2 seconds are not recommended to give
					     the kernel or volume management a chance
					     to learn the new state. */
					  " gracetime=4 ",

					  add_multi ? " -multi " : " ",

					  " ", task->burn.cdrecord.other_cdrecord_options, " ",

					  pad_string,

					  /* wait for data on stdin before opening SCSI device */
					  read_from_stdin ? " -waiti " : " ",

					  track_format,

					  /* tsize_value is just a string with number, so we have to add 's' */
					  " tsize=", tsize_value, "s ",

					  /* either get data from stdin, or from iso image;
					     path to iso file may contain spaces, so put
					     the path in quotes */
					  read_from_stdin ? " - " : "\"",
					  read_from_stdin ? "  " : task->burn.iso9660_file_fullpath,
					  read_from_stdin ? "  " : "\"",

					  /* last argument passed to concat() must be NULL */
					  (char *) NULL);

	if (command == (char *) NULL) {
	        cdw_vdm ("ERROR: failed to create command for writing with cdrecord\n");
		return (char *) NULL;
	} else {
		return command;
	}
}





bool cdw_cdrecord_add_multisession_info(cdw_task_t *task)
{
	if (task->burn.session_mode == CDW_SESSION_MODE_START_MULTI
	    || task->burn.session_mode == CDW_SESSION_MODE_CONTINUE_MULTI) {
		/* note that write_mode == CDW_SESSION_MODE_START_MULTI may
		   be also true for writing image */

		return true;

	} else if (task->burn.session_mode == CDW_SESSION_MODE_CREATE_SINGLE
		   || task->burn.session_mode == CDW_SESSION_MODE_WRITE_FINAL) {
		/* note that write_session_mode == CDW_SESSION_MODE_CREATE_SINGLE
		   may be also true for writing image */

		/* I want to make it explicit */
		return false;
	} else {
		/* I want to make it explicit too */
		return false;
	}
}





/**
  \brief Prepare string with mode of writing disc

  Prepare char string describing mode of writing to disc (tao, dao, etc.).
  Value of the string depends on value of task->write_disc_mode.
  Target buffer (\p disc_mode) must be prepared by caller and
  must be large enough to store disc mode string.

  \param task - variable describing current task
  \param disc_mode - target buffer
*/
void cdw_cdrecord_set_disc_mode_arg(cdw_task_t *task, char *disc_mode)
{
	if (task->burn.disc_mode == CDW_DISC_MODE_UNSPECIFIED) {
		/* cdrecord will use its default mode */
		disc_mode[0] = '\0';
	} else if (task->burn.disc_mode == CDW_DISC_MODE_TAO) {
		strcpy(disc_mode, " -tao ");
	} else if (task->burn.disc_mode == CDW_DISC_MODE_DAO) {
		strcpy(disc_mode, " -dao ");
	} else if (task->burn.disc_mode == CDW_DISC_MODE_SAO) {
		strcpy(disc_mode, " -sao ");
#if 0 /* currently unsupported modes */
	} else if (task->burn.disc_mode == CDW_DISC_MODE_RAW) {
		strcpy(disc_mode, " -raw ");
	} else if (task->burn.disc_mode == CDW_DISC_MODE_RAW96R) {
		strcpy(disc_mode, " -raw96r ");
	} else if (task->burn.disc_mode == CDW_DISC_MODE_RAW96P) {
		strcpy(disc_mode, " -raw96p ");
	} else if (task->burn.disc_mode == CDW_DISC_MODE_RAW16) {
		strcpy(disc_mode, " -raw16 ");
#endif
	} else {
		/* cdrecord will use its default mode */
		disc_mode[0] = '\0';
	}

	cdw_vdm ("INFO: disc mode = \"%s\"\n", disc_mode);

	return;

}





/**
   General error message displayed when something goes wrong in
   cdw_cdrecord_create_command_for_writing(), without going too much into details.

   At least we give _some_ explanation if selected action is not performed.

   Probably to be replaced with something more verbose and useful.

   \p id - small number identifying caller of this function
 */
void cdw_cdrecord_internal_error_dialog(cdw_id_t id)
{
	/* 2TRANS: this is title of dialog window */
	cdw_buttons_dialog(_("Error message"),
			   /* 2TRANS: this is message in dialog window */
			   _("An error occurred. Try to write disc again."),
			   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

	cdw_vdm ("INFO: caller id is %lld\n", id);

	return;
}





/* cdrecord pipe regexp has set values of some fields of "disc" data
   structure, and this function interprets these values; typically this
   function should be called by "disc" resolver function which doesn't need
   to be bothered with *how* cdrecord sees disc (i.e. last_sess_start and
   next_sess_start sectors), but is interested *what* cdrecord sees
   (i.e. "disc is empty" or "disc is not appendable anymore") */
cdw_rv_t cdw_cdrecord_set_disc_states(cdw_disc_t *disc)
{
	if (disc->cdrecord_info.last_sess_start == -1
	    && disc->cdrecord_info.next_sess_start == -1) {

		/* could not get information needed for
		   creating next session - disc is closed */
		disc->state_writable = CDW_FALSE;

		/* which at the same time means that disc
		   is not empty */
		disc->state_empty = CDW_FALSE;

	} else if (disc->cdrecord_info.last_sess_start == 0
		   && disc->cdrecord_info.next_sess_start == 0) {

		/* could not find information about any
		   existing session, so the disc is empty */
		disc->state_empty = CDW_TRUE;

		/* and since it is empty, it is appendable */
		disc->state_writable = CDW_TRUE;

	} else if (disc->cdrecord_info.last_sess_start >= 0
		   && disc->cdrecord_info.next_sess_start > 0) {

		/* cdrecord found information needed for
		   creating second or following session,
		   which means that disc is appendable */
		disc->state_writable = CDW_TRUE;

		/* it also means that disc is non-blank (at
		   least one session exists */
		disc->state_empty = CDW_FALSE;

	} else {
		/* some strange situation */
		disc->state_writable = CDW_UNKNOWN;
		disc->state_empty = CDW_UNKNOWN;

		cdw_vdm ("ERROR: strange values of msinfo data:\n");
		cdw_vdm ("ERROR: last_sess_start = %ld, next_sess_start = %ld:\n",
			 disc->cdrecord_info.last_sess_start, disc->cdrecord_info.next_sess_start);

		return CDW_ERROR;
	}


	/* exceptions:
	   - wodim 1.1.9 reports last_sess_start == next_sess_start == 0
	     for non-empty DVD-ROM, DVD-R, DVD+R, DVD+R DL, so the logic
	     above would set "appendable" = yes, "empty" = yes;
	     Cdrecord-ProDVD-ProBD-Clone 2.01.01a37 doesn't seem to have
	     problems with DVD-ROM nor DVD+/-R, but may have problems with
	     DVD+R DL  */
	if (cdw_ext_tools_is_cdrecord_wodim()) {
		if (disc->type == CDW_DVD_ROM
		    || disc->type == CDW_DVD_RP
		    || disc->type == CDW_DVD_R
		    || disc->type == CDW_DVD_R_RES
		    || disc->type == CDW_DVD_R_SEQ

		    || (disc->type == CDW_DVD_RP_DL && cdw_config_support_dvd_rp_dl())) {

			if (disc->cdrecord_info.has_toc) {
				/* we know now that it is non-empty, but
				   we still don't know if it is appendable */
				cdw_vdm ("INFO: cdrecord is wodim, disc has toc\n");
				disc->state_empty = CDW_FALSE;
				disc->state_writable = CDW_UNKNOWN;
			} else {
				cdw_vdm ("INFO: cdrecord is wodim, disc has no toc\n");
				disc->state_empty = CDW_TRUE;
				disc->state_writable = CDW_TRUE;
			}
		}
	}

	/* sometimes cdrecord / wodim can't recognize (using "cdrecord -toc" command) if
	   DVD+R DL is empty or not, so in case of DVD+R DL I can't rely on "has_toc" */
	if (disc->type == CDW_DVD_RP_DL && cdw_config_support_dvd_rp_dl()) {
		if (disc->state_writable == CDW_UNKNOWN
		    || disc->state_writable == CDW_TRUE) {

			disc->state_empty = CDW_UNKNOWN;
			disc->state_writable = CDW_UNKNOWN;
		} else {
			/* disc is recognized as non-appendable,
			   and my tests indicate that this is
			   probably the only state that is correctly
			   recognized by Cdrecord-ProDVD-ProBD-Clone 2.01.01a37 */
			disc->state_empty = CDW_FALSE;
			disc->state_writable = CDW_FALSE;
		}
	}

	return CDW_OK;
}






/**
   \brief Check if growisofs supports "dummy" option for given
   type of current disc

*/
bool cdw_cdrecord_allow_dummy(cdw_disc_t *disc)
{
	/* "dummy" option is not supported for some disc types;
	   to be on the safe side, I limit the subset of disc
	   types to only a few */
	if (disc->type == CDW_CD_R
	    || disc->type == CDW_CD_RW
	    || disc->type == CDW_DVD_R
	    || disc->type == CDW_DVD_R_SEQ
	    || disc->type == CDW_DVD_RW_SEQ
	    || disc->type == CDW_DVD_RW_RES) {

		return true;
	} else {
		return false;
	}
}





/* *** unused code below *** */

#if 0


/**
   \brief Create process writing audio CD; this code is currently not maintained

   \param task - variable describing current task

   \return CDW_OK
 */
cdw_rv_t run_command_cdrecord_write_audio(cdw_task_t *task)
{
	char command[500];
	cdw_rv_t crv = CDW_OK;

	crv = cdw_config_ui_conditional_volume_label_dialog();
	bool use_scsi = cdw_config_has_scsi_device();
	const char *device = use_scsi ? global_config.scsi : cdw_drive_get_drive_fullpath();

	if (crv == CDW_OK) { /* user entered label: empty or non-empty, but continue in both cases */
		cdw_assert (task->burn.tool.fullpath != (char *) NULL, "ERROR: tool fullpath is NULL\n");
		sprintf(command, "%s -v speed=%d dev=%s -audio", task->burn.tool.fullpath, task->burn.speed, device);
		if (global_config.pad) {
			/* when you go back to maintaining this code, don't
			   forget to add padsize */
			strcat(command, " -pad ");
		}
		if (task->burn.disc_mode == CDW_DISC_MODE_DAO) {
			strcat(command, " -dao ");
		}
		if (global_config.dummy) {
			strcat(command, " -dummy ");
		}
		if (global_config.burnproof) {
			strcat(command, " -driveropts=burnproof ");
		}
		sprintf(command, "%s %s/*.wav", command, global_config.audiodir);
		/* 2TRANS: this is dialog window title
		   (writing audio files to CD) */
		cdw_processwin_create(_("Write audio"),
				      /* 2TRANS: this is message in dialog
					 window: writing audio CD is in
					 progress */
				      _("Writing audio..."), true);

		run_command(command, task);

		/* 2TRANS: this is message in dialog window; result of
		   operation is unknown */
		cdw_processwin_destroy(_("Writing audio finished"), true);

		/* 2TRANS: this string will be used as title of window
		   displaying log of operation (of writing audio CD) */
		after_event(_("Write audio log"), 1);
	}

	return CDW_OK;
}





/**
   \brief Create process copying CD disk; this code is currently not maintained

   \param task - variable describing current task

   \return CDW_OK
 */
cdw_rv_t run_command_cdrecord_copy_disk(cdw_task_t *task)
{
	char command[500];
	int in_fd;
	bool use_scsi = cdw_config_has_scsi_device();
	const char *device = use_scsi ? global_config.scsi : cdw_drive_get_drive_fullpath();

	if ((in_fd = open("/dev/cdrom", O_RDONLY)) != -1) {
		close(in_fd);
		cdw_assert (task->burn.tool.fullpath != (char *) NULL, "ERROR: tool fullpath is NULL\n");
		sprintf(command, "%s -v speed=%d dev=%s %s", task->burn.tool.fullpath, task->burn.speed, device, lobal_config.other_cdrecord_options);
		if (global_config.eject) {
			strcat(command, " -eject");
		}
		if (global_config.dummy) {
			strcat(command, " -dummy");
		}

		sprintf(command, "%s -isosize %s", command, global_config.cdrw_device);
		/* 2TRANS: this is dialog window title */
		cdw_processwin_create(_("Copy data CD"), (char *) NULL, true);

		run_command(command, task);

		/* 2TRANS: this is message in dialog window:
		   result of operation (success/failure) unknown */
		cdw_processwin_destroy(_("Copying data CD finished"), true);

		/* 2TRANS: this string will be used as title of
		   window displaying log of operation (of copying CD) */
		after_event(_("Copy data CD log"), 1);
	}

	return CDW_OK;
}



#endif

