/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



/**
   \file cdw_list_display.c

   \brief File implements "list display" widget

   "list display" widget is a data structure consisting of following
   basic items:
   \li doubly linked list of items, each item being some data that has
       some printable representation
   \li ncurses window, on which this data (its printable representation)
       can be drawn, one item from list in one row
   \li pointer to a function (referred to in comments as "item display
       function"), used to draw single item from list (print its printable
       representation) in ncurses window

   This is assisted by some glue code: driver that handles keys, scroll()
   function that calls "item display function" repetitively for some range
   of list items and window rows. Constructor and destructor of a display
   widget are also provided.

   List of items is a field of display structure, but should be grown,
   shrinked, and filled with data by code using given instance of display.
   Display itself doesn't know anything about data format stored on list.
   All that a display can do with the list (knows how to do this and does it)
   is deallocating list data structure, but not the data stored on the list.

   Implementation of "item display function" must be provided by code
   creating and using given instance of display. The function must know
   what is the printable representation of data stored in list's item, and
   must know how to print the data properly.
   Correct declaration of such function is
   void name(void *display, void *data, size_t row, size_t h_offset, bool isreverse);
   The function should cast "void *display" argument to "CDW_LIST_DISPLAY *"
   type to access display's subwindow (display->subwindow) in which to
   print.

   The widget seem to handle displays with empty lists quite well.
   TODO: test it thoroughly.

   New display by default doesn't react to any keys other than movement keys
   and Escape key. This can be changed later by using
   cdw_list_display_add_return_char()

   TODO: decouple list and display - list should be separate and independent
   data structure (perhaps coupled with two pointers to functions that can:
   a: deallocate content of list; b: display one item of list)
*/


#include <stdlib.h>

#include "cdw_debug.h"
#include "cdw_list_display.h"
#include "cdw_ncurses.h"


static void cdw_list_display_scroll_to_current(CDW_LIST_DISPLAY *display, size_t h_offset, bool highlight);



/**
   \brief Create new display widget with empty list

   Function creates new display widget. List of items associated with given
   display is initialized as empty, and can be later accessed as display->list.
   Data on the list belongs to code creating and using the display - the data
   won't be destroyed by the display. The list itself should be created,
   expanded and shrinked by code creating and using the display. List data
   structure (but not payload of the list) will be destroyed when display is
   destroyed. You should deallocate all content of list before deleting a
   display.

   New display is embedded into a given existing \p parent window: the
   function creates a new subwindow (a child window of \p parent).
   The subwindow is then used as main drawing area of the display. Main
   window (\p parent window) can be accessed by parent->window.

   It is up to creator of a given display to draw or redraw its border
   if needed (by accessing display->window).

   Pointer to function drawing a single line in display ("item display
   function") is initialized as NULL. It can be set by owner of display
   by accessing display->display_item.

   \param parent is an existing ncurses window, into which new display should
   be embedded. \p parent always belongs to caller, but may be refreshed by
   display functions.

   \p n_lines and \p n_cols are sizes of new subwindow created as a child of \p parent

   \p begin_y and \p begin_x are offset of new subwindow created as a child
   of \p parent, the offset relative to beginning of \p parent.

   \param colors - color scheme of new display

   \return pointer to new display on success
   \return NULL on errors
*/
CDW_LIST_DISPLAY *cdw_list_display_new(WINDOW *parent, int n_lines, int n_cols, int begin_y, int begin_x, cdw_colors_t colors)
{
	cdw_assert (parent != (WINDOW *) NULL, "ERROR: called the function with null parent window\n");

	CDW_LIST_DISPLAY *display = (CDW_LIST_DISPLAY *) malloc(sizeof(CDW_LIST_DISPLAY));
	if (display == (CDW_LIST_DISPLAY *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for display\n");
		return (CDW_LIST_DISPLAY *) NULL;
	}

	/* begin_y and begin_x are beginning of embedded window relative to
	   beginning of parent window, and n_lines and n_cols are
	   sizes of subwindow */
	display->n_lines = n_lines;
	display->n_cols = n_cols;
	display->begin_y = begin_y;
	display->begin_x = begin_x;

	display->colors = colors;

	display->n_return_chars = 0;
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		display->return_chars[i] = 0;
	}

	display->window = parent;
	display->subwindow = derwin(display->window,
				    (int) display->n_lines, (int) display->n_cols,
				    (int) display->begin_y, (int) display->begin_x);
	if (display->subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create subwindow for embedded display\n");

		free(display);
		display = (CDW_LIST_DISPLAY *) NULL;

		return (CDW_LIST_DISPLAY *) NULL;
	}

	display->display_format = CDW_LIST_DISPLAY_FORMAT_SHORT;
	display->display_item = NULL;
	display->n_items = 0;
	display->current_item_ind = 0;
	display->list = (cdw_dll_item_t *) NULL;

	keypad(display->window, TRUE);

	wbkgd(display->window, A_NORMAL | COLOR_PAIR(display->colors));
	(void) wattrset(display->window, A_NORMAL | COLOR_PAIR(display->colors));
	wbkgd(display->subwindow, A_NORMAL | COLOR_PAIR(display->colors));
	(void) wattrset(display->subwindow, A_NORMAL | COLOR_PAIR(display->colors));

	return display;
}





/**
   \brief Deallocated resources of given display

   Function deallocates some resources related to given display: its
   subwindow, its main window (if created by display code), associated
   list data structure, and the display itself. Function does not
   deallocate data stored on the list, it is a task of code using the
   display to free the data before destroying given display.

   \param display - display to destroy
*/
void cdw_list_display_delete(CDW_LIST_DISPLAY **display)
{
	if (*display == (CDW_LIST_DISPLAY *) NULL) {
		cdw_vdm ("ERROR: \"display\" argument is NULL\n");
		return;
	}

	if ((*display)->subwindow == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: \"display->subwindow\" is NULL\n");
	} else {
		delwin((*display)->subwindow);
		(*display)->subwindow = (WINDOW *) NULL;
	}

	if ((*display)->window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: \"display->window\" is NULL\n");
	} /* no "else", we can't free parent window */

	(*display)->n_lines = 0;
	(*display)->n_cols = 0;
	(*display)->begin_y = 0;
	(*display)->begin_x = 0;

	if ((*display)->list != (cdw_dll_item_t *) NULL) {
		if ((*display)->list->data != (void *) NULL) {
			cdw_vdm ("ERROR: caller didn't deallocate properly data stored on the list\n");
			cdw_vdm ("ERROR: data stored on display's list belongs to owner\n");
		}
		cdw_dll_clean((*display)->list);
		(*display)->list = (cdw_dll_item_t *) NULL;
	}
	(*display)->n_items = 0;

	free(*display);
	*display = (CDW_LIST_DISPLAY *) NULL;

	return;
}





/**
   \brief Function handling keys pressed when a display has focus

   Function reacts to keys pressed in given display's window. By default
   a display reacts only to movement keys and Escape key, moving highlight
   (selection) on the list up and down. Content of list associated with
   the display and displayed in display's subwindow is scrolled if necessary
   (i.e. if there are more items on the list than rows in the window) to
   follow cursor's movement. If function ("item display function") provided
   by code using a display supports horizontal offset of displayed item,
   then the driver reacts to Left/Right arrows as well.

   Supporting of other keys (characters) can be enabled by adding these
   characters with cdw_list_display_add_return_char() function. The only
   "reaction" to enabled key is to return with value corresponding to key
   pressed. It is up to caller of the driver function to act upon such
   event, including calling the driver function again to resume reacting
   to keys in driver. "item_i" field of the display struct can be used by
   caller to check for which item a key was pressed.

   "item_i" field of display struct is constantly updated by driver to always
   indicate (point at) item which currently is highlighted (selected) in the
   display.

   Function returns with CDW_ESCAPE when it catches Escape key pressed by
   user. Function can also return with value corresponding to other keys
   (e.g. Space, Enter, Delete) if handling such keys by display is enabled.

   \param display - display that currently has keyboard focus

   \return CDW_ESCAPE if user pressed escape key
   \return other specific key, if display was configured to react to other keys
*/
int cdw_list_display_driver(CDW_LIST_DISPLAY *display)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"display\" argument in NULL\n");
	cdw_assert (display->display_item != NULL, "ERROR: the display doesn't have ->display_item() set\n");

	size_t h_offset = 0; /* indicates if and how much content of the window was moved to left */

	if (display->n_items != 0) {
		/* initialization: highlight initial item */
		size_t subwindow_row = display->current_item_ind % (size_t) display->n_lines;
		cdw_dll_item_t *item = cdw_dll_ith_item(display->list, display->current_item_ind);
		display->display_item(display, item->data, subwindow_row, h_offset, true);
	}
	wrefresh(display->subwindow);

	int key = 'a'; /* safe (?) initial value */
	while (key != CDW_KEY_ESCAPE) {
		key = wgetch(display->window);
		switch (key) {
		        case KEY_HOME:
				/* first Home key should move content of
				   window to beginning of lines, leaving
				   vertical position unchanged */
				if (h_offset > 0) {
					h_offset = 0;
					cdw_list_display_scroll_to_current(display, h_offset, true);
				} else {
					if (display->current_item_ind > 0) {
						display->current_item_ind = 0;
						h_offset = 0;
						cdw_list_display_scroll_to_current(display, h_offset, true);
					}
				}
				break;

			case KEY_END:
				/* go to last item */
				if (display->current_item_ind < display->n_items - 1) {
					display->current_item_ind = display->n_items - 1;
					cdw_list_display_scroll_to_current(display, h_offset, true);
				}
				break;

			case KEY_DOWN:
				/* go to next item */
				if (display->current_item_ind < display->n_items - 1) {
					display->current_item_ind++;
					cdw_list_display_scroll_to_current(display, h_offset, true);
				}
				break;

			case KEY_UP:
				/* go to previous item */
				if (display->current_item_ind > 0) {
					display->current_item_ind--;
					cdw_list_display_scroll_to_current(display, h_offset, true);
				}

				break;

			case KEY_RIGHT:
				/* some modules using list display may use
				   "item display function" that does not
				   support non-zero h_offset, it will be
				   ignored by such "item display function" */
				h_offset++;
				cdw_list_display_scroll_to_current(display, h_offset, true);
				break;
			case KEY_LEFT:
				/* see comment for KEY_RIGHT */
				if (h_offset > 0) {
					h_offset--;
					cdw_list_display_scroll_to_current(display, h_offset, true);
				}
				break;

			case KEY_PPAGE:
				/* Previous Page, Page Up */
				if (display->current_item_ind > 0) {
					if (display->current_item_ind < (size_t) display->n_lines) {
						/* we are on top of listing,
						   so there is only one place
						   to move cursor and item index */
						display->current_item_ind = 0;
					} else {
						/* move item index exactly one
						   screen size up */
						display->current_item_ind -= (size_t) display->n_lines;
					}
					cdw_list_display_scroll_to_current(display, h_offset, true);
				}
				break;

			case KEY_NPAGE:
				/* Next Page, Page Down */
				if (display->current_item_ind < display->n_items - 1) {
					display->current_item_ind += (size_t) display->n_lines;
					if (display->current_item_ind > display->n_items - 1) {
						display->current_item_ind = display->n_items - 1;
					}
					cdw_list_display_scroll_to_current(display, h_offset, true);
				}
				break;

			default:
				break;

		} /* switch (c) */

		if (cdw_list_display_is_return_char(display, key)) {
			break;
		}
	} /* while (c != CDW_ESCAPE) */

	if (display->n_items != 0) {
		/* control (and focus) is leaving display area, and to indicate this,
		   we un-highlight current item */
		cdw_dll_item_t *item = cdw_dll_ith_item(display->list, display->current_item_ind);
		size_t subwindow_row = display->current_item_ind % (size_t) display->n_lines;
		display->display_item(display, item->data, subwindow_row, h_offset, false);
	}

	wrefresh(display->subwindow);

	return key;
}





/**
   \brief Refresh windows associated with given display

   Function calls wrefresh() function for its window and subwindow

   \param display - display that you want to refresh
*/
void cdw_list_display_refresh(CDW_LIST_DISPLAY *display)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"display\" argument is NULL\n");
	cdw_assert (display->window != (WINDOW *) NULL, "ERROR: \"display->window\" is NULL\n");
	cdw_assert (display->subwindow != (WINDOW *) NULL, "ERROR: \"display->subwindow\" is NULL\n");

	wrefresh(display->subwindow);
	redrawwin(display->window);
	wrefresh(display->window);

	return;
}





/**
   \brief Display part of list associated with given display

   Function displays given \p item_i item (item from display's list, indicated
   by index \p item_i), and all surrounding items that fit in a single
   display's subwindow. If \p highlight is true, then the item is additionally
   highlighted.

   \p h_offset indicates how many columns to left should the lines in the display
   be moved. This is useful when displayed item is represented by line with more
   chars than there is columns in display's subwindow - the line should be moved
   (shifted) to left to display content truncated by display's border. This value
   affects all lines displayed in the display, however: read next paragraph.

   This function doesn't draw representation of item itself. It uses pointer to
   a function: display->display_item ("item display function"), which must be
   set and provided by code using the display (set by display->display_item).
   "item display function" is allowed not to implement support for h_offset,
   and to ignore its value.

   \param display - display, in which given item exists, and which you want to
                    use to display the item
   \param h_offset - value of horizontal offset, with which lines should be
                     displayed in the display
   \param highlight - should an item indicated by \p item_i be highlighted?
*/
void cdw_list_display_scroll_to_current(CDW_LIST_DISPLAY *display, size_t h_offset, bool highlight)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"display\" argument is NULL\n");
	cdw_assert (display->subwindow != (WINDOW *) NULL, "ERROR: \"display->subwindow\" is NULL\n");
	cdw_assert (display->display_item != NULL, "ERROR: ->display_item() function pointer is NULL\n");

	if (display->n_items == 0) {
		/* perfectly normal situation, but it means that we have
		   completely nothing to do */
		return;
	}

	/* we use werase() here, because this function call may display
	   only tail of list, so old items at the bottom of subwindow would
	   be still visible */
	werase(display->subwindow);

	/* get first item to be displayed in 0-th row of subwindow; get it
	   with expensive cdw_dll_ith_item() call, we will get rest of items
	   by traversing list "manually" in loop */
	size_t item_ind = display->current_item_ind - (display->current_item_ind % (size_t) display->n_lines);
	cdw_dll_item_t *item = cdw_dll_ith_item(display->list, item_ind);

	/* increment subwindow row number and traverse list at the same time */
	for (int subwindow_row = 0;
	     subwindow_row < display->n_lines && item != (cdw_dll_item_t *) NULL;
	     subwindow_row++, item = item->next, item_ind++) {

		display->display_item(display, item->data, (size_t) subwindow_row, h_offset, false);

		/* since we already traverse list of items, we can update
		   value of "current_item" field: since this function was called,
		   it is *very* probable that display->current_item_ind has been
		   updated, so updating display->current_item is also necessary */
		if (item_ind == display->current_item_ind) {
			display->current_item = item;
		}
	}

	if (highlight) {
		/* highlight current item */
		item = cdw_dll_ith_item(display->list, display->current_item_ind);
		size_t subwindow_row = display->current_item_ind % (size_t) display->n_lines;
		display->display_item(display, item->data, subwindow_row, h_offset, true);
	}

	wrefresh(display->subwindow);

	return;
}





/**
   \brief Add char that, if captured by display's driver, will cause return of control by driver

   Function registers a char \p c, which, if captured by driver, will cause
   the driver to return control to it's (driver's) caller.

   You may want to use this function e.g. when you want to perform some
   action whenever ENTER key is hit. In that case you have to configure
   a display before using it by calling this function with CDW_ENTER as
   second argument. Similarly for other keys: Space, Delete, any digit or
   letter. Digits should be passed as characters.

   \param display - display that you want to configure
   \param c - character that you want to add to given \p display
*/
void cdw_list_display_add_return_char(CDW_LIST_DISPLAY *display, int c)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"display\" argument is NULL\n");
	cdw_assert (display->n_return_chars < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return chars in the display, can't add another one\n",
		    display->n_return_chars, N_RETURN_KEYS_MAX);
	cdw_assert (c != 0, "ERROR: trying to add char == 0, but 0 is an initializer value\n");

	if (cdw_list_display_is_return_char(display, c)) {
		cdw_vdm ("WARNING: attempting to add char %d / \"%s\", but it is already on the list of return chars\n",
			 c, cdw_ncurses_key_label(c));
	} else {
		display->return_chars[display->n_return_chars++] = c;
	}

	return;
}




/**
   \brief Add key(s) that, if captured by display's driver, will cause return of control by driver

   Function registers keys, which, if captured by driver, will cause
   the driver to return control to it's (driver's) caller.

   You may want to use this function e.g. when you want to perform some
   action whenever ENTER key is hit. In that case you have to configure
   a display before using it by calling this function with CDW_KEY_ENTER as
   second argument. Similarly for other keys: Space, Delete, any digit or
   letter. Digits should be passed as characters.

   \param display - display that you want to configure
   \param ... - list of keys to add; use '0' as a last key (guard)
*/
void cdw_list_display_add_return_keys(CDW_LIST_DISPLAY *display, ...)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"display\" argument is NULL\n");
	cdw_assert (display->n_return_chars < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    display->n_return_chars, N_RETURN_KEYS_MAX);

	va_list ap;
	va_start(ap, display);
	int key = 'a';
	while ((key = va_arg(ap, int)) != 0) {
		cdw_list_display_add_return_char(display, key);
	}
	va_end(ap);

	return;
}






/**
  \brief Check if given char is one of configured "return" chars

  Function checks if given \p c char was earlier added to given \p display
  as "return on the char" character.

  \param display - display that you want to query
  \param c - character you want to check if it is added to display

  \return true if given character was added to given \p display
  \return false if given character was not added to given \p display
*/
bool cdw_list_display_is_return_char(CDW_LIST_DISPLAY *display, int c)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: \"display\" argument is NULL\n");
	cdw_assert (c != 0, "ERROR: asking for 0, which is an initialization value\n");

	int i = 0;
	for (i = 0; i < display->n_return_chars; i++) {
		if (display->return_chars[i] == c) {
			return true;
		}
	}

	return false;
}





/**
   \brief Remove i-th item from given display

   Function looks up \p item_i-th item in given display, removes it from
   associated window and associated list. It also updates value of 'number
   of items' variable associated with the display.

   The function does not deallocate payload associated with list item:
   the payload is application-specific, and this function doesn't know (yet)
   how to deallocate all the various data types that can be put as payloads.
   So: you have to get the payload and free it yourself before calling the
   function.

   \param display - display from which to remove an item
   \param item_i - index of item to be removed

   \return CDW_OK if removing was successfull
   \return CDW_GEN_ERROR if it failed
*/
cdw_rv_t cdw_list_display_remove_item(CDW_LIST_DISPLAY *display, size_t item_i)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: display is null\n");
	cdw_assert (display->n_items > 0, "ERROR: you called the function for display with empty list\n");
	cdw_assert (display->list != (cdw_dll_item_t *) NULL, "ERROR: list associated with the display is null\n");

	display->current_item_ind = item_i;

#ifndef NDEBUG
	cdw_dll_item_t *item = cdw_dll_ith_item(display->list, display->current_item_ind);
	cdw_assert (item->data == NULL, "ERROR: item payload is not NULL, trying to leak memory?\n");
#endif

#ifndef NDEBUG
	size_t len1 = cdw_dll_length(display->list);
#endif

	cdw_rv_t crv = cdw_dll_remove_ith_item(&(display->list), display->current_item_ind);

#ifndef NDEBUG
	size_t len2 = cdw_dll_length(display->list);
#endif

	if (crv == CDW_OK) {
		display->n_items--; /* to avoid calling cdw_dll_length() */

		cdw_assert (len2 == len1 - 1, "ERROR: lengths of list do not differ by one: len before: %zd, len after: %zd\n", len1, len2);
		cdw_assert (len2 == display->n_items, "ERROR: length mismatch\n");

		if (display->n_items > 0) {
			if (display->current_item_ind == 0) {
				; /* removed item was first, no need to
				     reposition cursor in new list; new
				     current item will also be first */
			} else if (display->current_item_ind == display->n_items) {
				/* removed item was last one;
				   here n_files has new, decreased value */
				display->current_item_ind = display->n_items - 1;
				/* we could have achieved the same effect
				   by simple display->current_item_ind--; */
			} else {
				; /* removed item was somewhere in between
				     beginning and end, no need to change item_i */
			}

			cdw_list_display_scroll_to_current(display, 0, true);
		} else {
			werase(display->subwindow);
			wrefresh(display->subwindow);
		}
		return CDW_OK;
	} else {
		return CDW_ERROR;
	}
}




cdw_rv_t cdw_list_display_add_item(CDW_LIST_DISPLAY *display, void *data, bool (*pred)(const void *, const void *))
{

#ifndef NDEBUG
	size_t len1 = cdw_dll_length(display->list);
#endif
	cdw_rv_t crv = cdw_dll_append(&(display->list), data, pred);
#ifndef NDEBUG
	size_t len2 = cdw_dll_length(display->list);
#endif

	if (crv == CDW_OK) {
		display->n_items++; /* this is to avoid using cdw_dll_length() */

		if (display->n_items == 1) {
			/* this is kind of initialization of 'current_item_ind' and 'current_item' */
			display->current_item_ind = 0;
			display->current_item = display->list;
		}

		cdw_assert (len2 == len1 + 1, "ERROR: length didn't increase\n");
		cdw_assert (len2 == display->n_items, "ERROR: list len mismatch");


	} else if (crv == CDW_NO) {
		/* given data was already on a list, no need to do anything */
		;
	} else {
		cdw_vdm ("ERROR: failed to add data\n");
		/*  this  warning was all we could do */
	}
	return crv;
}




void cdw_list_display_scroll_to(CDW_LIST_DISPLAY *display, size_t current_item_ind, size_t h_offset, bool highlight)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: passed NULL display to function\n");
	if (display->n_items > 0) {
		cdw_assert (current_item_ind < display->n_items, "ERROR: item index %zd is larger than number of items %zd\n",
			    current_item_ind, display->n_items);
		display->current_item_ind = current_item_ind;
		cdw_list_display_scroll_to_current(display, h_offset, highlight);
	} else {
		cdw_vdm ("%s: called the function for empty list, requested_item_ind = %zd\n",
			 current_item_ind == 0 ? "WARNING" : "ERROR", current_item_ind);
	}
	return;
}




void *cdw_list_display_get_current_item_data(CDW_LIST_DISPLAY *display)
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: passed NULL display to function\n");
	cdw_assert (display->n_items > display->current_item_ind,
		    "ERROR, current item index (0-based) is %zd but number of items is %zd\n",
		    display->current_item_ind, display->n_items);

	if (display->n_items > 0) {
		return display->current_item->data;
	} else {
		return (void *) NULL;
	}
}





/**
   \brief Wrapper for cdw_dll_is_member()

   This is a wrapper for cdw_dll_is_member() so that code using list display
   doesn't have to reach to lower-level code.

   \p pred is predicate function that will check if given \p data is member
   of list associated with \p display or not.

   \param display - display which you want to query for given data
   \param data - data for which you want to check in given display
   \param pred - predicate function

   \return true if \p data is on a list associated with \p display
   \return false if \p data isn't on a list associated with \p display
*/
bool cdw_list_display_is_member(CDW_LIST_DISPLAY *display, void *data, bool (*pred)(const void *, const void *))
{
	cdw_assert (display != (CDW_LIST_DISPLAY *) NULL, "ERROR: display is NULL\n");
	cdw_assert (pred != (bool (*)(const void *, const void *)) NULL, "ERROR: predicate function is NULL\n");
	return cdw_dll_is_member(display->list, data, pred);
}
