
/* Authors: Frank Mayer <mayerf@tresys.com> and Karl MacMillan <kmacmillan@tresys.com>
 *
 * Copyright (C) 2003 Tresys Technology, LLC
 *	This program is free software; you can redistribute it and/or modify
 *  	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, version 2.
 */
 
/* 
 * displaypol.c
 *
 * Test program to the contents of a binary policy in text
 * form.  This program currently only displays the
 * avtab (including conditional avtab) rules.
 *
 * 	displaypol binary_pol_file
 */

#include "../policydb.h"
#include "../avtab.h"
#include "../services.h"
#include "../conditional.h"
#include <getopt.h>
#include <assert.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>


extern policydb_t policydb;
extern unsigned int ss_initialized;

/* hack! */
int avc_ss_reset(__u32 seqno)
{
	return 0;
}

void usage(char *progname)
{
	printf("usage:  %s binary_pol_file\n\n", progname);
	exit(1);
}

/* borrowed from avtab.c */
#define HASH_BITS 15
#define HASH_BUCKETS (1 << HASH_BITS)
#define HASH_MASK (HASH_BUCKETS-1)

/* borrowed from checkpolicy.c */
static int find_perm(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	unsigned int *valuep;
	perm_datum_t *perdatum;

	valuep = (unsigned int *) p;
	perdatum = (perm_datum_t *) datum;

	if (*valuep == perdatum->value)
		return (int) key;

	return 0;
}

int render_access_mask(__u32 mask, avtab_key_t *key, policydb_t *p, FILE *fp)
{
	unsigned int i;
	class_datum_t *cladatum;
	char *perm;
	cladatum = p->class_val_to_struct[key->target_class -1];
	fprintf(fp, "{");
	for (i = 1; i <= sizeof(mask) * 8; i++) {
		if (mask & (1 << (i - 1))) {
			perm = (char *) hashtab_map(cladatum->permissions.table,
				  find_perm, &i);

			if (!perm && cladatum->comdatum) {
				perm = (char *) hashtab_map(cladatum->comdatum->permissions.table,
				  find_perm, &i);
			}
			if (perm)
				fprintf(fp, " %s", perm);
		}
	}
	fprintf(fp, " }");
	return 0;
}

int render_type(__u32 type, policydb_t *p, FILE *fp)
{
	fprintf(fp, "%s", p->p_type_val_to_name[type - 1]);
	return 0;
}

int render_key(avtab_key_t *key, policydb_t *p, FILE *fp)
{
	fprintf(fp, "%s %s : %s ", p->p_type_val_to_name[key->source_type - 1], 
			p->p_type_val_to_name[key->target_type - 1],
			p->p_class_val_to_name[key->target_class - 1]);
	return 0;
}

/* 'what' values for this function */
#define	RENDER_UNCONDITIONAL	0x0001	/* render all regardless of enabled state */
#define RENDER_ENABLED		0x0002
#define RENDER_DISABLED		0x0004
#define RENDER_CONDITIONAL	(RENDER_ENABLED|RENDER_DISABLED)

int render_av_rule(avtab_key_t *key, avtab_datum_t *datum, __u32 what, policydb_t *p, FILE *fp)
{
	if(!(what & RENDER_UNCONDITIONAL)) {
		if(what != RENDER_CONDITIONAL &&
		   (((what & RENDER_ENABLED) && !(datum->specified & AVTAB_ENABLED)) ||
		   ((what & RENDER_DISABLED) && (datum->specified & AVTAB_ENABLED)))) {
		   	return 0; /* doesn't match selection criteria */
		} 
	}

	if(!(what & RENDER_UNCONDITIONAL)) {
		if(datum->specified & AVTAB_ENABLED) 
			fprintf(fp, "[enabled] ");
		else if(!(datum->specified & AVTAB_ENABLED))
			fprintf(fp, "[disabled] ");
	} 

	if( datum->specified & AVTAB_AV) {
		if(datum->specified & AVTAB_ALLOWED) {
			fprintf(fp, "allow ");			
			render_key(key, p, fp);
			render_access_mask(avtab_allowed(datum),key, p, fp);
			fprintf(fp, ";\n");
		}
		if(datum->specified & AVTAB_AUDITALLOW) {
			fprintf(fp, "auditallow ");
			render_key(key, p, fp);
			render_access_mask(avtab_auditallow(datum),key, p, fp);
			fprintf(fp, ";\n");
		}
		if(datum->specified & AVTAB_AUDITDENY) {
			fprintf(fp, "dontaudit ");
			render_key(key, p, fp);
			/* We inverse the mask for dontaudit since the mask is internally stored
			 * as a auditdeny mask */
			render_access_mask(~avtab_auditdeny(datum),key, p, fp);
			fprintf(fp, ";\n");
		}
	}
	else if( datum->specified & AVTAB_TYPE){
		if(datum->specified & AVTAB_TRANSITION) {
			fprintf(fp, "type_transition ");
			render_key(key, p, fp);
			render_type(avtab_transition(datum), p, fp);
			fprintf(fp, ";\n");
		}
		if(datum->specified & AVTAB_MEMBER) {
			fprintf(fp, "type_member ");
			render_key(key, p, fp);
			render_type(avtab_member(datum), p, fp);
			fprintf(fp, ";\n");
		}
		if(datum->specified & AVTAB_CHANGE) {
			fprintf(fp, "type_change ");
			render_key(key, p, fp);
			render_type(avtab_change(datum), p, fp);
			fprintf(fp, ";\n");
		}
	}
	else {
		fprintf(fp, "     ERROR: no valid rule type specified\n");
		return -1;
	}
	return 0;
}

int display_avtab(avtab_t *a, __u32 what, policydb_t *p, FILE *fp)
{
	int i;
	avtab_ptr_t cur;

	for (i = 0; i < AVTAB_SIZE; i++) {
		for (cur = a->htable[i]; cur; cur = cur->next) {
			render_av_rule(&cur->key, &cur->datum, what, p, fp);
		}
	}
	fprintf(fp, "\n");
	return 0;
}

int display_bools(policydb_t *p, FILE *fp)
{
	int i;

	for (i = 0; i < p->p_bools.nprim; i++) {
		fprintf(fp, "%s : %d\n", p->p_bool_val_to_name[i],
			p->bool_val_to_struct[i]->state);
	}
	return 0;
}

void display_expr(policydb_t *p, cond_expr_t *exp, FILE *fp)
{

	cond_expr_t *cur;
	for (cur = exp; cur != NULL; cur = cur->next) {
		switch (cur->expr_type) {
		case COND_BOOL:
			fprintf(fp, "%s ", p->p_bool_val_to_name[cur->bool - 1]);
			break;
		case COND_NOT:
			fprintf(fp, "! ");
			break;
		case COND_OR:
			fprintf(fp, "|| ");
			break;
		case COND_AND:
			fprintf(fp, "&& ");
			break;
		case COND_XOR:
			fprintf(fp, "^ ");
			break;
		case COND_EQ:
			fprintf(fp, "== ");
			break;
		case COND_NEQ:
			fprintf(fp, "!= ");
			break;
		default:
			fprintf(fp, "error!");
			break;
		}
	}
}

int display_cond_expressions(policydb_t *p, FILE *fp)
{
	cond_node_t *cur;
	cond_av_list_t *av_cur;
	for (cur = p->cond_list; cur != NULL; cur = cur->next) {
		fprintf(fp, "expression: ");
		display_expr(p, cur->expr, fp);
		fprintf(fp, "current state: %d\n", cur->cur_state);
		fprintf(fp, "True list:\n");
		for (av_cur = cur->true_list; av_cur != NULL; av_cur = av_cur->next) {
			fprintf(fp, "\t");
			render_av_rule(&av_cur->node->key, &av_cur->node->datum,
				       RENDER_CONDITIONAL, p, fp);
		}
		fprintf(fp, "False list:\n");
		for (av_cur = cur->false_list; av_cur != NULL; av_cur = av_cur->next) {
			fprintf(fp, "\t");
			render_av_rule(&av_cur->node->key, &av_cur->node->datum,
				       RENDER_CONDITIONAL, p, fp);
		}
	}
	return 0;
}

int change_bool(char *name, int state, policydb_t *p, FILE *fp)
{
	cond_bool_datum_t *bool;

	bool = hashtab_search(p->p_bools.table, name);
	if (bool == NULL) {
		fprintf(fp, "Could not find bool %s\n", name);
		return -1;
	}
	bool->state = state;
	evaluate_conds(p);
	return 0;
}

int menu() {
	printf("\nSelect a command:\n");
	printf("1)  display unconditional AVTAB\n");
	printf("2)  display conditional AVTAB (entirely)\n");
	printf("3)  display conditional AVTAG (only ENABLED rules)\n");
	printf("4)  display conditional AVTAB (only DISABLED rules)\n");
	printf("5)  display conditional bools\n");
	printf("6)  display conditional expressions\n");
	printf("7)  change a boolean value\n");
	printf("\n");
	printf("f)  set output file\n");
	printf("m)  display menu\n");
	printf("q)  quit\n");
	return 0;
}


int main(int argc, char **argv)
{
	FILE *out_fp = stdout;
	char ans[81], OutfileName[121];
	int fd, ret;
	struct stat sb;
	void *map;
	char *name;
	int state;


	if(argc != 2) 
		usage(argv[0]);
		
	fd = open(argv[1], O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, "Can't open '%s':  %s\n",
			argv[1], strerror(errno));
		exit(1);
	}
	if (fstat(fd, &sb) < 0) {
		fprintf(stderr, "Can't stat '%s':  %s\n",
			argv[1], strerror(errno));
		exit(1);
	}
	map = mmap(NULL, sb.st_size, PROT_READ, MAP_SHARED, fd, 0);
	if (map == MAP_FAILED) {
		fprintf(stderr, "Can't map '%s':  %s\n",
			argv[1], strerror(errno));
		exit(1);
	}
	
	/* read the binary policy */
	fprintf(out_fp, "Reading policy...\n");
	ret = security_load_policy(map,sb.st_size);
	if (ret) {
		fprintf(stderr, "%s:  error(s) encountered while parsing configuration\n", argv[0]);
		exit(1);
	}

	fprintf(stdout, "binary policy file loaded\n\n");
	close(fd);
	
	menu();
	for(;;) {
		printf("\nCommand (\'m\' for menu):  ");
		fgets(ans, sizeof(ans), stdin);	
		switch(ans[0]) {
			
		case '1':
			display_avtab(&policydb.te_avtab, RENDER_UNCONDITIONAL, &policydb, out_fp);
			break;
		case '2':
			display_avtab(&policydb.te_cond_avtab, RENDER_CONDITIONAL, &policydb, out_fp);
			break;
		case '3':
			display_avtab(&policydb.te_cond_avtab, RENDER_ENABLED, &policydb, out_fp);
			break;
		case '4':
			display_avtab(&policydb.te_cond_avtab, RENDER_DISABLED, &policydb, out_fp);
			break;
		case '5':
			display_bools(&policydb, out_fp);
			break;
		case '6':
			display_cond_expressions(&policydb, out_fp);
			break;
		case '7':
			printf("name? ");
			fgets(ans, sizeof(ans), stdin);
			ans[strlen(ans) - 1] = 0;
			
			name = malloc((strlen(ans) + 1) * sizeof(char));
			if (name == NULL) {
				fprintf(stderr, "couldn't malloc string.\n");
				break;
			}
			strcpy(name, ans);


			printf("state? ");
			fgets(ans, sizeof(ans), stdin);
			ans[strlen(ans) - 1] = 0;

			if (atoi(ans))
				state = 1;
			else
				state = 0;

			change_bool(name, state, &policydb, out_fp);
			free(name);
			break;
		case 'f':
			printf("\nFilename for output (<CR> for screen output): ");
			fgets(OutfileName, sizeof(OutfileName), stdin);	
			OutfileName[strlen(OutfileName)-1] = '\0'; /* fix_string (remove LF) */
			if (strlen(OutfileName) == 0) 
				out_fp = stdout;
			else if ((out_fp = fopen(OutfileName, "w")) == NULL) {
				fprintf (stderr, "Cannot open output file %s\n", OutfileName);
				out_fp = stdout;
			}
			if (out_fp != stdout) 
				printf("\nOutput to file: %s\n", OutfileName);
			break;
		case 'q':
			policydb_destroy(&policydb);
			exit(0);
			break;
		case 'm':
			menu();
			break;
		default:
			printf("\nInvalid choice\n");
			menu();
			break;
		
		}
	}
}

/* FLASK */

