
/* Author : Stephen Smalley, <sds@epoch.ncsc.mil> */

/* Updated: Frank Mayer <mayerf@tresys.com> and Karl MacMillan <kmacmillan@tresys.com>
 *
 * 	Added conditional policy language extensions
 *
 * Copyright (C) 2003 - 2004 Tresys Technology, LLC
 *	This program is free software; you can redistribute it and/or modify
 *  	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, version 2.
 */

#include "ebitmap.h"
#include "avtab.h"
#include "mls.h"
#include "policydb.h"
#include "conditional.h"

int ebitmap_write(ebitmap_t * e, FILE * fp)
{
	ebitmap_node_t *n;
	__u32 buf[32], bit, count;
	__u64 map;
	size_t items;

	buf[0] = cpu_to_le32(MAPSIZE);
	buf[1] = cpu_to_le32(e->highbit);

	count = 0;
	for (n = e->node; n; n = n->next)
		count++;
	buf[2] = cpu_to_le32(count);

	items = fwrite(buf, sizeof(__u32), 3, fp);
	if (items != 3)
		return -1;

	for (n = e->node; n; n = n->next) {
		bit = cpu_to_le32(n->startbit);
		items = fwrite(&bit, sizeof(__u32), 1, fp);
		if (items != 1)
			return -1;
		map = cpu_to_le64(n->map);
		items = fwrite(&map, sizeof(__u64), 1, fp);
		if (items != 1)
			return -1;

	}

	return 0;
}

int avtab_write_item(avtab_ptr_t cur, FILE *fp)
{
	__u32 buf[32];
	size_t items, items2;

	items = 1;	/* item 0 is used for the item count */
	buf[items++] = cpu_to_le32(cur->key.source_type);
	buf[items++] = cpu_to_le32(cur->key.target_type);
	buf[items++] = cpu_to_le32(cur->key.target_class);
	buf[items++] = cpu_to_le32(cur->datum.specified);
	if (!(cur->datum.specified & (AVTAB_AV | AVTAB_TYPE))) {
		printf("security: avtab: null entry\n");
		return -1;
	}
	if ((cur->datum.specified & AVTAB_AV) &&
	    (cur->datum.specified & AVTAB_TYPE)) {
		printf("security: avtab: entry has both access vectors and types\n");
		return -1;
	}
	if (cur->datum.specified & AVTAB_AV) {
		if (cur->datum.specified & AVTAB_ALLOWED)
			buf[items++] = cpu_to_le32(avtab_allowed(&cur->datum));
		if (cur->datum.specified & AVTAB_AUDITDENY)
			buf[items++] = cpu_to_le32(avtab_auditdeny(&cur->datum));
		if (cur->datum.specified & AVTAB_AUDITALLOW)
			buf[items++] = cpu_to_le32(avtab_auditallow(&cur->datum));
	} else {
		if (cur->datum.specified & AVTAB_TRANSITION)
			buf[items++] = cpu_to_le32(avtab_transition(&cur->datum));
		if (cur->datum.specified & AVTAB_CHANGE)
			buf[items++] = cpu_to_le32(avtab_change(&cur->datum));
		if (cur->datum.specified & AVTAB_MEMBER)
			buf[items++] = cpu_to_le32(avtab_member(&cur->datum));
	}
	buf[0] = cpu_to_le32(items - 1);
	
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;
	return 0;
}

int avtab_write(avtab_t * a, FILE * fp)
{
	int i;
	avtab_ptr_t cur;
	__u32 nel;
	size_t items;

	nel = cpu_to_le32(a->nel);
	items = fwrite(&nel, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;

	for (i = 0; i < AVTAB_SIZE; i++) {
		for (cur = a->htable[i]; cur; cur = cur->next) {
			if (avtab_write_item(cur, fp))
			    return -1;
		}	
	}

	return 0;
}

#ifdef CONFIG_SECURITY_SELINUX_MLS
/*
 * Write a MLS level structure to a policydb binary 
 * representation file.
 */
int mls_write_level(mls_level_t * l,
		    FILE * fp)
{
	__u32 sens;
	size_t items;

	sens = cpu_to_le32(l->sens);
	items = fwrite(&sens, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;

	if (ebitmap_write(&l->cat, fp))
		return -1;

	return 0;
}


/*
 * Write a MLS range structure to a policydb binary 
 * representation file.
 */
static int mls_write_range_helper(mls_range_t * r,
				  FILE * fp)
{
	__u32 buf[3];
	size_t items, items2;
	int rel;

	rel = mls_level_relation(r->level[1], r->level[0]);

	items = 1;		/* item 0 is used for the item count */
	buf[items++] = cpu_to_le32(r->level[0].sens);
	if (rel != MLS_RELATION_EQ)
		buf[items++] = cpu_to_le32(r->level[1].sens);
	buf[0] = cpu_to_le32(items - 1);

	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items2 != items)
		return -1;

	if (ebitmap_write(&r->level[0].cat, fp))
		return -1;
	if (rel != MLS_RELATION_EQ)
		if (ebitmap_write(&r->level[1].cat, fp))
			return -1;

	return 0;
}

int mls_write_range(context_struct_t * c,
		    FILE * fp)
{
	return mls_write_range_helper(&c->range, fp);
}


/*
 * Write a MLS perms structure to a policydb binary 
 * representation file.
 */
int mls_write_class(class_datum_t * cladatum,
		    FILE * fp)
{
	mls_perms_t *p = &cladatum->mlsperms;
	__u32 buf[32];
	size_t items, items2;

	items = 0;
	buf[items++] = cpu_to_le32(p->read);
	buf[items++] = cpu_to_le32(p->readby);
	buf[items++] = cpu_to_le32(p->write);
	buf[items++] = cpu_to_le32(p->writeby);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items2 != items)
		return -1;

	return 0;
}

#define mls_write_perm(buf, items, perdatum) \
	buf[items++] = cpu_to_le32(perdatum->base_perms);

int mls_write_user(user_datum_t *usrdatum, FILE *fp)
{
	mls_range_list_t *r;
	__u32 nel;
	__u32 buf[32];
	int items;

	nel = 0;
	for (r = usrdatum->ranges; r; r = r->next)
		nel++;
	buf[0] = cpu_to_le32(nel);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;
	for (r = usrdatum->ranges; r; r = r->next) {
		if (mls_write_range_helper(&r->range, fp))
			return -1;
	}
	return 0;
}

int mls_write_nlevels(policydb_t *p, FILE *fp)
{
	__u32 buf[32];
	size_t items;

	buf[0] = cpu_to_le32(p->nlevels);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;
	return 0;
}

int mls_write_trusted(policydb_t *p, FILE *fp)
{
	if (ebitmap_write(&p->trustedreaders, fp))
		return -1;
	if (ebitmap_write(&p->trustedwriters, fp))
		return -1;
	if (ebitmap_write(&p->trustedobjects, fp))
		return -1;
	return 0;
}

int sens_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	level_datum_t *levdatum;
	__u32 buf[32];
	size_t items, items2, len;
	FILE *fp = p;

	levdatum = (level_datum_t *) datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(levdatum->isalias);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	if (mls_write_level(levdatum->level, fp))
		return -1;

	return 0;
}

int cat_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	cat_datum_t *catdatum;
	__u32 buf[32];
	size_t items, items2, len;
	FILE *fp = p;


	catdatum = (cat_datum_t *) datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(catdatum->value);
	buf[items++] = cpu_to_le32(catdatum->isalias);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	return 0;
}
#else
#define mls_write_range(c, fp) 0
#define mls_write_class(c, fp) 0
#define mls_write_perm(buf, items, perdatum) 
#define mls_write_user(u, fp) 0
#define mls_write_nlevels(p, fp) 0
#define mls_write_trusted(p, fp) 0
#endif


int cond_write_bool(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	cond_bool_datum_t *booldatum;
	__u32 buf[3], len;
	int items, items2;
	FILE *fp = p;

	booldatum = (cond_bool_datum_t*)datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(booldatum->value);
	buf[items++] = cpu_to_le32(booldatum->state);
	buf[items++] = cpu_to_le32(len);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;
	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;
	return 0;
}

/*
 * cond_write_cond_av_list doesn't write out the av_list nodes.
 * Instead it writes out the key/value pairs from the avtab. This
 * is necessary because there is no way to uniquely identifying rules
 * in the avtab so it is not possible to associate individual rules
 * in the avtab with a conditional without saving them as part of
 * the conditional. This means that the avtab with the conditional
 * rules will not be saved but will be rebuilt on policy load.
 */
int cond_write_av_list(cond_av_list_t *list, FILE *fp)
{
	__u32 buf[4];
	cond_av_list_t *cur_list;
	__u32 len, items;

	len = 0;
	for (cur_list = list; cur_list != NULL; cur_list = cur_list->next) {
		if (cur_list->node->parse_context)
			len++;
	}
	
	buf[0] = cpu_to_le32(len);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;
	if (items == 0)
		return 0;

	for (cur_list = list; cur_list != NULL; cur_list = cur_list->next) {
		if (cur_list->node->parse_context)
			if (avtab_write_item(cur_list->node, fp))
				return -1;
	}
	return 0;
}

int cond_write_node(cond_node_t *node, FILE *fp)
{
	cond_expr_t *cur_expr;
	__u32 buf[2];
	__u32 items, items2, len;

	buf[0] = cpu_to_le32(node->cur_state);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;

	/* expr */
	len = 0;
	for (cur_expr = node->expr; cur_expr != NULL; cur_expr = cur_expr->next)
		len++;

	buf[0] = cpu_to_le32(len);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;

	for (cur_expr = node->expr; cur_expr != NULL; cur_expr = cur_expr->next) {
		items = 0;
		buf[items++] = cpu_to_le32(cur_expr->expr_type);
		buf[items++] = cpu_to_le32(cur_expr->bool);
		items2 = fwrite(buf, sizeof(__u32), items, fp);
		if (items2 != items)
			return -1;
	}

	if (cond_write_av_list(node->true_list, fp) != 0)
		return -1;
	if (cond_write_av_list(node->false_list, fp) != 0)
		return -1;
	
	return 0;
}

int cond_write_list(cond_list_t *list, void *p)
{
	FILE *fp = p;
	cond_node_t *cur;
	__u32 len, items;
	__u32 buf[1];

	len = 0;
	for (cur = list; cur != NULL; cur = cur->next)
		len++;
	buf[0] = cpu_to_le32(len);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;

	for (cur = list; cur != NULL; cur = cur->next) {
		if (cond_write_node(cur, p) != 0)
			return -1;
	}
	return 0;
}

/*
 * Write a security context structure
 * to a policydb binary representation file.
 */
static int context_write(context_struct_t * c, FILE * fp)
{
	__u32 buf[32];
	size_t items, items2;

	items = 0;
	buf[items++] = cpu_to_le32(c->user);
	buf[items++] = cpu_to_le32(c->role);
	buf[items++] = cpu_to_le32(c->type);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items2 != items)
		return -1;
	if (mls_write_range(c, fp))
		return -1;

	return 0;
}


/*
 * The following *_write functions are used to
 * write the symbol data to a policy database
 * binary representation file.
 */

static int perm_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	perm_datum_t *perdatum;
	__u32 buf[32];
	size_t items, items2, len;
	FILE *fp = p;

	perdatum = (perm_datum_t *) datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(perdatum->value);
	mls_write_perm(buf, items, perdatum);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	return 0;
}


static int common_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	common_datum_t *comdatum;
	__u32 buf[32];
	size_t items, items2, len;
	FILE *fp = p;

	comdatum = (common_datum_t *) datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(comdatum->value);
	buf[items++] = cpu_to_le32(comdatum->permissions.nprim);
	buf[items++] = cpu_to_le32(comdatum->permissions.table->nel);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	if (hashtab_map(comdatum->permissions.table, perm_write, fp))
		return -1;

	return 0;
}


static int class_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	class_datum_t *cladatum;
	constraint_node_t *c;
	constraint_expr_t *e;
	__u32 buf[32], ncons, nexpr;
	size_t items, items2, len, len2;
	FILE *fp = p;

	cladatum = (class_datum_t *) datum;

	len = strlen(key);
	if (cladatum->comkey)
		len2 = strlen(cladatum->comkey);
	else
		len2 = 0;

	ncons = 0;
	for (c = cladatum->constraints; c; c = c->next) {
		ncons++;
	}

	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(len2);
	buf[items++] = cpu_to_le32(cladatum->value);
	buf[items++] = cpu_to_le32(cladatum->permissions.nprim);
	if (cladatum->permissions.table) 
		buf[items++] = cpu_to_le32(cladatum->permissions.table->nel);
	else
		buf[items++] = 0;
	buf[items++] = cpu_to_le32(ncons);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	if (cladatum->comkey) {
		items = fwrite(cladatum->comkey, 1, len2, fp);
		if (items != len2)
			return -1;
	}
	if (hashtab_map(cladatum->permissions.table, perm_write, fp))
		return -1;

	for (c = cladatum->constraints; c; c = c->next) {
		nexpr = 0;
		for (e = c->expr; e; e = e->next) {
			nexpr++;
		}
		buf[0] = cpu_to_le32(c->permissions);
		buf[1] = cpu_to_le32(nexpr);
		items = fwrite(buf, sizeof(__u32), 2, fp);
		if (items != 2)
			return -1;
		for (e = c->expr; e; e = e->next) {
			items = 0;
			buf[items++] = cpu_to_le32(e->expr_type);
			buf[items++] = cpu_to_le32(e->attr);
			buf[items++] = cpu_to_le32(e->op);
			items2 = fwrite(buf, sizeof(__u32), items, fp);
			if (items != items2)
				return -1;

			switch (e->expr_type) {
			case CEXPR_NAMES:
				if (ebitmap_write(&e->names, fp))
					return -1;
				break;
			default:
				break;
			}
		}
	}

	if (mls_write_class(cladatum, fp))
		return -1;

	return 0;
}

static int role_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	role_datum_t *role;
	__u32 buf[32];
	size_t items, items2, len;
	FILE *fp = p;

	role = (role_datum_t *) datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(role->value);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	if (ebitmap_write(&role->dominates, fp))
		return -1;

	if (ebitmap_write(&role->types, fp))
		return -1;

	return 0;
}

static int type_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	type_datum_t *typdatum;
	__u32 buf[32];
	size_t items, items2, len;
	FILE *fp = p;

	typdatum = (type_datum_t *) datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(typdatum->value);
	buf[items++] = cpu_to_le32(typdatum->primary);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	return 0;
}

static int user_write(hashtab_key_t key, hashtab_datum_t datum, void *p)
{
	user_datum_t *usrdatum;
	__u32 buf[32];
	size_t items, items2, len;
	FILE *fp = p;


	usrdatum = (user_datum_t *) datum;

	len = strlen(key);
	items = 0;
	buf[items++] = cpu_to_le32(len);
	buf[items++] = cpu_to_le32(usrdatum->value);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	items = fwrite(key, 1, len, fp);
	if (items != len)
		return -1;

	if (ebitmap_write(&usrdatum->roles, fp))
		return -1;

	return mls_write_user(usrdatum, fp);
}


static int (*write_f[SYM_NUM]) (hashtab_key_t key, hashtab_datum_t datum, void *datap) =
{
	common_write,
	class_write,
	role_write,
	type_write,
	user_write,
	mls_write_f
	cond_write_bool
};


/*
 * Write the configuration data in a policy database
 * structure to a policy database binary representation
 * file.
 */
int policydb_write(policydb_t * p, FILE * fp)
{
	struct role_allow *ra;
	struct role_trans *tr;
	ocontext_t *c;
	genfs_t *genfs;
	int i, j, num_syms;
	__u32 buf[32], config;
	size_t items, items2, len, nel;
	struct policydb_compat_info *info;
	char *policydb_str = POLICYDB_STRING;

	config = 0;
	mls_set_config(config);

	/* Write the magic number and string identifiers. */
	items = 0;
	buf[items++] = cpu_to_le32(POLICYDB_MAGIC);
	len = strlen(POLICYDB_STRING);
	buf[items++] = cpu_to_le32(len);
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;
	items = fwrite(policydb_str, 1, len, fp);
	if (items != len)
		return -1;

	/* Write the version, config, and table sizes. */
	items = 0;
	info = policydb_lookup_compat(policyvers);
	if (!info) {
		fprintf(stderr, "policydb_lookup_compat() failed for %d\n", policyvers);
		return -1;
	}

	buf[items++] = cpu_to_le32(policyvers);
	buf[items++] = cpu_to_le32(config);
	buf[items++] = cpu_to_le32(info->sym_num);
	buf[items++] = cpu_to_le32(info->ocon_num);
	
	items2 = fwrite(buf, sizeof(__u32), items, fp);
	if (items != items2)
		return -1;

	if (mls_write_nlevels(p, fp))
		return -1;

	num_syms = info->sym_num;
	for (i = 0; i < num_syms; i++) {
		buf[0] = cpu_to_le32(p->symtab[i].nprim);
		buf[1] = cpu_to_le32(p->symtab[i].table->nel);
		items = fwrite(buf, sizeof(__u32), 2, fp);
		if (items != 2)
			return -1;
		if (hashtab_map(p->symtab[i].table, write_f[i], fp))
			return -1;
	}

	if (avtab_write(&p->te_avtab, fp))
		return -1;

	if (policyvers < POLICYDB_VERSION_BOOL) {
		if (p->p_bools.nprim)
			fprintf(stderr, "warning: discarding booleans and conditional rules\n");

	} else {
		if (cond_write_list(p->cond_list, fp))
			return -1;
	}

	nel = 0;
	for (tr = p->role_tr; tr; tr = tr->next) 
		nel++;
	buf[0] = cpu_to_le32(nel);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;
	for (tr = p->role_tr; tr; tr = tr->next) {
		buf[0] = cpu_to_le32(tr->role);
		buf[1] = cpu_to_le32(tr->type);
		buf[2] = cpu_to_le32(tr->new_role);
		items = fwrite(buf, sizeof(__u32), 3, fp);
		if (items != 3)
			return -1;		
	}

	nel = 0;
	for (ra = p->role_allow; ra; ra = ra->next) 
		nel++;
	buf[0] = cpu_to_le32(nel);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;
	for (ra = p->role_allow; ra; ra = ra->next) {
		buf[0] = cpu_to_le32(ra->role);
		buf[1] = cpu_to_le32(ra->new_role);
		items = fwrite(buf, sizeof(__u32), 2, fp);
		if (items != 2)
			return -1;		
	}

	for (i = 0; i < info->ocon_num; i++) {
		nel = 0;
		for (c = p->ocontexts[i]; c; c = c->next)
			nel++;
		buf[0] = cpu_to_le32(nel);
		items = fwrite(buf, sizeof(__u32), 1, fp);
		if (items != 1)
			return -1;
		for (c = p->ocontexts[i]; c; c = c->next) {
			switch (i) {
			case OCON_ISID:
				buf[0] = cpu_to_le32(c->sid[0]);
				items = fwrite(buf, sizeof(__u32), 1, fp);
				if (items != 1)
					return -1;
				if (context_write(&c->context[0], fp))
					return -1;
				break;
			case OCON_FS:
			case OCON_NETIF:
				len = strlen(c->u.name);
				buf[0] = cpu_to_le32(len);
				items = fwrite(buf, sizeof(__u32), 1, fp);
				if (items != 1)
					return -1;
				items = fwrite(c->u.name, 1, len, fp);
				if (items != len)
					return -1;
				if (context_write(&c->context[0], fp))
					return -1;
				if (context_write(&c->context[1], fp))
					return -1;
				break;
			case OCON_PORT:
				buf[0] = c->u.port.protocol;
				buf[1] = c->u.port.low_port;
				buf[2] = c->u.port.high_port;
				for (j = 0; j < 3; j++) {
					buf[j] = cpu_to_le32(buf[j]);
				}
				items = fwrite(buf, sizeof(__u32), 3, fp);
				if (items != 3)
					return -1;
				if (context_write(&c->context[0], fp))
					return -1;
				break;
			case OCON_NODE:
				buf[0] = cpu_to_le32(c->u.node.addr);
				buf[1] = cpu_to_le32(c->u.node.mask);
				items = fwrite(buf, sizeof(__u32), 2, fp);
				if (items != 2)
					return -1;
				if (context_write(&c->context[0], fp))
					return -1;
				break;
			case OCON_FSUSE:
				buf[0] = cpu_to_le32(c->v.behavior);
				len = strlen(c->u.name);
				buf[1] = cpu_to_le32(len);
				items = fwrite(buf, sizeof(__u32), 2, fp);
				if (items != 2)
					return -1;
				items = fwrite(c->u.name, 1, len, fp);
				if (items != len)
					return -1;
				if (context_write(&c->context[0], fp))
					return -1;
				break;
			case OCON_NODE6:
				for (j = 0; j < 4; j++)
					buf[j] = cpu_to_le32(c->u.node6.addr[j]);
				for (j = 0; j < 4; j++)
					buf[j+4] = cpu_to_le32(c->u.node6.mask[j]);
				items = fwrite(buf, sizeof(__u32), 8, fp);
				if (items != 8)
					return -1;
				if (context_write(&c->context[0], fp))
					return -1;
				break;	
			}
		}
	}

	nel = 0;
	for (genfs = p->genfs; genfs; genfs = genfs->next) 
		nel++;
	buf[0] = cpu_to_le32(nel);
	items = fwrite(buf, sizeof(__u32), 1, fp);
	if (items != 1)
		return -1;	
	for (genfs = p->genfs; genfs; genfs = genfs->next) {
		len = strlen(genfs->fstype);
		buf[0] = cpu_to_le32(len);
		items = fwrite(buf, sizeof(__u32), 1, fp);
		if (items != 1)
			return -1;
		items = fwrite(genfs->fstype, 1, len, fp);
		if (items != len)
			return -1;
		nel = 0;
		for (c = genfs->head; c; c = c->next)
			nel++;
		buf[0] = cpu_to_le32(nel);
		items = fwrite(buf, sizeof(__u32), 1, fp);
		if (items != 1)
			return -1;
		for (c = genfs->head; c; c = c->next) {
			len = strlen(c->u.name);
			buf[0] = cpu_to_le32(len);
			items = fwrite(buf, sizeof(__u32), 1, fp);
			if (items != 1)
				return -1;
			items = fwrite(c->u.name, 1, len, fp);
			if (items != len)
				return -1;
			buf[0] = cpu_to_le32(c->v.sclass);
			items = fwrite(buf, sizeof(__u32), 1, fp);
			if (items != 1)
				return -1;
			if (context_write(&c->context[0], fp))
				return -1;
		}
	}

	if (mls_write_trusted(p, fp))
		return -1;

	return 0;
}

