/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <pwd.h>
typedef unsigned int in_addr_t;
#include <asn1.h>
#include <mib.h>
#include <parse.h>
#include <snmp_api.h>
#include <snmp_client.h>
#include <snmp.h>
#include <snmp_impl.h>
#include "cheops.h"



struct entry {
	struct entry *next;
	char *obj;
	char *value;
	char *community;
	int type;
};

struct browser {
	struct net_object *no;
	struct net_page *np;
	int count;
	int busy;
	int *cancel;
	char community[256];
	struct entry *el;
	GtkWidget *window;
	GtkWidget *hbox;
	GtkWidget *entry;
	GtkWidget *vbox;
	GtkWidget *tree;
	GtkWidget *combo;
	GtkWidget *status;
	GtkWidget *close;
};


static char *status2str(int status)
{
	switch(status) {
	case MIB_STATUS_MANDATORY:
	return "Mandatory";
	case MIB_STATUS_OPTIONAL:
	return "Optional";
	case MIB_STATUS_OBSOLETE:
	return "Obsolete";
	case MIB_STATUS_DEPRECATED:
	return "Deprecated";
	case MIB_STATUS_CURRENT:
	return "Current";
	default:
	return "Unknown";
	};
}

static void delete_entries(struct browser *b)
{
	/* Delete allocated entry space */
	struct entry *e, *e2;
	e = b->el;
	while(e) {
		e2 = e;
		e=e->next;
		g_free(e2->obj);
		g_free(e2->value);
		g_free(e2);
	}
	b->el=NULL;
}

int destroy_browser(GtkWidget *w)
{
	struct browser *b = (struct browser *)gtk_object_get_user_data(GTK_OBJECT(w));
	if (!b->busy) {
		delete_entries(b);
		gtk_widget_destroy(b->window);
		g_free(b);
		return FALSE;
	} else {
		return TRUE;
	}
}

static char *access2str(int access)
{
	switch(access) {
	case MIB_ACCESS_READONLY:
	return "Read Only";
	case MIB_ACCESS_READWRITE:
	return "Read/Write";
	case MIB_ACCESS_NOACCESS:
	return "No access";
	case MIB_ACCESS_NOTIFY:
	return "Notify";
	case MIB_ACCESS_CREATE:
	return "Create";
	default:
	return "Unknown";
	}
}

static int type2type(int type)
{
	/* For some reason we have to munge the types :( */
	switch(type){
	case 1:
	return ASN_OBJECT_ID;
	case 2:
	return ASN_OCTET_STR;
	case 3:
	return ASN_INTEGER;
	case 5:
	return ASN_IPADDRESS;
	case 6:
	return ASN_COUNTER;
	case 7:
	return ASN_GAUGE;
	case 8:
	return ASN_TIMETICKS;
	default:
	return type;
	}
}

static char *type2str(int type)
{
	static char buf[256];
	switch(type) {
	case ASN_IPADDRESS:
	return "IP Address";
	case ASN_COUNTER:
	return "Counter (wrap around)";
	case ASN_GAUGE:
	return "Gauge (no wrap wround)";
	case ASN_OPAQUE:
	return "Opaque Type";
	case ASN_BOOLEAN:
	return "Boolean (true/false)";
	case ASN_INTEGER:
	return "Integer";
	case ASN_BIT_STR:
	return "String of Bits";
	case ASN_OCTET_STR:
	return "Byte String";
	case ASN_NULL:
	return "Null Value";
	case ASN_OBJECT_ID:
	return "Object ID";
	case ASN_SEQUENCE:
	return "Sequence";
	case ASN_SET:
	return "Set of values";
	case ASN_TIMETICKS:
	return "1/100 of seconds";
	default:
	g_snprintf(buf, sizeof(buf), "Unknown: %d", type);
	return buf;
	}
}

static void do_set(GtkWidget *w, struct entry *e)
{
	char *c;	
	struct browser *b = (struct browser *)gtk_object_get_user_data(GTK_OBJECT(w));
	char buf[256];
	c = gtk_entry_get_text(GTK_ENTRY(b->entry));
#if 0
	printf("Setting %s's %s (%s) to %s\n", e->community, e->obj, type2str(e->type), c);
#endif
	c = snmp_set(b->no, e->community, &(e->obj), &(c), &(e->type), 1, NULL);
	if (!c || !strlen(c)) 
		g_snprintf(buf, sizeof(buf), "SNMP agent on %s did not accept our change of %s", b->no->hostname, e->obj);
	else {
		g_snprintf(buf, sizeof(buf), "%s changed to '%s' on %s", e->obj, c, b->no->hostname);
		gtk_entry_set_text(GTK_ENTRY(b->entry), c);
	}
	generic_set_status(b->status, buf);
	g_free(e->value);
	e->value = g_strdup(c);
#if 0
	printf("Result is %s\n", c);
#endif
}

static void show_value(GtkTree *root, struct entry *e)
{
	struct browser *b = (struct browser *)gtk_object_get_user_data(GTK_OBJECT(root));
	oid name[256];
	int namelen;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *hbox2;
	GtkWidget *vbox;
	GtkWidget *adjust;
	struct tree *t, *head;
	char desc[4096];
	int x, nflag;
	char *c;
	namelen = sizeof(name);
	if (b) {
		read_objid(e->obj, name, &namelen);
		head = get_tree_head();
		t = get_tree(name, namelen, head);
		if (t) {
			e->type = type2type(t->type);
			if (b->vbox)
				gtk_widget_destroy(b->vbox);
			b->vbox = gtk_vbox_new(FALSE, 5);

			/* Label */
			label = gtk_label_new(e->obj);
			gtk_widget_set_usize(label, 400, 0);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(b->vbox), label, FALSE, FALSE, 5);
			
			/* Value */
			hbox = gtk_hbox_new(FALSE, 5);
			vbox = gtk_vbox_new(FALSE, 0);
			gtk_widget_show(hbox);
			gtk_widget_show(vbox);
			label = gtk_label_new("Value: ");
			gtk_widget_set_usize(label, 80, 0);
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
			gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 5);

			if ((t->access == MIB_ACCESS_READWRITE) && strcasecmp(b->community, "public")) {
				b->entry = label = gtk_entry_new();
				adjust = gtk_button_new_with_label("   Set   ");
				hbox2 = gtk_hbox_new(FALSE, 5);
				gtk_object_set_user_data(GTK_OBJECT(adjust), b);
				gtk_signal_connect(GTK_OBJECT(adjust), "clicked", GTK_SIGNAL_FUNC(do_set), e);
				gtk_widget_show(hbox2);
				gtk_widget_show(adjust);
				gtk_entry_set_text(GTK_ENTRY(label), e->value);
				gtk_box_pack_start(GTK_BOX(hbox2),label, TRUE, TRUE, 5);
				gtk_box_pack_start(GTK_BOX(hbox2),adjust, FALSE, FALSE, 5);
				gtk_box_pack_start(GTK_BOX(hbox),hbox2, TRUE, TRUE, 5);
			} else {
				label = gtk_label_new(e->value);
				gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
				gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			}
			gtk_box_pack_start(GTK_BOX(b->vbox), hbox, FALSE, FALSE, 5);
			gtk_widget_show(label);
			
			/* Type */
			hbox = gtk_hbox_new(FALSE, 5);
			gtk_widget_show(hbox);
			label = gtk_label_new("Type: ");
			gtk_widget_set_usize(label, 80, 0);
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			g_snprintf(desc, sizeof(desc), "%s %s", type2str(e->type), t->units ? t->units : "");
			label = gtk_label_new(desc);
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(b->vbox), hbox, FALSE, FALSE, 5);

			/* Access */
			hbox = gtk_hbox_new(FALSE, 5);
			gtk_widget_show(hbox);
			label = gtk_label_new("Access: ");
			gtk_widget_set_usize(label, 80, 0);
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			label = gtk_label_new(access2str(t->access));
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(b->vbox), hbox, FALSE, FALSE, 5);
			
			/* Status */
			hbox = gtk_hbox_new(FALSE, 5);
			gtk_widget_show(hbox);
			label = gtk_label_new("Status: ");
			gtk_widget_set_usize(label, 80, 0);
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			label = gtk_label_new(status2str(t->status));
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(b->vbox), hbox, FALSE, FALSE, 5);

			/* Description */
			hbox = gtk_hbox_new(FALSE, 5);
			vbox = gtk_vbox_new(FALSE, 0);
			gtk_widget_show(hbox);
			gtk_widget_show(vbox);
			label = gtk_label_new("Description: ");
			gtk_widget_set_usize(label, 80, 0);
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
			gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 5);
			x=0;
			nflag=1;
			c=t->description;
			if (c) {
				while(*c) {
					switch(*c) {
					case '\n':
					case '\r':
						nflag=1;
						desc[x++]='\n';
						break;
					case ' ':
						if (nflag)
							break;
					default:
						desc[x++]=*c;
						nflag=0;
					}
					c++;			
				}
				desc[x]='\0';
			} else {
				strncpy(desc,"No description available\n", sizeof(desc));
			}
#if 0
			g_snprintf(desc, sizeof(desc), "         %s", t->description);
#endif
			label = gtk_label_new(desc);
			gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
			gtk_widget_show(label);
			gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
			gtk_box_pack_start(GTK_BOX(b->vbox), hbox, FALSE, FALSE, 5);

			gtk_widget_show(b->vbox);
			gtk_box_pack_start(GTK_BOX(b->hbox), b->vbox, TRUE, TRUE, 5);
#if 0
			printf("%s (%s, %s): %s\n", t->label, access2str(t->access), status2str(t->status), t->description);
#endif
		} else
			printf("No tree!\n");
	} else
		printf("No browser\n");
}

static void handle_click(GtkWidget *w)
{
	struct entry *e;
	
	e = (struct entry *)gtk_object_get_user_data(GTK_OBJECT(w));
#if 0
	printf("%s's %s is %s\n", e->community, e->obj, e->value);
#endif
	show_value(GTK_TREE(w->parent)->root_tree, e);
}


static void make_tree(struct browser *b)
{
	GtkWidget *t;
	GtkWidget *sw;
	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_show(sw);
	/* Maybe we'll want to have the option of  GtkCTree in the future */
	t=gtk_tree_new();
	gtk_widget_set_usize(sw, 300, 300);
	b->tree = t;
#if (GTK_MINOR_VERSION > 1) || ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 4))
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), b->tree);
#else
	gtk_container_add(GTK_CONTAINER(sw), b->tree);
#endif
	gtk_object_set_user_data(GTK_OBJECT(b->tree), b);
	gtk_box_pack_start(GTK_BOX(b->hbox), sw, FALSE, FALSE, 5);
}

static GtkWidget *get_tree_w(GtkWidget *tree, char *entry, char *next, int add)
{
	GList *l;
	char *c;
	char buf[256];
	GtkWidget *ti=NULL;
	GtkWidget *label;
	if (!strcmp(entry, "0") && !next) {
		/* Don't make a separate entry for .0's */
		if (GTK_IS_TREE(tree)) 
			ti = GTK_TREE(tree)->tree_owner;
		else
			ti = tree;
		if (ti) {
			label = GTK_BIN(ti)->child;
			gtk_label_get(GTK_LABEL(label), &c);
			g_snprintf(buf, sizeof(buf), "%s.0", c);
			gtk_label_set(GTK_LABEL(label), buf);
			return ti;
		}
	}
	if (GTK_IS_TREE_ITEM(tree)) {
		ti = tree;
		if (GTK_TREE_ITEM(ti)->subtree) {
			tree = GTK_TREE_ITEM(ti)->subtree;
		} else {
			tree = gtk_tree_new();
			gtk_widget_show(tree);
			gtk_tree_item_set_subtree(GTK_TREE_ITEM(ti), tree);
		}
	}
	l = GTK_TREE(tree)->children;
	while(l) {
		ti = GTK_WIDGET(l->data);
		label = GTK_BIN(ti)->child;
		gtk_label_get(GTK_LABEL(label), &c);
		if (!strcmp(c, entry))
			break;
		l = g_list_next(l);
	}
	if (!l) {
		if (add) {
			ti = gtk_tree_item_new_with_label(entry);
			gtk_widget_show(ti);
			gtk_tree_append(GTK_TREE(tree), ti);
		} else ti=NULL;
	}
	return ti;
}

static void add_to_tree(struct browser *b, char *community, char *obj, char *value)
{
	char *c, *d;
	char buf[256];
	GtkWidget *tree=b->tree;
	struct entry *e;
	strncpy(buf, obj, sizeof(buf));
	c = strtok(buf, ".");
	while(c) {
		d = strtok(NULL, ".");
		tree = get_tree_w(tree, c, d, 1);
		c=d;
	}
	e = g_new0(struct entry, 1);
	e->next = b->el;
	b->el = e;
	e->obj = obj;
	e->community = community;
	e->value = value;
	e->type = -1;
	gtk_object_set_user_data(GTK_OBJECT(tree), e);
	gtk_signal_connect(GTK_OBJECT(tree), "button_press_event", GTK_SIGNAL_FUNC(handle_click), NULL);
}

static void snmp_browser_callback(char *obj, char *value, void *data)
{
	struct browser *b = (struct browser *)data;
	char buf[256];
	b->count++;
	if (!(b->count % 100)) {
		g_snprintf(buf, sizeof(buf), "%d objects read", b->count);
		generic_set_status(b->status, buf);
	}
	add_to_tree(b, b->community, g_strdup(obj), g_strdup(value));
}

static void snmp_browser_show(GtkWidget *w)
{
	GtkWidget *tmpw;
	GtkWidget *label;
	char buf[256];
	int res;
	char *c;
	struct browser *b = (struct browser *)gtk_object_get_user_data(GTK_OBJECT(w));
	label = GTK_BUTTON(w)->child;
	gtk_label_get(GTK_LABEL(label), &c);
	if (!strcasecmp(c, "Cancel")) {
		(*(b->cancel))++;
		return;
	}
	if (b->vbox) {
		gtk_widget_destroy(b->vbox);
		b->vbox = NULL;
	}
	if (!valid_no(b->np, b->no))
		return;
	gtk_tree_clear_items(GTK_TREE(b->tree), 0, -1);
	delete_entries(b);
	generic_setcursor(b->window->window, GDK_WATCH);
	b->count=0;
	c=gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(b->combo)->entry));
	g_snprintf(buf, sizeof(buf), "Searching community '%s'", c);
	generic_set_status(b->status, buf);
	b->busy = 1;
	b->cancel = g_new0(int, 1);
	*(b->cancel) = 0;
	gtk_label_set(GTK_LABEL(label), "  Cancel  ");
	gtk_widget_set_sensitive(b->close, FALSE);
	strncpy(b->community, c, sizeof(b->community));
	res = snmp_walk(b->no, c, snmp_browser_callback, b, b->cancel);
	gtk_widget_set_sensitive(b->close, TRUE);
	gtk_label_set(GTK_LABEL(label), "  Lookup  ");
	g_free(b->cancel);
	b->busy = 0;
	if (res) {
		g_snprintf(buf, sizeof(buf), "No such community '%s'", c);
	} else {
		g_snprintf(buf, sizeof(buf), "%d objects read total", b->count);
	}
	generic_set_status(b->status, buf);
	tmpw = get_tree_w(b->tree, "system", NULL, 0);
	if (tmpw)
		gtk_tree_item_expand(GTK_TREE_ITEM(tmpw));
	gdk_window_set_cursor(b->window->window, NULL);
}

void snmp_browser(struct net_object *no)
{
	struct browser *b;
	char buf[256];
	
	GtkWidget *vbox;
	GtkWidget *bbox;
	GtkWidget *hbox;
	GtkWidget *lookup;
	GtkWidget *label;
	GList *l;
	
	b = g_new0(struct browser, 1);
	b->busy =0;
	b->el = NULL;
	b->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	b->hbox = gtk_hbox_new(FALSE, 5);
	hbox = gtk_hbox_new(FALSE, 5);
	vbox = gtk_vbox_new(FALSE, 5);
	b->status = gtk_statusbar_new();
	gtk_widget_show(b->status);
	b->vbox = NULL;
	b->no = no;
	b->np = no->np;
	make_tree(b);
	
	b->combo = gtk_combo_new();
	gtk_widget_show(b->combo);
	
	l = g_list_append(NULL, "public");
	l = g_list_append(l, "private");
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(b->combo)->entry), "public");
	gtk_combo_set_popdown_strings(GTK_COMBO(b->combo), l);
	gtk_widget_show(hbox);
	label = gtk_label_new("Community");
	gtk_widget_show(label);
	lookup = gtk_button_new_with_label("  Lookup  ");
	gtk_widget_show(lookup);
	
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), b->combo, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), lookup, FALSE, FALSE, 5);

	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	
	bbox = gtk_hbox_new(FALSE, 5);
	b->close = gtk_button_new_with_label("  Close  ");
	gtk_widget_realize(b->window);
	gtk_widget_show(vbox);
	gtk_widget_show(bbox);
	gtk_widget_show(b->close);
	
	gtk_object_set_user_data(GTK_OBJECT(b->window), b);
	gtk_object_set_user_data(GTK_OBJECT(b->close), b);
	gtk_object_set_user_data(GTK_OBJECT(lookup), b);
	
	gtk_signal_connect(GTK_OBJECT(b->window), "delete_event", GTK_SIGNAL_FUNC(destroy_browser), NULL);
	gtk_signal_connect(GTK_OBJECT(b->close), "clicked", GTK_SIGNAL_FUNC(destroy_browser), NULL);
	gtk_signal_connect(GTK_OBJECT(lookup), "clicked", GTK_SIGNAL_FUNC(snmp_browser_show), NULL);
	
	gtk_widget_show(b->hbox);
	gtk_widget_show(b->tree);

	gtk_box_pack_start(GTK_BOX(vbox), b->hbox, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(bbox), b->status, TRUE, TRUE, 5);
	gtk_box_pack_start(GTK_BOX(bbox), b->close, FALSE, FALSE, 5);
	
	fix_icon(b->window->window);
	gtk_container_add(GTK_CONTAINER(b->window), vbox);
	g_snprintf(buf, sizeof(buf), "SNMP Browser: %s", no->hostname);
	gtk_window_set_title(GTK_WINDOW(b->window), buf);
	gtk_widget_grab_focus(b->close);
	gtk_widget_show(b->window);
}
