/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <string.h>
#include <netdb.h>
#include "cheops.h"

#ifdef USE_CANVAS
#define DESTROY_LINK(a) gtk_object_destroy(GTK_OBJECT(a))
#else
#define DESTROY_LINK(a) gtk_widget_destroy(a)
#endif

char defname[256] = "";
struct net_win main_window;
struct net_page *current_page=NULL;
static struct pixmap_cache *pc=NULL;


static void show_page_menu(struct net_win *n);
static void hide_page_menu(struct net_win *n);
	
void set_status(char *s)
{
	GtkWidget *bar = main_window.status;
	char *tmp;
	gtk_label_get(GTK_LABEL(GTK_STATUSBAR(bar)->label), &tmp);
	if (strcmp(tmp, s)) {
		gtk_statusbar_pop(GTK_STATUSBAR(bar), 1);
		gtk_statusbar_push(GTK_STATUSBAR(bar), 1, s);
	}
}

static void arrange_cb(GtkWidget *w, int how)
{
	arrange(current_page, how);
}

static void make_saveas(GtkWidget *blah, GtkWidget *w)
{
	char *fn;
	fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(w));
	if (strlen(fn)) {
		do_save(NULL, fn);
		gtk_widget_destroy(w);
	}
}

static void remove_page()
{
	struct net_object *no, *no2;
	struct network *n, *n2;
	struct domain *d, *d2;
	struct net_page *np;
	int l;
	np = current_page;
	current_page = NULL;
	if (!np)
		return;
#ifdef USE_CANVAS
	if ((l=g_list_length(GTK_NOTEBOOK(np->canvas->parent->parent->parent)->children)) < 2)
#else
	if ((l=g_list_length(GTK_NOTEBOOK(np->pane->parent->parent)->children)) < 2)
#endif
		hide_page_menu(&main_window);
	no = np->objs;
	n = np->ncontents;
	d = np->dcontents;
	while(n) {
		n2=n;
		n=n->next;
		g_free(n2);
	}
	while(d) {
		d2=d;
		d=d->next;
		g_free(d2);
	}
	while(no) {
		no2=no;
		no = no->next;
		destroy_object(np, no2);
	}
	if (np->pe) {
		gtk_widget_destroy(np->pe->window);
		g_free(np->pe);
	}
	gtk_widget_destroy(np->notebook);
	g_free(np);
}

static void refresh_page()
{
	struct network *net;
	struct domain *d;
	struct in_addr a,m;
	char netmask[80];
	char ip[80];
#if 0
	printf("refreshing...\n");
#endif
	if (!current_page)
		return;
	net = current_page->ncontents;
	while(net) {
		a.s_addr = net->net_addr;
		m.s_addr = net->net_mask;
		strcpy(ip, inet_ntoa(a));
		strcpy(netmask, inet_ntoa(m));
#if 0
		printf("refreshing '%s' (%s)\n",ip, netmask);
#endif
		discover_network_a(current_page, ip,netmask, 0);
		net=net->next;
	} 
	d = current_page->dcontents;
	while(d) {
#if 0
		printf("refreshing '%s'\n",d->domain);
#endif
		gather_hosts(current_page, d->domain, d->recursive, 0);
		d=d->next;
	} 
}

struct auto_ref {
	GtkWidget *window;
	GtkWidget *checkbox;
	GtkWidget *timebox;
	GtkWidget *label;
	struct net_page *np;
};

static void destroy_refresh(GtkWidget *w)
{
	struct auto_ref *ref;
	ref = gtk_object_get_user_data(GTK_OBJECT(w));
	gtk_widget_destroy(ref->window);
	g_free(ref);
}

int do_refresh(void *data)
{
	struct net_page *np;
	fprintf(stderr, "Refreshing!\n");
	np = current_page;
	current_page = data;
	refresh_page();
	current_page = np;
	return 0;
}

static void click_refresh_box(GtkWidget *w)
{
	struct auto_ref *ref;
	ref = gtk_object_get_user_data(GTK_OBJECT(w));
	if (GTK_TOGGLE_BUTTON(ref->checkbox)->active) {
		gtk_widget_set_sensitive(ref->timebox, TRUE);
		gtk_widget_set_sensitive(GTK_BIN(ref->checkbox)->child, TRUE);
		gtk_widget_set_sensitive(ref->label, TRUE);
	} else {
		gtk_widget_set_sensitive(ref->timebox, FALSE);
		gtk_widget_set_sensitive(GTK_BIN(ref->checkbox)->child, FALSE);
		gtk_widget_set_sensitive(ref->label, FALSE);
	}
}
static void okay_refresh(GtkWidget *w)
{
	struct auto_ref *ref;
	ref = gtk_object_get_user_data(GTK_OBJECT(w));
	if (ref->np->refid > -1) {
		gtk_timeout_remove(ref->np->refid);
		ref->np->refid = -1;
	}
	if (GTK_TOGGLE_BUTTON(ref->checkbox)->active) {
		ref->np->refsec = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(ref->timebox)) * 60;
	} else {
		ref->np->refsec = 0;
	}
	if (ref->np->refsec) 
		ref->np->refid = gtk_timeout_add(ref->np->refsec * 1000, do_refresh, NULL);		

	destroy_refresh(w);
}

static void auto_refresh()
{
	struct auto_ref *ref;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *bbox;
	GtkWidget *ok;
	GtkWidget *cancel;
	GtkObject *adj;
	adj = gtk_adjustment_new(5.0, 1.0, 60.0, 1.0, 1.0, 1.0);
	fprintf(stderr, "Refresh: %d secs, id = %d\n", current_page->refsec, current_page->refid);
	ref = g_new0(struct auto_ref, 1);
	ref->np = current_page;
	ref->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_realize(ref->window);
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(ref->window), vbox);
	gtk_widget_show(vbox);
	hbox = gtk_hbox_new(FALSE, 5);
	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(hbox);
	gtk_widget_show(bbox);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);
	ok = gtk_button_new_with_label("  OK  ");
	cancel = gtk_button_new_with_label("Cancel");
	gtk_widget_show(ok);
	gtk_widget_show(cancel);
	gtk_box_pack_end(GTK_BOX(bbox), cancel, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), ok, FALSE, FALSE, 5);
	ref->checkbox = gtk_check_button_new_with_label("Auto-refresh this page every");
	gtk_widget_show(ref->checkbox);
	gtk_box_pack_start(GTK_BOX(hbox), ref->checkbox, FALSE, FALSE, 5);
	ref->timebox = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 5.0, 0);
	gtk_widget_show(ref->timebox);
	gtk_box_pack_start(GTK_BOX(hbox), ref->timebox, FALSE, FALSE, 5);
	fix_icon(ref->window->window);
	ref->label = gtk_label_new(" minutes");
	gtk_widget_show(ref->label);
	if (current_page->refsec) {
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(ref->timebox), (double)(current_page->refsec / 60));
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(ref->checkbox), TRUE);
	} else {
		gtk_widget_set_sensitive(ref->timebox, FALSE);
		gtk_widget_set_sensitive(GTK_BIN(ref->checkbox)->child, FALSE);
		gtk_widget_set_sensitive(ref->label, FALSE);
	}
	gtk_box_pack_start(GTK_BOX(hbox), ref->label, FALSE, FALSE, 5);
	gtk_window_set_title(GTK_WINDOW(ref->window), "Auto Refresh");
	gtk_object_set_user_data(GTK_OBJECT(ref->window), ref);
	gtk_object_set_user_data(GTK_OBJECT(ok), ref);
	gtk_object_set_user_data(GTK_OBJECT(cancel), ref);
	gtk_object_set_user_data(GTK_OBJECT(ref->checkbox), ref);
	gtk_signal_connect(GTK_OBJECT(ref->window), "delete_event", GTK_SIGNAL_FUNC(destroy_refresh), NULL);
	gtk_signal_connect(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(destroy_refresh), NULL);
	gtk_signal_connect(GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(okay_refresh), NULL);
	gtk_signal_connect(GTK_OBJECT(ref->checkbox), "clicked", GTK_SIGNAL_FUNC(click_refresh_box), NULL);
	gtk_widget_show(ref->window);
	
}

static void make_load(GtkWidget *blah, GtkWidget *w)
{
	char *fn;
	fn = gtk_file_selection_get_filename(GTK_FILE_SELECTION(w));
	if (strlen(fn)) {
		while(current_page)
		{
			remove_page();
		}	
		do_load(fn);
		gtk_widget_destroy(w);
	}
}

static void cancel_saveas(GtkWidget *w, GtkWidget *w2)
{
	gtk_widget_destroy(w2);
}


static void do_saveas()
{
	GtkWidget *fs;
	fs = gtk_file_selection_new("Save As...");
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", GTK_SIGNAL_FUNC(make_saveas), fs);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button), "clicked", GTK_SIGNAL_FUNC(cancel_saveas), fs);
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), defname);
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(fs));
	gtk_widget_show(fs);
}


static void do_open()
{
	GtkWidget *fs;
	fs = gtk_file_selection_new("Open...");
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button), "clicked", GTK_SIGNAL_FUNC(make_load), fs);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button), "clicked", GTK_SIGNAL_FUNC(cancel_saveas), fs);
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), defname);
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(fs));
	gtk_widget_show(fs);
}

struct net_object *get_object(struct net_page *page, unsigned int addr)
{
	struct net_object *no;
	if (!valid_np(page))
		return NULL;
	no = page->objs;
	while(no) {
		if (no->ip_addr == addr)
			break;
		no = no->next;
	}
	return no;
}

void do_quit()
{
	save_geometry();
	exit(0);
}

void object_link(struct net_page *np, struct net_object *no, struct net_object *no2)
{
	struct link *l1, *l2;
#ifdef USE_CANVAS
	GnomeCanvasPoints *pnts;
	double x1, y1, x2, y2;
#endif
	if (no == no2) {
#if 0
		printf("Being asked to link an object to itself (%s)!\n", no->hostname);
#endif
		return;
	}
	l1 = g_new0(struct link, 1);
#ifdef USE_CANVAS
	pnts = gnome_canvas_points_new(2);
	gnome_canvas_item_get_bounds(no->icon, &x1, &y1, &x2, &y2);
	pnts->coords[0] = x1 + (x2 - x1)/2;
	pnts->coords[1] = y1 + (y2 - y1)/2;
	gnome_canvas_item_get_bounds(no2->icon, &x1, &y1, &x2, &y2);
	pnts->coords[2] = x1 + (x2 - x1)/2;
	pnts->coords[3] = y1 + (y2 - y1)/2;
	l1->linkw = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS(np->canvas)),
                                       GNOME_TYPE_CANVAS_LINE,
				       "points", pnts,
                                       "fill_color", "black",
                                       "width_units", 1.0,
                                       NULL);
	gnome_canvas_item_lower_to_bottom(l1->linkw);
#else
	l1->linkw = gtk_link_new();
#endif
	l1->other = no2;
	l1->page = np;
	l1->next = no->links;
	no->links = l1;
	l2 = g_new0(struct link, 1);
	l2->linkw = l1->linkw;
	l2->other = no;
	l2->page = np;
	l2->next = no2->links;
	no2->links = l2;
#ifndef USE_CANVAS
	gtk_widget_show(l2->linkw);
	gtk_fixed_put(GTK_FIXED(np->fixed), l1->linkw, 500, 500);
	gtk_link_set_coords(GTK_LINK(l1->linkw),
			    GTK_WIDGET(no->eventbox)->allocation.x + GTK_WIDGET(no->eventbox)->allocation.width/2,
			    GTK_WIDGET(no->eventbox)->allocation.y + GTK_WIDGET(no->eventbox)->allocation.height/2,
			    GTK_WIDGET(no2->eventbox)->allocation.x + GTK_WIDGET(no2->eventbox)->allocation.width/2,
			    GTK_WIDGET(no2->eventbox)->allocation.y + GTK_WIDGET(no2->eventbox)->allocation.height/2);
#endif
}


char *get_pm_name(char *fn, char *extn)
{
	static char filename[256];
	char *c;
	strncpy(filename, fn, sizeof(filename));
	if ((c = strstr(filename, ".xpm"))) {
		strncpy(c, extn, sizeof(filename) - (c - filename));
	}
	return filename;
}

#ifndef USE_GNOME
struct pixmap_cache *get_pixmap(char *filename)
{
	struct pixmap_cache *tmp;
	GtkStyle *style = gtk_widget_get_default_style();
	tmp = pc;
	while(tmp) {
		if (!strcmp(filename,tmp->filename)) {
			break;
		}
		tmp=tmp->next;
	}
	if (!tmp) {
		tmp = g_new0(struct pixmap_cache, 1);
		strncpy(tmp->filename, filename, sizeof(tmp->filename));
		tmp->pixmap = gdk_pixmap_create_from_xpm (main_window.window->window, &tmp->mask, 
					   &style->bg[GTK_STATE_NORMAL],
					   find_file(filename));
		tmp->next = pc;
		pc = tmp;
	}
	return tmp;
}
#else
struct pixmap_cache_r *get_pixmap_r(char *filename)
{
	struct pixmap_cache_r *tmp;
	GtkStyle *style = gtk_widget_get_default_style();
	tmp = (struct pixmap_cache_r *)pc;
	while(tmp) {
		if (!strcmp(filename,tmp->filename)) {
			break;
		}
		tmp=(struct pixmap_cache_r *)(tmp->next);
	}
	if (!tmp) {
		tmp = g_new0(struct pixmap_cache_r, 1);
		strncpy(tmp->filename, filename, sizeof(tmp->filename));
		tmp->pixmap = gdk_pixmap_create_from_xpm (main_window.window->window, &tmp->mask, 
					   &style->bg[GTK_STATE_NORMAL],
					   find_file(filename));
		tmp->next = pc;
		pc = (struct pixmap_cache *)tmp;
	}
	return tmp;
}
#endif



void fix_tooltip(struct net_object *no)
{
	char buf[1024];
#ifdef SNMP
	char buf2[1024];
#endif
	struct in_addr ia;
	struct net_page *np=NULL;
	GdkPixmap *pm;
	GdkBitmap *bm;
	struct pixmap_cache_r *tmp;

#ifdef USE_CANVAS
	if (no->icon->parent)
#else
	if (no->eventbox->parent)
#endif
		np = no->np;

	ia.s_addr = no->ip_addr;
#ifdef SNMP
	if (no->hassnmp) {
		strncpy(buf, "SNMP available\n", sizeof(buf));
		if (strlen(no->sysName)) {
			g_snprintf(buf2, sizeof(buf2), "Name: %s %s\n", no->sysName, print_aliases(no, '\n'));
		} else {
			g_snprintf(buf2, sizeof(buf2), "Name: %s %s\n", no->hostname, print_aliases(no, '\n'));
		}
		strncat(buf, buf2, sizeof(buf) - strlen(buf));
		g_snprintf(buf2, sizeof(buf2), "IP: %s\n", inet_ntoa(ia));
		strncat(buf, buf2, sizeof(buf) - strlen(buf));
		if (strlen(no->sysDescr)) {
			g_snprintf(buf2, sizeof(buf2), "Description: %s\n", no->sysDescr);
		} else {
			g_snprintf(buf2, sizeof(buf2), "OS: %s\n", no->os);
		}
		strncat(buf, buf2, sizeof(buf) - strlen(buf));
		if (strlen(no->sysContact)) {
			g_snprintf(buf2, sizeof(buf2), "Contact: %s\n", no->sysContact);
			strncat(buf, buf2, sizeof(buf) - strlen(buf));
		}
	} else
#endif
		g_snprintf(buf,sizeof(buf), "Name: %s\n%sIP: %s\nOS: %s\n", no->hostname, print_aliases(no, '\n'),inet_ntoa(ia), no->os + 2);

	if (valid_np(np)) {
		gtk_clist_freeze(GTK_CLIST(np->list));
		if (no->id < 0)
			no->id = gtk_clist_append(GTK_CLIST(np->list), NULL);
		if (gtk_clist_get_pixmap(GTK_CLIST(np->list), no->id, 0, &pm, &bm)) 
			gtk_clist_set_pixtext(GTK_CLIST(np->list), no->id, 0, no->hostname, 4, pm, bm);
	 	else {
			tmp = get_pixmap_r(get_pm_name(no->fn, "-tiny.xpm"));
			gtk_clist_set_pixtext(GTK_CLIST(np->list), no->id, 0, no->hostname, 4, tmp->pixmap, tmp->mask);
		}
		gtk_clist_set_text(GTK_CLIST(np->list), no->id, 1, inet_ntoa(ia));
		gtk_clist_set_text(GTK_CLIST(np->list), no->id, 2, no->os + 2);
		gtk_clist_set_text(GTK_CLIST(np->list), no->id, 3, print_aliases(no, ' '));
		gtk_clist_thaw(GTK_CLIST(np->list));
	}
#ifdef USE_CANVAS
	/* fprintf(stderr, "I'd set the tooltip to '%s' if I knew how\n", buf); */
#else
	gtk_tooltips_set_tip(main_window.tips, no->eventbox, buf, NULL);
#endif
}

static void grow_pixmaps()
{
	struct net_object *no;
	if (!current_page->small_icons)
		return;
	current_page->icon_height *= 2;
	current_page->icon_width *=2;
	current_page->small_icons=0;
	no = current_page->objs;
	while(no) {
		set_icon(no, no->fn);
		check_links(no);
		fix_label(no);
		no = no->next;
	}
}

static void view_pictures(GtkWidget *w, void *data)
{
	if (data)
		shrink_pixmaps();
	else
		grow_pixmaps();
	gtk_notebook_set_page(GTK_NOTEBOOK(current_page->notebook), 0);
}

static void view_list(GtkWidget *w)
{
	gtk_notebook_set_page(GTK_NOTEBOOK(current_page->notebook), 1);
}

static int event_dispatch(GtkObject *thing, GdkEvent *event, struct net_object *no)
{
	/* General object event dispatcher */
	struct net_page *np = no->np;
#ifdef USE_CANVAS
	double newx, newy;
	GdkEventMotion *em = (GdkEventMotion *)event;
#else
	int newx, newy;
#endif
	struct pcache *pc;
	GdkEventButton *eb = (GdkEventButton *)event;
	switch(event->type) {
	case GDK_BUTTON_PRESS:
		if (valid_np(np)) {
			if (np->sel && (np->sel != no)) {
				highlight_object(np->sel, HIGHLIGHT_NORMAL);
			}
			np->sel = no;
			highlight_object(np->sel, HIGHLIGHT_SELECTED);
			no->drag = 2;
			set_infobox(no);
		}
		if (eb->button == 1) {
			if (event->type == GDK_2BUTTON_PRESS) {
				printf("Double click!\n");
			} else {
#ifdef USE_CANVAS
				no->xo = eb->x;
				no->yo = eb->y;
				gnome_canvas_item_raise_to_top(no->otherbox);
				gnome_canvas_item_raise_to_top(no->icon);
				gnome_canvas_item_grab ((GnomeCanvasItem*)thing,
							GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
							NULL,
							eb->time);			;
#else
				gtk_widget_get_pointer(np->fixed, &(no->xo), &(no->yo));
#endif			
				no->drag = 1;
			}
		} else 
			show_service_menu(no, eb);
		break;
	case GDK_MOTION_NOTIFY:
		if (no->drag == 1) {
#ifdef USE_CANVAS
			newx = (int)em->x;
			newy = (int)em->y;
#else
			gtk_widget_get_pointer(np->fixed, &newx, &newy);
#endif			
			if (newx < 0) newx=0;
			if (newy < 0) newy=0;
			move_object(no, newx - no->xo, newy - no->yo);
			no->xo = newx;
			no->yo = newy;
#ifdef USE_CANVAS
			gnome_canvas_item_get_bounds(no->icon, &newx, &newy, NULL, NULL);
#else
			newx = no->eventbox->allocation.x;
			newy = no->eventbox->allocation.y;
#endif
			pc = get_cache(np, no->ip_addr);
			pc->x = (int)newx;
			pc->y = (int)newy;
			if (!option_update_release)
				check_links(no);
		}
		break;
	case GDK_BUTTON_RELEASE:
#ifdef USE_CANVAS
		gnome_canvas_item_ungrab ((GnomeCanvasItem*)thing, eb->time);
#endif
		check_links(no);
		no->drag = 0;
		break;
	default:
		return FALSE;
	}
	if (thing)
		gtk_signal_emit_stop_by_name(thing, "event");
	return TRUE;
}

#ifdef USE_CANVAS

/* Canvas specific routines */
static void make_label(struct net_object *no) {
	#define gray50_width 2
	#define gray50_height 2
	static char gray50_bits[] = {
  		0x02, 0x01, };
	static GdkBitmap *stipple=NULL;
	if (!stipple)
		stipple = gdk_bitmap_create_from_data (NULL, gray50_bits, gray50_width, gray50_height);
	
	no->label=gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(no->np->canvas)),
					  GNOME_TYPE_CANVAS_TEXT, 
					 "anchor", GTK_ANCHOR_NW,
#ifdef OBJECT_FONT			
					  "font", OBJECT_FONT,
#endif
					  NULL);
	no->otherbox=gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(no->np->canvas)),
					   GNOME_TYPE_CANVAS_GROUP, NULL);
	no->rect = gnome_canvas_item_new(GNOME_CANVAS_GROUP(no->otherbox), GNOME_TYPE_CANVAS_RECT,
					 "fill_color", "white",
					 "outline_color", "black",
					 "fill_stipple", stipple,
					 "width_pixels", 1,
					 NULL);
	gnome_canvas_item_reparent(no->label, GNOME_CANVAS_GROUP(no->otherbox));
	gtk_object_set_user_data(GTK_OBJECT(no->label), no);
	gtk_signal_connect(GTK_OBJECT(no->label), "event",
			   GTK_SIGNAL_FUNC (event_dispatch), no);
}

static void make_icon(struct net_object *no) {
	no->icon = gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(no->np->canvas)),
					 gnome_canvas_image_get_type(),
					 "anchor", GTK_ANCHOR_NW,
					 NULL);
	gtk_object_set_user_data(GTK_OBJECT(no->icon), no);
	gtk_signal_connect(GTK_OBJECT(no->icon), "event",
			   GTK_SIGNAL_FUNC (event_dispatch), no);
}

#else

/* GTK routines */
static void make_label(struct net_object *no)
{
	no->otherbox = gtk_event_box_new();
	no->label = gtk_label_new("");
	gtk_object_set_user_data(GTK_OBJECT(no->otherbox), no);
	gtk_signal_connect(GTK_OBJECT(no->otherbox), "event",
			   GTK_SIGNAL_FUNC (event_dispatch), no);
	no->pixmap=NULL;
#if 0
	gtk_container_border_width(GTK_CONTAINER(no->otherbox), 5);
#endif
	gtk_container_add(GTK_CONTAINER(no->otherbox), no->label);
	gtk_widget_show(no->label);
	gtk_widget_show(no->otherbox);
}

static void make_icon(struct net_object *no)
{
	no->eventbox = gtk_event_box_new();
	gtk_object_set_user_data(GTK_OBJECT(no->eventbox), no);
	gtk_signal_connect(GTK_OBJECT(no->eventbox), "event",
			   GTK_SIGNAL_FUNC (event_dispatch), no);
	gtk_widget_show(no->eventbox);
}

#endif
struct net_object *network_object(struct net_page *np)
{
	struct net_object *no;
	no = g_new0(struct net_object, 1);

	memset(no, 0, sizeof(struct net_object));

	no->np = np;
	no->id=-1;
	make_label(no);
	make_icon(no);
	return no;
	
}

static int page_dispatch(GtkObject *thing, GdkEvent *event, struct net_page *np)
{
#ifndef GNOME
#ifdef USE_ITEM
	GtkWidget *pagemenu;
#else
	static GtkMenuPath *gmp;
#endif
#endif
	switch(event->type) {
	case GDK_2BUTTON_PRESS:
		printf("Double click!\n");
		break;
	case GDK_BUTTON_PRESS:
		if (!valid_np(np))
			return FALSE;
		if (np->sel)
			highlight_object(np->sel, HIGHLIGHT_NORMAL);
		set_infobox(NULL);
		np->sel = NULL;
#ifndef USE_CANVAS
		if (event->button.button != 1) {
#ifdef USE_ITEM
			pagemenu = gtk_item_factory_get_widget(main_window.menuf, "/Viewspace");
			gtk_menu_popup(GTK_MENU(pagemenu), NULL, NULL, NULL, NULL, event->button.button, event->button.time);
#else
			gmp = gtk_menu_factory_find(main_window.menuf, "<Main>/Viewspace");
			gtk_menu_popup(GTK_MENU(GTK_MENU_ITEM(gmp->widget)->submenu), NULL, NULL, NULL, NULL, event->button.button, event->button.time);
#endif
		}
#endif
		break;
	case GDK_MOTION_NOTIFY:
		if (np->sel)
			event_dispatch(NULL, (GdkEvent *)event, np->sel);
	default:
	}
	return FALSE;
}

void click_column(GtkWidget *w, GdkEventButton *event, struct net_page *np)
{
	int row=-1;
	int column=-1;
	struct net_object *no;
	if (event->window != GTK_CLIST(w)->clist_window)
		return;
	if (event->button < 2)
		return;
	gtk_clist_get_selection_info(GTK_CLIST(w), event->x, event->y, &row, &column);
	if (row > -1) {
		gtk_clist_select_row(GTK_CLIST(w), row, -1);
		no = (struct net_object *)gtk_clist_get_row_data(GTK_CLIST(w), row);
		show_service_menu(no, event);
	}
}

#ifdef USE_CANVAS
static void make_canvas(struct net_page *np)
{
#ifdef ANTI_ALIAS
	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
#else
	gtk_widget_push_visual (gdk_imlib_get_visual ());
	gtk_widget_push_colormap (gdk_imlib_get_colormap ());
#endif
	np->canvas = gnome_canvas_new();
	gtk_widget_pop_visual();
	gtk_widget_pop_colormap();
	gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(np->canvas), 1.0);
#ifdef ANTI_ALIAS
	GNOME_CANVAS(np->canvas)->aa = 1;
#endif
	gtk_widget_set_usize(np->canvas, FIXED_WIDTH, FIXED_HEIGHT);
	gnome_canvas_set_scroll_region(GNOME_CANVAS(np->canvas), 0.0, 0.0, (double)FIXED_WIDTH, (double)FIXED_HEIGHT);
	np->eventbox = np->canvas;
	np->pane = gtk_scrolled_window_new(GTK_LAYOUT(np->canvas)->hadjustment,
	                                   GTK_LAYOUT(np->canvas)->vadjustment);
	gtk_object_set_user_data(GTK_OBJECT(np->canvas), np);
	gtk_widget_show(np->canvas);
}
#else
static void make_canvas(struct net_page *np)
{
	np->fixed = gtk_fixed_new();
	gtk_widget_set_usize(np->fixed, FIXED_WIDTH, FIXED_HEIGHT);
	gtk_container_add(GTK_CONTAINER(np->eventbox), np->fixed);
	np->pane = gtk_scrolled_window_new(NULL, NULL);
	gtk_object_set_user_data(GTK_OBJECT(np->fixed), np);
	gtk_widget_show(np->fixed);
}
#endif

struct net_page *add_network(struct net_win *n, char *title)
{
	struct net_page *np;
	static char *titles[] = { "Hostname", "IP", "OS", "Aliases" };
	
	GtkWidget *label, *label2;

#if (GTK_MINOR_VERSION > 1) || ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 4))
	/* GTK 1.1.5 handles scrolled widgets differently and requires labels for
	   pages or it will subtly crash */
	GtkWidget *sw;
	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
#endif 
	label = gtk_label_new("Icon View");
	label2 = gtk_label_new("List View");
	gtk_widget_show(label);
	gtk_widget_show(label2);
	np = g_new0(struct net_page, 1);
	np->label = gtk_label_new(title);
	np->eventbox = gtk_event_box_new();

	make_canvas(np);

	np->list = gtk_clist_new_with_titles(4, titles);
	np->notebook = gtk_notebook_new();
	np->ncontents = NULL;
	np->dcontents = NULL;
	np->lastx = 20;
	np->lasty = 20;
	np->objs = NULL;
	np->refsec = 0;
	np->refid = -1;
	np->last = NULL;
	np->laststr[0]='\0';
	np->lasthand = NULL;
	np->lastid = 0;
	np->pending = NULL;
	np->automap = 0;
	np->pos = NULL;
	np->pe = NULL;
	np->sel = NULL;
	np->small_icons = 0;
	np->icon_height = DEFAULT_ICON_HEIGHT;
	np->icon_width = DEFAULT_ICON_WIDTH;
	
#if ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 2))  || (GTK_MAJOR_VERSION > 1)
	gtk_clist_set_column_auto_resize(GTK_CLIST(np->list), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(np->list), 1, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(np->list), 2, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(np->list), 3, TRUE);
#else
	gtk_clist_set_column_width(GTK_CLIST(np->list), 0,200);
	gtk_clist_set_column_width(GTK_CLIST(np->list), 1,100);
	gtk_clist_set_column_width(GTK_CLIST(np->list), 2,300);
	gtk_clist_set_column_width(GTK_CLIST(np->list), 3,200);
#endif	
	gtk_notebook_append_page(GTK_NOTEBOOK(np->notebook), np->pane, label);
#if ((GTK_MINOR_VERSION > 1) || ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 4))) && !(defined USE_CANVAS)
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(np->pane),
					      np->eventbox);
	gtk_container_add(GTK_CONTAINER(sw), np->list);
	gtk_object_set_user_data(GTK_OBJECT(sw), np);
	gtk_notebook_append_page(GTK_NOTEBOOK(np->notebook), sw, label2);
#else
	gtk_container_add(GTK_CONTAINER(np->pane), np->eventbox);
	gtk_notebook_append_page(GTK_NOTEBOOK(np->notebook), np->list, label2);
#endif
	gtk_widget_show(np->label);
	gtk_widget_show(np->eventbox);
	gtk_widget_show(np->list);
	gtk_widget_show(np->pane);
	gtk_widget_show(np->notebook);
	gtk_object_set_user_data(GTK_OBJECT(np->pane), np);
	gtk_object_set_user_data(GTK_OBJECT(np->eventbox), np);
	gtk_object_set_user_data(GTK_OBJECT(np->notebook), np);
	gtk_object_set_user_data(GTK_OBJECT(np->list), np);
	gtk_signal_connect(GTK_OBJECT(np->list), "button_press_event", GTK_SIGNAL_FUNC(click_column), np);
	gtk_signal_connect_after(GTK_OBJECT(np->eventbox), "event", GTK_SIGNAL_FUNC(page_dispatch), np);
	gtk_notebook_set_page(GTK_NOTEBOOK(np->notebook), 0);
	gtk_notebook_append_page(GTK_NOTEBOOK(n->notebook), np->notebook, np->label);
	gtk_widget_realize(np->notebook);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(np->notebook), FALSE);
	gtk_notebook_set_show_border(GTK_NOTEBOOK(np->notebook), FALSE);
	while(gtk_events_pending())
		gtk_main_iteration();
	return np;
}

static void cancel_page(GtkWidget *w, GtkWidget *win)
{
	gtk_widget_destroy(win);
}

static void reverse_dns()
{
	struct net_object *no;
	char *name;
	struct hostent *hp;
	struct in_addr ia;
	if (!current_page)
		return;
	set_status("Doing Reverse DNS...");
	immediate_setcursor(GDK_WATCH);
	no = current_page->objs;
	while(no) {
		ia.s_addr = no->ip_addr;
		/* Don't bother if it already is different */
		if (strcmp((name=inet_ntoa(ia)), no->hostname)) {
			no = no->next;
			continue;
		}
		hp = gethostbyaddr((char *)&no->ip_addr, sizeof(no->ip_addr), AF_INET);
		if (hp) 
			name = hp->h_name;
		strncpy(no->hostname, name, sizeof(no->hostname));
		fix_label(no);
		fix_tooltip(no);
		no = no->next;
	}
	gdk_window_set_cursor(main_window.window->window, NULL);
	set_status("Reverse DNS lookups complete");
}

static void make_page(GtkWidget *w, GtkWidget *dialog)
{
	char *c;
	GtkWidget *entry;
	entry = GTK_WIDGET(gtk_object_get_user_data(GTK_OBJECT(w)));
	c = gtk_entry_get_text(GTK_ENTRY(entry));
	if (strlen(c)) {
		add_network(&main_window, c);
	}
	gtk_widget_destroy(dialog);	
}

static void do_map(void *data)
{
	struct net_object *no;
	int style = (data - NULL);
	if (!current_page)
		return;
	no = current_page->objs;
	current_page->automap = 1;
	while(no) {
		if ((style == 2)) no->mapped = 0;
		if ((style == 1) && (no->links == NULL)) no->mapped = 0;
		if (!no->mapped)
			start_mapping(no);
		no = no->next;
	}
}

static void do_page()
{
	GtkWidget *dialog;
	GtkWidget *ok;
	GtkWidget *cancel;
	GtkWidget *label;
	GtkWidget *hbox;
	GtkWidget *entry;
	dialog = gtk_dialog_new();
	hbox = gtk_hbox_new(FALSE, 5);
	entry = gtk_entry_new();
	label = gtk_label_new("New page name:");
	cancel = gtk_button_new_with_label("Cancel");
	ok = gtk_button_new_with_label("OK");
	gtk_window_set_title(GTK_WINDOW(dialog), "Add Viewspace");
	gtk_object_set_user_data(GTK_OBJECT(ok), entry);
	gtk_object_set_user_data(GTK_OBJECT(entry), entry);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);
	gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 5);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), hbox, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), ok, TRUE,
                           TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), cancel, TRUE,
                           TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(make_page), dialog);
	gtk_signal_connect(GTK_OBJECT(entry), "activate", GTK_SIGNAL_FUNC(make_page), dialog);
	gtk_signal_connect(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(cancel_page), dialog);
	gtk_widget_show(hbox);
	gtk_widget_show(entry);
	gtk_widget_show(label);
	gtk_widget_show(cancel);
	gtk_widget_show(ok);
	gtk_widget_grab_focus(entry);
	gtk_widget_show(dialog);
}

void destroy_object(struct net_page *np, struct net_object *no)
{
	struct link *l, *l2;
	struct alias *a, *a2;
	struct monitor_tag *mt, *mt2;
	/* destroy graphical representation */
#ifdef USE_CANVAS
	gtk_object_destroy(GTK_OBJECT(no->icon));
	gtk_object_destroy(GTK_OBJECT(no->otherbox));
#else
	gtk_widget_destroy(no->eventbox);
	gtk_widget_destroy(no->otherbox);
#endif
#ifdef SNMP
	if (no->ss)
		do_snmp_close(no->ss);
#endif
	mt = no->mon;
	while(mt) {
		mt2 = mt;
		mt = mt->next;
		destroy_tag(mt2);	
	}
	l = no->links;
	while(l) {
		l2 = l;
		l = l->next;
		/* Only other objects should unlink */
		if (l2->other < no)
			DESTROY_LINK(l2->linkw);
		g_free(l2);
	}
	a = no->aliases;
	while(a) {
		a2 = a;
		a=a->next;
		g_free(a2);
	}
	g_free(no);
}

static void clear_page()
{
	struct net_object *no;
	if (!current_page)
		return;
	no = current_page->objs;
	while(no) {
		destroy_object(current_page, no);
		no = no->next;
	}
	gtk_clist_clear(GTK_CLIST(current_page->list));
	current_page->lastx=20;
	current_page->lasty=20;
	current_page->objs=NULL;
	current_page->sel = NULL;
}

void reload_page()
{
	clear_page();
	refresh_page();
}

static void unmap_page()
{
	struct net_object *no;
	struct link *l, *l2;
	if (!current_page)
		return;
	no = current_page->objs;
	while(no) {
		l = no->links;
		while(l) {
			l2 = l;
			l = l->next;
			/* Only other objects should unlink */
			if (l2->other < no)
				DESTROY_LINK(l2->linkw);
			g_free(l2);
		}
		no->links = NULL;
		no->mapped = 0;
		no = no->next;
	}
	current_page->automap = 0;

}

void show_hide_verbose(int force)
{
	if (current_page) {
		if (option_verbose_display) {
			gtk_widget_show(main_window.infobox);
			if (force || !main_window.container) {
				if (main_window.container) {
					delete_edit(main_window.container);
				}	
				main_window.container = gtk_vbox_new(FALSE, 0);
				gtk_widget_show(main_window.container);
				gtk_box_pack_start(GTK_BOX(main_window.cvbox), main_window.container, TRUE, TRUE, 0);
				build_edit(current_page, main_window.container);
			}
		} else {
			gtk_widget_hide(main_window.infobox);
			if (main_window.container) {
				delete_edit(main_window.container);
			}	
			main_window.container = NULL;
		}
	}
}

static void page_click(GtkWidget *w, GtkNotebookPage *page)
{
	struct net_page *np;
	char buf[256];
	int cnt=0;
	struct net_object *no;
	np = (struct net_page *)gtk_object_get_user_data(GTK_OBJECT(page->child));
	if (np->sel && np->sel->drag)
		return;
	if (np && !current_page)
		show_page_menu(&main_window);
	if (np) {
		current_page = np;
		show_hide_verbose(1);
	} else
		printf("np is null!\n");
	no = np->objs;
	while(no) {
		cnt++;
		no=no->next;
	}
	g_snprintf(buf, sizeof(buf), "%d host%s", cnt, (cnt == 1) ? "" : "s");
	set_status(buf);
}

static void show_page_menu(struct net_win *n)
{
#ifdef USE_ITEM
	GtkWidget *menu;
	menu = gtk_item_factory_get_widget(n->menuf, "/Viewspace");
	if (GTK_MENU(menu)->parent_menu_item)
		gtk_widget_set_sensitive(GTK_MENU(menu)->parent_menu_item, TRUE);
#else
	GtkMenuPath *gmp;
	gmp = gtk_menu_factory_find(n->menuf, "<Main>/Viewspace");
	gtk_widget_set_sensitive(gmp->widget, TRUE);
#endif
}

static void hide_page_menu(struct net_win *n)
{
#ifdef USE_ITEM
	GtkWidget *menu;
	menu = gtk_item_factory_get_widget(n->menuf, "/Viewspace");
	if (GTK_MENU(menu)->parent_menu_item)
		gtk_widget_set_sensitive(GTK_MENU(menu)->parent_menu_item, FALSE);
#else
	GtkMenuPath *gmp;
	gmp = gtk_menu_factory_find(n->menuf, "<Main>/Viewspace");
	gtk_widget_set_sensitive(gmp->widget, FALSE);
#endif
}

static void do_edit()
{
	if (!current_page)
		return;
	show_edit(current_page);
}

void fix_icon(GdkWindow *w)
{
	GdkPixmap *gdk_pixmap;
	GdkBitmap *gdk_pixmap_mask;
	gdk_pixmap = gdk_pixmap_create_from_xpm(w, &gdk_pixmap_mask, NULL, find_file("icon.xpm"));
	gdk_window_set_icon(w, NULL, gdk_pixmap, gdk_pixmap_mask);
}

static void show_window(struct net_win *n)
{
	static GdkColor *yellow=NULL;
#ifdef USE_ITEM
	/* you gtk bastards can't keep any backwards compatibility can you? */
	GtkAccelGroup *grp;
	static GtkItemFactoryEntry menu_items[]= {
		{ "/_File",	NULL,	NULL,	0, 	"<Branch>" },
		{ "/File/_New viewspace...", NULL, GTK_MENU_FUNC(do_page), 0 },
		{ "/File/_Open...", "<control>O", GTK_MENU_FUNC(do_open), 0 },
		{ "/File/_Save",  "<control>S", GTK_MENU_FUNC(do_save), 0 },
		{ "/File/Save _as...", NULL, GTK_MENU_FUNC(do_saveas), 0 },
		{ "/File/sep1", NULL, NULL, 0, "<Separator>" },
		{ "/File/O_ptions", NULL, GTK_MENU_FUNC(show_options), 0 },
		{ "/File/sep2", NULL, NULL, 0, "<Separator>" },
		{ "/File/_Quit", "<control>Q", GTK_MENU_FUNC(do_quit), 0 },
		{ "/_Viewspace", 	NULL,  NULL, 0, "<Branch>" },
		{ "/Viewspace/_New...", "<control>N", GTK_MENU_FUNC(do_page), 0 },
		{ "/Viewspace/_Edit...", "<control>E", GTK_MENU_FUNC(do_edit), 0 },
		{ "/Viewspace/_Remove", NULL, GTK_MENU_FUNC(remove_page), 0 },
		{ "/Viewspace/sep1", NULL, NULL, 0, "<Separator>" },
		{ "/Viewspace/Add _host...", "<control>H", GTK_MENU_FUNC(do_network), 0 },
		{ "/Viewspace/Add n_etwork...", "<control>A", GTK_MENU_FUNC(do_network), 1 },
		{ "/Viewspace/Add _domain...", "<control>D", GTK_MENU_FUNC(do_network), 2 },
		{ "/Viewspace/sep2", NULL, NULL, 0, "<Separator>" },
		{ "/Viewspace/_View", NULL, NULL, 0, "<Branch>" },
		{ "/Viewspace/_View/_List", NULL, GTK_MENU_FUNC(view_list), 0 },
		{ "/Viewspace/View/_Icons", NULL, GTK_MENU_FUNC(view_pictures), 0},
		{ "/Viewspace/View/_Small Icons", NULL, GTK_MENU_FUNC(view_pictures), 1 },
		{ "/Viewspace/_Arrange", NULL, NULL, 0, "<Branch>" },
#if 0
		{ "/Viewspace/Arrange/By _map", NULL, GTK_MENU_FUNC(arrange_cb), BY_MAP },
#endif
		{ "/Viewspace/Arrange/By _domain", NULL, GTK_MENU_FUNC(arrange_cb), BY_DOMAIN },
		{ "/Viewspace/Arrange/By _hostname", NULL, GTK_MENU_FUNC(arrange_cb), BY_NAME },
		{ "/Viewspace/Arrange/By _address", NULL, GTK_MENU_FUNC(arrange_cb), BY_ADDR },
		{ "/Viewspace/Arrange/By _type", NULL, GTK_MENU_FUNC(arrange_cb), BY_OS },
		{ "/Viewspace/_Map", NULL, NULL, 0, "<Branch>" },
		{ "/Viewspace/Map/_Map", "<control>M", GTK_MENU_FUNC(do_map), 0 },
		{ "/Viewspace/Map/_Unlinked hosts", NULL, GTK_MENU_FUNC(do_map), 1 },
		{ "/Viewspace/Map/_Remap Everything", NULL, GTK_MENU_FUNC(do_map), 2 },
		{ "/Viewspace/Map/_Unmap", "<control>U", GTK_MENU_FUNC(unmap_page), 0 },
		{ "/Viewspace/Reverse DNS _everything", NULL, GTK_MENU_FUNC(reverse_dns), 0 },
		{ "/Viewspace/sep1", NULL, NULL, 0, "<Separator>" },
		{ "/Viewspace/_Find in viewspace...", "<control>F", GTK_MENU_FUNC(do_search), 0 },
		{ "/Viewspace/Find a_gain...", "<control>G", GTK_MENU_FUNC(search_again), 0 },
		{ "/Viewspace/sep2", NULL, NULL, 0, "<Separator>" },
		{ "/Viewspace/_Refresh","<control>R", GTK_MENU_FUNC(refresh_page), 0 },
		{ "/Viewspace/_Auto Refresh...",NULL, GTK_MENU_FUNC(auto_refresh), 0 },
		{ "/Viewspace/Re_load", "<control>L", GTK_MENU_FUNC(reload_page), 0 },
		{ "/Viewspace/_Clear", "<control>C", GTK_MENU_FUNC(clear_page), 0 },
		{ "/_View", NULL, NULL, 0, "<Branch>" },
		{ "/_View/E_vent log...", "<control>V", GTK_MENU_FUNC(event_log), 0 },
		{ "/_View/_Status...", NULL, GTK_MENU_FUNC(show_status), 0 },
		{ "/_Help", NULL, NULL, 0, "<LastBranch>" },
		{ "/_Help/About...", NULL, GTK_MENU_FUNC(show_about), 0 },
		{ "/_Help/License...", NULL, GTK_MENU_FUNC(show_gpl), 0 },
		
	};
	
#else
	GtkMenuFactory *subfactory;
	GtkMenuPath *gmp;
	static GtkMenuEntry menu_items[] = 
	{
		{"<Main>/File/New viewspace...", NULL, do_page, NULL },
		{"<Main>/File/Open...", "<control>O", GTK_MENU_FUNC(do_open), NULL },
		{"<Main>/File/Save","<control>S", GTK_MENU_FUNC(do_save), NULL },
		{"<Main>/File/Save as...", NULL, GTK_MENU_FUNC(do_saveas), NULL },
		{"<Main>/File/<separator>",NULL, NULL, NULL },
		{"<Main>/File/Options",NULL, GTK_MENU_FUNC(show_options), NULL },
		{"<Main>/File/<separator>",NULL, NULL, NULL },
		{"<Main>/File/Quit","<control>Q", do_quit, NULL },
		{"<Main>/Viewspace/New...","<control>N", do_page, NULL },
		{"<Main>/Viewspace/Edit...", "<control>E", do_edit, NULL },
		{"<Main>/Viewspace/Remove",NULL, GTK_MENU_FUNC(remove_page), NULL }, 
		{"<Main>/Viewspace/<separator>",NULL, NULL, NULL },
		{"<Main>/Viewspace/Add host...","<control>H", GTK_MENU_FUNC(do_network), (void *)0},
		{"<Main>/Viewspace/Add network...","<control>A", GTK_MENU_FUNC(do_network), (void *)1},
		{"<Main>/Viewspace/Add domain...","<control>D", GTK_MENU_FUNC(do_network), (void *)2},
		{"<Main>/Viewspace/<separator>",NULL, NULL, NULL },
		{"<Main>/Viewspace/View/List",NULL, GTK_MENU_FUNC(view_list), NULL },
		{"<Main>/Viewspace/View/Icons",NULL, GTK_MENU_FUNC(view_pictures), (void *)0 },
		{"<Main>/Viewspace/View/Small Icons",NULL, GTK_MENU_FUNC(view_pictures), (void *)1 },
#if 0
		{"<Main>/Viewspace/Arrange/By map", NULL, GTK_MENU_FUNC(arrange_cb), (void *)BY_MAP },
#endif
		{"<Main>/Viewspace/Arrange/By domain", NULL, GTK_MENU_FUNC(arrange_cb), (void *)BY_DOMAIN },
		{"<Main>/Viewspace/Arrange/By hostname", NULL, GTK_MENU_FUNC(arrange_cb), (void *)BY_NAME },
		{"<Main>/Viewspace/Arrange/By address", NULL, GTK_MENU_FUNC(arrange_cb), (void *)BY_ADDR },
		{"<Main>/Viewspace/Arrange/By type", NULL, GTK_MENU_FUNC(arrange_cb), (void *)BY_OS },
		{"<Main>/Viewspace/Map/Map","<control>M", GTK_MENU_FUNC(do_map), NULL },
		{"<Main>/Viewspace/Map/Unlinked hosts",NULL, GTK_MENU_FUNC(do_map), (void *)1 },
		{"<Main>/Viewspace/Map/Remap everything",NULL, GTK_MENU_FUNC(do_map), (void *)2 },
		{"<Main>/Viewspace/Map/Unmap","<control>U", GTK_MENU_FUNC(unmap_page), NULL },
		{"<Main>/Viewspace/Reverse DNS everything",NULL, GTK_MENU_FUNC(reverse_dns), NULL },
		{"<Main>/Viewspace/<separator>",NULL, NULL, NULL },
		{"<Main>/Viewspace/Find in viewspace...", "<control>F", GTK_MENU_FUNC(do_search), NULL},
		{"<Main>/Viewspace/Find again...", "<control>G", GTK_MENU_FUNC(search_again), NULL},
		{"<Main>/Viewspace/<separator>",NULL, NULL, NULL },
		{"<Main>/Viewspace/Refresh","<control>R", GTK_MENU_FUNC(refresh_page), NULL},
		{"<Main>/Viewspace/Auto Refresh...",NULL, GTK_MENU_FUNC(auto_refresh), NULL},
		{"<Main>/Viewspace/Reload","<control>L", GTK_MENU_FUNC(reload_page), NULL},
		{"<Main>/Viewspace/Clear","<control>C", GTK_MENU_FUNC(clear_page), NULL},
		{"<Main>/View/Event log...", "<control>V", GTK_MENU_FUNC(event_log), NULL},
		{"<Main>/View/Status...", NULL, GTK_MENU_FUNC(show_status), NULL},
		{"<Main>/Help/About...",NULL, GTK_MENU_FUNC(show_about), NULL},
		{"<Main>/Help/License...",NULL, GTK_MENU_FUNC(show_gpl), NULL},
	};
#endif
	int nmenu_items = sizeof(menu_items)/sizeof(menu_items[0]);
	
	n->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_realize(n->window);
	if (!yellow) {
		yellow = g_new0(GdkColor, 1);
		yellow->red = 65535;
		yellow->green = 65535;
		yellow->blue = 65535;
		gdk_color_alloc(gdk_window_get_colormap(n->window->window), yellow);
	}		
	n->tips = gtk_tooltips_new();
	gtk_tooltips_set_colors(n->tips, NULL, yellow);
	gtk_tooltips_enable(n->tips);
	gtk_signal_connect(GTK_OBJECT(n->window), "delete_event", GTK_SIGNAL_FUNC(do_quit), NULL);

	/* The mess of menu creation */
	
#ifdef USE_ITEM
	grp = gtk_accel_group_new();
	n->menuf = gtk_item_factory_new(GTK_TYPE_MENU_BAR,"<main>", grp);
	gtk_item_factory_create_items(n->menuf, nmenu_items, menu_items, NULL);
	gtk_accel_group_attach(grp, GTK_OBJECT(n->window));
	n->menu = n->menuf->widget;
#else /* USE_ITEM */
	n->menuf = gtk_menu_factory_new(GTK_MENU_FACTORY_MENU_BAR);
	subfactory = gtk_menu_factory_new(GTK_MENU_FACTORY_MENU_BAR);
	gtk_menu_factory_add_subfactory(n->menuf, subfactory, "<Main>");
        gtk_menu_factory_add_entries(n->menuf, menu_items, nmenu_items);
#if (GTK_MINOR_VERSION < 1)
        gtk_window_add_accelerator_table(GTK_WINDOW(n->window), subfactory->table);
#else
	gtk_window_add_accel_group(GTK_WINDOW(n->window), subfactory->accel_group);
#endif
	/* Right Justify Help Menu */
	gmp = gtk_menu_factory_find(n->menuf, "<Main>/Help");
	gtk_menu_item_right_justify(GTK_MENU_ITEM(gmp->widget));
	n->menu = subfactory->widget;
#endif

	n->notebook = gtk_notebook_new();
	gtk_notebook_set_show_border(GTK_NOTEBOOK(n->notebook), FALSE);
	n->vbox = gtk_vbox_new(FALSE, 0);
	n->cvbox = gtk_vbox_new(FALSE, 0);
	n->hbox = gtk_hbox_new(FALSE, 0);
	n->infobox = gtk_vbox_new(FALSE, 5);
	n->status = gtk_statusbar_new();
	n->container=NULL;
	n->namelabel = gtk_label_new("Name: ");
	n->iplabel = gtk_label_new("IP: ");
	n->oslabel = gtk_label_new("OS: ");
#ifdef SNMP 
	n->snmplabel = gtk_label_new("SNMP: ");
#endif
	gtk_box_pack_start(GTK_BOX(n->infobox), n->namelabel, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(n->namelabel), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(n->infobox), n->iplabel, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(n->infobox), n->oslabel, FALSE, FALSE, 0);
#ifdef SNMP
	gtk_box_pack_start(GTK_BOX(n->infobox), n->snmplabel, FALSE, FALSE, 0);
#endif
	gtk_signal_connect(GTK_OBJECT(n->notebook), "switch_page", GTK_SIGNAL_FUNC(page_click), NULL);
	gtk_box_pack_start(GTK_BOX(n->vbox), n->menu, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(n->cvbox), n->infobox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(n->hbox), n->cvbox, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(n->hbox), n->notebook, TRUE,TRUE, 0);
	gtk_box_pack_start(GTK_BOX(n->vbox), n->hbox, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(n->vbox), n->status, FALSE, FALSE, 0);
	gtk_widget_show(n->status);
	gtk_widget_show(n->notebook);
	gtk_widget_show(n->menu);
	gtk_widget_show(n->vbox);
	gtk_widget_show(n->cvbox);
	gtk_widget_show(n->hbox);
	gtk_widget_show(n->namelabel);
	gtk_widget_show(n->oslabel);
	gtk_widget_show(n->iplabel);
#ifdef SNMP
	gtk_widget_show(n->snmplabel);
#endif
	gtk_object_set_user_data(GTK_OBJECT(n->window), n);
	gtk_container_add(GTK_CONTAINER(n->window), n->vbox);
	gtk_window_set_title(GTK_WINDOW(n->window), "Cheops Network User Interface version "  VERSION);

	set_geometry(n->window);
	fix_icon(n->window->window);
	hide_page_menu(n);
	gtk_window_set_policy(GTK_WINDOW(n->window), TRUE, TRUE, FALSE);
	
} 


int main(int argc, char *argv[])
{
	int doshow=0, loaded;
	main_window.next = NULL;
	if (geteuid()) {
		fprintf(stderr, "Must be root\n");
		exit(1);
	}
	/* queso */
	init_tcpip();
	/* Ping */
	net_init();
	/* halfscan */
	init_halfscan();
	setuid(getuid());
	if (!load_options())
		doshow = -1;

	load_plugins();
	init_monitoring();
	init_services();
	register_probes();
#ifdef SNMP
	cheops_snmp_init();
#endif
#ifdef USE_GNOME
	gnome_init("Cheops",NULL, argc, argv);
#else
	gtk_init(&argc, &argv);
#endif

	if (!getenv("HOME")) {
		fprintf(stderr, "No home directory!\n");
		return 1;
	}
	check_old_config(".cheops-map", ".cheops/map");
	g_snprintf(defname, sizeof(defname), "%s/.cheops/map", getenv("HOME"));
	show_window(&main_window);
	gtk_widget_show(main_window.window);
	doshow |= (loaded = do_load(NULL));
	if (!doshow) {
		save_options();
		show_about(NULL, 8000);
	}
	if (!loaded) {
		load_interfaces();
	}
	gtk_main();
	return 0;
}
