/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <netinet/in.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <gtk/gtk.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include "cheops.h"

#define DNS_PORT 53

/* What to look for */
static char *defhostname = "localhost";

static char *defip = "127.0.0.1";

static char *service = "dns";

static char *label = "DNS Server";

struct dns_setup {
	/* The request to make */
	char hostname[80];
	/* What to look for */
	char ip[80];

	GtkWidget *window;
	GtkWidget *hostnamew;
	GtkWidget *ipw;
};

struct dns_monitor {
	int level;		/* level of error we should report */
	int s;
	struct net_page *np;	/* Page our object is on */
	struct net_object *no;	/* Pointer to the object we're monitoring */
	int io_id;		/* ID for I/O attempts */
	struct dns_setup *setup;		/* Setup data */
};

GList *dnsmons=NULL;

char *plugin_name() {
	return service;
}

char *plugin_label() {
	return label;
}

int plugin_type() {
	return TYPE_MONITOR;
}

int default_level() {
	return MONITOR_CRITICAL;
}

void reset(struct dns_monitor *hm)
{
	if (hm->io_id > -1) {
		cancel_data(hm->io_id);
		hm->io_id = -1;
	}
	if (hm->s > -1) {
		close(hm->s);
		hm->s = -1;
	}
}

static int found_answer(char *buf, int buflen, char *hostname, char *ip)
{
	/* I need to work */
	HEADER *h;
	char *eom, *start;
	int qcount, acount;
	int len;
	char hostbuf[256];
	
	len = ntohs(*(unsigned short *)buf);
	buf+=2;
	h = (HEADER *)buf;
	start = buf;
	qcount = ntohs(h->qdcount);
	acount = ntohs(h->ancount);
	if (len != buflen - 2) {
		/* Theoretically we should have read exactly len bytes, but
		   I'm just assuming that we get it all at once, which we
		   should, but isn't guaranteed. */
		fprintf(stderr, "Error:  DNS monitor is too lazy\n");
		return 0;
	}
	eom = buf + len;
#if 0
	printf("len is %d (%d), %d questions, %d answers\n", len, buflen, qcount, acount);
#endif
	/* Return if there were no answers */
	if (acount < 0)
		return 0;
	/* Skip header */
	buf += HFIXEDSZ;
	/* Skip questions */
	while(qcount-- > 0) 
		buf += dn_skipname(buf, eom) + QFIXEDSZ;
	/* Look at answers */
	while(acount-- > 0 && (buf < eom)) {
		unsigned short type, class, dlen;
		int n;
		struct in_addr ia;
		n = dn_expand(start, eom, buf, hostbuf, sizeof(hostbuf));
		/* If it fails to expand, error, whatever the reason */
		if (n < 0)
			break;
		buf += n;
		type = _getshort(buf);
		buf += INT16SZ;
		class = _getshort(buf);
		buf += INT16SZ;
		buf += INT32SZ;
		dlen = _getshort(buf);
		buf += INT16SZ;
#if 0
		printf("type: %d, class: %d, dlen: %d\n", type, class, dlen);
#endif
		if ((type == T_A) && (class == C_IN) && (dlen == sizeof(struct in_addr))) {
			memcpy(&ia, buf, dlen);
#if 0
			printf("host %s is %s\n", hostbuf, inet_ntoa(ia));
#endif
			if (!strcmp(hostbuf, hostname) &&
			    !strcmp(inet_ntoa(ia), ip))
			    	return -1;
		}
	}
	return 0;
}

static int build_request(char *buf, int buflen, char *hostname)
{
	return res_mkquery(QUERY, hostname, C_IN, T_A, NULL, 0, 0, buf, buflen);
}

void stage3(void *data, int s, GdkInputCondition c)
{

	/* Called when data is ready for reading.  Make sure the
	   answer is good. */
	int res=0;
	struct dns_monitor *hm = (struct dns_monitor *)data;
	char buf[1024];
	char *ip;
	char *hostname;
	hm->io_id = -1;
#if 0
	printf("Stage 3 and condition is %d!\n", c);
#endif
	if (hm->setup) {
		hostname = hm->setup->hostname;
		ip = hm->setup->ip;
	} else {
		hostname = defhostname;
		ip = defip;
	}
	
	res=read(hm->s, &buf, sizeof(buf));
	
	res = found_answer(buf, res, hostname, ip);
	
	if (res) {
		monitor_report(hm->no, hm->np, MONITOR_NOMINAL, service, "Nominal condition");
	} else {
		monitor_report(hm->no, hm->np, hm->level, service, "DNS server provided invalid answer");
	}
	reset(hm);
}

int transmit(int s, char *buf, int buflen)
{
	unsigned short len;
	int res=0;

	__putshort(buflen, (unsigned char *)&len);
	if ((write(s, (char *)&len, INT16SZ) != INT16SZ) ||
	    ((res = write(s,  buf, buflen)) != buflen))
	   		return -1;
	return res;
}

void stage2(void *data, int s, GdkInputCondition c)
{
	/* Called when connect() has completed or failed, one or the other */
	struct dns_monitor *hm = (struct dns_monitor *)data;
	char writestr[1024];
	
	int res;
	
	res = get_socket_error(s);
	
#if 0
	printf("Error is %s (%d)\n", strerror(res), res);
#endif
	
#if 0
	printf("Stage 2 and condition is %d!\n",c);
#endif
	
	/* If the socket is connected (i.e. available for write), then
	   we wait for data to read on it */
	
	hm->io_id = -1;
	if (!res) {
		res = build_request(writestr, sizeof(writestr), (hm->setup ? hm->setup->hostname : defhostname));
		if (res < 0) {
			snprintf(writestr, sizeof(writestr), "res_mkquery() failed (local)");
			monitor_report(hm->no, hm->np, hm->level, service, writestr);
		} else {
			if (transmit(hm->s, writestr, res) != res) {
				snprintf(writestr, sizeof(writestr), "Connection failed: %s", strerror(errno));
				monitor_report(hm->no, hm->np, hm->level, service, writestr);
			} else {
				hm->io_id = wait_for_data(s, stage3, data);
				return;
			}
		}
	} else {
		snprintf(writestr, sizeof(writestr), "Connection failed: %s", strerror(res));
		monitor_report(hm->no, hm->np, hm->level, service, writestr);
	}
	reset(hm);
}

void monitor(struct net_object *no, int level, void **data, void *setupdata)
{
	struct dns_monitor *hm = (struct dns_monitor *)(*data);
	
	/* While the arguments to our monitoring function are straightforward
	   and the general idea (connect, write a request, read back and look
	   for our magic string) isn't that difficult, the function is somewhat
	   complicated by the requirement that we never make a blocking
	   I/O call.
	
	   This is the first step of the monitoring function.  First, we
	   allocate a monitor stucture to hold some important info like
	   the callback id's, our sockets, and the object and page. */
	
#if 0
	fprintf(stdout, "Monitoring DNS!\n");
#endif
	if (!*data) {
#if 0
		fprintf(stdout, "Making new monitor structure for %s\n", no->hostname);
#endif
		hm = g_new(struct dns_monitor, 1);
		hm->s = -1;
		hm->io_id = -1;
		hm->no = no;
		hm->np = no->np;
		hm->setup = setupdata;
		*data = hm;
	}
	hm->level = level;

	/* Remember, we're not guaranteed that we have delivered
	   a reply by this time, so we need to close the socket
	   if it isn't already closed after first removing any
	   pending io callbacks */

	reset(hm);
		   
	hm->s = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
	
	/* Immediately report an error if socket creation failed */
	if (hm->s < 0) {
		monitor_report(no, no->np, hm->level, service, "Unable to create socket (local)");
		return;
	}
	
	/* monitor.c provides this as a convenience function.  We try to asynchronously
	   connect the socket to the given address.  When it is available for writing
	   or an  error has occured, stage2 will be called */
	
	hm->io_id = try_to_connect(hm->s, no->ip_addr, htons(DNS_PORT), hm, stage2);
}


void cleanup(void **data)
{
	/* Here, we clean up our ftp_monitor structure and 
	   remove any pending inputs we had, since we are no longer
	   being asked to work */
	   
	struct dns_monitor *hm = (struct dns_monitor *)(*data);
	dnsmons = g_list_remove(dnsmons, hm);
	reset(hm);
	g_free(hm);
	*data = NULL;
#if 0
	fprintf(stdout, "Cleaned up!\n");
#endif
}

static void close_setup_window(GtkWidget *w)
{
	struct dns_setup *s = (struct dns_setup *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	/* When they close the window, grab their info */
	strncpy(s->hostname, gtk_entry_get_text(GTK_ENTRY(s->hostnamew)), sizeof(s->hostname));
	strncpy(s->ip, gtk_entry_get_text(GTK_ENTRY(s->ipw)), sizeof(s->ip));
	
	/* Destroy the window itself */
	gtk_widget_hide(s->window);
	gtk_widget_destroy(s->window);
	s->window = NULL;
}

void *setup(void *oldsetup, struct net_object *no)
{
	/* Allocate a ftp_setup structure, initialize some reasonable
	   defaults, display a window, and then return our structure.
	   
	   It's not important that the structure have final values in 
	   it when we return, since our callbacks will fill them in when
	   the user closes our setup window */
	   
	struct dns_setup *s;
	
	GtkWidget *label;
	GtkWidget *bbox;
	GtkWidget *close;
	GtkWidget *vbox;
	
	char buf[256];
	
	if (oldsetup)
		s = oldsetup;
	else {
		s = g_new(struct dns_setup, 1);
		memset(s, 0, sizeof(struct dns_setup));
		strncpy(s->hostname, defhostname, sizeof(s->hostname));
		strncpy(s->ip, defip, sizeof(s->ip));
	}
	
	if (s->window) {
		gtk_widget_show(s->window);
		return s;
	}
	
	/* Make our setup box */
	snprintf(buf, sizeof(buf), "DNS Monitor: %s", no->hostname);
	s->window = gtk_window_new(GTK_TOPLEVEL);
	gtk_widget_realize(s->window);
	gtk_window_set_title(GTK_WINDOW(s->window), buf);
	fix_icon(s->window->window);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vbox);

	label = gtk_label_new(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);
	
	label = gtk_label_new("Lookup Hostname:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);
		
	s->hostnamew = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(s->hostnamew), s->hostname);
	gtk_box_pack_start(GTK_BOX(vbox), s->hostnamew, FALSE, FALSE, 5);
	gtk_widget_show(s->hostnamew);

	label = gtk_label_new("Should have IP:");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	s->ipw = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(s->ipw), s->ip);
	gtk_box_pack_start(GTK_BOX(vbox), s->ipw, FALSE, FALSE, 5);
	gtk_widget_show(s->ipw);

	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	
	close = gtk_button_new_with_label("  Close  ");
	gtk_object_set_user_data(GTK_OBJECT(close), s);
	gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_object_set_user_data(GTK_OBJECT(s->window), s);
	gtk_signal_connect(GTK_OBJECT(s->window), "delete_event", GTK_SIGNAL_FUNC(close_setup_window), NULL);
	gtk_widget_show(close);
	
	gtk_box_pack_end(GTK_BOX(bbox), close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);	

	gtk_container_border_width(GTK_CONTAINER(s->window), 10);
	gtk_container_add(GTK_CONTAINER(s->window), vbox);
	gtk_widget_show(s->window);
	return s;
}

void setup_cleanup(void *data)
{
	struct dns_setup *setup = (struct dns_setup *)data;
	/* If we have a setup window open, destroy it, then free
	   our structure */
	if (setup->window)
		gtk_widget_destroy(setup->window);
#if 0
	printf("Cleaning up setup structure (%p)!\n", setup);
#endif
	g_free(setup);
}

char *setup2str(void *data)
{
	static char buf[256];
	struct dns_setup *s = (struct dns_setup *)data;
	
	/* When monitor stuff is saved, the saving code calls this
	   routing to convert the setup structure into a string that it
	   can include in the configuration file. */
	
	snprintf(buf, sizeof(buf), "%s!%s", s->hostname, s->ip);
	return buf;
}

void *str2setup(char *buf, struct net_object *no)
{

	/* Hre, we read a setup string and return a setup pointer.  We use
	   "!" as delimiters because ^ is already used and ! are unlikely
	   to be found in the request */
	char *c;
	struct dns_setup *s;

	s = g_new(struct dns_setup, 1);
	strncpy(s->hostname, defhostname, sizeof(s->hostname));
	strncpy(s->ip, defip, sizeof(s->ip));
	s->window = NULL;

	c = strtok(buf, "!");
	if (c)
		strncpy(s->hostname, c, sizeof(s->hostname));
	c = strtok(NULL, "!");
	if (c)
		strncpy(s->ip, c, sizeof(s->ip));
	return s;
}

