/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004, 2005 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common-internal.h"
#include "config_value.h"


ret_t 
cherokee_config_value_new  (cherokee_config_value_t **value)
{
	CHEROKEE_NEW_STRUCT (n, config_value);

	INIT_LIST_HEAD (&n->list_item);
	INIT_LIST_HEAD (&n->list);

	cherokee_buffer_init (&n->string);
	cherokee_buffer_init (&n->keyval.key);

	n->number       = 0;
	n->keyval.value = NULL;
	n->type         = cvalue_unknown;

	*value = n;
	return ret_ok;
}


ret_t 
cherokee_config_value_free (cherokee_config_value_t *value)
{
	list_t *i, *j;

	list_for_each_safe (i, j, &value->list) {
		list_del (i);
		cherokee_config_value_free (CONFVAL(i));
	}

	cherokee_buffer_mrproper (&value->string);

	cherokee_buffer_mrproper (&value->keyval.key);	   
	if (value->keyval.value != NULL) {
		cherokee_config_value_free (value->keyval.value);
		value->keyval.value = NULL;
	}
	   
	free (value);
	return ret_ok;
}

static void
write_string (cherokee_buffer_t *source, cherokee_buffer_t *buf)
{
	cherokee_buffer_add (buf, "(string \"", 9);
	cherokee_buffer_add_buffer (buf, source);
	cherokee_buffer_add (buf, "\")", 2);
}


ret_t 
cherokee_config_value_serialize (cherokee_config_value_t *value, cherokee_buffer_t *buf)
{
	list_t *i;

	switch (value->type) {
	case cvalue_number:
		cherokee_buffer_add_va (buf, "(number "FMT_OFFSET")", value->number);
		break;

	case cvalue_string:
		write_string (&value->string, buf);
		break;

	case cvalue_keyval:
		cherokee_buffer_add (buf, "(keyval ", 8);
		write_string (&value->string, buf);
		cherokee_config_value_serialize (value->keyval.value, buf);
		break;

	case cvalue_list:
		cherokee_buffer_add (buf, "(list ", 6);
		list_for_each (i, &value->list) {
			cherokee_config_value_serialize (CONFVAL(i), buf);
		}
		cherokee_buffer_add (buf, ")", 1);			 
		break;

	default:
		SHOULDNT_HAPPEN;
		return ret_error;
	}

	return ret_ok;
}


ret_t 
cherokee_config_value_parse (cherokee_config_value_t **value, char *string)
{
	int   error;
        void *bufstate;

	extern int  yy_config_protocol_parse             (void *);
	extern void yy_config_protocol__switch_to_buffer (void *);
	extern void yy_config_protocol__delete_buffer    (void *);
	extern int  yy_config_protocol__scan_string      (const char *);

	bufstate = (void *) yy_config_protocol__scan_string (string);
        yy_config_protocol__switch_to_buffer(bufstate);

	error = yy_config_protocol_parse((void *)value);
			 
        yy_config_protocol__delete_buffer (bufstate);

        return (error)? ret_error : ret_ok;
}
