/*
 * ImagePanel.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  ImagePanel.cpp

  The "images" panel of the Fl_Tabs within the primary window. 
  
  TODO: 
   o Verhindern, dass Tabellenspalten bis auf Null zusammengeschoben
     werden koennen.
   
   o Wenn erste Zeit veraendert wurde (die als Basis fuer Stops dient) und die
     uebrigen mit *demselben* Stops-Faktor wie bisher neu berechnet werden
     sollen, geht das nicht direkt, da das Menu bei Wahl des *alten* Items 
     keine Callback-Funktion aufruft. (Kann man das einstellen?) Man muss erst
     anderen `stops'-Wert waehlen und dann wieder den alten. 
   
   o Setzende und ausgebende Funktionen "set_...()" nennen, nur ausgebende
     "out_...()"; pauschaler und fuer beides offen "update_...()".
*/
#include <cstdio>               // sprintf() 
#include <cstring>

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/fl_draw.H>
#include <FL/filename.H>        // fl_filename_name()
#include <FL/fl_ask.H>
#include "Fl_Table_Row.H"

#include "Br.hpp"               // `Br', SPUR_BR_EVENT()
#include "BrGUI.hpp"            // `BrGUI'
#include "ImagePanel.hpp"       // class ImagePanel
#include "StatusLine.hpp"       // `__statusline'
#include "fl_print_event.hpp"   // debug utilities

//#define YS_DEBUG
//#undef  YS_DEBUG
//#include "YS/ys_dbg.hpp"

// Using an input widget in Fl_Table accordingly to Ercolano's Fl_Table example 
//   gives a weird bahavior of the activation button: the first click into 
//   a "virgin" panel has no effect. Un/define it for testing.
#define BR_TIME_INPUT

// const char* contextname(Fl_Table::TableContext context)
// {
//     static char s[32];  
//     switch (context) 
//     {
//     case Fl_Table::CONTEXT_NONE:       return "NONE";
//     case Fl_Table::CONTEXT_STARTPAGE:  return "STARTPAGE";
//     case Fl_Table::CONTEXT_ENDPAGE:    return "ENDPAGE";
//     case Fl_Table::CONTEXT_ROW_HEADER: return "ROW_HEADER";
//     case Fl_Table::CONTEXT_COL_HEADER: return "COL_HEADER";
//     case Fl_Table::CONTEXT_CELL:       return "CELL";
//     case Fl_Table::CONTEXT_TABLE:      return "TABLE";
//     default: sprintf(s,"context=%d", context); return s;
//     }
// }

/**================================================================
 *
 * ImageTable  -  class
 *
 * Diese Tabelle praesentiert Daten aus dem globalen Container Br.imgVec.
 *   Im Konstruktionsmoment konstruiert sie sich mit soviel Zeilen wie 
 *   Bilder in Br.imgVec (gewoehnlich 0). (Nach)Laden von Bildern erfordert
 *   dann Zeilenzahl anzupassen!
 * 
 *=================================================================*/
/**-------------------
 * Constructor...
 *--------------------*/
ImageTable::ImageTable(int X, int Y, int W, int H, const char *la) 
  : Fl_Table (X,Y,W,H,la)
{ 
  CTOR(la); 
  begin();
    selection_color(FL_YELLOW);
    //type(Fl_Table_Row::SELECT_NONE);
    callback((Fl_Callback*)cb_event_, this);
    when(FL_WHEN_RELEASE);  // handle table events on release
    //when(FL_WHEN_CHANGED|FL_WHEN_RELEASE);
#ifdef BR_TIME_INPUT    
    input_ = new Fl_Float_Input(W/2, H/2, 0, 0);
    input_ -> hide();
    input_ -> callback((Fl_Callback*)cb_input_, this);
    input_ -> when(FL_WHEN_ENTER_KEY_ALWAYS);
    input_ -> maximum_size(12);
#endif
  end(); 
    
  int frame     = Fl::box_dw(box()), // *after* box decision!
      w_name    = 100,
      w_size    = 100,
      w_brightn = 80,
      w_wrange  = 80,
      w_time;
    
  // Fit the last column into the rest of the table ('25' is the width
  //   of the activation button)...
  w_time = W - (25 + w_name + w_size + w_brightn + w_wrange) - frame;
  w_time = (w_time > 80) ? w_time : 80;
    
  cols(5);
  col_header(1);          // enable col header
  col_header_height(25);
  col_width(0,w_name);    // filename
  col_width(1,w_size);    // image size
  col_width(2,w_brightn); // brightness
  col_width(3,w_wrange);  // working range
  col_width(4,w_time);    // time
  col_resize(4);          // enable col resizing

  rows(Br.imgVec.size()); // number of images
  row_header(1);          // enable row header
  row_header_width(25);   // width of activation button
  row_height_all(25);
  row_resize(4);          // enable row resizing
}                   

/**-----------------------------------------------------
 * draw_cell()  --  Handle drawing all cells in table...
 *
 * TODO: alignment at decimal points
 *------------------------------------------------------*/
void ImageTable::draw_cell(TableContext context, 
                           int R, int C, int X, int Y, int W, int H)
{
  //printf("draw_cell(): %s, R=%d, C=%d, X=%d, Y=%d, W=%d, H=%d\n", contextname(context),R,C,X,Y,W,H);
  
  static char s[128];
  int wcell;        // width of *visible* cell (not always == W)
  Fl_Align align;   // alignment inside the visible cell

  switch ( context )
  {
  case CONTEXT_STARTPAGE:
      //fl_font(FL_HELVETICA, 16);  
        // im Prinzip noetig, aber vorerst verlasse ich mich auf den
        // aktuell aktiven Font wegen Font-Ungeklaertheiten
      return;

  case CONTEXT_COL_HEADER:
      fl_push_clip(X, Y, W, H);
      {
        //fl_draw_box(FL_EMBOSSED_BOX, X, Y, W, H, color());
        fl_draw_box(FL_THIN_UP_BOX, X, Y, W, H, color());
        fl_color(FL_BLACK);
        wcell = wix + wiw - X;
        wcell = wcell < W ? wcell : W; // printable width never wider than W
        switch(C) 
        {
        case 0:  fl_draw("Name",  X, Y, wcell, H, FL_ALIGN_CENTER); break;
        case 1:  fl_draw("Pixel", X, Y, wcell, H, FL_ALIGN_CENTER); break;
        case 2:  fl_draw("Brigthness", X, Y, wcell, H, FL_ALIGN_CENTER); break;
        case 3:  fl_draw("Wk-Range",   X, Y, wcell, H, FL_ALIGN_CENTER); break;
        case 4:  fl_draw("Time",  X, Y, wcell, H, FL_ALIGN_CENTER); break;
        default: sprintf(s, "%d/%d", R, C); // something forgotten?
                 fl_draw(s,       X, Y, wcell, H, FL_ALIGN_CENTER);
        }
      }
      fl_pop_clip();
      return;

  case CONTEXT_ROW_HEADER:
      fl_push_clip(X, Y, W, H);
      {
        // Seltsamer Bug, dass bei einem "jungfraeulichen" Klick manchmal
        //   in anderes Fenster oder ungueltiges Drawable gezeichnet wird.
        //   Deshalb vorlaeufig dieser Hack, der aber auch nur manchmal hilft.
        if (window() != window()->current()) {
          printf("BUG! draw_cell(): window() = %p,  current-win = %p\n",
              window(), window()->current()); 
          window()-> make_current();   // HACK
        }
        sprintf(s, "%d", R+1);  // begin with 1, not 0
        if (Br.imgVec.active(R)) fl_draw_box(FL_DOWN_BOX, X,Y,W,H, FL_GREEN);
        else                     fl_draw_box(FL_UP_BOX,   X,Y,W,H, color());
        fl_color(FL_WHITE); 
        fl_draw(s, X,Y,W,H, FL_ALIGN_CENTER);
      }
      fl_pop_clip();
      return;
    
  case CONTEXT_CELL:
#ifdef BR_TIME_INPUT
      // don't overwrite input widget...
      if (R == row_edit && C == col_edit && input_->visible())
        { return; }     
#endif    
      fl_push_clip(X, Y, W, H);
      {
        // BG COLOR
        //Fl_Color c = row_selected(R) ? selection_color() : FL_WHITE;
        Fl_Color c = FL_WHITE;
        if (!Br.imgVec.active(R)) c = fl_color_average(c, FL_GRAY, 0.66f);
        draw_box(FL_THIN_UP_BOX, X,Y,W,H, c);

        // TEXT
        Br.imgVec.active(R) ? fl_color(FL_BLACK) : fl_color(fl_inactive(FL_BLACK));
        wcell = wix + wiw - X;
        wcell = wcell < W ? wcell : W; // printable width never wider than W
        switch (C)
        {
        case 0:
            fl_draw(fl_filename_name(Br.imgVec[R].name()), X+3, Y, W, H, FL_ALIGN_LEFT); 
            break;
        
        case 1:
            snprintf(s, 128, "%d x %d", Br.imgVec[R].width(), Br.imgVec[R].height());
            if (wcell > fl_width(s)) align = FL_ALIGN_CENTER; 
            else                     align = FL_ALIGN_LEFT;
            fl_draw(s, X, Y, wcell, H, align);
            break;
        
        case 2:
            snprintf(s, 128, "%.4f ", Br.imgVec[R].brightness);
            if (wcell > fl_width(s)) align = FL_ALIGN_RIGHT; 
            else                     align = FL_ALIGN_LEFT;
            fl_draw(s, X, Y, wcell, H, align);
            break;
        
        case 3:
            snprintf(s, 128, "%.3f %% ", Br.imgVec[R].r_wrange*100.0);
            if (wcell > fl_width(s)) align = FL_ALIGN_RIGHT; 
            else                     align = FL_ALIGN_LEFT;
            fl_draw(s, X, Y, wcell, H, align);
            break;
        
        case 4:
            snprintf(s, 128, "%f ", Br.imgVec[R].time);
            if (wcell > fl_width(s)) align = FL_ALIGN_RIGHT; 
            else                     align = FL_ALIGN_LEFT;
            fl_draw(s, X, Y, wcell, H, align);
            break;
        
        default:
            sprintf(s, "%d/%d", R, C);    // something forgotten?
            fl_draw(s, X, Y, wcell, H, FL_ALIGN_CENTER);
        }

        // BORDER (example)
        //fl_color(FL_LIGHT2); 
        //fl_rect(X, Y, W, H);
      }
      fl_pop_clip();
      return;

#ifdef BR_TIME_INPUT         
  case CONTEXT_RC_RESIZE:
      // Input widget mitscrollen...
      if (!input_-> visible()) return;
      find_cell(CONTEXT_TABLE, row_edit, col_edit, X, Y, W, H);
      if (X==input_->x() && Y==input_->y() && W==input_->w() && H==input_->h())
        return;
      input_ -> resize(X,Y,W,H);
      return;
#endif
      
  case CONTEXT_ENDPAGE:
      // Draw a box in the "headers corner"; X,Y are in STARTPAGE/ENDPAGE
      //   those of the data table *without* headers, x() and y() are the
      //   outer coords; what we need are wix, wiy, the (protected) inner
      //   coords of the widget without border (same as X - row_header_width(), 
      //   Y - col_header_height)!)...
      draw_box(FL_EMBOSSED_BOX, wix, wiy, row_header_width(), col_header_height(), color());
      return;
      
  default:
      //fprintf(stderr, "ImageTable::%s(): not processed context %d\n",__func__, context);
      return;
  }
}

/**----------------------------------------------------
 * cb_event()  --  callback whenever someone clicks on
 *                   different parts of the table...
 *-----------------------------------------------------*/
void ImageTable::cb_event()
{
#if 0
  fprintf(stderr, "ImageTable::cb_event(): %s, row=%d col=%d, %s, clicks=%d\n",
                  //(const char*)label(),
                  contextname(callback_context()),
                  (int)callback_row(),
                  (int)callback_col(),
                  fl_eventname(Fl::event()),
                  (int)Fl::event_clicks());
#endif
  TableContext context = callback_context();
  int R = callback_row(),
      C = callback_col();
  
  switch (Fl::event())
  {
  case FL_PUSH:
      switch (context)
      {
      case CONTEXT_ROW_HEADER:
          if (Br.imgVec.active(R)) Br.deactivate(R);
          else                     Br.activate(R);
          // table callback by default allready redraws *inner* table, i.e.
          //   nessecary here only is adding redrawing of row header...
          int XX,YY,WW,HH;
          find_cell(CONTEXT_ROW_HEADER, R,C, XX, YY, WW, HH);
          draw_cell(CONTEXT_ROW_HEADER, R,C, XX, YY, WW, HH);
          return;

#ifdef BR_TIME_INPUT      
      case CONTEXT_CELL:    
          if (C == col_edit) {   // time column
            if (input_->visible()) { input_->do_callback(); }
            row_edit = R;
            int XX,YY,WW,HH;
            find_cell(CONTEXT_CELL, R,C, XX, YY, WW, HH);
            input_ -> resize(XX,YY,WW,HH);
            char s[30];
            snprintf(s, 30, "%f", Br.imgVec.at(R).time);
            input_ -> value(s);
            input_ -> show();
            input_ -> take_focus();    
          }
          return;
#endif          
      default:           
          return;
      }
  
  default: ;
  }
}


/**------------------------------------------------------------
 * cb_input()  --  callback for ImageTable's Float_Input widget
 *   Translate current input->value() into double and set the
 *   global variable.
 *-------------------------------------------------------------*/
void ImageTable::cb_input ()
{
  //printf("cb_input(): [%d] = \"%s\")...\n", row_edit, input_->value());
  char*  endptr;
  double res = strtod (input_->value(), &endptr);

  // accept only correct values...
  if (*endptr) {                         
    printf("[%s]: wrong float format: \"%s\"\n", __func__, input_->value());
    // global variable remains unchanged!
  } 
  else if (res <= 0.0) {
    printf("[%s]: times <= 0 not possible\n", __func__);
    // global variable remains unchanged!
  } 
  else if (Br.imgVec.at(row_edit).time != res) { // ok, take over
    Br.imgVec.at(row_edit).time = res;        // Sets global variable
    Br.update_camera_times();
  }
  input_ -> hide();                     // verbirgt input    
}


/**================================================================
 *
 * ImagePanel  --  class
 *
 * The main panel of the Fl_Tabs in the main menu, contains the ImageTable
 *
 *=================================================================*/
 
// The static elements...
//
// Structures for the stops menus...
//
//struct { const char* text; float value; }
ImagePanel::MenuMapItem_stops  ImagePanel::menuMap_stops[] = {
  {"0.33...", 1./3.   }, 
  {"0.5"    , 0.5     },
  {"0.66...", 2./3.   },
  {"1"      , 1.      },
  {"1.33...", 1.+1./3.},
  {"1.5"    , 1.5     },
  {"1.66...", 1.+2./3.},
  {"2"      , 2.      }
};  
Fl_Menu_Item  ImagePanel::menu_choice_stops[] = {
  {menuMap_stops[0].text},
  {menuMap_stops[1].text},
  {menuMap_stops[2].text},
  {menuMap_stops[3].text},
  {menuMap_stops[4].text},
  {menuMap_stops[5].text},
  {menuMap_stops[6].text},
  {menuMap_stops[7].text},
  {0}
};
//const int n_items_stops = sizeof(menuMap_stops) / sizeof(menuMap_stops[0]);

//
// tooltips (have to be global or static or allocated)
//
const char* ImagePanel::tooltip_activate = 
//"Deaktivierungbutton: Hier koennen Sie einzelne Bilder aus den HDR-Rechnungen ausschliessen.";
"Deactivation button: Here you can exclude single images from the HDR calculations.";

const char* ImagePanel::tooltip_stops = 
//"Hier koennen Sie Belichtungszeiten generieren, die mit einem festen Faktor F=2^(stops) wachsen. Traditionell wird dabei nicht F, sondern der 2er-Exponent \"stops\", der den Differenzen der Blendenzahlen entspricht, angegeben. Bsp: stops=1 --> F=2, stops=2 --> F=4.";
"Here you can generate exposure times which increase by a constant factor F=2^(stops). Traditionally thereby F is stated by specifying the exponent \"stops\", which corresponds to the differences of aperture stops.";
 
/**---------------
 * Constructor... 
 *----------------*/
ImagePanel::ImagePanel (int X, int Y, int W, int H, const char* la)
  : Fl_Group (X,Y,W,H, la)
{ 
  CTOR(la)
  
  // in Distributor(en) einloggen...
  Br.distrib_event.login ((Callback*)event_msg_, this);

  int y_table  = Y + 11,
      y_grid   = Y + H - 75,  // "95": space for two buttons
      dy_table = y_grid - y_table - 10; // 10: between table and buttons
      
  // default for `stops' value...
  int i_stops_default = 3;          // 0 = first item, 3 -> "1.0"
  val_stops_ = menuMap_stops [i_stops_default].value;

  Br.list();
    
  begin();
    { table_ = new ImageTable (X+5, y_table, W-10, dy_table);
    }
    { Fl_Int_Input* o = new Fl_Int_Input(90, y_grid, 50, 25, "Grid Points");
      input_n_grid_points_ = o;                           // "Stuetzstellen"
      set_n_grid_points (Br.n_grid_points_);  // --> o->value()
      o->callback((Fl_Callback*)cb_grid_points_, this);
    }
    { Fl_Float_Input* o = new Fl_Float_Input(260, y_grid, 50, 25, "Smoothing");
      input_smoothing_ = o;
      set_smoothing (Br.smoothing_);  // --> o->value()
      o->callback((Fl_Callback*)cb_smoothing_, this);
    }
    { Fl_Choice* o = new Fl_Choice(400, y_grid, 60, 25, "Stops"); 
      choice_stops_ = o;
      o->menu(menu_choice_stops);
      o->align(FL_ALIGN_LEFT);
      o->callback((Fl_Callback*)cb_stops_choice_, this);
      o->value(i_stops_default);             
      o->tooltip(tooltip_stops);
    }
    { Fl_Button* o = new Fl_Button(15, y_grid+40, 85, 25, "Init");
      button_init_camera_ = o;
      o->callback((Fl_Callback*)cb_init_camera_, this);
      o->deactivate();
    }
    { Fl_Button* o = new Fl_Button(120, y_grid+40, 140, 25, "Compute Response");
      button_compute_CCD_ = o;
      o->callback((Fl_Callback*)cb_compute_CCD_, this);
      o->deactivate();
    }
    { Fl_Button* o = new Fl_Button(260, y_grid+40, 70, 25, "HDR");
      button_compute_HDR_ = o;
      o->callback((Fl_Callback*)cb_compute_HDR_, this);
      o->deactivate();
    }
    // Voruebergehend stillgelegt...
    { Fl_Button* o = new Fl_Button(330, y_grid+35, 120, 25, "Make HDR");
      button_make_HDR_ = o;
      o->callback((Fl_Callback*)cb_make_HDR_, this);
      o->deactivate();
      o->hide(); // Voruebergehend stillgelegt, vielleicht nochmal gebraucht 
    }
    { Fl_Button* o = new Fl_Button(360, y_grid+35, 120, 25, "FollowUpCurves");
      button_follow_ = o;        //260 
      o->callback((Fl_Callback*)cb_FollowCurveWindow_, this);
      o->deactivate();
      o->hide(); // Voruebergehend stillgelegt, vielleicht nochmal gebraucht 
    }
  end();
  resizable(table_);
}


//-----------
// Dtor...
//-----------
ImagePanel::~ImagePanel() 
{ 
  DTOR(label())
  
  // aus Distributor(en) ausloggen...
  Br.distrib_event.logout (this);
}


/**----------------------------------------------
 * Callback for the stops choice menu           MODIFIES Br.imgVec
 *
 *  @param w: choice menu (calling widget)
 *----------------------------------------------*/
void ImagePanel::cb_stops_choice(Fl_Menu_* w) 
{
  const Fl_Menu_Item* m = w->mvalue();  // BTW: mvalue() != value()!
  if (!m)                                   
    printf("[%s]: Menu not initialized --> Programmer!\n",__func__);
  else {
    //printf("%s, %d\n", m->label(), w->value());
    
    // Calc. new times accordingly to stops value...
    val_stops_ = menuMap_stops[ w->value() ].value;
    Br.imgVec.set_times_by_stops(val_stops_);
    
    // Update output (nessecary would be only time column)...
    table_ -> redraw();  
    
    // info to menubar's choice not nessecary, as there no value is shown
    
    Br.update_camera_times();
  }
}

/**----------------------------------------------
 * Callback for the grid points input
 *
 *  @param w: Fl_Int_Input (calling widget)
 *-----------------------------------------------*/
void ImagePanel::cb_grid_points(Fl_Int_Input* in) 
{
  int res = atoi(in->value());
  
  if (res < 2)   res = 2;  else      
  if (res > 255) res = 255;     // < 255: 8-bit minus both border points
  
  set_n_grid_points (res);  // sets Br.n_grid_points_ and output
  TRACE(("new n_grid_points = %d\n", Br.n_grid_points_))
  Br.distrib_event.value (BracketingCore::CCD_OUTDATED);
}

/**----------------------------------------------
 * Callback for smoothing coefficient input
 *
 *  @param w: Fl_Int_Input (calling widget)
 *-----------------------------------------------*/
void ImagePanel::cb_smoothing(Fl_Float_Input* in) 
{
  char*  endptr;
  double res = strtod (in->value(), &endptr);

#if 1   // take over only correct values...
  if (*endptr || (res < 0.0)) {    // not ok, redraw old value
    if (*endptr)                          
      printf("[%s]: wrong float format: \"%s\"\n",__func__, in->value());
    else  
      printf("[%s]: smoothing coefficient < 0 not possible\n",__func__);
    
    set_smoothing (Br.smoothing_);  // redraw old value
  } 
  else {                            // ok, take over
    Br.smoothing_ = res;            // set global variable
    set_smoothing (res);            // output new value
    Br.distrib_event.value (BracketingCore::CCD_OUTDATED);
  }

#else  // emend as far as possible and take over...

  if (*endptr) {             
    printf("[%s]: wrong float format: \"%s\"\n",__func__, in->value());
  }
  else if (res < 0.0) {
    Br.smoothing_ = 0.0;
    printf("[%s]: smoothing coefficient < 0 not possible\n",__func__);
  } 
  else {
    Br.smoothing_ = res;
    Br.distrib_event.value (BracketingCore::CCD_OUTDATED);
  }
  
  // for all three cases...
  set_smoothing (Br.smoothing_);   // output old resp. new value
#endif  
  
  TRACE(("new smoothing coefficient = %f\n", Br.smoothing_))
}

//----------------------------------------------------------
// Set `Br.n_grid_points_' and output new value...
//----------------------------------------------------------
void ImagePanel::set_n_grid_points (int n)
{
  Br.n_grid_points_ = n;     // Zuweiseung auch schon in cb_n_grid_points() 
  char s[8];  snprintf(s, 8, "%d", n);
  input_n_grid_points_ -> value(s);         
}

//------------------------------------------------------
// Set `Br.smoothing_' and output new value...
//------------------------------------------------------
void ImagePanel::set_smoothing (double coeff)
{
  Br.smoothing_ = coeff;     // Zuweiseung auch schon in cb_smoothing() 
  char s[32];  snprintf(s, 32, "%.2f", coeff);
  input_smoothing_ -> value(s);         
}


/**---------------------------------------------------------------
 * event_msg()  --  Distributor callback
 *
 * What shall express the hi-color of the "init" button? That an camera-init
 *   "should" do! We hi-color at every CAMERA_OUTDATED event. But at present
 *   this event is sended also, if only one image is loaded and NO camera
 *   exists, because can't init for one. Well, this case wasn't intended, but
 *   is makes sense though: Camera "should" init - because it doesn't exist.
 * Nowadays the "Init" button is then blue colored indicating "no camera
 *   exists". Better!
 *----------------------------------------------------------------*/
void ImagePanel::event_msg (BracketingCore::Event e)
{
  static Fl_Color color_hi = fl_color_average (FL_YELLOW, FL_WHITE, 0.5);
  static Fl_Color color_hi2 = fl_color_average (FL_BLUE, FL_WHITE, 0.25);
  
  SPUR_BR_EVENT(("ImagePanel::%s(%d): ", __func__, e));
  switch (e) 
  {
  case BracketingCore::IMAGE_LOADED:
  case BracketingCore::IMAGES_CHANGED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      // Update ImageTable...
      table_ -> rows (Br.size()); // new row number
      table_ -> redraw();
      break;
 
  case BracketingCore::CAMERA_INIT:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      // Downlight "Init" button....
      button_init_camera_ -> color(color()); // bg color of ImagePanel
      button_init_camera_ -> redraw();
      // Highlight2 "Compute CCD" button denoting "CCDs don't exist"...
      button_compute_CCD_ -> color(color_hi2);
      button_compute_CCD_ -> redraw();
      break;
  
  case BracketingCore::CAMERA_OUTDATED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      if (Br.camera()) {
        // Highlight "Init" button...
        button_init_camera_ -> color(color_hi);
        button_init_camera_ -> redraw();
      } else {
        // Highlight2 "Init" && "Compute CCD" button...
        button_init_camera_ -> color(color_hi2);
        button_init_camera_ -> redraw();
        button_compute_CCD_ -> color(color_hi2);
        button_compute_CCD_ -> redraw();
      }
      break;
    
  case BracketingCore::CAMERA_DELETED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      button_init_camera_ -> color(color()); // if highlighted before
      button_init_camera_ -> redraw();
      button_compute_CCD_ -> color(color()); // if highlighted before
      button_compute_CCD_ -> redraw();
      break;
          
  case BracketingCore::CCD_UPDATED:    
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      // Downlight "Compute CCD" button...
      button_compute_CCD_ -> color(color()); // bg color of ImagePanel
      button_compute_CCD_ -> redraw();
      break;
      
  case BracketingCore::CCD_OUTDATED:
      SPUR_BR_EVENT(("%s\n", br_eventnames[e]));
      // Highlight "Compute CCD" button if ccd plots && curves exist...
      if (BrGUI.ccd_plot_created() && Br.ccd_curves_ready()) {
        button_compute_CCD_ -> color(color_hi);
        button_compute_CCD_ -> redraw();
      }
      break;
  
  default: 
      SPUR_BR_EVENT(("not handled\n"));    
  }
  
  // De|activate the items/buttons depending on imgVec.size_active()...
  if (Br.imgVec.size_active() > 1) {
    if (!button_init_camera_->active()) button_init_camera_-> activate();
    if (!button_make_HDR_->active())    button_make_HDR_-> activate();
  } else {
    if (button_init_camera_->active()) button_init_camera_-> deactivate();
    if (button_make_HDR_->active())    button_make_HDR_-> deactivate();
  }
  
  // De|activate the items/buttons depending on camera...
  if (Br.camera() && Br.camera()->nImages() > 1) {
    if (!button_compute_CCD_->active()) button_compute_CCD_-> activate();
    if (!button_compute_HDR_->active()) button_compute_HDR_-> activate();
    if (!button_follow_->active())      button_follow_-> activate();
  } else {
    if (button_compute_CCD_->active()) button_compute_CCD_-> deactivate();
    if (button_compute_HDR_->active()) button_compute_HDR_-> deactivate();
    if (button_follow_->active())      button_follow_-> deactivate();
  }
} 


// END OF FILE
