/*
 * response.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright 2005  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
 * response.hpp  -  class template CCDResponse
 */
#ifndef RESPONSE_HPP
#define RESPONSE_HPP


// TNT debug switches (to define before TNTs included)...
//#define DEBUG
//#define TNT_DEBUG
//#define TNT_DEBUG_1
//#define TNT_BOUNDS_CHECK


#include <fstream>

#include "TNT/tnt_misc.hpp"
#include "TNT/tnt_stopwatch.hpp"
#include "TNT/jama_svd.hpp"
#include "TNT/jama_qr.hpp"

#include "FilePtr.hpp"      
#include "Rgb.hpp"

//#define YS_DEBUG
#undef YS_DEBUG
#include "YS/ys_dbg.hpp"


/**
 * OBACHT: Die Differenz zweier Integer muss bei signed int's nicht mehr
 * in den Datentyp passen: 128 - (-127) = 255 > 128. Hier muss eigentlich
 * immer dann die unsigned Variante stehen!
 * Oder wir legen uns von vorherein auf unsigned Datentypen fest.
 * Dann "Unsign" statt "Integer" nennen. Aber auch dann passt die
 * ANZAHL moeglicher Werte nicht notwendig in diesen Typ, denn von 0..255
 * sind 255+1 = 256 Werte > 255.
 * size_t (==uint) waere da der vorlaeufige Behelf, der freilich fuer
 * sizeof(size_t)==sizeof(Unsign) auch versagt (16-Bit Unsign?).
 */

 
template <class Tx, class Ty>
class Curve2D {

public:

  Array1D <Tx> X;
  Array1D <Ty> Y;

  Curve2D ()          : X(), Y()      {}
  Curve2D (int n)     : X(n), Y(n)    {}

  bool is_empty()                     { return X.dim()==0; }
  void writeFile (const char* fname);
  void list ();
};

template <class Tx, class Ty>
void Curve2D<Tx,Ty>::writeFile (const char* fname)
{
  std::ofstream f(fname);              // ueberschreibt ohne Nachfrage!
  if (!f) 
  { 
    std::cerr << "Konnte " << fname << " nicht oeffnen\n";
    return;
  }
  
  for (int i=0; i < X.dim(); i++)
    f << X[i] << " " << Y[i] << "\n";
}

template <class Tx, class Ty>
void Curve2D<Tx,Ty>::list ()
{
  for (int i=0; i < X.dim(); i++)
  {
     std::cout << " [" << i << "]\tx=" << X[i] << " \tf(x)=" << Y[i] << "\n";
  }
}

/**
 * zWeighting  -  Wichtungsfunktion als separates Funktionsobjekt.
 * 
 * FRAGE: Soll Funktion Unsign-Typ zurueckliefern oder besser gleich
 *   einen Real-Typ. Wenn letzteres, ist zweiter Templateparamter 'Real'
 *   oder `ReturnType' erforderlich. Was ist effizienter? Vermtl. egal.
 */
template <class Unsign>
class zWeighting 
{
  Unsign zmin, zmax, zmid;

public:

  zWeighting (Unsign zMin, Unsign zMax)           // Konstruktor
    : zmin(zMin), zmax(zMax)
    {
      zmid = (zmin + zmax) / 2;
    }

  Unsign operator () (Unsign z)                   // Funktionssyntax
    {
      if (z <= zmid)  return z - zmin;
      else            return zmax - z;
    }

  // Version mit einem Rgb als Argument und als Rueckgabetyp...
#ifdef rgb_HPP
  
  Rgb<Unsign> operator () (const Rgb<Unsign>& z)
    {
      //return Rgb<Unsign>(*this(z.r), *this(z.g), *this(z.b)); Unerlaubt
      
      return Rgb<Unsign>(
                z.r <= zmid ? z.r - zmin : zmax - z.r,
                z.g <= zmid ? z.g - zmin : zmax - z.g,
                z.b <= zmid ? z.b - zmin : zmax - z.b);
    }
#endif
};


/**
 * CCDResponse  -  class
 *
 * NOTE: The output of an CCD-Element ("pixel") is called pixel-value or
 *   z-value and denoted by z. The number of possible z-values is n_zval; 
 *   i.e for a 8-bit resolution n_zval=256, for 10-bit n_zval=1024 etc.
 */
template <class Unsign, class Real>
class CCDResponse 
{
  size_t n_zval;            // Zahl moegl. z-Werte: z_max - z_min + 1
  Unsign z_min;             // min. z-Wert; = 0
  Unsign z_max;             // max. z-Wert; = 255 bei 8 Bit

  Array2D<Unsign> Z;        // Z[i,j]: Pixelwert am Ort i in Aufname j
  Array1D<Real> B;          // B[j]: Log der Belichtungszeit in Aufn j

  Array1D<Real> Xcrv;       // Xcrv[z]: exposure X to z-value z.
                            // Inverse of the characteristic function
                            // z=f(X) and the main effort of the procedure.
  Array1D<Real> logXcrv;    // ==log(Xcrv); primary result of the calculation

  Real wicht (Unsign z);    // Gewicht des Pixelwertes z

public:

  CCDResponse (const Array2D<Unsign> & Z_,
               const Array1D<Real>   & B_, 
               unsigned int          CCDbits);

  void calc (Real lambda=1.0, bool debug=false);
  void write_Xcrv (const char* fname, bool log=false);
  void exchange_Z (const Array2D<Unsign>& Z_)    { Z = Z_; } // no dim-check?
  void exchange_B (const Array2D<Unsign>& B_)    { B = B_; } // no dim-check?

  Array1D<Real> get_Xcrv()        { return Xcrv; }
  Array1D<Real> get_logXcrv()     { return logXcrv; }
};


template <class Unsign, class Real>
CCDResponse<Unsign,Real>::CCDResponse (
    
    const Array2D<Unsign> & Z_,
    const Array1D<Real>   & B_,
    unsigned int          CCDbits )
  :
  Z (Z_),
  B (B_)
{
  assert (CCDbits <= sizeof(Unsign)*8);   // so nur fuer unsigned sicher!
  assert (Z.dim2() == B.dim());

  z_min = 0;
  z_max = (1 << CCDbits) - 1;
  n_zval = (size_t)z_max - z_min + 1;     // 256 for 8 bit
}


template <class Unsign, class Real>
Real
CCDResponse<Unsign,Real>::wicht (Unsign z)
{
  //return 1.0;
  static Unsign zmid = (z_min + z_max) / 2;

  if (z <= zmid)  return z - z_min;
  else            return z_max - z;
};


/**
 * calc() 
 *
 * @param lambda: weight of smothness
 * @param debug: if true, debug output
 *
 *  The number of equations is
 *      n_eq  =  N*P + 1 + (n_zval-2)  =  N*P + n_zval - 1,
 *  where
 *      N*P .... for N selected pixel and P photographs
 *      +1 ..... for the fixing equation g(z_mid) = 0.
 *      n_zval-2 for the smothness equations (excluding the both borders).
 *  The number of unknowns to determine is 
 *      n_var = n_zval + N,
 *  where
 *      n_zval: the n_zval (logarithm) exposure values corresponding to the
 *              n_zval possible output values (0..255) of an CCD-element
 *              ("pixel").
 *      N.....: the N unknow (logarithm) irradiance values at the N selected
 *              pixels (assumed to be constant over the P photographes of
 *              the same scene with different exposure times).
 */
template <class Unsign,class Real>
void
CCDResponse<Unsign,Real>::calc (Real lambda, bool debug)
{
  int N = Z.dim1();                   // Zahl ausgewaehlter Orte (Kamerapixel)
  int P = Z.dim2();                   // Zahl der Aufnahmen
  int n_eq  = N * P + n_zval - 1;     // Zahl der Gleichungen
  int n_var = n_zval + N;             // Zahl der Unbekannten

  std::cout << "N=" << N << ", P=" << P << ", n_zval=" << n_zval << "\n";
  std::cout << "n_eq=" << n_eq << ", n_var=" << n_var << "\n";

  Array2D<Real> A (n_eq, n_var, 0.);  // Koeff.-Matrix von Ax=b
  Array1D<Real> b (n_eq, 0.);         // rechte Seite von Ax=b

  // A und b belegen:
  // Include the (N*P) data-fitting equations...
  int k=0;
  for (int i=0; i < N; i++)           // ueber alle Orte
      for (int j=0; j < P; j++)       // ueber alle Aufn., dh Zeiten
      {
          Real wij = wicht( Z[i][j] );
          if (wij == 0.0)
              continue;               // Nullzeilen ueberspringen
          A [k][ Z[i][j]  ] = wij;
          A [k][n_zval + i] = -wij;
          b [k] = wij * B[j];
          k++;
      }

  // Fix the curve by setting its middle value to 0...

  A [k][n_zval/2] = 1.0;              // dh. g_{mid} = b[k] = 0
  k++;

#if 0
  // Fix the curve by setting its middle value to the corresponding
  // original CCD value (for testing)

  double X_mid = 0.5 * (ccd.get_min_in() + ccd.get_max_in());
  Unsign z_mid = ccd.kennlinie (X_mid,false);
  std::cout << "X_mid=" << X_mid
            << ",  z_mid=" << z_mid
            << ",  log(X_mid)=" << log(X_mid) << "\n";
  A [k][z_mid] = 1.0;
  b [k] = log(X_mid);            // dh. g_{mid} = log(X_mid)
  k++;
#endif

  // Include the (n-2) smoothness equations...

  for (size_t i=0; i < n_zval-2; i++)
  {  
      Real val = lambda * wicht(i+1);
      A [k][i  ] = val;           // lambda * wicht(i+1);
      A [k][i+1] = -2.0 * val;    // -2. * lambda * wicht(i+1);
      A [k][i+2] = val;           // lambda * wicht(i+1);
      k++;
  }

  // If null-rows were skipped reduce A and b to their effektiv size

  if (k < n_eq)
  {   std::cout << n_eq - k << " Nullzeilen. Effektiv ein ("
                << k << "," << n_var << ")-Problem\n";
      A = A.subarray (0,k-1, 0,n_var-1);
      b = b.subarray (0,k-1);
  }
  if (debug)
  {   std::cout << "A = " << A;
      std::cout << "b = " << b;
  }

  Stopwatch uhr;
  double zeit;

  // SVD...
/*
  uhr.start();
  JAMA::SVD <Real> svd(A);
  zeit = uhr.stop();
  std::cout << "Zeit = " << zeit << " sec\n";

  std::cout << "Rang = " << svd.rank() << '\n';
  std::cout << "cond = " << svd.cond() << '\n';
  std::cout << "norm = " << svd.norm2() << '\n';

  TNT::Array2D <Real> U,V,S, Aback;
  TNT::Array1D <Real> s;

  svd.getU (U);
  svd.getV (V);
  svd.getS (S);
  svd.getSingularValues (s);
  std::cout << "s = " << s;

  Aback = mat_dot_mat(U, mat_dot_matT(S,V));

  // std::cout << "U * S * V^T = " << Aback;
  // std::cout << "U * S * V^T - A = " << Aback - A;
  list (Aback, 1e-13);
*/

  // QR-Decompostion...

  uhr.start();
  JAMA::QR <Real> qr(A);
  zeit = uhr.stop();
  std::cout << "Zeit = " << zeit << " sec\n";

  std::cout << "QR::isFullRank = " << qr.isFullRank() << "\n";


  // Quadratmittelproblem:

//Array1D <Real> x_svd = svd.solve (b);
  Array1D <Real> x_qr = qr.solve (b);

  if (debug)
  {   //std::cout << "SVD-Loesung = " << x_svd;
      std::cout << "QR-Loesung = " << x_qr;
      //std::cout << "svd - qr = " << x_svd - x_qr;
  }
  
  // std::cout << "Proof SVD...\n";
  // proof_least_square_fit (x_svd, A, b);

  std::cout << "Proof QR...\n";
  proof_least_square_fit (x_qr, A, b);


  //logXcrv = x_qr;         // Schlecht: dim(x_qr) > n_zval!!

  logXcrv = Array1D<Real> (n_zval);
  Xcrv    = Array1D<Real> (n_zval);

  for (size_t z=0; z < n_zval; z++) 
  {
    logXcrv[z] = x_qr [z];    
    Xcrv   [z] = exp (logXcrv[z]);
  }

  //write_Xcrv ("xcrv.dat");
}

/**
 * write_Xcrv()  
 * 
 * @param fname: Dateiname
 * @param log: Falls true, wird logarithm. Xcrv geschrieben und automatisch
 *    "log_" vor den Dateinamen gehaengt.
 */
template <class Unsign, class Real>
void
CCDResponse<Unsign,Real>::write_Xcrv (const char* fname, bool log)
{
  char* name;

  if (log) {
    name = (char*)malloc (strlen(fname)+5);    // +5 == "log_" + '\0'
    sprintf (name, "log_%s", fname);
  }
  else
    name = (char*)fname;

  printf ("Schreibe Datei %s\n", name);
  FilePtr f(name, "wt");                  // ueberschreibt ohne Warnung

  if (log) {
    for (size_t i=0; i < n_zval; i++)
      fprintf (f, "%i  %f\n", i, logXcrv[i]);
    
    free (name);
  }
  else 
    for (size_t i=0; i < n_zval; i++)
      fprintf (f, "%i  %f\n", i, Xcrv[i]);
}



#if 0

/*
 * Wann lief das Untige denn als Hauptprogramm, was ist [war] denn 'camera'
 * fuer ein Objekt, was 'pixel'??? Hergott nochmal! Klingt nach fruehem Anfang,
 * gehoerte dann wohl kaum noch hier her.
 */

typedef unsigned char uint8;

//typedef float Real;
typedef double Real;
//typedef uint8 Unsign;
typedef int Unsign;

int main ()
{
    CCDResponse<Unsign,Real> brack (camera, pixel);
    Real lambda = 1.0;
    do {
        std::cout << "Eingeben: lambda ["<<lambda<<"] = ";
        std::cin >> lambda;
        std::cout << "lambda = " << lambda << "\n";
        brack.calc(lambda, false);
    } while(true);

    return 0;
}

#endif  // if 0|1

#endif  // RESPONSE_HPP

// END OF FILE
