/*
 * Br2HdrManager.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file Br2HdrManager.hpp
*/
#ifndef Br2HdrManager_hpp
#define Br2HdrManager_hpp

//#include <iostream>
#include "br_types.hpp"            // uint8, uint16
#include "br_Image.hpp"            // BrImage, BrImgVector, ImageHDR
#include "HdrCalctor_RGB_U8.hpp"   // HdrCalctor_RGB_U8
#include "HdrCalctor_RGB_U16_as_U8.hpp"  // HdrCalctor_RGB_U16_as_U8
#include "Distributor.hpp"         // ValueDistributor<>
#include "StatusLineBase.hpp"      // StatusLineBase
#include "ResponseSolverBase.hpp"  // ResponseSolverBase::SolveMode



namespace br {

/**===========================================================================

  @class  Br2HdrManager
  
  This class manages interplay between input image container and HdrCalctor.
   GUI independent!

  Note: Distribution of the enum type `Event' with ValueDistributor<T>.value(..)
   requires `value(T)' or `value(const T&)'; not works `value(T&)'.

  <b>Leistungen:</b>
   - Kontaktierung des input image containers `BrImgVector' mit dem `HdrCalctor'.
   - BrImgVector is type-independent, HdrCalctor's are typed.
   - Kreieren eines Pack aus BrImgVector. Verschiedene Pack-Typen denkbar.
   - Einfuegen eventl. Verschiebungskorrekturen
   - Initialisierung (Erzeugen) des HdrCalctor's passenden Typs.
   - Zentraler Ansprechpartner fuer Auftraege und Anfragen der GUI, Vorhalten 
      aller (numerischen) Parameter. Muss also globales singulaeres Objekt geben.
   - Entgegennahme der Nutzerwuensche, deren Umsetzung + Versenden informierender
      Rundrufe -- das Objekt "strahlt".
   - REGEL: Nur Br2HdrManager soll senden!
  
  Die beiden virtuellen Funktionen `cursor_wait()' und `cursor_default()' hier
   angesiedelt, weil hier naeher an den Rechnungen und besserer Einblick, wann
   etwas "dauert". Die GUI sollte diese Funktionen passend dann definieren, in
   Fltk durch fl_cursor(FL_CURSOR_WAIT) und fl_cursor(FL_CURSOR_DEFAULT).
  
  Warum numerische Parameter ausser im Calctor auch in Br2HdrManager? 
   Weil Calctoren initialisiert werden muessen und vernichtet werden koennen.
   Nur die Br2HdrManager-Werte ueberdauern.
        
  <b>refpic:</b>
   Wann *automatisch* auf "guenstigen" Wert setzen? Bei jedem init_Calctor()
   oder nur beim Hinzufuegen von Bildern? Z.Z. bei jedem init.
  
  - Namensgebung ganz schoener Mischmasch. "get_ResponseCurve" oder "getResponseCurve"
     oder "get_response_curve"? Ferner "ExternResponse" oder "ResponseExtern"?

  @NOTE Often the weight functions for response computation and HDR-merging will
   be the same. For saving memory the pointers <tt>pWeightResp_</tt> and
   <tt>pWeightMerge_</tt> could be pointed then to the same weight function object.
   For a <i>tabled</i> WeightFunc_U16 of 512 kB this would be even a necessisty,
   but less urgent for WeightFunc_U8 with 2 kB. Currently we handle U16 data with
   <tt>HdrCalctor_U16_as_U8</tt>. Here is needed one WeightFunc_U16 only, whereas
   for the response computation a WeightFunc_U8 is used. So maximal either two
   WeightFunc_U8's or one WeightFunc_U8 + one WeightFunc_U16 are needed at the same
   time. To hold the code clearer we create at present always two weight function
   <i>objects</i>.
   
*============================================================================*/
class Br2HdrManager 
{
public:
    
    enum Event 
    {  NO_EVENT          = 0,   ///< not set
       CALCTOR_INIT      = 1,   ///< Calctor-init happend
       CALCTOR_OUTDATED  = 2,   ///< activated images != images used in Calctor
       CALCTOR_REDATED   = 3,   ///< Calctor got up-to-date again without init
       CALCTOR_DELETED   = 4,   ///< Calctor==0
       IMAGE_LOADED      = 5,   ///< image loaded, imgVec changed
       IMAGES_CHANGED    = 6,   ///< imgVec changed (e.g. cleared)
                                ///<   (Can '5' subsumes '4'? We will see.)
       CCD_UPDATED       = 7,   ///< new response (CCD) curves computed
       CCD_OUTDATED      = 8,   ///< any change happend that could give another
                                ///<   CCD curves than the existing
       CCD_DELETED       = 9,   ///< Subsumed by CALCTOR_INIT & ~_DELETED?
       HDR_UPDATED       = 10,  ///< HDR image created or updated
       TIMES_CHANGED     = 11,  ///< exposure times changed
       FOLLOWUP_UPDATED  = 12,  ///< Follow-up values updated
       FOLLOWUP_OUTDATED = 13,  ///< Follow-up values outdated
       EXTERN_RESPONSE   = 14,  ///< change in usage of extern response curves
       WEIGHT_CHANGED    = 15,  ///< weight function has been changed
       EVENT_UNKNOWN     = 16   ///< marks also the end of this set
    };                          
    ///  NOTE: Update always "br_eventnames.hpp" too, if you change `Event'!
        
    enum WhichCurves
    {  USE_NEXT          = 0,
       COMPUTED          = 1,
       EXTERNAL          = 2,
       CALCTOR_S         = 3
    };
        
   
private:
    //  For functions returning a reference, we need a static "null curve"
    static TNT::Array1D<double>  null_response_curve_;

    //  Our input image container
    BrImgVector  imgVec_;
    
    //  The (dynamic created) Hdr-Calculator(s)
    HdrCalctor_RGB_U8*         pCalctor_RGB_U8;  
    HdrCalctor_RGB_U16_as_U8*  pCalctor_RGB_U16_as_U8;  

    //  Base pointer of the current created HdrCalctor
    HdrCalctorBase*  pCalctorBase_;
    
    //  Shape of our weight functions (intended or realized)
    WeightFuncBase::Shape  weight_func_shape_resp_; 
    WeightFuncBase::Shape  weight_func_shape_merge_; 
    
    //  The dynamic created weight function objects for U8 and U16 data
    WeightFunc_U8*   pWeightResp_U8_;
    WeightFunc_U8*   pWeightMerge_U8_;
    WeightFunc_U16*  pWeightMerge_U16_;
    
    //  Is the set of images used in Calctor identic with that currently activated in image container?
    bool  images_uptodate_;
       // Regime for setting: 
       //  Set it true by: init Calctor
       //  Set it false by: Loading of an image, de/activation of an image

    //  Are the last computed response curves `Xcrv_computed_ up-to-date?
    bool  response_uptodate_;   
           
    //  Update follow-up values automatically when ever they have got out-of-date?
    bool  auto_update_followup_;
       
    //  Use extern response curves for HDR merging (instead of computing them from image data)
    bool  use_extern_response_;
    
    //  For which reference picture the follow-up values shall be computed?
    int  refpic_;

    //  Numeric parameter: number of selected spot points for response computation
    int  n_selectpoints_;
    
    //  Numeric parameter: smoothing coefficent for response computation
    double  smoothing_;
    
    //  Mode for solving the rectangular `Ax=b' problem: AUTO, USE_QR, USE_SVD
    ResponseSolverBase::SolveMode  solve_mode_;
    
    //  Method for generating of the "Z-Matrix" (maybe temporarily)
    int  method_Z_;
    
    //  The "stop value" for automatic time generation
    double  stopvalue_;
    
    //  Mark "bad (unresolved) pixel" in the generated HDR-image by contrast colors?
    bool  mark_bad_pixel_;
    
    //  Generate a bad pixel protocol to file or to stdout while merging?
    bool  protocol_to_file_;
    bool  protocol_to_stdout_;

    
    //  Response curves (X vs. z) for the 3 channels
    TNT::Array1D<double>  Xcrv_use_next_[3];    // using next for merging
    TNT::Array1D<double>  Xcrv_computed_[3];    // last computed curves
    TNT::Array1D<double>  Xcrv_extern_[3];      // loaded from file
    
    //  Which "use_next" response curve is a computed one?
    bool is_usenext_curve_computed_[3];
    
    //  Filenames of last loaded external response curves
    char* fname_curve_extern_[3];

    //  Distributor(en):
    ValueDistributor<Event>  distrib_event_;
    ValueDistributor<int>    distrib_refpic_;

        
protected:    
    //  Pointer to our ProgressInfo instance (given in Ctor)  
    ProgressInfo *    progressinfo_;  
    StatusLineBase *  statusLineBase_;

        
public:
    //  Ctor & Dtor...
    Br2HdrManager (ProgressInfo* = 0);
    virtual ~Br2HdrManager();
    
    //  Hands out the base pointer of the (current) Calctor...
    const HdrCalctorBase * calctor() const    {return pCalctorBase_;}
    HdrCalctorBase       * calctor()          {return pCalctorBase_;}
    
    ValueDistributor<Event> & distribEvent()  {return distrib_event_;}
    ValueDistributor<int> &   distribRefpic() {return distrib_refpic_;}

    //==================================================================
    //  General Initialisation && Clearing...
    //==================================================================
          
    //  Init (create) a Calctor
    void init_Calctor();
    
    //  Destroy the existing Calctor
    void clear_Calctor();     
    
    //  Clear the input image container
    void clear_Container();
    
    //  Clear Calctor && container
    void clear_ContAndCalc()            {clear_Calctor(); clear_Container();}
    
    //  Clear the filenames of the last loaded external response curves
    void clear_fname_curve_extern();
    
    //  Clear set of external response curves incl. filenames
    void clear_ResponseExtern();
    
    //==================================================================
    //  Adding of input images...
    //==================================================================
    
    //  Add an image to the input container
    void add_Image (BrImage & img);
    bool check_image_to_add (BrImage & img);
    
    //==================================================================
    //  General status information...
    //==================================================================

    //  Number of images in input container
    int  size() const                   {return imgVec_.size();}
    
    //  Number of activated images in input container
    int  size_active() const            {return imgVec_.size_active();}
    
    //  Are the images used in Calctor up-to-date?
    bool areImagesUptodate() const      {return images_uptodate_;}
    
    //  Are the computed response curves up-to-date?
    bool isComputedResponseUptodate() const {return response_uptodate_;}
                      
    //  The "stop value" for automatic time generation; set via `set_TimesByStop()'
    double stopvalue() const            {return stopvalue_;}
    //void   reset_stopvalue(double val) {stopvalue_ = val;}
    
    //=====================================================================
    //  Hand through input container functionality (hides the container)...
    
    DataType data_type() const          {return imgVec_.data_type();}
    
    //==================================================================
    //  Hand through data of the input image with index 'i'...
    //==================================================================
    BrImage &       getImage (int i)            {return imgVec_[i];}
    const BrImage & getImage (int i) const      {return imgVec_[i];}
    ImageID     imageID    (int i) const;
    const char* name       (int i) const {return imgVec_.at(i).name();}
    bool        isActive   (int i) const {return imgVec_.at(i).active();}
    int         width      (int i) const {return imgVec_.at(i).width();}
    int         height     (int i) const {return imgVec_.at(i).height();}
    double      time       (int i) const {return imgVec_.at(i).exposure.time;}
    double      brightness (int i) const {return imgVec_.at(i).brightness();}
    double      relWkRange (int i) const {return imgVec_.at(i).statistics.r_wk_range;}

    //==================================================================
    //  Setting of data of the input images... (with broadcasting)
    //==================================================================
    
    //  Activate and deactivate the i-th image in input container
    void activate (int i);
    void deactivate (int i);
    
    //  Set exposure time of input image `i' to the value `tm'
    void set_Time (int i, double tm);
    
    //  Automatic time generation by a stop value; sets also `stopvalue_'.
    void set_TimesByStop (double stop); 
    
    //==================================================================
    //  Setting && getting of numeric Calctor parameters...
    // 
    //  Function name scheme:
    //   T  param() const         - return the current value
    //   void set_param (T val)   - set with broadcasting
    //   void reset_param (T val) - set w/out broadcasting
    //==================================================================
    
    int    nSelectPoints() const         {return n_selectpoints_;}
    void   set_nSelectPoints (int n);
    void   reset_nSelectPoints (int n);    
    
    double smoothing() const             {return smoothing_;}
    void   set_smoothing (double val);
    void   reset_smoothing (double val);

    int    refpic() const                {return refpic_;}
    void   set_refpic (int pic); 
    void   reset_refpic (int pic);
    
    //==========================
    //  Other Calctor parameter
    //==========================
    
    bool   mark_bad_pixel()  const  {return mark_bad_pixel_;}
    void   mark_bad_pixel (bool b)  {mark_bad_pixel_ = b; 
                                     if (calctor()) calctor()->mark_bad_pixel(b);}
    
    bool   protocol_to_file() const  {return protocol_to_file_;}
    void   protocol_to_file (bool b) {protocol_to_file_ = b;
                                      if (calctor()) calctor()->protocol_to_file(b);}
    
    bool   protocol_to_stdout() const  {return protocol_to_stdout_;}
    void   protocol_to_stdout (bool b) {protocol_to_stdout_ = b;
                                        if (calctor()) calctor()->protocol_to_stdout(b);}
    
    ResponseSolverBase::SolveMode solve_mode() const    {return solve_mode_;}
    void   solve_mode (ResponseSolverBase::SolveMode m) {solve_mode_ = m;
                                                         if (calctor()) calctor()->solve_mode(m);}

    int    method_Z() const             {return method_Z_;}
    void   method_Z (int m)             {if (m<1 || m>2) return;
                                         method_Z_ = m;                                                        
                                         if (calctor()) calctor()->method_Z(m);}
                                         
    //==================================================================
    //  Relation between image container and Calctor...
    //==================================================================
    
    //  Return input index (in imgVec) of image with Calctor index `k'
    int get_InputIndex (int k) const;
    
    //  Return Calctor index of input image with index `k'
    int get_CalctorIndex (int k) const;

    //  Return true, if input image with index `k' is used in Calctor
    bool isUsedInCalctor (int k) const;

    //  Update times in `Calctor' by those of the image container
    void update_CalctorTimes();  
      
    //==================================================================
    //  Follow-up curves...
    //==================================================================

    //  Have we follow-up curves ready in Calctor?
    bool haveFollowUpCurves() const     {return calctor() && calctor()->haveFollowUpCurves();}
    
    //  Shall we update follow-up curves automatically?
    bool isAutoUpdateFollowUp() const   {return auto_update_followup_;} 
    void setAutoUpdateFollowUp (bool b) {auto_update_followup_ = b;}

    //==================================================================
    //  Response curves...
    //==================================================================

    TNT::Array1D<double> & getResponseCurveComputed (int channel)   {return Xcrv_computed_[channel];}
    TNT::Array1D<double> & getResponseCurveExtern (int channel)     {return Xcrv_extern_[channel];}
    TNT::Array1D<double> & getResponseCurveUsenext (int channel)    {return Xcrv_use_next_[channel];}
    TNT::Array1D<double> & getResponseCurve (WhichCurves, int channel);
    
    //  Is the set of use_next|computed|external response curves ready for use?
    bool isUsenextResponseReady() const;
    bool isComputedResponseReady() const;
    bool isExternResponseReady() const;
    bool isResponseReady (WhichCurves) const;
    
    //  Is the set of use_next|computed|external curves empty?
    bool isUsenextResponseEmpty() const;
    bool isComputedResponseEmpty() const;
    bool isExternResponseEmpty() const;
    bool isResponseEmpty (WhichCurves) const;
    
    //  Is any of the "use_next" response curves a computed one?
    bool anyUsenextCurveComputed() const;
    //  Is the "use_next" curve for `channel' a computed one?
    bool isUsenextCurveComputed (int channel)   {return is_usenext_curve_computed_[channel];}
        
    //  Use external response curves only and switch off response computation
    bool isUseExternResponse() const            {return use_extern_response_;}
    void setUseExternResponse (bool b);
    
    bool applyResponseCurveComputed (int channel);
    bool applyResponseCurveExtern (int channel);

    bool write_ResponseCurve (WhichCurves, int channel, const char* fname);
    bool read_ResponseCurve  (int channel, const char* fname);
    
    bool init_ResponseForMerging();
    
    const char* getFilenameCurveExtern (int channel)    {return fname_curve_extern_[channel];}
    
    //==================================================================
    //  Weight function...
    //==================================================================

    void setWeightResp  (WeightFuncBase::Shape);
    void setWeightMerge (WeightFuncBase::Shape);
    void setWeight      (WeightFuncBase::Shape);
    WeightFuncBase::Shape weightShapeResp() const       {return weight_func_shape_resp_;}
    WeightFuncBase::Shape weightShapeMerge() const      {return weight_func_shape_merge_;}
    
    //==================================================================
    //  Computation calls...
    //==================================================================
    
    //  Compute & update follow-up values for the current `refpic_'
    void compute_FollowUpValues();
    void update_FollowUpValues(); 
    void update_FollowUpValues (int channel)    {update_FollowUpValues();}
    void watch_FollowUpValues();
    
    //  Compute response curves (incl. *update* of FollowUpValues)
    void compute_Response();
    
    //  Pure merging of an HDR image
    ImageHDR merge_HDR();
    
    //  Pure merging of an log. HDR image
    ImageHDR merge_LogHDR();
    
    //  Compute response if not existent + merge HDR
    ImageHDR complete_HDR();
    
    //  Compute response if not existent + merge log HDR
    ImageHDR complete_LogHDR();
    
    //  Init Calctor + response + merge
    ImageHDR make_HDR();
    
    //==================================================================
    //  Miscellaneous...
    //==================================================================
    
    //  "Abstract" cursor functions: to implement by a GUI-dependent heir
    virtual void cursor_wait()          {std::cout<<__func__<<'\n';}
    virtual void cursor_default()       {std::cout<<__func__<<'\n';}

            
    //  Set/Change the ProgressInfo pointer
    void set_progressinfo (ProgressInfo* pr) {
      progressinfo_ = pr;
      if (calctor()) calctor()-> progressinfo (pr);
    }
    
    //  Simplifying usage of `progressinfo_' pointer...
    void progressinfo_value (float v) 
      {if (progressinfo_) progressinfo_-> value(v);}
    
    void progressinfo_text (const char* s)
      {if (progressinfo_) progressinfo_-> text(s);}
    
    void progressinfo_show()
      {if (progressinfo_) progressinfo_-> show();}
    
    void progressinfo_hide()
      {if (progressinfo_) progressinfo_-> hide();}
     
      
    //  Set statusline pointer:
    void set_statusline (StatusLineBase* st)  {statusLineBase_ = st;}
      
    //  Simplifying usage of `statusLineBase_' pointer...
    void statusline_out (const char* s)
      {if (statusLineBase_) statusLineBase_-> out(s);}
    
    void statusline_out_default()
      {if (statusLineBase_) statusLineBase_-> out_default();}  
      
    StatusLineBase* statusline()
      {return statusLineBase_;}  
      
    //==================================================================
    //  List some debug information on console...
    //==================================================================
    
    void report_NumericParams() const;
    void report_CalctorParams() const;
    void report_Indices() const;
    void report_Images(ReportWhat w=REPORT_BASE) const  {imgVec_.report(w);}
    void report_StatusResponseCurves() const;
    void report_WeightFunc() const;
    

private:
    //  Helper of init_Calctor();
    void create_Calctor_RGB_U8_();
    void create_Calctor_RGB_U16_as_U8_();
    
    //  Helper of create_Calctor_RGB_... and setWeightShape()
    void createWeightResp_U8_(WeightFuncBase::Shape);
    void createWeightMerge_U8_(WeightFuncBase::Shape);
    void createWeightMerge_U16_(WeightFuncBase::Shape);
    
    //  Helper of compute_Response()
    void update_FollowUpValues_aux();
};


}  // namespace

#endif  // Br2HdrManager_hpp

// END OF FILE
