/*
 * br_ImgScheme2D.hpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  @file br_ImgScheme2D.hpp  -  typcasting 2D-views on `Image' objects
  
  Content:
   - ImgArray2DView<>
   - ImgArray2D<>
   - ImgScheme2DView<>
   - ImgScheme2D<>
  
  FRAGE: Wie macht man das ImgScheme2D eines Image-Objekts bzgl. des Image-
   Bildpuffers referenzzaehlend und damit speicherblockierend?  
  
  ANTWORT: Indem die Scheme-Klasse ein Array1D<uchar>-Element erhaelt, das im
   Scheme-Ctor mit dem uebergebenen Image-Objekt initialisiert wird (Image ist
   von Array1D<uchar> abgeleitet!). Der Copy-Ctor des Array1D erhoeht dann den
   Ref-Zaehler des Image-Bildpuffers. Ebenso wird bei jedem Kopieren von
   ImgScheme2D's (default Copy-Op!) das Array1D-Element als Element kopiert, dh.
   dessen ref-zaehlender Copy-Ctor springt an.
  [Nur fuer ImgArray2DView<>:]
   Die auch moegliche Variante, das ImgScheme von Array1D<uchar> *abstammen* zu
   lassen und dafuer das Array2D<T> als Element zu halten, hat den Nachteil,
   dass die SubArray-Funktionalitaet fuer das Scheme dann erst nachgebaut werden
   muss.
   
  FRAGE: Bei den nicht ref-zaehlenden View's Kopieren nicht besser verbieten?
*/
#ifndef br_ImgScheme2D_hpp
#define br_ImgScheme2D_hpp


#include "TNT/tnt_array2d.hpp"  // TNT::Array2D<>
#include "Scheme2D.hpp"         // Scheme2D<>
#include "Scheme2D_utils.hpp"   // '<<' for Scheme2D<>
#include "br_Image.hpp"         // Image


namespace br {

/**============================================================================

  ImgArray2D<>  &  ImgArray2DView<>
 
  Typecasting 2D-view of an Image object using TNT::Array2D<> (C-array Ctor). 
   Provides "[][]"-syntax. TNT::Array2D<> allocates an auxillary array of 
   dim1() T-pointers. 

  Alternative zu ImgScheme2D***<>.
   Vorteil: Sub-Array-Unterstuetzung.
   Nachteil: Alloziiert und wartet [dim1]-Feld von Zeigern.
 
*============================================================================\n
  @class ImgArray2DView  -  template
 
  - No increasing of the img-buffers ref-counter (no data locking!) 
  - No ref-counting copy-semantik concerning the image buffer.
*============================================================================*/
template <typename T>  
class ImgArray2DView : public TNT::Array2D<T>
{
public:
    ImgArray2DView (Image & img) 
      : TNT::Array2D<T> (img.dim1(), img.dim2(), (T*)img.buffer()) {}
    
    ImgArray2DView (const Image & img) 
      : TNT::Array2D<T> (img.dim1(), img.dim2(), (T*)img.buffer()) {}
};


/**==========================================================================\n
  @class ImgArray2D  -  template
 
  - Increases the ref-counter of the image buffer (data locking!) 
  - Ref-counting copy-semantik for the image buffer.
*============================================================================*/
template <typename T>  
class ImgArray2D : public TNT::Array2D<T>
{
    TNT::Array1D<unsigned char>  my_refcopy_of_img_buffer;

public:
    ImgArray2D (Image & img) 
      : TNT::Array2D<T> (img.dim1(), img.dim2(), (T*)img.buffer()),
        my_refcopy_of_img_buffer (img)
      {}
    
    ImgArray2D (const Image & img) 
      : TNT::Array2D<T> (img.dim1(), img.dim2(), (T*)img.buffer()),
        my_refcopy_of_img_buffer (img)
      {}
};



/**============================================================================

  ImgScheme2D<>  &  ImgScheme2DView<>

  Typecasted 2D-view of an Image object using Scheme2D<>. Provides "[][]" syntax. 
  
  Alternative zu ImgArray2D***<>.
   Vorteil: alloziiert und wartet kein [dim1]-Feld von Zeigern
   Nachteil: Element-Zugriff hat eine int-Multpl. mehr. Aber oft ohnehin 
             wegoptimiert. Keine Sub-Array-Unterstuetzung.

*============================================================================\n
  @class ImgScheme2DView  -  template

  - No increasing of the img buffers ref-counter (no data locking!) 
  - No ref-counting copy-semantik concerning the image buffer.
*============================================================================*/
template <typename T>           
class ImgScheme2DView : public Scheme2D <T>
{
public:
    ImgScheme2DView (Image & img) 
      : Scheme2D<T> (img.dim1(), img.dim2(), img.buffer())  {}
    
    ImgScheme2DView (const Image & img) 
      : Scheme2D<T> (img.dim1(), img.dim2(), img.buffer())  {}
      
    ImgScheme2DView() {}  // Zum Anlegen von Vektoren Default-Ctor noetig
};   


/**==========================================================================\n
  @class ImgScheme2D  -  template
 
  - Increases the ref-counter of the image buffer (data locking!) 
  - Ref-counting copy-semantik for the image buffer.
*============================================================================*/  
template <typename T>           
class ImgScheme2D : public Scheme2D <T>
{
    TNT::Array1D<unsigned char>  my_refcopy_of_img_buffer;

public:
    ImgScheme2D (Image & img) 
      : Scheme2D<T> (img.dim1(), img.dim2(), img.buffer()),
        my_refcopy_of_img_buffer (img)
      {}
    
    ImgScheme2D (const Image & img) 
      : Scheme2D<T> (img.dim1(), img.dim2(), img.buffer()),
        my_refcopy_of_img_buffer (img)
      {}
      
    ImgScheme2D() {}    // Zum Anlegen von Vektoren Default-Ctor noetig
};   


template <typename T>
std::ostream& operator<<(std::ostream &s, const ImgScheme2D<T> &A)
{
    s << "Image-" << (Scheme2D<T>&)A; 
    return s;
}

template <typename T>
std::ostream& operator<<(std::ostream &s, const ImgScheme2DView<T> &A)
{
    s << "Image-View-" << (Scheme2D<T>&) A; 
    return s;
}



#if 0
//=============================================================================
// GEDANKE: ImgScheme2D von Image ableiten. Konstruieren dann mit *default* 
//  Copy-Ctor von Image (kopiert nur Zeiger). Vorteil: Alle imgData-Funktionen
//  sogleich verfuegbar. Nachteil: Kopiert auch unnoetige Elemente wie `DataType'.
// BUG: Da Dtor von Image freigibt, koennen keine automatischen ImgScheme2D-
//  Objekte erzeugt werden von Image-Objekten, die eingereiht sind. LOesung(en):
//  Image-Dtor virtual und Scheme2D ueberschreibt diesen.
//=============================================================================
template <typename T>           
class ImgScheme2D : public Image
{
public:
    ImgScheme2D (Image       & img) : Image (img)  {}
    ImgScheme2D (const Image & img) : Image (img)  {}  
    
    const T* operator[] (int i) const {return & ((T*)buffer())[ i * dim2() ];}
    T*       operator[] (int i)       {return & ((T*)buffer())[ i * dim2() ];}
};     
#endif



}  // namespace "br"

#endif  // br_ImgScheme2D_hpp

// END OF FILE
