/*
 * buttons.cpp  --  Part of the CinePaint plug-in "Bracketing_to_HDR"
 *
 * Copyright (c) 2005-2006  Hartmut Sbosny  <hartmut.sbosny@gmx.de>
 *
 * LICENSE:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/**
  buttons.cpp
*/
#include <FL/fl_draw.H>                 // fl_color_average()
#include "buttons.hpp"
#include "../br_core/br_defs.hpp"       // BR_DEBUG_RECEIVER
#include "../bracketing_to_hdr.hpp"     // cpaint_show_image()

#ifdef BR_DEBUG_RECEIVER
#  include <iostream>
   using std::cout;
#endif


static Fl_Color col_needed = fl_color_average (FL_YELLOW, FL_WHITE, 0.5);
static Fl_Color col_nonexist = fl_color_average (FL_BLUE, FL_WHITE, 0.25);



using namespace br;

//============================================================================
//
//  class InitCalctorButton
//
//============================================================================

InitCalctorButton::InitCalctorButton (int X, int Y, int W, int H, const char* L)
  : Fl_Button (X,Y,W,H,L),
    the_Br2Hdr (Br2Hdr::Instance())
{
    CTOR("");
    col_default = color();  // FLUID's changes would come behind (-> extra_init())
    callback (cb_fltk_, this);
}  

//  Put it in FLUID's "Extra Code" field to consider changes made in FLUID
void InitCalctorButton::extra_init() 
{
    col_default = color();
    handle_Event(Br2Hdr::NO_EVENT);  // sets start-color and activation
}


void InitCalctorButton::handle_Event (Br2Hdr::Event e) 
{
#ifdef BR_DEBUG_RECEIVER
    cout<<"InitCalctorButton::"; EventReceiver::handle_Event(e);
#endif
    
    //  Button color: depending on Calctor existence as well as the Event...
    if (!the_Br2Hdr.calctor()) 
    {
      if (color() != col_nonexist) {color(col_nonexist); redraw();}
    }
    else switch (e)        // Calctor exists    
    {
      case Br2Hdr::CALCTOR_INIT:
      case Br2Hdr::CALCTOR_REDATED:
          BR_EVENT_HANDLED(e);
          color(col_default); redraw();
          break;
        
      case Br2Hdr::CALCTOR_OUTDATED:
          BR_EVENT_HANDLED(e);
          color(col_needed); redraw(); 
          break;

      default: BR_EVENT_NOT_HANDLED(e);
          break;
    }
  
    //  De|activation: depending on size_active():
    if (the_Br2Hdr.size_active() > 1) 
      {if (!active()) activate();}
    else 
      {if (active()) deactivate();}
}      


//============================================================================
//
//  class ComputeResponseButton
//
//============================================================================

ComputeResponseButton::ComputeResponseButton (int X, int Y, int W, int H, const char* L)
  : Fl_Button (X,Y,W,H,L),
    the_Br2Hdr (Br2Hdr::Instance())
{
    CTOR("");
    col_default = color();  // FLUID's changes would come behind (-> extra_init())
    callback (cb_fltk_, this);
}  

/**+*************************************************************************\n
  Put it in FLUID's "Extra Code" field to consider changes made in FLUID
******************************************************************************/
void ComputeResponseButton::extra_init() 
{
    col_default = color();
    handle_Event(Br2Hdr::NO_EVENT);  // sets start-color and activation
}

/**+*************************************************************************\n
  "No-existence" color only, if computation possibly, i.e. if calctor exists.
  
  @NOTE Falls gewuenscht, dass ein inaktiver Button keine Farbaenderungen erleidet,
   waere in den case-Zweigen die Abfrage "if (active())" vorzuschalten und im 
   De/Aktivierungs-Teil im "activate()"-Zweig die Farbe gemaess des momentanten 
   Zustandes einzustellen. Denn waehrend der Inaktivphase wurde Farbe ja nicht
   aktuell gehalten.
******************************************************************************/
void ComputeResponseButton::handle_Event (Br2Hdr::Event e)
{
#ifdef BR_DEBUG_RECEIVER
    cout<<"ComputeResponseButton::"; EventReceiver::handle_Event(e);
#endif

    //  Button color: depending on Calctor, Response and the Event
    if (! the_Br2Hdr.calctor())  {
      if (color() != col_default) {color(col_default); redraw();}
    }
    else if (! the_Br2Hdr.isComputedResponseReady()) { // Calctor exists
      if (color() != col_nonexist) {color(col_nonexist); redraw();}
    }
    else switch (e)     // Calctor && Response curves exist
    {
      case Br2Hdr::CALCTOR_INIT:        // should be unreachable by cond above  
        BR_EVENT_HANDLED(e);
        color(col_nonexist); redraw();
        break;
      
      //case Br2Hdr::CALCTOR_OUTDATED: break; // ?
      
      case Br2Hdr::CCD_UPDATED:         
        BR_EVENT_HANDLED(e);
        color(col_default); redraw();
        break;
        
      case Br2Hdr::CCD_OUTDATED:        // i.e. num. params changed
        BR_EVENT_HANDLED(e);
        if (color()!=col_needed) {color(col_needed); redraw();}
        break;  
      
      case Br2Hdr::WEIGHT_CHANGED:      // weight for resp. and/or merging, but
        BR_EVENT_HANDLED(e);            //  only resp. outdates response curves
        if (! the_Br2Hdr.isComputedResponseUptodate())
          if (color()!=col_needed) {color(col_needed); redraw();}
        break;  
      
      default: BR_EVENT_NOT_HANDLED(e);
        break;
    }
  
    //  De|activation: depending on isUseExternResponse and Calctor:
    if (the_Br2Hdr.isUseExternResponse())
      {if (active()) deactivate();}     // denn dann wird nicht gerechnet
    else if (the_Br2Hdr.calctor())
      {if (!active()) activate();}
    else                                // no calctor
      {if (active()) deactivate();}
}      


//============================================================================
//
//  class HDRButton
//
//============================================================================

HDRButton::HDRButton (int X, int Y, int W, int H, const char* L)
  : Fl_Button (X,Y,W,H,L),
    the_Br2Hdr (Br2Hdr::Instance())
{
    CTOR("");
    col_default = color();  // FLUID's changes would come behind (-> extra_init())
    callback (cb_fltk_, this);
}  

/**+*************************************************************************\n
  Put it in FLUID's "Extra Code" field to consider changes made in FLUID
******************************************************************************/
void HDRButton::extra_init() 
{
    col_default = color();
    handle_Event(Br2Hdr::NO_EVENT);  // sets start-color and activation
}

/**+*************************************************************************\n
  Deactivate if: !calctor OR (use_extern_curves AND !ready_usenext_response).
   Denn fuer !use_extern werden fehlende Kurven automatisch berechnet.
   Test auf Calctor solange Merging von Calctor ausgefuehrt.
******************************************************************************/
void HDRButton::handle_Event (Br2Hdr::Event e)
{
#ifdef BR_DEBUG_RECEIVER
    cout<<"HDRButton::"; EventReceiver::handle_Event(e);
#endif

#if 0
    if (! the_Br2Hdr.calctor() || (
        the_Br2Hdr.isUseExternResponse() && ! the_Br2Hdr.isUsenextResponseReady()))
      {if (active()) deactivate();}
    else 
      {if (!active()) activate();}
#else      
    //  De|activation: depending on Calctor:
    if (the_Br2Hdr.calctor())
      {if (!active()) activate();}
    else 
      {if (active()) deactivate();}
#endif      
}      

void HDRButton::cb_fltk_(Fl_Widget*, void*) 
{
    cpaint_show_image (Br2Hdr::Instance().complete_HDR());
}


//============================================================================
//
//  class ComputeFllwUpButton
//
//============================================================================

ComputeFllwUpButton::ComputeFllwUpButton (int X, int Y, int W, int H, const char* L)
  : Fl_Button (X,Y,W,H,L),
    the_Br2Hdr (Br2Hdr::Instance())
{
    CTOR("");
    col_default = color();  // FLUID's changes would come behind (-> extra_init())
    callback (cb_fltk_, this);
}  

/**+*************************************************************************\n
  Put it in FLUID's "Extra Code" field to consider changes made in FLUID
******************************************************************************/
void ComputeFllwUpButton::extra_init() 
{
    col_default = color();
    handle_Event(Br2Hdr::NO_EVENT);  // sets start-color and activation
}


void ComputeFllwUpButton::handle_Event (Br2Hdr::Event e)
{
#ifdef BR_DEBUG_RECEIVER
    cout<<"ComputeFllwUpButton::"; EventReceiver::handle_Event(e);
#endif

    //  Button color: depending on calctor, curves and the Event...
    if (! the_Br2Hdr.calctor()) {
      if (color() != col_default) {color(col_default); redraw();}
    }
    else if (!the_Br2Hdr.haveFollowUpCurves()) {  // calctor exists
      if (color() != col_nonexist) {color(col_nonexist); redraw();}
    }
    else switch (e)     // Calctor && follow-up curves exist
    {
      case Br2Hdr::CALCTOR_INIT:        
        BR_EVENT_HANDLED(e);
        color(col_nonexist); redraw();  // should be unreachable by cond above
        break;
        
      case Br2Hdr::FOLLOWUP_UPDATED:         
        BR_EVENT_HANDLED(e);
        color(col_default); redraw();
        break;
        
      case Br2Hdr::FOLLOWUP_OUTDATED:
        BR_EVENT_HANDLED(e);
        if (color() != col_needed) {color(col_needed); redraw();}
        break;  
      
      default: BR_EVENT_NOT_HANDLED(e);
        break;
    }
  
    //  De|activation: depending on Calctor's existence
    if (the_Br2Hdr.calctor())
      {if (!active()) activate();}
    else 
      {if (active()) deactivate();}
}      

// END OF FILE
