// Copyright (c) 2011 Google, Inc.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <cstdio>
#include <iostream>
#include <string.h>
#include "city.h"
#ifdef __SSE4_2__
#include "citycrc.h"
#endif

using std::cout;
using std::cerr;
using std::hex;

static const uint64 k0 = 0xc3a5c85c97cb3127ULL;
static const uint64 kSeed0 = 1234567;
static const uint64 kSeed1 = k0;
static const uint128 kSeed128(kSeed0, kSeed1);
static const int kDataSize = 1 << 20;
static const int kTestSize = 300;

static char data[kDataSize];

static int errors = 0;  // global error count

// Initialize data to pseudorandom values.
void setup() {
  uint64 a = 9;
  uint64 b = 777;
  for (int i = 0; i < kDataSize; i++) {
    a = (a ^ (a >> 41)) * k0 + b;
    b = (b ^ (b >> 41)) * k0 + i;
    uint8 u = b >> 37;
    memcpy(data + i, &u, 1);  // uint8 -> char
  }
}

#if 1

#define C(x) 0x ## x ## ULL
static const uint64 testdata[kTestSize][15] = {
{C(9ae16a3b2f90404f), C(75106db890237a4a), C(3feac5f636039766), C(3df09dfc64c09a2b), C(3cb540c392e51e29), C(6b56343feac0663), C(5b7bc50fd8e8ad92),
C(3df09dfc64c09a2b), C(3cb540c392e51e29), C(6b56343feac0663), C(5b7bc50fd8e8ad92),
C(95023ca8f708594b), C(1a53a4d030ae7a4c), C(6dcd2262ea627129), C(4d74ec7516ab8204)},
{C(75e9dee28ded761d), C(931992c1b14334c5), C(245eeb25ba2c172e), C(1290f0e8a5caa74d), C(ca4c6bf7583f5cda), C(e1d60d51632c536d), C(cbc54a1db641910a),
C(1290f0e8a5caa74d), C(ca4c6bf7583f5cda), C(e1d60d51632c536d), C(cbc54a1db641910a),
C(f4c86421d64a4389), C(294d0d59683eb5db), C(80cdaadae9b53b1f), C(430875b396e01610)},
{C(75de892fdc5ba914), C(f89832e71f764c86), C(39a82df1f278a297), C(b4af8ae673acb930), C(992b7acb203d8885), C(57b533f3f8b94d50), C(bbb69298a5dcf1a1),
C(b4af8ae673acb930), C(992b7acb203d8885), C(57b533f3f8b94d50), C(bbb69298a5dcf1a1),
C(268ef94b0a4078cd), C(fe947b396ffec889), C(3fa904353795d77d), C(77a18f0f07ac9f7f)},
{C(69cfe9fca1cc683a), C(e65f2a81e19b8067), C(20575ea6370a9d14), C(8f52532fc6f005b7), C(4ebe60df371ec129), C(c6ef8a7f8deb8116), C(83df17e3c9bb9a67),
C(8f52532fc6f005b7), C(4ebe60df371ec129), C(c6ef8a7f8deb8116), C(83df17e3c9bb9a67),
C(847adc1e007da730), C(7d49c8095815cdfd), C(cff14fc5a2491665), C(761b5d3b787a424f)},
{C(675b04c582a34966), C(53624b5ef8cd4f45), C(c412e0931ac8c9b1), C(798637e677c65a3), C(83e3b06adc4cd3ff), C(f3e76e8a7135852f), C(111e66cfbb05366d),
C(798637e677c65a3), C(83e3b06adc4cd3ff), C(f3e76e8a7135852f), C(111e66cfbb05366d),
C(77163a370a416689), C(1c27c0ee7688c56f), C(8b6e8437bbf4f621), C(37f4f4147ad4b53d)},
{C(46fa817397ea8b68), C(cc960c1c15ce2d20), C(e5f9f947bafb9e79), C(b342cdf0d7ac4b2a), C(66914d44b373b232), C(261194e76cb43966), C(45a0010190365048),
C(b342cdf0d7ac4b2a), C(66914d44b373b232), C(261194e76cb43966), C(45a0010190365048),
C(bd92dffa3bc91d9), C(1854247ba582a26c), C(c1c6edb93d443899), C(520dcb565f49bf4e)},
{C(406e959cdffadec7), C(e80dc125dca28ed1), C(e5beb146d4b79a21), C(e66d5c1bb441541a), C(d14961bc1fd265a2), C(e4cc669d4fc0577f), C(abf4a51e36da2702),
C(e66d5c1bb441541a), C(d14961bc1fd265a2), C(e4cc669d4fc0577f), C(abf4a51e36da2702),
C(1df2adbb7fa956d8), C(b166453cb97efb61), C(d3214c1e1596aa87), C(21eb0f13dff1f383)},
{C(46663908b4169b95), C(4e7e90b5c426bf1d), C(dc660b58daaf8b2c), C(b298265ebd1bd55f), C(4a5f6838b55c0b08), C(fc003c97aa05d397), C(2fb5adad3380c3bc),
C(b298265ebd1bd55f), C(4a5f6838b55c0b08), C(fc003c97aa05d397), C(2fb5adad3380c3bc),
C(ffd1d16a13a0273c), C(46c7ca881c23a15a), C(88d8b2597f333fea), C(513cb03a061ba356)},
{C(f214b86cffeab596), C(5fccb0b132da564f), C(86e7aa8b4154b883), C(763529c8d4189ea8), C(860d77e7fef74ca3), C(3b1ba41191219b6b), C(722b25dfa6d0a04b),
C(763529c8d4189ea8), C(860d77e7fef74ca3), C(3b1ba41191219b6b), C(722b25dfa6d0a04b),
C(d38bce10e9cb7046), C(bd1dbdc98876e297), C(3a0a7b7e6a9e5255), C(814aacf7e8f7637)},
{C(eba670441d1a4f7d), C(eb6b272502d975fa), C(69f8d424d50c083e), C(313d49cb51b8cd2c), C(6e982d8b4658654a), C(dd59629a17e5492d), C(81cb23bdab95e30e),
C(313d49cb51b8cd2c), C(6e982d8b4658654a), C(dd59629a17e5492d), C(81cb23bdab95e30e),
C(bea50d61120cc62f), C(6d1ce78f0bc0b80b), C(e909a964599f968), C(6f5d2ab1656e6d1d)},
{C(172c17ff21dbf88d), C(1f5104e320f0c815), C(1e34e9f1fa63bcef), C(3506ae8fae368d2a), C(59fa2b2de5306203), C(67d1119dcfa6007e), C(1f7190c648ad9aef),
C(3506ae8fae368d2a), C(59fa2b2de5306203), C(67d1119dcfa6007e), C(1f7190c648ad9aef),
C(86cfcfc1cc6a11db), C(6c0890c7270e0dbe), C(8e89d2e6861ecb5), C(d6a2151582a36609)},
{C(5a0838df8a019b8c), C(73fc859b4952923), C(45e39daf153491bd), C(a9b91459a5fada46), C(de0fbf8800a2da3), C(21800e4b5af9dedb), C(517c3726ae0dbae7),
C(a9b91459a5fada46), C(de0fbf8800a2da3), C(21800e4b5af9dedb), C(517c3726ae0dbae7),
C(a0074f54a45db108), C(99218eadf0603a25), C(16cc650b8dded2ee), C(23358fe215f748f0)},
{C(8f42b1fbb2fc0302), C(5ae31626076ab6ca), C(b87f0cb67cb75d28), C(2498586ac2e1fab2), C(e683f9cbea22809a), C(a9728d0b2bbe377c), C(46baf5cae53dc39a),
C(2498586ac2e1fab2), C(e683f9cbea22809a), C(a9728d0b2bbe377c), C(46baf5cae53dc39a),
C(47927391c3a2af3), C(172127d21e88bc24), C(e3e8e9c10fc2f5af), C(90887b1e55a6fd19)},
{C(72085e82d70dcea9), C(32f502c43349ba16), C(5ebc98c3645a018f), C(c7fa762238fd90ac), C(8d03b5652d615677), C(a3f5226e51d42217), C(46d5010a7cae8c1e),
C(c7fa762238fd90ac), C(8d03b5652d615677), C(a3f5226e51d42217), C(46d5010a7cae8c1e),
C(ddb4eae471da4140), C(863d5d01a2ca98d5), C(97532c6592d84c3d), C(5db95b869784afde)},
{C(32b75fc2223b5032), C(246fff80eb230868), C(a6fdbc82c9aeecc0), C(c089498074167021), C(ab094a9f9ab81c23), C(4facf3d9466bcb03), C(57aa9c67938cf3eb),
C(c089498074167021), C(ab094a9f9ab81c23), C(4facf3d9466bcb03), C(57aa9c67938cf3eb),
C(fd467cc2f458325c), C(dd66e7703b05a582), C(7e4ceb0c62bdd4a9), C(dcfd5f0b5b7236fc)},
{C(e1dd010487d2d647), C(12352858295d2167), C(acc5e9b6f6b02dbb), C(1c66ceea473413df), C(dc3f70a124b25a40), C(66a6dfe54c441cd8), C(b436dabdaaa37121),
C(1c66ceea473413df), C(dc3f70a124b25a40), C(66a6dfe54c441cd8), C(b436dabdaaa37121),
C(ae1dc1de7ee66758), C(f3be41b059b8d25d), C(ba692ade4ded17f4), C(e4fb583d3ce32b7d)},
{C(2994f9245194a7e2), C(b7cd7249d6db6c0c), C(2170a7d119c5c6c3), C(8505c996b70ee9fc), C(b92bba6b5d778eb7), C(4db4c57f3a7a4aee), C(3cfd441cb222d06f),
C(8505c996b70ee9fc), C(b92bba6b5d778eb7), C(4db4c57f3a7a4aee), C(3cfd441cb222d06f),
C(8a4b50e7c91ecdcf), C(b75dfba44eee3725), C(92ce15e3cfcdb2f6), C(39db1f066b2c66d0)},
{C(32e2ed6fa03e5b22), C(58baf09d7c71c62b), C(a9c599f3f8f50b5b), C(1660a2c4972d0fa1), C(1a1538d6b50a57c), C(8a5362485bbc9363), C(e8eec3c84fd9f2f8),
C(1660a2c4972d0fa1), C(1a1538d6b50a57c), C(8a5362485bbc9363), C(e8eec3c84fd9f2f8),
C(3039e44e78d5c0), C(987bc225df21d1e3), C(5e3bceacac08114b), C(aec827944a4c64ec)},
{C(37a72b6e89410c9f), C(139fec53b78cee23), C(4fccd8f0da7575c3), C(3a5f04166518ac75), C(f49afe05a44fc090), C(cb01b4713cfda4bd), C(9027bd37ffc0a5de),
C(3a5f04166518ac75), C(f49afe05a44fc090), C(cb01b4713cfda4bd), C(9027bd37ffc0a5de),
C(ed1415a9018d5a89), C(1323b029f4887b3a), C(5b09eb0ad88ed7dc), C(4494d452a6fb4048)},
{C(10836563cb8ff3a1), C(d36f67e2dfc085f7), C(edc1bb6a3dcba8df), C(bd4f3a0566df3bed), C(81fc8230c163dcbe), C(4168bc8417a8281b), C(7100c9459827c6a6),
C(bd4f3a0566df3bed), C(81fc8230c163dcbe), C(4168bc8417a8281b), C(7100c9459827c6a6),
C(f28457d4df6004bb), C(a0946ff2f4b842cd), C(ad95004da4fe697d), C(f0b57f4dcf7676c2)},
{C(4dabcb5c1d382e5c), C(9a868c608088b7a4), C(7b2b6c389b943be5), C(c914b925ab69fda0), C(6bafe864647c94d7), C(7a48682dd4afa22), C(40fe01210176ba10),
C(c914b925ab69fda0), C(6bafe864647c94d7), C(7a48682dd4afa22), C(40fe01210176ba10),
C(f7860d80f8c6cc96), C(bedbe5f366fc00d6), C(315a4d69ad10746c), C(479067287e848907)},
{C(296afb509046d945), C(c38fe9eb796bd4be), C(d7b17535df110279), C(dd2482b87d1ade07), C(662785d2e3e78ddf), C(eae39994375181bb), C(9994500c077ee1db),
C(dd2482b87d1ade07), C(662785d2e3e78ddf), C(eae39994375181bb), C(9994500c077ee1db),
C(92925daa51764b59), C(628e0eb360a63e67), C(aab5bcf342898229), C(aa162029312227e6)},
{C(f7c0257efde772ea), C(af6af9977ecf7bff), C(1cdff4bd07e8d973), C(fab1f4acd2cd4ab4), C(b4e19ba52b566bd), C(7f1db45725fe2881), C(70276ff8763f8396),
C(fab1f4acd2cd4ab4), C(b4e19ba52b566bd), C(7f1db45725fe2881), C(70276ff8763f8396),
C(3e5f524573938add), C(8cdbba98db29868e), C(c8f17269bef879a), C(c9db2c6a9391c4ac)},
{C(61e021c8da344ba1), C(cf9c720676244755), C(354ffa8e9d3601f6), C(44e40a03093fbd92), C(bda9481cc5b93cae), C(986b589cbc0cf617), C(210f59f074044831),
C(44e40a03093fbd92), C(bda9481cc5b93cae), C(986b589cbc0cf617), C(210f59f074044831),
C(28173177855537e8), C(a8d855dd9179c337), C(a75961d94295428e), C(87b6abda336f08fc)},
{C(c0a86ed83908560b), C(440c8b6f97bd1749), C(a99bf2891726ea93), C(ac0c0b84df66df9d), C(3ee2337b437eb264), C(8a341daed9a25f98), C(cc665499aa38c78c),
C(ac0c0b84df66df9d), C(3ee2337b437eb264), C(8a341daed9a25f98), C(cc665499aa38c78c),
C(825f2d7ee87071ff), C(71a802c44d48c5c2), C(e660bce32fe96a74), C(bf7a93427b02e1dd)},
{C(35c9cf87e4accbf3), C(2267eb4d2191b2a3), C(80217695666b2c9), C(cd43a24abbaae6d), C(a88abf0ea1b2a8ff), C(e297ff01427e2a9d), C(935d545695b2b41d),
C(cd43a24abbaae6d), C(a88abf0ea1b2a8ff), C(e297ff01427e2a9d), C(935d545695b2b41d),
C(6da9611e0a97cb7d), C(4488df3286c81c95), C(e2e8c4da17d388f3), C(862af26f430477f8)},
{C(e74c366b3091e275), C(522e657c5da94b06), C(ca9afa806f1a54ac), C(b545042f67929471), C(90d10e75ed0e75d8), C(3ea60f8f158df77e), C(8863eff3c2d670b7),
C(b545042f67929471), C(90d10e75ed0e75d8), C(3ea60f8f158df77e), C(8863eff3c2d670b7),
C(3f2d5bd313010994), C(699bd0164ff7bc9b), C(fa45cc2fd36c475e), C(d0145b594db1a47f)},
{C(a3f2ca45089ad1a6), C(13f6270fe56fbce4), C(1f93a534bf03e705), C(aaea14288ae2d90c), C(1be3cd51ef0f15e8), C(e8b47c84d5a4aac1), C(297d27d55b766782),
C(aaea14288ae2d90c), C(1be3cd51ef0f15e8), C(e8b47c84d5a4aac1), C(297d27d55b766782),
C(2829328c6db9cfad), C(88475938af5c2b1a), C(e334da6d8055039c), C(e72e645453eaa62)},
{C(e5181466d8e60e26), C(cf31f3a2d582c4f3), C(d9cee87cb71f75b2), C(4750ca6050a2d726), C(d6e6dd8940256849), C(f3b3749fdab75b0), C(c55d8a0f85ba0ccf),
C(4750ca6050a2d726), C(d6e6dd8940256849), C(f3b3749fdab75b0), C(c55d8a0f85ba0ccf),
C(de0353b2bfd40c2a), C(59193afed9f5155d), C(f4274b63b02fb840), C(e7e1924d404c621c)},
{C(fb528a8dd1e48ad7), C(98c4fd149c8a63dd), C(4abd8fc3377ae1f), C(d7a9304abbb47cc5), C(7f2b9a27aa57f99), C(353ab332d4ef9f18), C(47d56b8d6c8cf578),
C(d7a9304abbb47cc5), C(7f2b9a27aa57f99), C(353ab332d4ef9f18), C(47d56b8d6c8cf578),
C(2062bddb82a97056), C(1a57e6cc6d8a2964), C(6fd90cbb2109caf4), C(82f3d6c446b5b3a0)},
{C(da6d2b7ea9d5f9b6), C(57b11153ee3b4cc8), C(7d3bd1256037142f), C(90b16ff331b719b5), C(fc294e7ad39e01e6), C(d2145386bab41623), C(7045a63d44d76011),
C(90b16ff331b719b5), C(fc294e7ad39e01e6), C(d2145386bab41623), C(7045a63d44d76011),
C(b346cec7a0d071ee), C(56e5bc7b05bc6ab4), C(67b90085206f2a13), C(622c7d5f99aeea5e)},
{C(61d95225bc2293e), C(f6c52cb6be9889a8), C(91a0667a7ed6a113), C(441133d221486a3d), C(fb9c5a40e19515b), C(6c967b6c69367c2d), C(145bd9ef258c4099),
C(441133d221486a3d), C(fb9c5a40e19515b), C(6c967b6c69367c2d), C(145bd9ef258c4099),
C(db8ecb75eaa297e1), C(3c02b85694c68851), C(62d832b4c10fb6db), C(7a2b185996a2a63e)},
{C(81247c01ab6a9cc1), C(fbccea953e810636), C(ae18965000c31be0), C(15bb46383daec2a5), C(716294063b4ba089), C(f3bd691ce02c3014), C(14ccaad685a20764),
C(15bb46383daec2a5), C(716294063b4ba089), C(f3bd691ce02c3014), C(14ccaad685a20764),
C(892122c01b9bf3b7), C(38b1f330dbd7ed55), C(ceacbb1b17339af1), C(38230aa39400e543)},
{C(c17f3ebd3257cb8b), C(e9e68c939c118c8d), C(72a5572be35bfc1b), C(f6916c341cb31f2a), C(591da1353ee5f31c), C(f1313c98a836b407), C(e0b8473eada48cd1),
C(f6916c341cb31f2a), C(591da1353ee5f31c), C(f1313c98a836b407), C(e0b8473eada48cd1),
C(ddeac1b84bea27db), C(f60cfc74d2505bf8), C(995d7bc8055998ca), C(2432f8a04d6af08d)},
{C(9802438969c3043b), C(6cd07575c948dd82), C(83e26b6830ea8640), C(d52f1fa190576961), C(11d182e4f0d419cc), C(5d9ccf1b56617424), C(c8a16debb585e452),
C(d52f1fa190576961), C(11d182e4f0d419cc), C(5d9ccf1b56617424), C(c8a16debb585e452),
C(9553bfd7b6886d7c), C(216ed409de5b6b93), C(8d0e0ce56dac43ff), C(8b58cfdd161c34e1)},
{C(3dd8ed248a03d754), C(d8c1fcf001cb62e0), C(87a822141ed64927), C(4bfaf6fd26271f47), C(aefeae8222ad3c77), C(cfb7b24351a60585), C(8678904e9e890b8f),
C(4bfaf6fd26271f47), C(aefeae8222ad3c77), C(cfb7b24351a60585), C(8678904e9e890b8f),
C(bd674a9139dc4546), C(cfbe8eb024d612a5), C(ec6fd5d41fad739a), C(d226c882e0c71c7e)},
{C(c5bf48d7d3e9a5a3), C(8f0249b5c5996341), C(c6d2c8a606f45125), C(fd1779db740e2c48), C(1950ef50fefab3f8), C(e4536426a6196809), C(699556c502a01a6a),
C(fd1779db740e2c48), C(1950ef50fefab3f8), C(e4536426a6196809), C(699556c502a01a6a),
C(135b17b83dbbf0c), C(1b96a2429cdd4d15), C(cfce0995c1f0fd4c), C(e5b4b9706b2e56b)},
{C(bc4a21d00cf52288), C(28df3eb5a533fa87), C(6081bbc2a18dd0d), C(8eed355d219e58b9), C(2d7b9f1a3d645165), C(5758d1aa8d85f7b2), C(9c90c65920041dff),
C(8eed355d219e58b9), C(2d7b9f1a3d645165), C(5758d1aa8d85f7b2), C(9c90c65920041dff),
C(a0c43ee3ec6a79cd), C(4c915fa16e8b8c81), C(b1339b5fe42fab21), C(89175650bdd1f8cd)},
{C(172c8674913ff413), C(1815a22400e832bf), C(7e011f9467a06650), C(161be43353a31dd0), C(79a8afddb0642ac3), C(df43af54e3e16709), C(6e12553a75b43f07),
C(161be43353a31dd0), C(79a8afddb0642ac3), C(df43af54e3e16709), C(6e12553a75b43f07),
C(63a2ca06572153c2), C(5232482e2f3634d7), C(be176997776ffc37), C(1540d53974cdc6bd)},
{C(17a361dbdaaa7294), C(c67d368223a3b83c), C(f49cf8d51ab583d2), C(666eb21e2eaa596), C(778f3e1b6650d56), C(3f6be451a668fe2d), C(5452892b0b101388),
C(666eb21e2eaa596), C(778f3e1b6650d56), C(3f6be451a668fe2d), C(5452892b0b101388),
C(8a9fb8df05a47ccd), C(6f6de5fd8d5a0583), C(178e2fd588c96844), C(dd8265ff40669c7d)},
{C(5cc268bac4bd55f), C(232717a35d5b2f1), C(38da1393365c961d), C(2d187f89c16f7b62), C(4eb504204fa1be8), C(222bd53d2efe5fa), C(a4dcd6d721ddb187),
C(2d187f89c16f7b62), C(4eb504204fa1be8), C(222bd53d2efe5fa), C(a4dcd6d721ddb187),
C(86db745bb31b6713), C(f1a3ce35719bf8a8), C(c971171a7ae89af0), C(e4e624d3c0ee0f7)},
{C(db04969cc06547f1), C(fcacc8a75332f120), C(967ccec4ed0c977e), C(ac5d1087e454b6cd), C(c1f8b2e284d28f6c), C(cc3994f4a9312cfa), C(8d61606dbc4e060d),
C(ac5d1087e454b6cd), C(c1f8b2e284d28f6c), C(cc3994f4a9312cfa), C(8d61606dbc4e060d),
C(89bc8be6e4cc91f7), C(3ce128305856b46e), C(45fa557debb74f94), C(f1d01d4419f24cfc)},
{C(25bd8d3ca1b375b2), C(4ad34c2c865816f9), C(9be30ad32f8f28aa), C(7755ea02dbccad6a), C(cb8aaf8886247a4a), C(8f6966ce7ea1b6e6), C(3f2863090fa45a70),
C(7755ea02dbccad6a), C(cb8aaf8886247a4a), C(8f6966ce7ea1b6e6), C(3f2863090fa45a70),
C(3780c8671bf69bde), C(1e02ed476a91a1b8), C(bd055fd2164db36e), C(6381a699096e41c2)},
{C(166c11fbcbc89fd8), C(cce1af56c48a48aa), C(78908959b8ede084), C(19032925ba2c951a), C(a53ed6e81b67943a), C(edc871a9e8ef4bdf), C(ae66cf46a8371aba),
C(19032925ba2c951a), C(a53ed6e81b67943a), C(edc871a9e8ef4bdf), C(ae66cf46a8371aba),
C(10650d7854cf2c01), C(3c8f739a82406c84), C(ef00140edd1f0ef5), C(bad40d52a178d073)},
{C(3565bcc4ca4ce807), C(ec35bfbe575819d5), C(6a1f690d886e0270), C(1ab8c584625f6a04), C(ccfcdafb81b572c4), C(53b04ba39fef5af9), C(64ce81828eefeed4),
C(1ab8c584625f6a04), C(ccfcdafb81b572c4), C(53b04ba39fef5af9), C(64ce81828eefeed4),
C(5cb28ef42b8fd9a2), C(1b216fbd2a7187ad), C(210eaa73f53bb956), C(66e46a98311d257b)},
{C(b7897fd2f274307d), C(6d43a9e5dd95616d), C(31a2218e64d8fce0), C(664e581fc1cf769b), C(415110942fc97022), C(7a5d38fee0bfa763), C(dc87ddb4d7495b6c),
C(664e581fc1cf769b), C(415110942fc97022), C(7a5d38fee0bfa763), C(dc87ddb4d7495b6c),
C(4b9cd641187b2496), C(5b9f4e64fd9dfa83), C(c005f0afd66a8a2e), C(bde202bf0cc52141)},
{C(aba98113ab0e4a16), C(287f883aede0274d), C(3ecd2a607193ba3b), C(e131f6cc9e885c28), C(b399f98d827e4958), C(6eb90c8ed6c9090c), C(ec89b378612a2b86),
C(e131f6cc9e885c28), C(b399f98d827e4958), C(6eb90c8ed6c9090c), C(ec89b378612a2b86),
C(3846308b68506215), C(8014bdd6e9b54908), C(d4e05cce1a158797), C(6a7176f691c9eaa5)},
{C(17f7796e0d4b636c), C(ddba5551d716137b), C(65f9735375df1ada), C(a39e946d02e14ec2), C(1c88cc1d3822a193), C(663f8074a5172bb4), C(8ad2934942e4cb9c),
C(a39e946d02e14ec2), C(1c88cc1d3822a193), C(663f8074a5172bb4), C(8ad2934942e4cb9c),
C(ff21f1868b83ca0f), C(cf32b7705c582889), C(29de36887ca200fe), C(bd6ceff8ed2cdeb9)},
{C(33c0128e62122440), C(b23a588c8c37ec2b), C(f2608199ca14c26a), C(acab0139dc4f36df), C(9502b1605ca1345a), C(32174ef1e06a5e9c), C(d824b7869258192b),
C(acab0139dc4f36df), C(9502b1605ca1345a), C(32174ef1e06a5e9c), C(d824b7869258192b),
C(8e1da3b98755187d), C(46d82dcf4cecd82d), C(ef108e3c1ea81140), C(50b572c4c3e077b6)},
{C(988bc5d290b97aef), C(6754bb647eb47666), C(44b5cf8b5b8106a8), C(a1c5ba961937f723), C(32d6bc7214dfcb9b), C(6863397e0f4c6758), C(e644bcb87e3eef70),
C(a1c5ba961937f723), C(32d6bc7214dfcb9b), C(6863397e0f4c6758), C(e644bcb87e3eef70),
C(81989b4fe935f80a), C(11c9c21ee2de4d2e), C(324ce7187b355cae), C(be226da53be39d0d)},
{C(23c8c25c2ab72381), C(d6bc672da4175fba), C(6aef5e6eb4a4eb10), C(3df880c945e68aed), C(5e08a75e956d456f), C(f984f088d1a322d7), C(7d44a1b597b7a05e),
C(3df880c945e68aed), C(5e08a75e956d456f), C(f984f088d1a322d7), C(7d44a1b597b7a05e),
C(8e218be176a82f7d), C(c2c2e53669acb5c4), C(f4f9ab73295a87f), C(5179e3b16fa98921)},
{C(450fe4acc4ad3749), C(3111b29565e4f852), C(db570fc2abaf13a9), C(35107d593ba38b22), C(fd8212a125073d88), C(72805d6e015bfacf), C(6b22ae1a29c4b853),
C(35107d593ba38b22), C(fd8212a125073d88), C(72805d6e015bfacf), C(6b22ae1a29c4b853),
C(294e61f18339cb15), C(a63bca6ab461de80), C(2e929d7f213d5f10), C(204154e8836156ee)},
{C(48e1eff032d90c50), C(dee0fe333d962b62), C(c845776990c96775), C(8ea71758346b71c9), C(d84258cab79431fd), C(af566b4975cce10a), C(5c5c7e70a91221d2),
C(8ea71758346b71c9), C(d84258cab79431fd), C(af566b4975cce10a), C(5c5c7e70a91221d2),
C(126ef69504e48815), C(3d5a42a88d9865d4), C(76c880ec9720e4), C(c0dfa1b80d093331)},
{C(c048604ba8b6c753), C(21ea6d24b417fdb6), C(4e40a127ad2d6834), C(5234231bf173c51), C(62319525583eaf29), C(87632efa9144cc04), C(1749de70c8189067),
C(5234231bf173c51), C(62319525583eaf29), C(87632efa9144cc04), C(1749de70c8189067),
C(9977eb016fb429b), C(a3d5e0bf931d42d6), C(8c66bd015dc06fa4), C(b7b24e70dc14d470)},
{C(67ff1cbe469ebf84), C(3a828ac9e5040eb0), C(85bf1ad6b363a14b), C(2fc6c0783390d035), C(ef78307f5be5524e), C(a46925b7a1a77905), C(fea37470f9a51514),
C(2fc6c0783390d035), C(ef78307f5be5524e), C(a46925b7a1a77905), C(fea37470f9a51514),
C(8e634d63b75c1837), C(821ebab006d4ca8), C(2f1afc923db3f52e), C(720d79b9c49532e5)},
{C(b45c7536bd7a5416), C(e2d17c16c4300d3c), C(b70b641138765ff5), C(a5a859ab7d0ddcfc), C(8730164a0b671151), C(af93810c10348dd0), C(7256010c74f5d573),
C(a5a859ab7d0ddcfc), C(8730164a0b671151), C(af93810c10348dd0), C(7256010c74f5d573),
C(35eeb326710c212e), C(a6b808ca3e92a9ab), C(b6ac65c3594dcbe6), C(c9b2e5fd4e0d3d8)},
{C(215c2eaacdb48f6f), C(33b09acf1bfa2880), C(78c4e94ba9f28bf), C(981b7219224443d1), C(1f476fc4344d7bba), C(abad36e07283d3a5), C(831bf61190eaaead),
C(981b7219224443d1), C(1f476fc4344d7bba), C(abad36e07283d3a5), C(831bf61190eaaead),
C(7a8cfa48d8516601), C(5597551f26a24ee1), C(859b9cc44d905413), C(8a07d88bb72979e1)},
{C(241baf16d80e0fe8), C(b6b3c5b53a3ce1d), C(6ae6b36209eecd70), C(a560b6a4aa3743a4), C(b3e04f202b7a99b), C(3b3b1573f4c97d9f), C(ccad8715a65af186),
C(a560b6a4aa3743a4), C(b3e04f202b7a99b), C(3b3b1573f4c97d9f), C(ccad8715a65af186),
C(fb42b6e5a7d900fd), C(d439c2dd4ae54899), C(ba6fb2d419bcd0b6), C(8fe0d33a814587dc)},
{C(d10a9743b5b1c4d1), C(f16e0e147ff9ccd6), C(fbd20a91b6085ed3), C(43d309eb00b771d5), C(a6d1f26105c0f61b), C(d37ad62406e5c37e), C(75d9b28c717c8cf7),
C(43d309eb00b771d5), C(a6d1f26105c0f61b), C(d37ad62406e5c37e), C(75d9b28c717c8cf7),
C(ae18d930ba857d0c), C(96376cc9f1faa1f3), C(e27bce91bf41ff11), C(e475f347ca20e7f7)},
{C(919ef9e209f2edd1), C(684c33fb726a720a), C(540353f94e8033), C(26da1a143e7d4ec4), C(55095eae445aacf4), C(31efad866d075938), C(f9b580cff4445f94),
C(26da1a143e7d4ec4), C(55095eae445aacf4), C(31efad866d075938), C(f9b580cff4445f94),
C(1fa8df59147fad3e), C(cb2912a109d910d0), C(85ee391d283e6c3), C(31f860f0b7d43ede)},
{C(b5f9519b6c9280b), C(7823a2fe2e103803), C(d379a205a3bd4660), C(466ec55ee4b4302a), C(714f1b9985deeaf0), C(728595f26e633cf7), C(25ecd0738e1bee2b),
C(466ec55ee4b4302a), C(714f1b9985deeaf0), C(728595f26e633cf7), C(25ecd0738e1bee2b),
C(e3d25610e5aaf0ce), C(fca7b84ddde4d245), C(a5724a609112a5d3), C(c3c8cafce1f39369)},
{C(77a75e89679e6757), C(25d31fee616b5dd0), C(d81f2dfd08890060), C(7598df8911dd40a4), C(3b6dda517509b41b), C(7dae29d248dfffae), C(6697c427733135f),
C(7598df8911dd40a4), C(3b6dda517509b41b), C(7dae29d248dfffae), C(6697c427733135f),
C(bb3c2d45a33cb4ad), C(63759088b45065fb), C(e3c07d5968858439), C(6854efd588619d00)},
{C(9d709e1b086aabe2), C(4d6d6a6c543e3fec), C(df73b01acd416e84), C(d54f613658e35418), C(fcc88fd0567afe77), C(d18f2380980db355), C(ec3896137dfbfa8b),
C(d54f613658e35418), C(fcc88fd0567afe77), C(d18f2380980db355), C(ec3896137dfbfa8b),
C(65722e87ddb7efcb), C(438b520e0e6e979c), C(eb5ae2229115057a), C(288e8fe824b48eab)},
{C(91c89971b3c20a8a), C(87b82b1d55780b5), C(bc47bb80dfdaefcd), C(87e11c0f44454863), C(2df1aedb5871cc4b), C(ba72fd91536382c8), C(52cebef9e6ea865d),
C(87e11c0f44454863), C(2df1aedb5871cc4b), C(ba72fd91536382c8), C(52cebef9e6ea865d),
C(9a251a4f6dec73d0), C(ffe0ac4598453e46), C(cbf53e723b4c6977), C(7678a9934b49ce9)},
{C(16468c55a1b3f2b4), C(40b1e8d6c63c9ff4), C(143adc6fee592576), C(4caf4deeda66a6ee), C(264720f6f35f7840), C(71c3aef9e59e4452), C(97886ca1cb073c55),
C(4caf4deeda66a6ee), C(264720f6f35f7840), C(71c3aef9e59e4452), C(97886ca1cb073c55),
C(dea7c1f629182595), C(d70abe8308fb8ff6), C(20bd73ef8e62748e), C(393b86d02205a46)},
{C(8015f298161f861e), C(3b4a12bf2e24a16), C(37b223562c48b473), C(d82489179f16d4e8), C(a3c59f65e2913cc5), C(36cbaecdc3532b3b), C(f1b454616cfeca41),
C(d82489179f16d4e8), C(a3c59f65e2913cc5), C(36cbaecdc3532b3b), C(f1b454616cfeca41),
C(84b7ca0b97f40c09), C(b83b76cc780674e6), C(bfcd22ad536b4bb1), C(373927fdd98b5603)},
{C(71e244d7e2843a41), C(2132bd3f394ac99d), C(1791b7a5b93ad1f9), C(fd7feb3d2956875e), C(d7192a886b8b01b6), C(16e71dba55f5b85a), C(93dabd3ff22ff144),
C(fd7feb3d2956875e), C(d7192a886b8b01b6), C(16e71dba55f5b85a), C(93dabd3ff22ff144),
C(a2cf5b992e172715), C(4bfd8e8291470259), C(37424609d330da7c), C(b81684e4acf6ae11)},
{C(5d3cb0d2c7ccf11f), C(1215f183d5a24092), C(ea833d94dca4809a), C(b5b472960ece11ec), C(13935c99b9abbf53), C(3e80d95687f0432c), C(3516ab536053be5),
C(b5b472960ece11ec), C(13935c99b9abbf53), C(3e80d95687f0432c), C(3516ab536053be5),
C(6dd659e52a203d9), C(56113335fd45136c), C(6c8fec792e683c2), C(ff8278f06fd7b168)},
{C(d6cffe6c223aba65), C(b19224aad3a69ef1), C(67268f8829a9f99d), C(62e33ba258712d51), C(fa085c15d779c0e), C(2c15d9142308c5ad), C(feb517011f27be9e),
C(62e33ba258712d51), C(fa085c15d779c0e), C(2c15d9142308c5ad), C(feb517011f27be9e),
C(c64b2639773de03a), C(dff150c89c409027), C(38f55cf3472e7f43), C(a178f9518e66c24a)},
{C(8a17c5054e85e2be), C(15e35d5a33726681), C(9b345fa359c4e8e3), C(e4041579de57c879), C(bbf513cb7bab5553), C(66ad0373099d5fa0), C(44bb6b21b87f3407),
C(e4041579de57c879), C(bbf513cb7bab5553), C(66ad0373099d5fa0), C(44bb6b21b87f3407),
C(71a75a9eea752287), C(89e4646caa4d61ed), C(16dcea6d62871d1e), C(329110ddf8e15c6e)},
{C(77d112a0b7084c6a), C(2f869c2d79d95e45), C(a3c8f877e8ebc840), C(16fde90d02a1343b), C(ad14e0ed6e165185), C(8df6e0b2f24085dd), C(caa8a47292d50263),
C(16fde90d02a1343b), C(ad14e0ed6e165185), C(8df6e0b2f24085dd), C(caa8a47292d50263),
C(7a89e3eebe2939ea), C(d00e46213690a008), C(cf21eb7264ec9839), C(2661cb9947b6e01f)},
{C(708f2a6e8bd57583), C(688e0faea5f15272), C(d28955c99ed63d38), C(8459801016414808), C(6fbf75735353c2d1), C(6e69aaf2d93ed647), C(85bb5b90167cce5e),
C(8459801016414808), C(6fbf75735353c2d1), C(6e69aaf2d93ed647), C(85bb5b90167cce5e),
C(6694139503e11ece), C(262c6b886d57cb2c), C(59fa9c33d1764117), C(27de71d654f9930c)},
{C(50bc8f76b62c8de9), C(88b4d8ebe13cbd79), C(da08ee1bf528e82e), C(aad20d70e231582b), C(eab92d70d9a22e54), C(cc5ab266375580c0), C(85091463e3630dce),
C(aad20d70e231582b), C(eab92d70d9a22e54), C(cc5ab266375580c0), C(85091463e3630dce),
C(64a6468ae196d852), C(9f0666ed430c5ced), C(72276ddc483452f1), C(3e20002aa0d2f037)},
{C(8b15a656b553641a), C(611c74d4137bf21b), C(a051cbbf796013c1), C(38a42e0db55a4275), C(585971da56bb56d6), C(cd957009adc1482e), C(d6a96021e427567d),
C(38a42e0db55a4275), C(585971da56bb56d6), C(cd957009adc1482e), C(d6a96021e427567d),
C(10183eefcb6c149a), C(dc7b2e95dd0a59d4), C(249ecfd99a352126), C(90eb499e5bc3016c)},
{C(6ba74ccf722a52be), C(75e2d5362c0050b1), C(32e95f14d29a1c01), C(3526d9b950a1d910), C(a58ba01135bca7c0), C(cbad32e86d60a87c), C(adde1962aad3d730),
C(3526d9b950a1d910), C(a58ba01135bca7c0), C(cbad32e86d60a87c), C(adde1962aad3d730),
C(6978adb645f9d67a), C(8ecd8010c6c14e6c), C(a4792a2a4fd8b4d7), C(80ec738f89e172d7)},
{C(fb317bb7533454d0), C(15f9898fd61f0209), C(593d179631ddf22c), C(7c909e8cd5261727), C(c5acb3d5fbdc832e), C(54eff5c782ad3cdd), C(9d54397f3caf5bfa),
C(7c909e8cd5261727), C(c5acb3d5fbdc832e), C(54eff5c782ad3cdd), C(9d54397f3caf5bfa),
C(66c61e7ac32d8e8a), C(15e13d7e20653581), C(1fb2a076673d4ed7), C(df1d517cbac90f4d)},
{C(8eec643f62c90fea), C(df17fc55b1d4a915), C(62fa77ed321e937f), C(479f936b6d496dca), C(dc2dc93d63739d4a), C(27e4151c3870498c), C(3a3a22ba512d13ba),
C(479f936b6d496dca), C(dc2dc93d63739d4a), C(27e4151c3870498c), C(3a3a22ba512d13ba),
C(e5c26a6a1f34c8b), C(f8e2d55618cfd437), C(8f34b1e8afb7c487), C(1e0b5027a3de88a8)},
{C(81ce6becdf10dff2), C(4182c78d3d609461), C(db5cb16e44cb1e37), C(464f1adf4c68577), C(acf3961e1c9d897f), C(985b01ab89b41fe1), C(6972d6237390aac0),
C(464f1adf4c68577), C(acf3961e1c9d897f), C(985b01ab89b41fe1), C(6972d6237390aac0),
C(a35b31c92616a65a), C(277fea4f1033e80), C(b93848f7799af5), C(aa7ac2e7c2ca0c3f)},
{C(549c669fb0049f69), C(a01f16549b0a628e), C(675a9a86499cd4e4), C(8af42343888843c), C(191433ffcbab7800), C(7eb45fc94f88a71), C(31bc5418ffb88fa8),
C(8af42343888843c), C(191433ffcbab7800), C(7eb45fc94f88a71), C(31bc5418ffb88fa8),
C(2a83688059015d3d), C(6ebaace04b37faee), C(2711dbf9175e05af), C(dc48f30c0c5485c)},
{C(2b6a3433940bbf2d), C(dda5e942a8098f8b), C(812bcb2a17f1f652), C(dc46069eec17bfdf), C(cacb63fe65d9e3e), C(362fb57287d530c6), C(5854a4fbe1762d9),
C(dc46069eec17bfdf), C(cacb63fe65d9e3e), C(362fb57287d530c6), C(5854a4fbe1762d9),
C(ea16deeb7b10c9e9), C(c41455bd2891035), C(234df0018e17777f), C(2af60b0142554ad3)},
{C(d80b7a3c691401b7), C(e205b8266ea761cb), C(8e44beb4b7cde31b), C(69437142dae5a255), C(f2980cc4816965ac), C(dbbe76ba1d9adfcf), C(49c18025c0a8b0b5),
C(69437142dae5a255), C(f2980cc4816965ac), C(dbbe76ba1d9adfcf), C(49c18025c0a8b0b5),
C(fb3b9e51f05e41eb), C(2b98c335d35a255a), C(e28669bdcfd7e2fc), C(8fc1628f98097f89)},
{C(ab3bf6b494f66ef3), C(530b0467dcaf3c4b), C(383cc50df33afc6f), C(5e351e20f30377bf), C(91b3805daf12972c), C(94417fa6452a265e), C(bfa301a26765a7c),
C(5e351e20f30377bf), C(91b3805daf12972c), C(94417fa6452a265e), C(bfa301a26765a7c),
C(f24404822e88caf7), C(8016a4180e38ae78), C(554bcd2999907b4d), C(8b992ca8b140921c)},
{C(83f7b824a3911d44), C(921947a8a2668a44), C(13c001ebba408aaa), C(fd39b7642cecf78f), C(104f1af4e9201df5), C(ab1a3cc7eaeab609), C(cee3363f210a3d8b),
C(fd39b7642cecf78f), C(104f1af4e9201df5), C(ab1a3cc7eaeab609), C(cee3363f210a3d8b),
C(80e6f0488fcf2b74), C(5ae0c867f4678d7d), C(c7dd9616090bbb02), C(3d9f95de498cdc96)},
{C(3fb8d482d0d9d03f), C(d911bf94d3017ee1), C(96ebbf1ceac7b4cb), C(18865ff87619fd8f), C(dec5293e665663d8), C(ea07c345872d3201), C(6fce64da038a17ab),
C(18865ff87619fd8f), C(dec5293e665663d8), C(ea07c345872d3201), C(6fce64da038a17ab),
C(dcb0f7df521354a8), C(acf6cf3cc75f69cf), C(63f90b70122e9e1a), C(31ee0f81f9b46373)},
{C(ad346a1f100b3944), C(3934eb0f8d35a797), C(77664abec282db4c), C(b5f630ac75a8ce03), C(7cf71ae74fa8566a), C(e068f2b4618df5d), C(369df952ad3fd0b8),
C(b5f630ac75a8ce03), C(7cf71ae74fa8566a), C(e068f2b4618df5d), C(369df952ad3fd0b8),
C(264d8dc45552cbeb), C(beb7f7004c8430d9), C(e2ac51459c50db66), C(713432ce41894f55)},
{C(db210eb547a3dbc5), C(e1013615221cb0d7), C(4ca87abbb73194d8), C(1b0118c5c60a99c7), C(6ae919ef932301b8), C(cde25defa089c2fc), C(c2a3776e3a7716c4),
C(1b0118c5c60a99c7), C(6ae919ef932301b8), C(cde25defa089c2fc), C(c2a3776e3a7716c4),
C(2d7584f581337ebd), C(1c2ad68bb3841da4), C(24642f82c846d102), C(7ae8189fce271273)},
{C(e55fab4f920abdc0), C(7fa81600f789f5a6), C(6f67cf7344c18fce), C(2a5e555fd35627db), C(55d5da439c42f3b8), C(a758e451732a1c6f), C(18caa6b46664b484),
C(2a5e555fd35627db), C(55d5da439c42f3b8), C(a758e451732a1c6f), C(18caa6b46664b484),
C(f1d9a09efaa3b23f), C(96514bf9d3a14a7c), C(54bad54ea8aa08a3), C(8455a4cbe4357da3)},
{C(3b530fff7e848c5e), C(152e3fec5a21ed68), C(4340e5798860241a), C(1944ec723253392b), C(7ea6aa6a2f278ea5), C(5ff786af8113b3d5), C(194832eb9b0b8d0f),
C(1944ec723253392b), C(7ea6aa6a2f278ea5), C(5ff786af8113b3d5), C(194832eb9b0b8d0f),
C(147c80ac8c21e312), C(e8402f00491dd9fe), C(55f719b61bef6c41), C(7e40aced35c86182)},
{C(bde3379279d1cae1), C(f3596e48364bdaac), C(9f070e7509abc6bf), C(81d90ddff0d00fdb), C(2c8c7ce1173b5c77), C(18c6b6c8d3f91dfb), C(415d5cbbf7d9f717),
C(81d90ddff0d00fdb), C(2c8c7ce1173b5c77), C(18c6b6c8d3f91dfb), C(415d5cbbf7d9f717),
C(fb619d0453fff356), C(dac13cf12f023e76), C(f78a099853b565fd), C(c8181c3dbac54897)},
{C(4008062bc7755b37), C(8873f772dc7d1ea2), C(4502cd3133c94d7d), C(381068e0f65f708b), C(b4f3762e451b12a6), C(6d61ed2f6d4e741), C(8b3b9df537b91a2c),
C(381068e0f65f708b), C(b4f3762e451b12a6), C(6d61ed2f6d4e741), C(8b3b9df537b91a2c),
C(ef102f6286c7c1af), C(9afc95cde29a7aa6), C(2ff712f91ee9494f), C(9c6c76046b425b10)},
{C(76a66ce0ee8094d1), C(99dc87d517229612), C(83b12c2aff5dd46e), C(6e8e8ff107799274), C(24a2ef180891b531), C(c0eaf33a074bcb9d), C(1fa399a82974e17e),
C(6e8e8ff107799274), C(24a2ef180891b531), C(c0eaf33a074bcb9d), C(1fa399a82974e17e),
C(b29167d8432fafe), C(cc56b9205b528504), C(10d59e2938d5e43a), C(a90586bd2fb67b13)},
{C(2bc3dfb3b1756918), C(3e0269476ab76f14), C(52c60b61184e08de), C(f5f8b21ec30bd3a0), C(80a442fd5c6482a8), C(4fde11e5ccde5169), C(55671451f661a885),
C(f5f8b21ec30bd3a0), C(80a442fd5c6482a8), C(4fde11e5ccde5169), C(55671451f661a885),
C(b16048988fb7ee6), C(f885f2afd0dd0a35), C(43de0587ae1224c4), C(88e34aaa139bc84c)},
{C(d060dc1e8ca204ee), C(ce494f4b2198e36f), C(1f120ffb0524d537), C(caac64f5865d87f3), C(771b9fdbd3aa4bd2), C(88446393c3606c2d), C(bc3d3dcd5b7d6d7f),
C(caac64f5865d87f3), C(771b9fdbd3aa4bd2), C(88446393c3606c2d), C(bc3d3dcd5b7d6d7f),
C(c3cc4037483d6ffb), C(c0611d122df0a94a), C(bd0e8372d4ed9bda), C(6073eb94cae8b36f)},
{C(c8ec4fc839254a74), C(4d8b8b116ea60b09), C(d6a77d7a8c6d11f4), C(401a0581221957e2), C(fc04e99ae3a283ce), C(fe895303ab2d1e3e), C(35ab7c498403975b),
C(401a0581221957e2), C(fc04e99ae3a283ce), C(fe895303ab2d1e3e), C(35ab7c498403975b),
C(90e3d118304fc229), C(d8c2a402a3880687), C(790743ff50592460), C(655741d07bfda16a)},
{C(7cdf98a07b1315b0), C(5b7132d0a9ee6608), C(d2480e4e97602ad), C(330b7e93663affbd), C(3c59913fcf0d603f), C(e207e6572672fd0a), C(8a5dc17019c8a667),
C(330b7e93663affbd), C(3c59913fcf0d603f), C(e207e6572672fd0a), C(8a5dc17019c8a667),
C(6a650c716f0fd07b), C(1b31a47fc8bab8b), C(61d2ac5f64a95753), C(d833304729a56db5)},
{C(78284cb5c0143ed8), C(ac6af8e6f7820e82), C(71d171a63a6187b5), C(77fbb70409d316e2), C(c864432c5208e583), C(d3f593922668c184), C(23307562648bdb54),
C(77fbb70409d316e2), C(c864432c5208e583), C(d3f593922668c184), C(23307562648bdb54),
C(cd179db40527fcd9), C(5eb42e394975dc42), C(736ecffd66526be4), C(ac87ae424e7a6584)},
{C(5c2c485bdc8e3317), C(7bfe5915c5e0fa2d), C(6b433526b05fc4d8), C(20085827a39ff749), C(42e6c504df174606), C(839da16331fea7ac), C(7fd768552b10ffc6),
C(20085827a39ff749), C(42e6c504df174606), C(839da16331fea7ac), C(7fd768552b10ffc6),
C(2e5d1151c72c6bc4), C(8c55f4c8b55b8d97), C(bb3d972b6d146973), C(73f9202bc741c379)},
{C(6e38acb798627f75), C(55ac9c4d9d32fed7), C(766ef46cf807f655), C(d0884af223fd056b), C(bb33aafc7b80b3e4), C(36b722fea81a4c88), C(6e72e3022c0ed97),
C(d0884af223fd056b), C(bb33aafc7b80b3e4), C(36b722fea81a4c88), C(6e72e3022c0ed97),
C(299b25079cd24931), C(9524a42a403e3a3c), C(63cf882a484d9a7c), C(eef97121c5a5a2bb)},
{C(c5fb48f0939b4878), C(7b773f9bcd0ec27a), C(6d36a844bb3f3360), C(984cf3f611546e28), C(d7d9c9c4e7efb5d7), C(b3152c389532b329), C(1c168b512ec5f659),
C(984cf3f611546e28), C(d7d9c9c4e7efb5d7), C(b3152c389532b329), C(1c168b512ec5f659),
C(4a0caa396f16d3e9), C(d812c505c18cb004), C(d52562645e5d6d29), C(23a69405f4a7a1fb)},
{C(292da6390260110), C(7608d31cc4c96e48), C(f843ecb8366f0809), C(24940a3adac420b8), C(5adf73051c52bce0), C(1aa5030247ed3d32), C(e1ae74ab6804c08b),
C(24940a3adac420b8), C(5adf73051c52bce0), C(1aa5030247ed3d32), C(e1ae74ab6804c08b),
C(5fb9715b2bd92e3f), C(d9936c85aa5cb454), C(48f7c7fa87fe0886), C(33384b5bf3cc0a0f)},
{C(1e0ee26b7044741b), C(1b7f67a75b435af5), C(b24891afcb0faa49), C(2a55645640911e27), C(4fac2eefbd36e26f), C(79ad798fb4c5835c), C(359aa2faec050131),
C(2a55645640911e27), C(4fac2eefbd36e26f), C(79ad798fb4c5835c), C(359aa2faec050131),
C(b08464da8fa19778), C(76225cd3b6ab507b), C(c2b3fedc26c3070f), C(2318b4cc1f1eaef)},
{C(69b8f7e762db77ec), C(d845fd95e4f669e0), C(b1e8e3f0f5c9037e), C(10a7228693eb083e), C(1054fb19cbacf01c), C(a8f389d24587ebd8), C(afcb783a39926dba),
C(10a7228693eb083e), C(1054fb19cbacf01c), C(a8f389d24587ebd8), C(afcb783a39926dba),
C(9cea7dd3c9d7c13e), C(359371f6c4b1f07d), C(a09f9aef856aeecf), C(47f8fd69174f57c2)},
{C(9b321366d6585031), C(8ffcf9094b4ed2e2), C(3b7321189816fdcc), C(39756960441fbe2f), C(fb68e5fedbe3d874), C(3ff380fbdd27b8e), C(f48832fdda648998),
C(39756960441fbe2f), C(fb68e5fedbe3d874), C(3ff380fbdd27b8e), C(f48832fdda648998),
C(5f15cd83a154bb24), C(b7f8f0acd027320e), C(cabef155f2bc6cc1), C(3f71cadd80d07c62)},
{C(9375c89169bf70cf), C(45d697d09989365f), C(3ab599efd811ae97), C(cba4c10e65410ba0), C(3c250c8b2d72c1b6), C(177e82f415595117), C(8c8dcfb9e73d3f6),
C(cba4c10e65410ba0), C(3c250c8b2d72c1b6), C(177e82f415595117), C(8c8dcfb9e73d3f6),
C(d7351ad4cced7536), C(4b642eb64bd85529), C(8210191af35a2baf), C(6ad4db2f9a5f7f96)},
{C(a8db1643cc52d94d), C(47d3bfec129f7edd), C(925b29c3dbfea463), C(951f2078aa4b8099), C(e68b7fefa1cfd190), C(41525a4990ba6d4a), C(c373552ef4b51712),
C(951f2078aa4b8099), C(e68b7fefa1cfd190), C(41525a4990ba6d4a), C(c373552ef4b51712),
C(4a6206e99d0547b1), C(f2f9350de5aa024c), C(1c61958a2c4f08fe), C(a49542310488f889)},
{C(cf7a9ea6a7a30dee), C(9573ae0f07cb7c2e), C(6793c6e1fad303dd), C(b57ec44bc7101b96), C(6cb710e77767a25a), C(2f446152d5e3a6d0), C(cd69172f94543ce3),
C(b57ec44bc7101b96), C(6cb710e77767a25a), C(2f446152d5e3a6d0), C(cd69172f94543ce3),
C(d5fe33eaa00b2cd), C(6339cc4808b80a67), C(c171e9e64ab11197), C(c423dff6ab960cae)},
{C(42c2e9f84dc7f129), C(7b482774b391095c), C(216a0d505d49b80), C(ed094f47671e359d), C(d9ebdb047d57611a), C(1c620e4d301037a3), C(df6f401c172f68e8),
C(ed094f47671e359d), C(d9ebdb047d57611a), C(1c620e4d301037a3), C(df6f401c172f68e8),
C(7660fea207a306b2), C(b0b345f811948268), C(9af19dfcd71a3507), C(f6b3906d362aa47b)},
{C(394c2c1cca4e9271), C(2573fb79ecb7111f), C(5113e80f2555b54c), C(5d765af4e88f3277), C(d2abe1c63ad4d103), C(342a8ce0bc7af6e4), C(31bfda956f3e5058),
C(5d765af4e88f3277), C(d2abe1c63ad4d103), C(342a8ce0bc7af6e4), C(31bfda956f3e5058),
C(9e2f370f3ca72d39), C(934672274dae1bbf), C(a01f2897c4ade8b8), C(c3ee6a2df09f4e4a)},
{C(d38df9e9740cb16c), C(79be3445c5491402), C(a15ead26a317837e), C(a6814d3dc578b9df), C(3372111a3292b691), C(e97589c81d92b513), C(74edd943d1b9b5bf),
C(a6814d3dc578b9df), C(3372111a3292b691), C(e97589c81d92b513), C(74edd943d1b9b5bf),
C(ffe5b2d4ad351538), C(57be9ef9d48108b3), C(16506856c7346dac), C(f4f96dcf2d36b005)},
{C(ec12466d1379cfdf), C(84aae38bd5b56932), C(1407e7cad8d977df), C(63672de7951e1853), C(3ca0c763273b99db), C(29e04fa994cccb98), C(b02587d792be5ee8),
C(63672de7951e1853), C(3ca0c763273b99db), C(29e04fa994cccb98), C(b02587d792be5ee8),
C(12bcd1cd2ed88cbe), C(868db516f6cfaf86), C(70325ea7a64556c3), C(a1532526ba7d70ba)},
{C(9050986d9ced6a2e), C(8bc353d8f72e4f9c), C(b16a21f3ae8ddaf4), C(a16cd2e8b445a3fd), C(f0d4f9fb613c38ef), C(eee7755d444d8f2f), C(b530591eb67ae30d),
C(a16cd2e8b445a3fd), C(f0d4f9fb613c38ef), C(eee7755d444d8f2f), C(b530591eb67ae30d),
C(b803af84df15e736), C(e6d43b5e062f99e8), C(8edf24b490ca760e), C(7af8278b4d638935)},
{C(c7362967930e8a48), C(a61695f6772f5336), C(96e9b973fe114561), C(5386ef0b438d0330), C(d39e03c686f8a2da), C(9555249bb9073d78), C(8c0b3623fdf0b156),
C(5386ef0b438d0330), C(d39e03c686f8a2da), C(9555249bb9073d78), C(8c0b3623fdf0b156),
C(2d4bc6594c7a5295), C(31e13fa47c586321), C(2d68fdb034b0fbe1), C(dfc819d4cb95e6ed)},
{C(47bd8137d464eab3), C(236db8fed274d4d7), C(499063daa6e4eae3), C(e381f24ee1d9a97d), C(7c5d95b2a3af2e08), C(ca714acc461cdc93), C(1a8ee94bc847aa3e),
C(e381f24ee1d9a97d), C(7c5d95b2a3af2e08), C(ca714acc461cdc93), C(1a8ee94bc847aa3e),
C(82e217c1596deac5), C(747ce14ecb967425), C(7999db153c12cd13), C(66995512db3e71b9)},
{C(cff30d9303db2dfe), C(1afb5899ab9c8653), C(b2d9cc739ab9f148), C(4cbef49086e62678), C(d77dfecc2819ef19), C(c327e4deaf4c7e72), C(b4d58c73a262a32d),
C(4cbef49086e62678), C(d77dfecc2819ef19), C(c327e4deaf4c7e72), C(b4d58c73a262a32d),
C(cc79b6db913dc3c2), C(1aed3e358bee3387), C(6f785322ea91daf1), C(2524fdd521691a3c)},
{C(8d086fc30b6694b2), C(90533a6a1124ec0b), C(f24a7ec2f48b6809), C(becb065dc12d8b4e), C(ebee135492a2018), C(d3f07e65bcd9e13a), C(85c933e85382e9f9),
C(becb065dc12d8b4e), C(ebee135492a2018), C(d3f07e65bcd9e13a), C(85c933e85382e9f9),
C(4d47e3ea7334ca85), C(2dbad2622bcd261e), C(7f41afc6a29b148b), C(a127381c5f036c23)},
{C(b7d681356bdd9e4f), C(9e8e19b5cdbfb229), C(e8f5fbafde7bea61), C(bc944c1b5ba2184d), C(ab3d57e5e60e9714), C(5d8d27e7dd0a365a), C(4dd809e11740af1a),
C(bc944c1b5ba2184d), C(ab3d57e5e60e9714), C(5d8d27e7dd0a365a), C(4dd809e11740af1a),
C(b8bf591b4926fccf), C(9dbbca9cc696d802), C(1346415c65076fab), C(f68d75f0b2ce488b)},
{C(5bb01fcb2e6ad355), C(895c355e71191ef4), C(1f7a98978f1bf049), C(aaa144fbe3e6fda2), C(52a9291d1e212bc5), C(2b4c68291f26b570), C(45351ab332855267),
C(aaa144fbe3e6fda2), C(52a9291d1e212bc5), C(2b4c68291f26b570), C(45351ab332855267),
C(e97b427d76af6ce8), C(c5475d8608b9f1b7), C(ec7bc03fa014a705), C(532a752515fbad7f)},
{C(cd2ff001a80d1b11), C(c0f8d9d7d08c74b3), C(5df56e499e9ca980), C(b8c18d66154ac51), C(5807350371ad7388), C(81f783f4f5ab2b8), C(fa4e659f90744de7),
C(b8c18d66154ac51), C(5807350371ad7388), C(81f783f4f5ab2b8), C(fa4e659f90744de7),
C(f40172efee95ea91), C(790d87b29b591f28), C(5312ecbb4f9d51aa), C(7e4fec68c1f7cfb0)},
{C(8bfbf611401100cd), C(599edd1f5154a546), C(56b61ed81d29796), C(b744f5056e74ca86), C(88aa27b96f3d84a5), C(b4b1ee0470ac3826), C(aeb46264f4e15d4f),
C(b744f5056e74ca86), C(88aa27b96f3d84a5), C(b4b1ee0470ac3826), C(aeb46264f4e15d4f),
C(fd904427792f38c2), C(3ed5c93003a67d42), C(e55b5825b0daad8e), C(b0fcbf7ea7240f37)},
{C(ec9ae0cf9290d012), C(14368811a4a1621d), C(de04cc7d2c562fcf), C(4323852cc57e4af3), C(1f5f638bbf9d2e5b), C(578fb6ac89a31d9), C(7792536d9ac4bf12),
C(4323852cc57e4af3), C(1f5f638bbf9d2e5b), C(578fb6ac89a31d9), C(7792536d9ac4bf12),
C(40e652b441ced5ed), C(70abd80b280284a2), C(59a9fde500b7020d), C(9aab0532ecd500f5)},
{C(4ac2a5e9dc03176d), C(78df6aca1dd90b2b), C(e5fcbc1dfe65f7db), C(766b71bff7d6f461), C(b004f2c910a6659e), C(4c0eb3848e1a7c8), C(3f90439d05c3563b),
C(766b71bff7d6f461), C(b004f2c910a6659e), C(4c0eb3848e1a7c8), C(3f90439d05c3563b),
C(ce4a4c5e98f3ece3), C(500bd4273e7d4819), C(72cc628b298ca6a), C(c7760b25c80513eb)},
{C(5fd51f635bc557a8), C(7e810b3d0b3db6dc), C(603d74dd65a3baf6), C(d178444a236c1f2d), C(5576deee27f3f103), C(943611bb5b1b0736), C(a0fde17cb5c2316d),
C(d178444a236c1f2d), C(5576deee27f3f103), C(943611bb5b1b0736), C(a0fde17cb5c2316d),
C(cd8d6caab9dfd706), C(1ffe8429dfef2301), C(2bf61b63c0ce54da), C(80f164c017afbb5b)},
{C(ec3521e8efdb1779), C(422da247696eedc), C(a372b6a2f501313c), C(7a265e37da616168), C(6a1f06c34bafa27), C(fbae175e7ed22a9c), C(b144e84f6f33c098),
C(7a265e37da616168), C(6a1f06c34bafa27), C(fbae175e7ed22a9c), C(b144e84f6f33c098),
C(8f8e0fd7460992e7), C(238397859829ccfb), C(ba50181729d67168), C(2e5afacc63032432)},
{C(a9147f0fb2e38bb1), C(d2da27a1045272e7), C(7bfb951842162784), C(9fd4d9362494cbbc), C(e562bc615befb1b9), C(8096808d8646cfde), C(c4084a587b9776ec),
C(9fd4d9362494cbbc), C(e562bc615befb1b9), C(8096808d8646cfde), C(c4084a587b9776ec),
C(fe762782827f2a3), C(37fb8895239f65b6), C(37194c9ed5c85f0), C(ff55d85de34a85cf)},
{C(a080e609751f2e81), C(2df3057a20f24b2e), C(b9a0f8b45d79c318), C(c84bb7b3881ab070), C(36fe6c51023fbda0), C(d62838514bb87ea4), C(9eeb5e7934373d86),
C(c84bb7b3881ab070), C(36fe6c51023fbda0), C(d62838514bb87ea4), C(9eeb5e7934373d86),
C(c8ce51475e7447af), C(e49577031c3482b5), C(6b9f34f6d01f844d), C(5cfba1d77493e3c2)},
{C(3bc578f69905fa2d), C(bb2c316b7f3eb8c2), C(18646edbbf14b0ed), C(4473c8e2a3458ee0), C(258053945ab4a39a), C(f8d745ca41962817), C(7afb6d40df9b8f71),
C(4473c8e2a3458ee0), C(258053945ab4a39a), C(f8d745ca41962817), C(7afb6d40df9b8f71),
C(ac337c1953b598aa), C(5f74b5c6c9ea57a6), C(c699f64c67042a5a), C(170a611794cd9a0a)},
{C(9e6a5e0641d1c0d9), C(ba3905a52abd5805), C(73c0bcd5a7366e15), C(b5f52041a698da7), C(29864874b5f1936d), C(49b3a0c6d78f98da), C(93a1a8c7d90de296),
C(b5f52041a698da7), C(29864874b5f1936d), C(49b3a0c6d78f98da), C(93a1a8c7d90de296),
C(b90225f5eeb7f6c3), C(31be9ab5998656bc), C(d11c1b51a077518), C(cec3b33b76a971e2)},
{C(83b0cdb3c934c679), C(88e72fd2e9c85618), C(25cbcd575a21c446), C(664ec3fad8521859), C(406f082beb9ca29a), C(b6b0fb3a7981c7c8), C(3ebd280b598a9721),
C(664ec3fad8521859), C(406f082beb9ca29a), C(b6b0fb3a7981c7c8), C(3ebd280b598a9721),
C(1a76fb666b936652), C(5f1d4f347534db27), C(29527543299312f0), C(dcbf8d2bfa78a9c3)},
{C(f174161497c5fa97), C(db3717129871fa98), C(5eb7a9d5a724daed), C(5414e385f5677a6d), C(41ef105f8a682a28), C(94ae9f66d82d71f0), C(6b698643f5924cc6),
C(5414e385f5677a6d), C(41ef105f8a682a28), C(94ae9f66d82d71f0), C(6b698643f5924cc6),
C(43b734974ec0c76a), C(4ced7086d84d105e), C(f86223fe44450eb1), C(5ced41d863a4d226)},
{C(d7262cb2f2755e70), C(f87e6123e1e56dc7), C(76a474a551662078), C(d4bd358fed3e6aa5), C(8a1ba396356197d9), C(5bcf3e13a8184f6d), C(5f17183906e40929),
C(d4bd358fed3e6aa5), C(8a1ba396356197d9), C(5bcf3e13a8184f6d), C(5f17183906e40929),
C(4d19331b3ec930ec), C(53030189a9c901fb), C(93bff69e4c985550), C(cb289a8356a7ad3a)},
{C(1444ce264e8784b7), C(72dd36167600c3c4), C(7e52afec3d9d166e), C(db0c32f76f5b7fc1), C(5e41b711f0abd1a0), C(41f387462b60ec17), C(20cd7feaf6b0d5ea),
C(db0c32f76f5b7fc1), C(5e41b711f0abd1a0), C(41f387462b60ec17), C(20cd7feaf6b0d5ea),
C(52b22d58c4850902), C(7cdc51eb1705c332), C(caef4d284e0a6bd5), C(195aa229881bcd1)},
{C(532e6b5c95a2e229), C(2e74b7cb427c4e8a), C(90febf2e15deaaa5), C(32d3a29cf49e2dc9), C(3079c0b0c2269bd0), C(906379f72fdcdff1), C(ea076563ae6ed4ce),
C(32d3a29cf49e2dc9), C(3079c0b0c2269bd0), C(906379f72fdcdff1), C(ea076563ae6ed4ce),
C(9ade243380413403), C(1a1edd18efe953a0), C(375cfbeadcefb824), C(2f245730c19edfeb)},
{C(183d112159f539eb), C(e0a272f5325ccfce), C(82051bab809a3bf3), C(a54eaa5d7f3a7227), C(9d26922965d54727), C(858c68ea2e46ec2e), C(e157ea542fd04d72),
C(a54eaa5d7f3a7227), C(9d26922965d54727), C(858c68ea2e46ec2e), C(e157ea542fd04d72),
C(4482da9f5920b9a7), C(9eb3d1148a6bb2b), C(7607715a95ebcfe1), C(72780826506d32a1)},
{C(8f18272400b3ace9), C(f936fba4e6c7f6f5), C(a395a3524727f255), C(ea37f61c0c2f6d53), C(9b0c2174f14a01f5), C(631f4fbb52f7b4e1), C(882e8ea542c5a526),
C(ea37f61c0c2f6d53), C(9b0c2174f14a01f5), C(631f4fbb52f7b4e1), C(882e8ea542c5a526),
C(1ec92e2a2420d7c1), C(2ac6c4b00c99a2f7), C(a51a134fabb20833), C(210b66fdd86d39db)},
{C(43761e6a5f6f2fd6), C(49ff8647ef7de2cd), C(9db86e928d9c50d8), C(a32c192f6e3c3f66), C(8f10077b8a902d00), C(85dcc88c159d5666), C(2277bf43094b85be),
C(a32c192f6e3c3f66), C(8f10077b8a902d00), C(85dcc88c159d5666), C(2277bf43094b85be),
C(94a19f055590ffde), C(9fa196a7fbee216b), C(2dd7eb1748e1938e), C(8d362a7e503ef13d)},
{C(44f615fcd096fbfe), C(f91b1ef287c2b469), C(b0153a7a31076599), C(a82a7bb790678fc9), C(d197682c421e4373), C(92183ad53c7546c2), C(80e483f6e47eb386),
C(a82a7bb790678fc9), C(d197682c421e4373), C(92183ad53c7546c2), C(80e483f6e47eb386),
C(70138535c3e5336d), C(4eb80eb6b56b8009), C(5405df59eeb1b89e), C(c1966a88b1dd7355)},
{C(27613f9db818cf78), C(ffe86d35e2b6af5c), C(bf8519db808bc3fa), C(33322363b5f45216), C(7e83f1fe4189e843), C(e9fbbdca5076a660), C(a025a44ad19f89a6),
C(33322363b5f45216), C(7e83f1fe4189e843), C(e9fbbdca5076a660), C(a025a44ad19f89a6),
C(ec03c179728e7004), C(98e742a81a7d4523), C(bebbab82d82216fe), C(ed2f3f8a5219e953)},
{C(3f6984c7afaebd0b), C(ce1a85c17d6eaa95), C(5b4fdfaf084a67cb), C(33c6690937582317), C(fe6d61a77985d7bb), C(d33afeed3012068c), C(9769fde4cc576926),
C(33c6690937582317), C(fe6d61a77985d7bb), C(d33afeed3012068c), C(9769fde4cc576926),
C(7dd0a0f3b49c71e2), C(5647e9c2d9e75f7), C(ed0e4a8939988a28), C(32a7e3586cd9fab8)},
{C(8fc511284f47c772), C(b4c0724c9dd30a7a), C(d04d61891a700659), C(f3d41b3d4717eb83), C(2670d457dde68842), C(8af262ab62293c), C(c4bbf2c513a0d201),
C(f3d41b3d4717eb83), C(2670d457dde68842), C(8af262ab62293c), C(c4bbf2c513a0d201),
C(76e978704a11a16b), C(b0724f72c1f82e9d), C(45712558287f76ed), C(822ac276e9a84f89)},
{C(15ae5f12f88592e2), C(aeec43559dd10188), C(5bd91d548ffefd6e), C(dd70e407984cfa80), C(66996d6066db6e1a), C(ccdf7065727283f8), C(11e3ac9d9fa63e16),
C(dd70e407984cfa80), C(66996d6066db6e1a), C(ccdf7065727283f8), C(11e3ac9d9fa63e16),
C(1de97ef223cb1fe0), C(af9d731f864de29d), C(90b5d80578123019), C(f38f41986f3aba8f)},
{C(905f995bddf92cb7), C(f7847475e7e1e12), C(4dc5bb06cd995cee), C(9432536dd9f65229), C(192dc54522da3e3d), C(5eecf6aff92e88d4), C(2d450eaa62f5f8cf),
C(9432536dd9f65229), C(192dc54522da3e3d), C(5eecf6aff92e88d4), C(2d450eaa62f5f8cf),
C(76e3d90a7955b8fe), C(4e1884d2082708af), C(10a38e951ca848dd), C(402eef96359fa8af)},
{C(a23ac6bef8905fec), C(f675b3d658210f05), C(b363b9add66e1684), C(b9aa5bead3352801), C(8a6d9e02a19a4229), C(481ba75a9a2f85cd), C(17f5920e749edd3c),
C(b9aa5bead3352801), C(8a6d9e02a19a4229), C(481ba75a9a2f85cd), C(17f5920e749edd3c),
C(d105b754b4015341), C(36b8eac51f68cb8b), C(2fe079ec9fbe8825), C(8bce2434b0bc343c)},
{C(403b94a75160a06b), C(c6eb4d9351f81e3a), C(aa0739cef99075f8), C(8d8e7c63385df78e), C(16d55add72a5e25e), C(5fc4a1f7b4b489f1), C(b6bb01a378d6ff12),
C(8d8e7c63385df78e), C(16d55add72a5e25e), C(5fc4a1f7b4b489f1), C(b6bb01a378d6ff12),
C(694e481b95f51b9e), C(6bd91d351a87a04), C(568dc4871bb02524), C(319de7513b93791c)},
{C(14d1ee05672fc19b), C(703c1a85efd79788), C(d065eb50cfb5c451), C(ce218d5b44f7825a), C(2ae0c64765800d3a), C(176c27079a2958b8), C(9c886360f29d425d),
C(ce218d5b44f7825a), C(2ae0c64765800d3a), C(176c27079a2958b8), C(9c886360f29d425d),
C(48fee499ab87acfe), C(832fd5f475e2d5c3), C(4cdd0b88b9cd19e5), C(ff991c415dcc6ca)},
{C(f59376c617951a2a), C(32082c76de4710ff), C(5d291a873d8d9f9a), C(1682f54521c291e2), C(17ca7ab8d97ba0d9), C(d8d6d62c4e314ae1), C(269df5967d606f43),
C(1682f54521c291e2), C(17ca7ab8d97ba0d9), C(d8d6d62c4e314ae1), C(269df5967d606f43),
C(fe5f6273a99d92d2), C(c6acba21b2e1211f), C(f4e1eb97c0a53ac), C(5f9fad37e48c2e7f)},
{C(63982fdc37a9dc5), C(8f7025774c4faea9), C(e7d5ccc2c570c989), C(8868a216a8f9946), C(136b5ce2ede03238), C(c19f0f6f73f2e30e), C(9219f59f493aa238),
C(8868a216a8f9946), C(136b5ce2ede03238), C(c19f0f6f73f2e30e), C(9219f59f493aa238),
C(adfaf5e6efe814d5), C(49259b1d5b37655c), C(9161c70ec49c1228), C(c7f3381c1f29cf87)},
{C(eb480334ed838b48), C(87f74500f9dd7bec), C(39dfa7182359c93d), C(6b7d8d04ae2db2b2), C(703b85004b504bd6), C(6b08bbdf158d17eb), C(aa84994c75c80c1b),
C(6b7d8d04ae2db2b2), C(703b85004b504bd6), C(6b08bbdf158d17eb), C(aa84994c75c80c1b),
C(7e8e99af6a4272d6), C(c254a89bc1ab8a09), C(2823ea97135168a4), C(d4779384def89768)},
{C(d0b9004efa0a1164), C(9f68c1718dbde89f), C(b5d9a7ab23808f7), C(9463491a8439eb54), C(2bd049dd3b9307dd), C(6b8387ee8561f97e), C(2db67c3e7c0097f2),
C(9463491a8439eb54), C(2bd049dd3b9307dd), C(6b8387ee8561f97e), C(2db67c3e7c0097f2),
C(c0bf9b2f81a1713), C(fc543e0d193a6893), C(bdccc5c843dffb71), C(eb5521c7deb6fe9f)},
{C(b31f2b6cc2a15506), C(28ad14cab53f717a), C(341f7d6f1212356b), C(eee5254624f99323), C(50f935ad0b11b484), C(56944e9d03e9b415), C(886a205391ac0526),
C(eee5254624f99323), C(50f935ad0b11b484), C(56944e9d03e9b415), C(886a205391ac0526),
C(7d2993188d92fefa), C(8fe8f54e4a57f263), C(48c0e62b15bc56af), C(e205bc74c8bc8684)},
{C(4f9da8a709bec12f), C(69347f992ba0b479), C(d82b32d7831a57aa), C(5e1aaa3d321c7c55), C(d808df942eb2b92b), C(6f303048a4f26df9), C(caea0391727bdafd),
C(5e1aaa3d321c7c55), C(d808df942eb2b92b), C(6f303048a4f26df9), C(caea0391727bdafd),
C(196fc0853f68a0e0), C(92d3147565562aa4), C(5fa3bdb80def17a8), C(7d5e40ed55b3f1b5)},
{C(5504000602e6f8cf), C(17b80b6681b62db0), C(d83dda14a4423630), C(7926e11179d2b882), C(e94fcaa22d091d09), C(ea51d34158cd60a8), C(b6d56df73b12bacc),
C(7926e11179d2b882), C(e94fcaa22d091d09), C(ea51d34158cd60a8), C(b6d56df73b12bacc),
C(e65162ce7807404e), C(31eb63f0c4a123b4), C(a7874f0d3640e515), C(420cee9f66be7433)},
{C(2d022d82f513a883), C(da474eedd3ed528), C(b339f8f80f556971), C(bba2b0a0f1b18055), C(1e72c71e6c54cb2e), C(e43bfe7158c31dc3), C(416e04cd551cf777),
C(bba2b0a0f1b18055), C(1e72c71e6c54cb2e), C(e43bfe7158c31dc3), C(416e04cd551cf777),
C(64aec39dc4f1f6c7), C(6cdf3dc50db736e4), C(ca8664901b3a00d), C(ace03c7e1867c414)},
{C(a87268205997eddb), C(e58cd6fb05cb965), C(c69eac6e1256a4d9), C(e1dbe62ee1aec1a2), C(db97b9879cfc5bb0), C(c1a96637db293163), C(cdacf3525efecb1e),
C(e1dbe62ee1aec1a2), C(db97b9879cfc5bb0), C(c1a96637db293163), C(cdacf3525efecb1e),
C(412d6debef79c3d2), C(d97655e1602121c9), C(472fa4096222bb9), C(fda8628a5e21990d)},
{C(fde5f0a803c3affc), C(9db3e206a3deda39), C(6a678f40a8058ab0), C(dd00bf5b13f1b349), C(9441065c7bddf30a), C(e6005534fbe17f8e), C(32346eb79b5ac530),
C(dd00bf5b13f1b349), C(9441065c7bddf30a), C(e6005534fbe17f8e), C(32346eb79b5ac530),
C(7d5ad1e595c0857d), C(366fe6c4c020aa85), C(8dab547b7baadbe9), C(87661c1cc5a9a08d)},
{C(fa46e0e215c1aabd), C(6b795034438df194), C(14571d59c36d29ea), C(fa36d64aa42643b7), C(d2c8ac46f42c7ed), C(f3d36aa6e975be3f), C(e7f022266f7e0f3e),
C(fa36d64aa42643b7), C(d2c8ac46f42c7ed), C(f3d36aa6e975be3f), C(e7f022266f7e0f3e),
C(52276ebcac2790e3), C(713a9516f78e3920), C(e9712e1e52f5d4df), C(b12eb6f45a6d2080)},
{C(7e1f98b2c16f8b2b), C(c492c4d36704af4c), C(d23b89c36294385a), C(244cf0d778a05ec5), C(8380a5edd5a19005), C(29a9617055f5a9d9), C(7005e59f480fa82),
C(244cf0d778a05ec5), C(8380a5edd5a19005), C(29a9617055f5a9d9), C(7005e59f480fa82),
C(636e521af2d13f5), C(452d0180cfd54a1f), C(4e6bccbc4820cc1b), C(4291165870a4f268)},
{C(65a58d22d8665e60), C(41d64e7bbfe54f80), C(14ce593a63d5a590), C(97f748c8a8d4f327), C(3445886827d5e08c), C(634b32b334de0fa5), C(5ff94df6197c1bc1),
C(97f748c8a8d4f327), C(3445886827d5e08c), C(634b32b334de0fa5), C(5ff94df6197c1bc1),
C(e5e878f3d27813bb), C(8f2c727633cb701c), C(b9ff0e0fd561c99d), C(dc1a794a2d1dec63)},
{C(b781b9a55e7d6ab9), C(ed16ad5400bc87e), C(b2ae79ab4b601bec), C(1cfc0839227804c3), C(af51c631d33d0e65), C(dd36799ebc883ef5), C(548b458a566154a8),
C(1cfc0839227804c3), C(af51c631d33d0e65), C(dd36799ebc883ef5), C(548b458a566154a8),
C(d90598c43816265b), C(27d6a04ebf7cd129), C(2054183f5a23ec40), C(a05700bb0668541d)},
{C(a88c857b1aeb0835), C(1b2cdaa3a001518c), C(b03b42ed85390bea), C(47a8abfd4b36b416), C(c28d691c93c456ae), C(7b0890fd2c0a3e45), C(757b11be8b925e40),
C(47a8abfd4b36b416), C(c28d691c93c456ae), C(7b0890fd2c0a3e45), C(757b11be8b925e40),
C(81447174c18aedf8), C(6619a9228d979a37), C(6d363a425957a36d), C(3bbaf2ae71a999a3)},
{C(2a25994979124643), C(3a3e0c52948d116a), C(8667e3420bb8d50), C(ee4ae633d60287f3), C(baf02bf6e52c4b9e), C(31a0d0505fb5b745), C(bed2029003e3afe5),
C(ee4ae633d60287f3), C(baf02bf6e52c4b9e), C(31a0d0505fb5b745), C(bed2029003e3afe5),
C(b01c41401ac2bf5c), C(f5ecb340b107e5c4), C(3437ab8eb435721d), C(b3de626fb21560b7)},
{C(17236ed61e669c6f), C(b049f89783cd2c3f), C(28fe56765c1b7ed), C(e8e11d96868fd11e), C(d7323d276887576a), C(b6a62449ac769727), C(dfcf0d8773df717a),
C(e8e11d96868fd11e), C(d7323d276887576a), C(b6a62449ac769727), C(dfcf0d8773df717a),
C(409b51403f988323), C(478637bdbcab0148), C(98443aeea28bfb90), C(510cb8c9de50ffbb)},
{C(304f56359ac375a8), C(9b303c940f78c5b5), C(d52df3445a1a027c), C(fa406ca71ad32716), C(103ee9820a95f912), C(4d4cc990a9c5f759), C(7a70381414b621cf),
C(fa406ca71ad32716), C(103ee9820a95f912), C(4d4cc990a9c5f759), C(7a70381414b621cf),
C(d73781d294e28941), C(27a74f93060e4850), C(2d2412d7f302b7b4), C(c5e86edf2d014368)},
{C(2e236ded6ce34194), C(55cc633269f53f75), C(b0b18d56850de1a5), C(eb648960aa937c06), C(ac4d4ac05a66c83e), C(b56ebc918ee8e99c), C(acf505efb66a8d78),
C(eb648960aa937c06), C(ac4d4ac05a66c83e), C(b56ebc918ee8e99c), C(acf505efb66a8d78),
C(61fd4a2fe0f6c7b4), C(44caa61737900d59), C(a0618b159774d0ce), C(7862dbfe6d79d542)},
{C(837ecb10d69f9bb9), C(a51533872cb4f5b1), C(620aaae18ac52da4), C(94728cfb8d580043), C(98dc3ea6890f66e4), C(3373ffcf756e1a31), C(5742e8e69476ef0e),
C(94728cfb8d580043), C(98dc3ea6890f66e4), C(3373ffcf756e1a31), C(5742e8e69476ef0e),
C(872d0f3464fa1d16), C(ea7cbc3848c1dc6e), C(d397294ddb7b2613), C(669688d4162e8061)},
{C(c94bc80993d726f2), C(13d4514a68dca054), C(4b2b9d78cda610a0), C(c6a9ed722f850d2), C(ff91629f0e2bda83), C(637d74f5a36dc3d3), C(788b506986bfbaba),
C(c6a9ed722f850d2), C(ff91629f0e2bda83), C(637d74f5a36dc3d3), C(788b506986bfbaba),
C(403b192883817625), C(cab571b4e769cae8), C(31f8b3dc88b037ae), C(fb278eb2bd7f3027)},
{C(463b54729349357a), C(381b0b67f1b0e6cf), C(ab16e09e606fd74f), C(783cc746f3e0ad94), C(4cb37580460063a4), C(bc693f29e8c8e068), C(d8d02963f0137472),
C(783cc746f3e0ad94), C(4cb37580460063a4), C(bc693f29e8c8e068), C(d8d02963f0137472),
C(31ccc2873c41544b), C(c06dc76468b1ae0b), C(c1f3df4a93369f85), C(3f3cef75bab5a6d6)},
{C(52e298a69bc61248), C(f0c389219e5b280d), C(45d0e8cbf8999a16), C(abf9fdcb39df4494), C(e5f29602c7c58f65), C(41b808023b066957), C(e2158bf17bd49509),
C(abf9fdcb39df4494), C(e5f29602c7c58f65), C(41b808023b066957), C(e2158bf17bd49509),
C(7c768c4f020b5cda), C(5dab22904e7a1284), C(4b4c3bd118bba731), C(5ad2ff39ddea315c)},
{C(f31bde28294be223), C(6fcf217a39638e31), C(b57c20d0886767cc), C(9ff176e872b59750), C(b79b82d5e03a36a7), C(a05cd10a650a945c), C(bf270f3283985d7d),
C(9ff176e872b59750), C(b79b82d5e03a36a7), C(a05cd10a650a945c), C(bf270f3283985d7d),
C(caa5f83a21cb9b18), C(9f202e882fe0aa5d), C(16dd89da53eb5adc), C(5216e37d06741216)},
{C(d1d98f3bbaf26f1e), C(6e1bb3126ee93a70), C(fdb28b9faa7455cb), C(77bccd28f0502dc2), C(4704cff4be86c032), C(2d7083cbb8dabf9), C(40f403385f91540f),
C(77bccd28f0502dc2), C(4704cff4be86c032), C(2d7083cbb8dabf9), C(40f403385f91540f),
C(16c3dae3a8334001), C(f147cc672486c8a0), C(6662f9789e91fd00), C(26c7a414998ca045)},
{C(77969267e761a5e2), C(ef907093519fd3ff), C(6a50d507180b9dd8), C(f7671fba16e03cdd), C(1d6da71263c3c627), C(1f3fd89b2729c1a7), C(823b6d2b1df14ad9),
C(f7671fba16e03cdd), C(1d6da71263c3c627), C(1f3fd89b2729c1a7), C(823b6d2b1df14ad9),
C(30aa3260a9f7a2fb), C(12d37b090f5199e0), C(e782082d20c2c888), C(d54934578119df6b)},
{C(763f1101a3d8e5d6), C(6de2bec8b2a9f0d9), C(bdc2cec3b0c72126), C(c684b15ed14c4849), C(25370610ff0d1b07), C(7b933b223c95a22f), C(3a44eb381671ea69),
C(c684b15ed14c4849), C(25370610ff0d1b07), C(7b933b223c95a22f), C(3a44eb381671ea69),
C(40db0baa0fbf6530), C(310bc1302e4c0bce), C(8f42002c428a01d2), C(9ebdd24c2ec70c0)},
{C(b6ffcab942c26180), C(1261443adbb2a65b), C(8bc905dfc7b85595), C(ada824a1b0baddee), C(dadce601b47afe19), C(46c3679d5436da89), C(c6251e4bebbd57da),
C(ada824a1b0baddee), C(dadce601b47afe19), C(46c3679d5436da89), C(c6251e4bebbd57da),
C(9a1d11ab6326b9ee), C(4d2264ede0a8fa6a), C(8e0d28c46f0a230f), C(8768a43ff488e65e)},
{C(65a85965268277a5), C(f29a5b2c9a51c439), C(aabf0324ece8814), C(6521fe578234c086), C(88294a1221e85ad5), C(685d8b1b46910a3a), C(600a02f24903a0a2),
C(6521fe578234c086), C(88294a1221e85ad5), C(685d8b1b46910a3a), C(600a02f24903a0a2),
C(df46eb75e5832634), C(cdf8f3fa8487d50a), C(1682d092975d5d0), C(6a319eaed17f1a5f)},
{C(6579248c4cabcf91), C(21b4fdd6da56df76), C(727c8a9161e8cc9e), C(f5ef59c60d0513ff), C(a2c01a296e191385), C(8a7be6557ac395e3), C(fa772f9eee991c2e),
C(f5ef59c60d0513ff), C(a2c01a296e191385), C(8a7be6557ac395e3), C(fa772f9eee991c2e),
C(e974088527162452), C(b2b17f57e5d511d5), C(a711d336d29e02e7), C(f07f7400f36629bc)},
{C(fcea6deb6fbc95de), C(fa083e475037dd9b), C(a402e8b120895dcf), C(128da31d726a4316), C(6b12ccb52a7d436b), C(86a73e06caadae8c), C(f0106fd320ce8874),
C(128da31d726a4316), C(6b12ccb52a7d436b), C(86a73e06caadae8c), C(f0106fd320ce8874),
C(64aa8b7daea0891c), C(5ec1bc1ff5d391cc), C(32fc55155de8c3ed), C(7f8a38b37601393a)},
{C(a5afb4dac88f15f0), C(fa2d11898daaa5e1), C(c118493d086eb030), C(932d16df5d691fb1), C(5dde659f15fe9b64), C(6a55cbf2cd4cf502), C(2c69b73b509c23e4),
C(932d16df5d691fb1), C(5dde659f15fe9b64), C(6a55cbf2cd4cf502), C(2c69b73b509c23e4),
C(9722df40bfbb4999), C(35c866515c4d4148), C(90da458940690443), C(ce5154f6d0a54e02)},
{C(35f437b7acbfd454), C(23e56c12e72a5eee), C(b29d3d3f9fea4832), C(bd6947ba8f8af57), C(e2c925a1262dbc65), C(cec8c16603d2616e), C(4acfe590ee1f5942),
C(bd6947ba8f8af57), C(e2c925a1262dbc65), C(cec8c16603d2616e), C(4acfe590ee1f5942),
C(80da14790406c57e), C(18fb591204bcd01a), C(dc5bbccb9ccb5186), C(fc31e73fe76a9aa0)},
{C(8f45f63a2f2d77d5), C(2f251880d8b83227), C(7b457c2c096a4fef), C(60386ba5baeb0a28), C(4c8db172ac835d05), C(c9f3245a03b60ffa), C(1e40258aeb689328),
C(60386ba5baeb0a28), C(4c8db172ac835d05), C(c9f3245a03b60ffa), C(1e40258aeb689328),
C(46a8d30c192531cc), C(793478b18882b435), C(a531e89a8c8db780), C(10df06050225f784)},
{C(62258e6fe64ea749), C(e61da1411606c50d), C(1e383042228a6524), C(ed27416d233bcc8c), C(c36902f1d5236598), C(72cc7a789bdc0df8), C(16e1c593759df8ba),
C(ed27416d233bcc8c), C(c36902f1d5236598), C(72cc7a789bdc0df8), C(16e1c593759df8ba),
C(2bcf22e9ea2bbe06), C(9baf6583531b76f5), C(83d9215c96f6211b), C(88c8c75776b2bdaa)},
{C(fc109f4192ba2587), C(da1eb26735719b82), C(da0616341a86759b), C(bae2015903c43013), C(ded1d087f02e51df), C(4865349424eee47d), C(cfab56ebb783561b),
C(bae2015903c43013), C(ded1d087f02e51df), C(4865349424eee47d), C(cfab56ebb783561b),
C(5d5d1fad719ab4c1), C(f33b42b15b262e3d), C(128672e5d3caef78), C(f63471af4bb28509)},
{C(5364968136715e44), C(4d6df79cca67503c), C(1b04a476a5af6319), C(38769eab2c8333ca), C(3babeb25b54f33e4), C(70e9850467e4d7c4), C(4b23b70a8351c781),
C(38769eab2c8333ca), C(3babeb25b54f33e4), C(70e9850467e4d7c4), C(4b23b70a8351c781),
C(f380a2b5efd93ecf), C(7778a9b5a9a58073), C(7cda8c1a01ae53f9), C(ebf7ddf24a99ec41)},
{C(dd84538848e07acb), C(7e8b2c2ea4b31867), C(4a1b7795ab30febc), C(6264b317e74f1956), C(e7baa170c2d01227), C(1d6e44e71bfb4f79), C(96c3307376e7264a),
C(6264b317e74f1956), C(e7baa170c2d01227), C(1d6e44e71bfb4f79), C(96c3307376e7264a),
C(316241819c488f50), C(e688950b08de8201), C(aa85fb3cbca8745a), C(62eb90d7fd588cbf)},
{C(397d78f9c2fb2a8a), C(a35668e29dd4493e), C(4b5d814365f300c3), C(b47c59e2352ab2de), C(fee916bf69dca4f2), C(31720599f9b05204), C(cefd056a9a8cdb6d),
C(b47c59e2352ab2de), C(fee916bf69dca4f2), C(31720599f9b05204), C(cefd056a9a8cdb6d),
C(3a9f817985de55da), C(d3f86f21fa53638c), C(c80df25daac6a39d), C(e9372b43dbd3e5be)},
{C(a3a22aed573f4128), C(6f441959a6a745a4), C(61b303cec0f02d33), C(7b245212b57b6416), C(483eb1c2247030ca), C(9e73ba2d0cf42603), C(7caaf3b868c1fa6d),
C(7b245212b57b6416), C(483eb1c2247030ca), C(9e73ba2d0cf42603), C(7caaf3b868c1fa6d),
C(587c30a367b868f1), C(102ed86d3bddb02e), C(1a6b6420d412f047), C(93f5919b9f81a96f)},
{C(94bcd5be64b0caf0), C(3eee5a2a3b25c93d), C(465dd18f325b992d), C(fd45ffd2387a5d25), C(35a79122b1ac38d2), C(45c691735698b5c), C(87dfa15417044ee0),
C(fd45ffd2387a5d25), C(35a79122b1ac38d2), C(45c691735698b5c), C(87dfa15417044ee0),
C(7279e6eacd0e3cc5), C(d4fb6388249a8e45), C(14c73952dd243da5), C(4bd8eaef54668ea0)},
{C(81d9fe1f35fe8dc), C(2c5e1e21c57e9a0f), C(9cee67a7a5c2455), C(f92e222a71b413fe), C(ec8f28b36b4b2d43), C(73416a88b65b673b), C(cf89cf0d3d4148ae),
C(f92e222a71b413fe), C(ec8f28b36b4b2d43), C(73416a88b65b673b), C(cf89cf0d3d4148ae),
C(eb12721f666ed5f4), C(9b4d30144ffeb98e), C(a1334e47f3139dc2), C(582ecf73171747b7)},
{C(aa21f88e4310c4aa), C(fd678eaf1a2ef449), C(9eb788a336487a2f), C(6c43d6ff49cff14b), C(88c3f8f93676751c), C(2322a939c0b8bef3), C(4adf71d19b5e462b),
C(6c43d6ff49cff14b), C(88c3f8f93676751c), C(2322a939c0b8bef3), C(4adf71d19b5e462b),
C(7828ca80e7f716f), C(b9f318f725e85721), C(9b3e4ed34a015d91), C(a952dda20242fc40)},
{C(88e65c8bd8fd0dc3), C(67506bfcc1350ac), C(8e35bd6404cd4a19), C(10f30b6a6ec67afe), C(aeb8adee4a38102c), C(c7c31205261346ba), C(ed3a7eb75f0bd93c),
C(10f30b6a6ec67afe), C(aeb8adee4a38102c), C(c7c31205261346ba), C(ed3a7eb75f0bd93c),
C(b775ebcf84957a11), C(17a623164b5489f6), C(c6cfa336bdca2d4f), C(1924f92fa79468d4)},
{C(ee7c287c7a74eaf6), C(a4716c9b01821875), C(15b4a29d0e8621fd), C(de6e60dce8b3c61), C(8c97da4b7dca945c), C(315027b9093d6a5), C(c60067ac2204a60),
C(de6e60dce8b3c61), C(8c97da4b7dca945c), C(315027b9093d6a5), C(c60067ac2204a60),
C(86cfd4ffdea36ead), C(c6936cae4e4828cb), C(9ddcada9edd726dd), C(82c0fee5b2be94a9)},
{C(59492bfd26df7a46), C(88ef9253d909b4e7), C(2dabd903fca564b6), C(d154dee359c629dd), C(61da6808b6b00605), C(1c862110cb1fa740), C(877bc417d7060346),
C(d154dee359c629dd), C(61da6808b6b00605), C(1c862110cb1fa740), C(877bc417d7060346),
C(855ba7d65e75c91b), C(8e389434d7cbed18), C(a867f0e5bc8ec887), C(225ca7717041376e)},
{C(79471e68a2e7b4c3), C(e2ecc811cd228f6c), C(1c0c9618d88b7cf9), C(c0eff8b2276af95f), C(3ec1a9c95964b183), C(9b480684e56c25dd), C(8b849da7c70dbd8f),
C(c0eff8b2276af95f), C(3ec1a9c95964b183), C(9b480684e56c25dd), C(8b849da7c70dbd8f),
C(908e5d09c4861494), C(83a76c29318afdae), C(b90c04d057223c0d), C(af640ded7aa860a4)},
{C(f806f8b0f54bbbf4), C(c55ef4c3c37c3e8), C(c57fbf8500049609), C(48180ec114a019d9), C(5dde07c8bebb3f74), C(f3c14ac2fd15afc7), C(18ba219f867fa279),
C(48180ec114a019d9), C(5dde07c8bebb3f74), C(f3c14ac2fd15afc7), C(18ba219f867fa279),
C(2b7b5ace013b7952), C(d406dc879ee138d6), C(f5c56dea1f74811b), C(3cb3d87ed825def8)},
{C(af0a9fa8d197fc2a), C(83821f8be6e51915), C(297afa7cc277321a), C(5006b740dd5776ee), C(f139123a5edbaaca), C(40b1400056322a19), C(b8d5d524f5b2d708),
C(5006b740dd5776ee), C(f139123a5edbaaca), C(40b1400056322a19), C(b8d5d524f5b2d708),
C(7c6f8579423fd3a7), C(91d265e096ea544a), C(df81c4b86b793348), C(6f211c19875236fe)},
{C(a93491c935028bfd), C(2c36d238aac64cd6), C(fc5dcf1134041bb3), C(d813172f90f9ff47), C(c65afeef14c7926f), C(e0b91c18356984dd), C(5b8bcaba09e98375),
C(d813172f90f9ff47), C(c65afeef14c7926f), C(e0b91c18356984dd), C(5b8bcaba09e98375),
C(a958355d9d9ef12d), C(15dfe0e1301fb990), C(fb004df606238c5f), C(3a6d4e39068c8d6c)},
{C(35fb344f57414e7e), C(b4b10fbd03ba41bd), C(f084e04093c69f21), C(1f4862ff7005835b), C(e3281b9e914c7ab1), C(b551d8d1d75dfe2d), C(cc64942fb0af0ab5),
C(1f4862ff7005835b), C(e3281b9e914c7ab1), C(b551d8d1d75dfe2d), C(cc64942fb0af0ab5),
C(ded13b1da3ced22b), C(92d399d2b3fe6c63), C(46f71f9118bc6d34), C(4060ba4d225575a2)},
{C(650c588ae7997006), C(7b6516105738f40b), C(fd225f2b4d05e97b), C(76d74223d7c381f4), C(84445fec43ac4ba9), C(e457a2d9de2f0cb), C(4f80e43844d5a721),
C(76d74223d7c381f4), C(84445fec43ac4ba9), C(e457a2d9de2f0cb), C(4f80e43844d5a721),
C(6e228e27a8f32d2), C(9954e3c2df675edf), C(19c5266d5c3a052e), C(ec7b3184c3cc2c48)},
{C(8e83c18ec4fac9b2), C(dd53e5013eea5434), C(15a4f84e1fbb081f), C(2494508d174f0a6a), C(f8f421c15e7812a6), C(15d14a45d852a3c1), C(a12e6ceaa7b91862),
C(2494508d174f0a6a), C(f8f421c15e7812a6), C(15d14a45d852a3c1), C(a12e6ceaa7b91862),
C(fa15ecaaf16ab412), C(f97aa90da810e7f9), C(3c60ed31a6b864af), C(b9ec2bbfff57377)},
{C(35422c6582e3fa2e), C(babd15f9e3a1362d), C(aac1901ece0f6dbe), C(a1d206f25bcb2e04), C(630172f8ae1fbfa7), C(4df60c162fd13077), C(24fcca03cbfd0d36),
C(a1d206f25bcb2e04), C(630172f8ae1fbfa7), C(4df60c162fd13077), C(24fcca03cbfd0d36),
C(4e04f489279c4ee7), C(50dce0384874852b), C(c0cae8ce2251502b), C(1d5f6b7dfac0ef99)},
{C(fc0cb7f55d516f4e), C(a92f59467a97a76e), C(f058bdc360787c2e), C(2a428170995dc60e), C(90bf382200c6050d), C(522302fa40bdb9fa), C(7d84077cabfda564),
C(2a428170995dc60e), C(90bf382200c6050d), C(522302fa40bdb9fa), C(7d84077cabfda564),
C(1706c4478cb26489), C(92dc4805ee38f9ab), C(c0aa6786a003d75e), C(bce5aa22f7c7f483)},
{C(e6245e6273cd7da4), C(bb6b62af69cce44f), C(def05217f2ba3b7e), C(40bd603028fabdfe), C(897f7885daba93), C(d343910442a51554), C(fe448c8b278337b5),
C(40bd603028fabdfe), C(897f7885daba93), C(d343910442a51554), C(fe448c8b278337b5),
C(95a10203c4b4f72e), C(8ca0cabffde487e7), C(94f77ae93eadec4d), C(689c08b515b0a64f)},
{C(bfb40261b25b0146), C(fd74edf7a1e0a088), C(d9de3386702c1efa), C(ac61387d34b8eba3), C(f658eceff68e4f98), C(77eb7ea3faf6af6f), C(f2ae3f731cc6c3d1),
C(ac61387d34b8eba3), C(f658eceff68e4f98), C(77eb7ea3faf6af6f), C(f2ae3f731cc6c3d1),
C(8012cd6cc04c9d2f), C(c962c7159c064b23), C(606d0b4e750545b5), C(8f81a71197887df9)},
{C(298876b240a1f937), C(81c826252bc3fd08), C(f4ec81df6387b639), C(a87f848ac3739101), C(50a679588482b1c6), C(5ac0460a499bdd23), C(49cf4b738f9b6eb1),
C(a87f848ac3739101), C(50a679588482b1c6), C(5ac0460a499bdd23), C(49cf4b738f9b6eb1),
C(18fbe25e26136d71), C(9a49ca8398f46d58), C(996cfa5fc309641d), C(75983d8d8f22f3d1)},
{C(bf26833d8f21542e), C(62d95f72f9dc8b6a), C(978d2ce692fc2cd8), C(b2fc06ae50b5bdc4), C(b5ebdc6f7e689092), C(1324ae1b8c471019), C(235375036a20b675),
C(b2fc06ae50b5bdc4), C(b5ebdc6f7e689092), C(1324ae1b8c471019), C(235375036a20b675),
C(cc7585555d64049b), C(30d719f341c76e2a), C(79b4f9a243d91bfc), C(293b7273babf033a)},
{C(ff85120bd8fa3cd4), C(f768d9c178d8e1a3), C(a7a136668d023fec), C(227a859ccfb575f7), C(8fc9e6749312034b), C(fe1358976e8d1d7f), C(47e95007f7949411),
C(227a859ccfb575f7), C(8fc9e6749312034b), C(fe1358976e8d1d7f), C(47e95007f7949411),
C(61fa1c3806199041), C(c266a31216641d96), C(fa81946db7497216), C(9240d38d4387d001)},
{C(a37277b9eb9b16fc), C(2b780c1cd1dad364), C(dd4a2f266cd6aa0f), C(cce760ed2d17daf7), C(2c5e908016d4ff3c), C(c68cab0c465584c1), C(67a3eafc44e4d677),
C(cce760ed2d17daf7), C(2c5e908016d4ff3c), C(c68cab0c465584c1), C(67a3eafc44e4d677),
C(8daaac6fd2019b6f), C(52ac66567930f82b), C(ea93a3fefe0ef68b), C(ee62d455548ceff6)},
{C(b95c558eb132482f), C(86deccc1c79e3da4), C(8b43da48c6f5fd49), C(d35063f44069c518), C(bfa680a6399cae70), C(14af385219857c8b), C(1689871cfa0f2813),
C(d35063f44069c518), C(bfa680a6399cae70), C(14af385219857c8b), C(1689871cfa0f2813),
C(1623e108e2d9a0ae), C(c779b3052b04f6c8), C(c3c517367c3c2986), C(a716c8ca76706a57)},
{C(eb2a51b23ea2f82d), C(7d3802b6045fbd2b), C(a8670a39b06b41ce), C(851073e9d44dc921), C(b0aa6451924617c8), C(7b3ca65185907aad), C(4c1e01ba84283e44),
C(851073e9d44dc921), C(b0aa6451924617c8), C(7b3ca65185907aad), C(4c1e01ba84283e44),
C(d92787c23734a22d), C(a1b488d127ea79c5), C(15b41f35ba38902d), C(cf595df5f53be3ef)},
{C(c85dcc13ce7d29c0), C(3a520e93ccdeb592), C(ea19bd7ee74ed003), C(3628daac258a4a48), C(afb35f7b660a441), C(5cb7c82d5974037a), C(f285ff762ef24753),
C(3628daac258a4a48), C(afb35f7b660a441), C(5cb7c82d5974037a), C(f285ff762ef24753),
C(b56d83b9494581bd), C(d26ae4b301ccbf42), C(326937bae1dd44fa), C(6772af88666b0e2)},
{C(8a8707d80cb54c7a), C(a4135e3dcd129a6c), C(411cbfa152de1998), C(674142656124b5c7), C(1280123d0a21a062), C(1afc0b67eb7c5bf7), C(6a666c0d97040f27),
C(674142656124b5c7), C(1280123d0a21a062), C(1afc0b67eb7c5bf7), C(6a666c0d97040f27),
C(204e54dafa0472d), C(a3f8980d0d8f8872), C(7deba057ab58bb4), C(27f582ab633e1c50)},
{C(12c7ffecff1800ba), C(1172a3993fff1057), C(44bd867e67fc8ae4), C(81b2a78e5319365c), C(b3aa90916d02335f), C(d36241c3a0745572), C(5c6e74ae8dd3d88b),
C(81b2a78e5319365c), C(b3aa90916d02335f), C(d36241c3a0745572), C(5c6e74ae8dd3d88b),
C(5ffb3be405ecde49), C(54e8d94250ec6c3d), C(afa435e69d9a0c20), C(59d47e1e82e13cfb)},
{C(cb16c5c1e342e34d), C(9087c1dc80a18b68), C(d48a476057f23dc8), C(41ff65495500e2e4), C(b8fab1ad95574a61), C(dbfd0326a7599b91), C(ae5eb38b876d3fa9),
C(41ff65495500e2e4), C(b8fab1ad95574a61), C(dbfd0326a7599b91), C(ae5eb38b876d3fa9),
C(67bb0e231945b9da), C(19bacbfe4d30bec6), C(4403017a97509174), C(12508312752f4d5e)},
{C(27fddd06bd368c50), C(f7efaa7ef3bed090), C(b505f7b0690e3f70), C(ed194c89f81522b9), C(272a0528540527e1), C(57e1e98c484b9f28), C(e0d5a808989c1b7),
C(ed194c89f81522b9), C(272a0528540527e1), C(57e1e98c484b9f28), C(e0d5a808989c1b7),
C(5b4949a7af2a254e), C(c751a6e3760e34a5), C(ed3edc0227801ce8), C(2a40a423a8c5938e)},
{C(5e6c6ee85cec7703), C(dc33500e0dea1513), C(b0d9d93b584d752f), C(596dad7ffc69035), C(1a31664d3d509c10), C(8dcf0b12245cbae5), C(6a01673fd8b513d),
C(596dad7ffc69035), C(1a31664d3d509c10), C(8dcf0b12245cbae5), C(6a01673fd8b513d),
C(a06973b2de272d06), C(e34cb809a6943631), C(c04f78fc94affedd), C(13c85bafb9d380db)},
{C(2117190446b50f9d), C(8b48030b85d083ab), C(68a79e0698163241), C(716bb879de7fba1d), C(a388cdfa3c61ab73), C(411fada7499faf84), C(14ef89ce95bee4a3),
C(716bb879de7fba1d), C(a388cdfa3c61ab73), C(411fada7499faf84), C(14ef89ce95bee4a3),
C(c6984631f66b37a3), C(206f294644d75453), C(f00322fa947d8d57), C(21be185cc02ddcbf)},
{C(f3f12b62f51a9b55), C(2866854cc1edef6c), C(1476fb5f05a37391), C(c40dceb74330b7b7), C(b84db83f7ccc5f9), C(9d0117552050ed7f), C(25b93de89e85456c),
C(c40dceb74330b7b7), C(b84db83f7ccc5f9), C(9d0117552050ed7f), C(25b93de89e85456c),
C(d2cbfa684c41278f), C(f22b05f4ef2e3e3a), C(107a732e941c0415), C(1b08db44fdf2e654)},
{C(2ee01b9e2a7692a6), C(2a9920e8a8923bea), C(c6cfbcd09cc47583), C(80e53ddb1a7abd17), C(d585cba8c327e538), C(f4d313c03777336c), C(7d136bd0afbaf9dc),
C(80e53ddb1a7abd17), C(d585cba8c327e538), C(f4d313c03777336c), C(7d136bd0afbaf9dc),
C(d893509d490446b4), C(807c4679d3adee0a), C(807cbbafbcbc9957), C(db70f0c59df2b387)},
{C(53ca5e2da19191b7), C(ab8fba8b2c21655), C(ede0ddbd23d66d73), C(ce7dca3935c2fdc1), C(bd19a32205fcc165), C(a0a12ad442bce1f6), C(15eae05c9c6a0e03),
C(ce7dca3935c2fdc1), C(bd19a32205fcc165), C(a0a12ad442bce1f6), C(15eae05c9c6a0e03),
C(53bbdc49336b431b), C(6f40b3e0ff2197e), C(e1b3ebb501897509), C(d70e6c577947fc6)},
{C(ce6d0917744faa2f), C(42d66b23a164a48a), C(ecece7b7f05004f), C(e08c6a73a5559556), C(889313a96c9a1323), C(3253f1f2a3a66b1a), C(a2767544b53fee4e),
C(e08c6a73a5559556), C(889313a96c9a1323), C(3253f1f2a3a66b1a), C(a2767544b53fee4e),
C(eb4fb516b5e848c5), C(3952d5e0a0889e61), C(593708024334fdd3), C(c173fddd549ff969)},
{C(f9b8ca6b46052208), C(d693bb3a5a89c965), C(6e975d1339c4774a), C(d72e8efd275e3db8), C(c471d5f3d0c9b1b6), C(c62c1d0fc7800ec2), C(acb7f73f9b5a1109),
C(d72e8efd275e3db8), C(c471d5f3d0c9b1b6), C(c62c1d0fc7800ec2), C(acb7f73f9b5a1109),
C(141c9d6939ba929c), C(f44c580a3d5216d9), C(6170bee9a5840f2a), C(cd5fad5f272d3fdc)},
{C(fb1cb91d94d6cddb), C(dfb1e3d6b795c8ac), C(ec4df9c3e90f1e24), C(6c487cdea2c507de), C(82d42d569ff4c955), C(7c3767d754e62f5f), C(3916c993f8cdca8c),
C(6c487cdea2c507de), C(82d42d569ff4c955), C(7c3767d754e62f5f), C(3916c993f8cdca8c),
C(ac72b4669841cd23), C(edcf79abeac4526d), C(8f3a034b2dc14555), C(477788b7ae98d30a)},
{C(a39e2eab5f174f15), C(9f2a05a63d015c7b), C(c2e46128224f249b), C(3d2b520d3144119b), C(c8ade99b9d8d5092), C(65882e9c99a0ed12), C(42a89a2be41263d9),
C(3d2b520d3144119b), C(c8ade99b9d8d5092), C(65882e9c99a0ed12), C(42a89a2be41263d9),
C(5ff12f788782e5f4), C(eea9a70a50dfdfe7), C(82eb7ea3da4e0475), C(13577a0dfeb8735c)},
{C(e9bfc7e088623326), C(a79c980a7458736f), C(e7a09424c5bd6f77), C(d623ef8d9e4750dd), C(b329a5d0ce2c4320), C(724fc6ee18c04a2f), C(6f288c76ecde63bb),
C(d623ef8d9e4750dd), C(b329a5d0ce2c4320), C(724fc6ee18c04a2f), C(6f288c76ecde63bb),
C(42aa6d38b0f188c9), C(f86f54915bb475ec), C(24e7d21a88f38bf5), C(224b199acaf0960d)},
{C(24d3561ce4eda075), C(74fdf369a4ba7bd8), C(c979f4ef12661fbd), C(b2e2bf501c9bd4ee), C(f66a2607c4d22a24), C(b9709df0c8fa8889), C(f70db2a5a9e6f385),
C(b2e2bf501c9bd4ee), C(f66a2607c4d22a24), C(b9709df0c8fa8889), C(f70db2a5a9e6f385),
C(cc69cecc1c9a05a0), C(f9b4d34f357ca309), C(e699f628bcf84dc9), C(be7639e5e0c56de8)},
{C(3edb299037e41adc), C(8e3327b45d22677f), C(85d3af0877d1b233), C(41a0a96292eebd12), C(d331d1a9960dd15e), C(45c06e443e3580ef), C(8c9a4b60297b5822),
C(41a0a96292eebd12), C(d331d1a9960dd15e), C(45c06e443e3580ef), C(8c9a4b60297b5822),
C(5948fbcf970977c1), C(86c05412d73409d3), C(da15a8dbbf18bb5d), C(6e6715045bcec260)},
{C(4ccafed99120c34c), C(c061a2298aacd9ee), C(d8a2a419bbd61dd4), C(ee848fe0fa5feec3), C(a1c2bbe4bea46de1), C(f72fee59825eaba4), C(1c6cbfcae94f761a),
C(ee848fe0fa5feec3), C(a1c2bbe4bea46de1), C(f72fee59825eaba4), C(1c6cbfcae94f761a),
C(90c8a885e242533a), C(7986e3b258f40896), C(ba24b200f3d810d8), C(7fe5a2fd8ec10220)},
{C(811039d76b0f5c10), C(81c01b119d95abfb), C(b9f230a525dd1a79), C(82a0d5833ef0fe08), C(bcab840f326aa717), C(ed7f80003ad9c7a8), C(e1a7e9e27bfbb5ce),
C(82a0d5833ef0fe08), C(bcab840f326aa717), C(ed7f80003ad9c7a8), C(e1a7e9e27bfbb5ce),
C(3427ad55fcb9b73b), C(f492301179a35f7b), C(341984e0ea3b64cc), C(5cc0b09c58f1b385)},
{C(f26eca16e4f6b311), C(32720d4ea0a72e4), C(7e13d7dbee27de4e), C(5c37936e56cf7e46), C(295f982a83b30c99), C(b0241eb8061d0f95), C(e4dc70591f41cea4),
C(5c37936e56cf7e46), C(295f982a83b30c99), C(b0241eb8061d0f95), C(e4dc70591f41cea4),
C(6ea3615e4a3179d7), C(ff10e1277bb02e99), C(39694fc5ce56fe6f), C(262dbd8035b2dcd5)},
{C(8ce51e30cf1501bb), C(35e452a0a514fbf6), C(e12df99407eac10b), C(2a4a1228a520332a), C(7746e7c2193f936e), C(814a4661f92c5f06), C(df8cbc1191bb982b),
C(2a4a1228a520332a), C(7746e7c2193f936e), C(814a4661f92c5f06), C(df8cbc1191bb982b),
C(6de56359803681b4), C(8c6f79130e30ea94), C(637b9a34f4aec7df), C(da7d862440bd78c6)},
{C(80d0fa7707773de4), C(4be9c7b015a574a9), C(d4cb1cb66a739318), C(df8dc7766f988303), C(8188a46bb7a98536), C(d9fcbdd211e305cb), C(2c798285814ddf2e),
C(df8dc7766f988303), C(8188a46bb7a98536), C(d9fcbdd211e305cb), C(2c798285814ddf2e),
C(cc629e3efd52659a), C(61f3438f013395c0), C(dc9d0b73a8e8d3f8), C(ab12e1f8eeafbdbe)},
{C(698d6cc716818773), C(845fe2403582149), C(9502bc1422758522), C(3d59e8cf894ec921), C(5310828b8dbedfdb), C(238dce16320651dd), C(7b38b1f93ce8749b),
C(3d59e8cf894ec921), C(5310828b8dbedfdb), C(238dce16320651dd), C(7b38b1f93ce8749b),
C(59265c4b0f075023), C(f87957df23e72e2), C(ec2fac13f9ef4044), C(2b1beec5fcd735d7)},
{C(caaa5ff55032cbcf), C(538e9005d8665c92), C(e174f0f93d30f0bc), C(9ad16d0b0a5892be), C(a2ce93130b6539eb), C(e50402009848b944), C(5bf398fdd39286d5),
C(9ad16d0b0a5892be), C(a2ce93130b6539eb), C(e50402009848b944), C(5bf398fdd39286d5),
C(7f8c3c638a634842), C(ab819d8e31baaf2b), C(2f9c646d5a6d8deb), C(ad39cc0b51e2f3e1)},
{C(3333d53faadbec42), C(3f8857090ee7798b), C(5c95401451994dac), C(ca985ee7a329cd7e), C(76ecafcc948c9562), C(268ce4e1a2a5a074), C(21d353fba6630d78),
C(ca985ee7a329cd7e), C(76ecafcc948c9562), C(268ce4e1a2a5a074), C(21d353fba6630d78),
C(2482a3fa5eae582d), C(ce72c382f03b66c8), C(c2038b03d34a2406), C(37da6d3df431c413)},
{C(10882aac3dd3587), C(e0963a96a791586f), C(2d2e1c962520b6de), C(b2a78656df8faaa1), C(ddfae1420e3e858d), C(1912b4f86123a4d6), C(986e18713086add0),
C(b2a78656df8faaa1), C(ddfae1420e3e858d), C(1912b4f86123a4d6), C(986e18713086add0),
C(3ef7c89bc1a43c6a), C(2c1c7da4d72670aa), C(19dd3d16ea03f45), C(d82894ff9284f697)},
{C(b11fde1059b22334), C(efc2d98538f4ecfc), C(36af0ce3f9940bdf), C(d7a26a8e9c020084), C(f5177c6dc6d8a5), C(6926948892e970c8), C(840b22073cf60998),
C(d7a26a8e9c020084), C(f5177c6dc6d8a5), C(6926948892e970c8), C(840b22073cf60998),
C(c8a28fb310816e17), C(75b12ec8bd7276b0), C(22f16cd0cce663a9), C(860fe142526731d7)},
{C(8977ae72ed603d45), C(60f4ffd92231c25b), C(e2b1b66758d158fc), C(70caf8189b6e929b), C(1b80d6fcc87b4d5e), C(77ae1691bcc4bbea), C(5c619855527e1200),
C(70caf8189b6e929b), C(1b80d6fcc87b4d5e), C(77ae1691bcc4bbea), C(5c619855527e1200),
C(35b6841ff41ee0fc), C(189ab15fb444bda1), C(229a3e368d65cf4b), C(14b411551973ebd9)},
{C(f65b17f58e2f82f6), C(7b5e65ee7c85e9b9), C(5e28d4218467b771), C(c0a7d673c0ae5225), C(31b05cd187dce5fc), C(3bdaf6c1c3992de9), C(af7d1ee6c8d8e3ae),
C(c0a7d673c0ae5225), C(31b05cd187dce5fc), C(3bdaf6c1c3992de9), C(af7d1ee6c8d8e3ae),
C(8ef5d35dc9db068e), C(b6809da31f611d66), C(7a5a690553e0535), C(85b557b00f1bc064)},
{C(63689bb426fad75), C(e65b123bfc973da6), C(db08275d11847a43), C(92dc01e5daa6f8cf), C(e2c4e337f7e3c4bf), C(35339b7ca3a1be91), C(80a30021da4c2964),
C(92dc01e5daa6f8cf), C(e2c4e337f7e3c4bf), C(35339b7ca3a1be91), C(80a30021da4c2964),
C(44da9837fd17c778), C(99fc4212a05f1b65), C(dad7bfd429058f87), C(95fb0bd2b24cc484)},
{C(f09d687ab01da414), C(89e97db87314fab1), C(2c48cf28ff80b17f), C(9443d8392ae194e1), C(929e316b17083568), C(117dea737f1df80b), C(1fc14678b5adf5ff),
C(9443d8392ae194e1), C(929e316b17083568), C(117dea737f1df80b), C(1fc14678b5adf5ff),
C(c606f1606c6d1dea), C(b1c90efa2758acfc), C(44aeb51f03e8b379), C(efc95bbe6873f5e5)},
{C(f9946308ce8bcec0), C(cdfe313f59a7c805), C(43f83dac819e8271), C(f7ee9f4f36e9cfcb), C(16c3965ae72f209e), C(56f7bd99b0d467f3), C(c7bc0fdcc7a4f542),
C(f7ee9f4f36e9cfcb), C(16c3965ae72f209e), C(56f7bd99b0d467f3), C(c7bc0fdcc7a4f542),
C(911407e639fe7d24), C(ea7ff196af48b6df), C(cc9883ec8ed620bf), C(d3a17a13a8e980aa)},
{C(5f2a932916c5c63f), C(f4787134f14a7108), C(fba7efcc1e2629e6), C(9b0287c30033872d), C(7fe7cd37d8292591), C(a402891b4a428cbe), C(473f085727dea256),
C(9b0287c30033872d), C(7fe7cd37d8292591), C(a402891b4a428cbe), C(473f085727dea256),
C(6a9aa99d1a75c109), C(1bca8cbace83a214), C(b62c98f67ec1e05b), C(6369037390f1d525)},
{C(3a7933b10ff2e831), C(22ff143fefbbd3b9), C(2e552c66fb8678d2), C(f427ee42d5ee8003), C(1eebdcf751988c45), C(a4262cebab700e78), C(f3b14f368783f74),
C(f427ee42d5ee8003), C(1eebdcf751988c45), C(a4262cebab700e78), C(f3b14f368783f74),
C(64741e1158e8bf97), C(5513da7cb02cc5c6), C(7b26d550167439bd), C(f1159bc4e6e73f42)},
{C(41f45d562a6689b), C(e23f0e34570f037c), C(990e39e880dc1533), C(bd8a072257c813e4), C(547537b7deeece82), C(f0a854abf63d7f2c), C(6ad10c54bbf5f37e),
C(bd8a072257c813e4), C(547537b7deeece82), C(f0a854abf63d7f2c), C(6ad10c54bbf5f37e),
C(bb9bdd0b7c40eafe), C(273793f98e346d0f), C(2798a2e8d0bed8d4), C(1bbbd60b67f4a82d)},
{C(bcec7d59b5858e63), C(ce27686675aca1b9), C(6dbc1f5cd79cec30), C(35792372c0a1f9ec), C(b5c79c04405b7d56), C(78be8b169f1d27ec), C(ee96813ea6366da7),
C(35792372c0a1f9ec), C(b5c79c04405b7d56), C(78be8b169f1d27ec), C(ee96813ea6366da7),
C(be76cd4c6c875fca), C(29f858ea491a3b37), C(bbe166defb235e50), C(ac9152dc13fbbba1)},
{C(82ea92d6830c37ad), C(8839b9de78d0ead5), C(15f84a79fe513c18), C(ccd8ee9c81f0fd31), C(c052c7f03a00caf1), C(e685a7c5e2dfefd8), C(5d1e5b7cad442fc8),
C(ccd8ee9c81f0fd31), C(c052c7f03a00caf1), C(e685a7c5e2dfefd8), C(5d1e5b7cad442fc8),
C(be56991d85c30622), C(d3bde6352ce246b7), C(a999374fd2208e7a), C(f937ba03a20de6b6)},
{C(27cc4624e3a8fd6c), C(c2c8aae889151aa7), C(ac3dce81ee301ebd), C(27d10cde7611dbf4), C(b1df219237f18451), C(fdd47fef8b61284b), C(529b0e44e4875fc5),
C(27d10cde7611dbf4), C(b1df219237f18451), C(fdd47fef8b61284b), C(529b0e44e4875fc5),
C(fc621051be44dac4), C(82ea6e5f62f9b858), C(33f1beb7b2de0c64), C(3b931dc39a5fb667)},
{C(bfa129745aeb3923), C(490ee8b72fb3248e), C(5daefa90bb67c95e), C(d08e3a551657f581), C(174c60b071a111d9), C(e42f0ffcf7a8a264), C(2dcd1114c8457f34),
C(d08e3a551657f581), C(174c60b071a111d9), C(e42f0ffcf7a8a264), C(2dcd1114c8457f34),
C(a0faab1b5ad9289d), C(6b749fe852367c13), C(49719a6b1c7bc133), C(840897c86bf15bb1)},
{C(9b19fb3f08515329), C(85dea11f9278c39e), C(5d8dd543eb53b5c5), C(62792122b242fbb0), C(1339a529c030fb61), C(d95fd1afa65a5ded), C(8e6bffc81a339dd0),
C(62792122b242fbb0), C(1339a529c030fb61), C(d95fd1afa65a5ded), C(8e6bffc81a339dd0),
C(29ffc4d9e170776f), C(d5c1aec0263b9a5d), C(1fb9b94eef7c6592), C(ffe487ff8dfe37a3)},
{C(b944c2c819b2038d), C(fc4cf4ef53de6f83), C(bec7b4166eb32958), C(98560aae6f6c1e35), C(eea46b496f45722b), C(74a0e05c8ef8afb6), C(63efa8fb5359a688),
C(98560aae6f6c1e35), C(eea46b496f45722b), C(74a0e05c8ef8afb6), C(63efa8fb5359a688),
C(a832fc3675dce851), C(67039741c7a9ab7f), C(b5928cae803e0057), C(d67d8b6837ecd2fa)},
{C(6e8d2803df3b267a), C(b94287ee66ec3f05), C(4b19fa3db0bb8ae1), C(c72b486a73ddfdb2), C(f87aad46e1a788da), C(5fae4d0974a5384e), C(d7864668291c713d),
C(c72b486a73ddfdb2), C(f87aad46e1a788da), C(5fae4d0974a5384e), C(d7864668291c713d),
C(10bc4d083f706d3c), C(e1a8c45b4894c855), C(27d56a458a01d8b4), C(e88ffda4ffc4faf5)},
{C(a5ed64048af45d9d), C(7b48feba418052fb), C(7a4a22451f57afc2), C(6ca5a10bb6dde0cf), C(916d9f2c62b33970), C(a7005fe34edfbc94), C(14ed78a60e348f2),
C(6ca5a10bb6dde0cf), C(916d9f2c62b33970), C(a7005fe34edfbc94), C(14ed78a60e348f2),
C(f94c14a4fdb28cfe), C(621f771911614a7d), C(935de8bd92b5bee3), C(d3a2665d95d317e4)},
{C(6d56acb61a9abe8e), C(fdb4477c368483a0), C(352075394f788b74), C(4f626288a601b303), C(fa445e36c5fc1bd8), C(487f76509190057e), C(c0c2ec27c850d93f),
C(4f626288a601b303), C(fa445e36c5fc1bd8), C(487f76509190057e), C(c0c2ec27c850d93f),
C(ce77b29d23910bf9), C(c5207ac4b8456659), C(3edcad5599286d87), C(b8f29aaec7f64d35)},
{C(4f03f6750128b16f), C(c2307541c9970984), C(eba0e38bdfb2f415), C(844490b6c94ff01d), C(97d2f6acf11431c5), C(31e7de47b6d4d6ad), C(fcab3de1e8f50d67),
C(844490b6c94ff01d), C(97d2f6acf11431c5), C(31e7de47b6d4d6ad), C(fcab3de1e8f50d67),
C(6f7217b984b7cc1a), C(22b89fcb86c080d5), C(e8b98847fbf6c19f), C(ff072b5bf2f0c889)},
{C(6e717510c8e732c4), C(d3e508a9e3855fab), C(18e0ba1d43b19fe8), C(848ad83b6e0d60f), C(c09d282f51da855d), C(82e7f74688a014ed), C(d3fb00a2bfb9f821),
C(848ad83b6e0d60f), C(c09d282f51da855d), C(82e7f74688a014ed), C(d3fb00a2bfb9f821),
C(187b6a1790272da0), C(cc37f88497fde17e), C(fa6a383d6c7edd02), C(4ac922b3820d55b2)},
{C(6167f57448c6559b), C(1bdac82270ba5daf), C(99414798c789a18b), C(5ca3088d1d613904), C(b514e80fa4bd6173), C(c7f333680ad450a), C(ba41a35ce36b4fdd),
C(5ca3088d1d613904), C(b514e80fa4bd6173), C(c7f333680ad450a), C(ba41a35ce36b4fdd),
C(87d78285f7af2c2d), C(6c8323b8ea9b1040), C(d25ef64eae72e387), C(75b243c6861ae71e)},
{C(4c445bb3cc5dc033), C(2f468b3f6e9ccf43), C(60e78440ab5f7a3f), C(c0a78fbbb4d9c7fe), C(5d5372b2750b6a97), C(9fc2a31931008d5a), C(d236ea6530b29183),
C(c0a78fbbb4d9c7fe), C(5d5372b2750b6a97), C(9fc2a31931008d5a), C(d236ea6530b29183),
C(b8583ce049f864b9), C(bcf728cad3a28f6e), C(3fb1305967daff3b), C(fc4890163b21bfee)},
{C(3d63ec327c84a0bf), C(40dd9339cd2e68ff), C(f0ba798fa143e548), C(3ad5fe46fea96c61), C(5b78b66fef8dea6f), C(686332310340452d), C(a826cb8d2394f95f),
C(3ad5fe46fea96c61), C(5b78b66fef8dea6f), C(686332310340452d), C(a826cb8d2394f95f),
C(523dcbdacd37c38b), C(930405cb36b18ed), C(9669175e7aefab7), C(bd631d3a801cfbd3)},
{C(eab5f4a8d3ec6334), C(a7e1065573315d35), C(3381e6aeaa8906cd), C(bb2fca5617f2c8c8), C(a68ae975813669d), C(21eb53b81a1608d7), C(96a0a8cef0ab1adc),
C(bb2fca5617f2c8c8), C(a68ae975813669d), C(21eb53b81a1608d7), C(96a0a8cef0ab1adc),
C(2fbcb941f76cc3e6), C(ba24db10a1c06940), C(62e14a95c9bec91b), C(e83b14e4708092f5)},
{C(1ffad87ddc8ca76a), C(e6b143d6ed7f42a0), C(51fc65a5f15337a2), C(281b10815ee6b36c), C(131460a7e307fb49), C(323ac05bb6f260f), C(86504e553eeeb51b),
C(281b10815ee6b36c), C(131460a7e307fb49), C(323ac05bb6f260f), C(86504e553eeeb51b),
C(244166b9e84caef8), C(2b3a71e12a020931), C(5c3c1d0c143ac17d), C(b2da89c77a7dbe65)},
{C(fcc3b1db7bb174a0), C(61bea0ed7dc160e8), C(7882e4ab6c8cb280), C(ab914bed4e97e8d2), C(6071c5a779cc97fc), C(e5472aa9a23a7d31), C(628dea5a3164c608),
C(ab914bed4e97e8d2), C(6071c5a779cc97fc), C(e5472aa9a23a7d31), C(628dea5a3164c608),
C(65ff115faeacc3f7), C(fcf7f354d9d3f42e), C(33c1c482957e9247), C(b4c03837010d19d1)},
{C(cffe79062bb4e7cd), C(9e01b50f95301ea2), C(2a616a3eb9110b32), C(325894413570e9b0), C(1ba9bdb939ee6d9b), C(6a7c5f758b0f8a22), C(5613c8af1381df60),
C(325894413570e9b0), C(1ba9bdb939ee6d9b), C(6a7c5f758b0f8a22), C(5613c8af1381df60),
C(3a760d64558560a2), C(b199b6b9f138e20e), C(92d4d79a6c2a4eba), C(f79f6b69a4475cd6)},
{C(a21717e2b3d282ee), C(1ac0595d4f40cda4), C(429a8a47cea11c02), C(e680b930b66396ed), C(d017b03635aece79), C(b5547e06d64d2394), C(cd8ee2c6d0f48658),
C(e680b930b66396ed), C(d017b03635aece79), C(b5547e06d64d2394), C(cd8ee2c6d0f48658),
C(66070c4498569d0d), C(5cf3fb6be2590228), C(c70d152cb6e19cfe), C(2171726d2c323507)},
{C(7e4143da4d878be5), C(de837a44b83df910), C(c5096fd7847216c), C(d1ffc6a0e63da251), C(fc0b63ceaef917c0), C(ac5020cdbb7db31a), C(f600187306f3f361),
C(d1ffc6a0e63da251), C(fc0b63ceaef917c0), C(ac5020cdbb7db31a), C(f600187306f3f361),
C(d97e5425e390ec63), C(71833c8a2329940f), C(5c9ddc2802e33d37), C(58ff256c73d9602d)},
{C(23b80b8bc4e75405), C(29de38e234555045), C(b58684b753996de8), C(7be7ad6fb131552), C(4c4e822573890072), C(1b1bffc34b326bfe), C(d4ca629a7c07f772),
C(7be7ad6fb131552), C(4c4e822573890072), C(1b1bffc34b326bfe), C(d4ca629a7c07f772),
C(b045866ce1f2e9fe), C(80758cc9dc9af28b), C(adf5f4fbe1ff3079), C(9bdd487f5cce4293)},
{C(a6ae749a1ed10838), C(a9c706bcdcfa891e), C(de2661c018abc48c), C(97ad0eeccf7beed9), C(bf70a97987134ce1), C(b0988e5caa387cb2), C(c762a4b2a2126a63),
C(97ad0eeccf7beed9), C(bf70a97987134ce1), C(b0988e5caa387cb2), C(c762a4b2a2126a63),
C(92090927bb9a14de), C(3b75dac9fa07439b), C(4b4015ccade7be93), C(bc41810b9827a7cc)},
{C(d4b4a81be36638f2), C(92aabc9931541a5a), C(43ab0147e976e855), C(c825b61dceb4e636), C(e0809d70e918ada7), C(c35b1dca85adbea), C(f77b1cd8381a85a9),
C(c825b61dceb4e636), C(e0809d70e918ada7), C(c35b1dca85adbea), C(f77b1cd8381a85a9),
C(f16ee42414ffae5b), C(3f5315e1c3abda64), C(72bcff88bf794f5a), C(98e287de601f12e5)},
{C(5bab2890f354896d), C(a9d1f89bd9868dd), C(3dcc900485630f0f), C(4f22b5392f0b094e), C(b0f6c85f71e717a6), C(46957a3b2d65a038), C(b083716110d971b4),
C(4f22b5392f0b094e), C(b0f6c85f71e717a6), C(46957a3b2d65a038), C(b083716110d971b4),
C(590eb05886da3ca3), C(72a6705f42ae3360), C(539af2631737f900), C(eca90bc7cf8e5a9a)},
{C(4c0a184632b0499a), C(e9c2e26d5bd7346), C(3e4fd6dfe99c67d2), C(84adabbc4885d2c), C(7f13d1c57c1436ee), C(816789354e143b64), C(3df2247f878cc4a9),
C(84adabbc4885d2c), C(7f13d1c57c1436ee), C(816789354e143b64), C(3df2247f878cc4a9),
C(299027e5cd6bb193), C(eec646fd89651947), C(3f577a6318d67d51), C(a178d50920bc66d9)},
{C(b45a39714746ec86), C(96f58143107477ac), C(2dd11909380bb2cd), C(d91b8bb8672fd8f5), C(740fccac7b4f751b), C(30dece8f93a98d22), C(4dfc62e32800ede8),
C(d91b8bb8672fd8f5), C(740fccac7b4f751b), C(30dece8f93a98d22), C(4dfc62e32800ede8),
C(cbb8048c9d423752), C(e50d29a6cb06675a), C(1cc701fcfd0c7eb4), C(85039c3883e7f77b)},
{C(c4b90839e91abfb2), C(e82891efce710c00), C(4a7ed592a3a82dd4), C(7e607fe600517cd0), C(e532f493827b0237), C(395cda8e4fe45809), C(54c07a612f99b802),
C(7e607fe600517cd0), C(e532f493827b0237), C(395cda8e4fe45809), C(54c07a612f99b802),
C(5e6cd2a69a70cb19), C(79ec437368efb985), C(4910aa3b78f20a98), C(27ca0a6592390c2f)},
{C(e81d35c8ed7827fe), C(1a262c26a7b07276), C(207dc323cb840325), C(c248f06ca75157a0), C(d89f50212f3ce653), C(be6f8171f28a86d0), C(429a914e8bcd778e),
C(c248f06ca75157a0), C(d89f50212f3ce653), C(be6f8171f28a86d0), C(429a914e8bcd778e),
C(549831597c30b677), C(77b327c31cc9df9b), C(842b5e028095a9e6), C(55454e622b5ba351)},
{C(587c5ee43e034ebd), C(17d007f9bc666c2d), C(af9c82c94dfcda1f), C(2ca7857ac0ec7867), C(4c7405c3f345264c), C(7a5fcd4b620e0939), C(fac9f4b7677b447f),
C(2ca7857ac0ec7867), C(4c7405c3f345264c), C(7a5fcd4b620e0939), C(fac9f4b7677b447f),
C(8d9855416604a380), C(12d008ea1ef252fe), C(751ebd2c8c3f5860), C(3406afedfe8b3cf2)},
{C(b1ec87f8823040ac), C(a7b6d2e2223d8bb0), C(d41677026942ade4), C(ac5aadb9c48b988d), C(850ad5a0d3650159), C(e6f67795d6a04567), C(5f9ba2bbfa36e575),
C(ac5aadb9c48b988d), C(850ad5a0d3650159), C(e6f67795d6a04567), C(5f9ba2bbfa36e575),
C(2611cb09da2480a), C(3c39a5994ab1930), C(d83f3ffe11cce91d), C(28a4e12d589a26e7)},
{C(7677dff12f92fbd9), C(a1c9bc7d32f35ca1), C(edcd974aa7488258), C(e38fa487026a5a0b), C(3b9b7540bf4802a5), C(f3118d8cf9507c02), C(6e8147b6eab1fe87),
C(e38fa487026a5a0b), C(3b9b7540bf4802a5), C(f3118d8cf9507c02), C(6e8147b6eab1fe87),
C(4fb094c3c1e8223d), C(23661d08e0a8abe2), C(21c5355030a29935), C(b826eeb5c0c6dbf2)},
{C(b69cea6e5a0e28fd), C(3e24ba0592afe2e), C(57dedae1b68ddd05), C(6e5f6aded4efd69f), C(41aaf253fd433093), C(811156ade1688bfb), C(f6be0584b63c47f3),
C(6e5f6aded4efd69f), C(41aaf253fd433093), C(811156ade1688bfb), C(f6be0584b63c47f3),
C(ce21b17027ede42c), C(f6a8636d32ca379), C(88b6b8b90ebf5a6b), C(fda2054503a9ed37)},
{C(f7180ae2e0f325e5), C(d853c886b1187cb8), C(500adca11f8e94de), C(f3c31d687579578f), C(2c3f467e63c5225f), C(5b92432e0e17d7aa), C(ac7174bf58f98dec),
C(f3c31d687579578f), C(2c3f467e63c5225f), C(5b92432e0e17d7aa), C(ac7174bf58f98dec),
C(fd2ae3f49aa1be8c), C(95df781cdf1bf382), C(dba87a0e2301f8f0), C(1475c16876ee1569)},
{C(a08d214869e84ccf), C(12011850607c0dc0), C(6ca21154a2193f1e), C(7941a5bb3a6fb8e0), C(31785fae4ea5cbd3), C(1614b3e8ba5368d0), C(7f23439c6eced206),
C(7941a5bb3a6fb8e0), C(31785fae4ea5cbd3), C(1614b3e8ba5368d0), C(7f23439c6eced206),
C(64893e462a240f79), C(235a739c42b95d66), C(20a62c4ba35b4110), C(7c71136d6f83a83a)},
{C(cfff666740e2f99f), C(434be89d9bd1b14), C(d12ba6e50904b61f), C(eaac07a345101168), C(3ab8690ec91d6ed0), C(a3e4855a45efe602), C(34c0bbd016958641),
C(eaac07a345101168), C(3ab8690ec91d6ed0), C(a3e4855a45efe602), C(34c0bbd016958641),
C(e1f661d930d7b066), C(d741e0e8c9fedbf8), C(3983140fc0cf74e), C(710084232663041e)},
{C(2fc743551c71634e), C(5a39844593357e83), C(5ae234ee018ecf0b), C(4b0b4fda75dafd7a), C(1dbc42dfe207f246), C(5a0d608f0148d695), C(3fea6c290b1dd217),
C(4b0b4fda75dafd7a), C(1dbc42dfe207f246), C(5a0d608f0148d695), C(3fea6c290b1dd217),
C(ca0169256d064f4c), C(b9c328a139717836), C(ebb88619d12aa4d1), C(716e494d69ab7212)},
{C(9bf4d77b464c9435), C(3b3e99dd8dbe902d), C(bad6a2d23cb69f3f), C(694a09a072fb55d0), C(554f4bef9ec86462), C(62b67efddc71a65e), C(742a639df5c8e97f),
C(694a09a072fb55d0), C(554f4bef9ec86462), C(62b67efddc71a65e), C(742a639df5c8e97f),
C(9f41e5f1462fce48), C(5e4092a0d77e33f4), C(9fa5b35f31ca4646), C(4dcb137b92826a76)},
{C(5e6b758083214c84), C(cb50a5ebbc51fe29), C(4a8d4e03e37c386a), C(c39c9c99c57f6ca5), C(b42e5c0f6facbc5d), C(b61e56112182b0bd), C(1e47e5eae9d39cc1),
C(c39c9c99c57f6ca5), C(b42e5c0f6facbc5d), C(b61e56112182b0bd), C(1e47e5eae9d39cc1),
C(a35ea8c79805b3ca), C(ddd09523ba1c1f2e), C(50a159b97c3e0a72), C(974d196fcc65005)},
{C(40548138ef68aa78), C(5ba2ddd6b6414839), C(9b08bb741da55929), C(d1a6d32fe1544ce7), C(2116dc3d7c295fd2), C(da9bf65c101dfb2b), C(28e657eb1e7ec91b),
C(d1a6d32fe1544ce7), C(2116dc3d7c295fd2), C(da9bf65c101dfb2b), C(28e657eb1e7ec91b),
C(63bec2cc093cfa03), C(e326d7714c20dbd3), C(93577ee4e096f514), C(ef60f32e2047f758)},
{C(7c6b73ef50249070), C(253fa259c87d5773), C(6d3fd907de23f5ee), C(48293bc212b01988), C(bc97363f6593b1c6), C(eb58e37ccf58a370), C(43958d7b7ea6e3ea),
C(48293bc212b01988), C(bc97363f6593b1c6), C(eb58e37ccf58a370), C(43958d7b7ea6e3ea),
C(cce5c8ab68ad833c), C(37e404cdd4574032), C(908a6cdbc893f819), C(cfd2036019d4e772)},
{C(462a1dc5b9cb1b3b), C(7904069a5fa90c8b), C(6b34023aa308a14d), C(30ca8d4223d71e5c), C(265806e3aa04ad6e), C(99003da804259b22), C(852d586b40d5fdc9),
C(30ca8d4223d71e5c), C(265806e3aa04ad6e), C(99003da804259b22), C(852d586b40d5fdc9),
C(5739025f2fd35c6d), C(da6315eb3c055536), C(8b78cc73b4ef5c2b), C(d8ee48a1f3a0504d)},
{C(b8b156aa6c884b21), C(6320356e9c8220eb), C(dab9fa981067044f), C(d87813c08592d18c), C(b382d881f2c03851), C(984c34fa745f8617), C(e83b77ea9b8cb55),
C(d87813c08592d18c), C(b382d881f2c03851), C(984c34fa745f8617), C(e83b77ea9b8cb55),
C(dfbbd862f43a7ce2), C(60cf024675a3f981), C(90489388325b3895), C(6ee812876a0d8280)},
{C(c7afcc722488f9e6), C(ad6f0985312a64a6), C(97c923f4604fdcf4), C(9ea08f9e3a9dcd7a), C(c43489710d913809), C(d93c98c6519cbb12), C(440d5b0518ebbba7),
C(9ea08f9e3a9dcd7a), C(c43489710d913809), C(d93c98c6519cbb12), C(440d5b0518ebbba7),
C(8b251e1446fe95e8), C(779411be7eeb235b), C(3a6c55bf8697a6c9), C(37c650eb87619094)},
{C(7a45b5b10dc24dbc), C(954a638cab780f0a), C(7f807ade405b7144), C(a06a1ff81f995ca6), C(3f86a498bc53f3fb), C(5754b5a8eaa5f9ae), C(c9525aa3857aeac2),
C(a06a1ff81f995ca6), C(3f86a498bc53f3fb), C(5754b5a8eaa5f9ae), C(c9525aa3857aeac2),
C(282a71910cce4c), C(4cb5187f1bb213a9), C(88a0b48d4d6cd82), C(7549a0d22628ef1f)},
{C(efe499d7a567391d), C(cc2e34ce91112f84), C(db5b75c40a8d6871), C(9baddec72034af7c), C(76093c1151a9c334), C(87e40485b73a54da), C(60016735dade9c79),
C(9baddec72034af7c), C(76093c1151a9c334), C(87e40485b73a54da), C(60016735dade9c79),
C(7f6898c8b8f0cf75), C(4e3ded8b54a231a2), C(bffa17838d1e8a93), C(f22673526561be89)},
{C(b60d26b461d05e25), C(85e44ec2d4c15b5f), C(a33941259a71f695), C(db65a2f99f10b462), C(3a8a521b95aa4061), C(926cb95f25d21bf0), C(bd55cebbdeb0995b),
C(db65a2f99f10b462), C(3a8a521b95aa4061), C(926cb95f25d21bf0), C(bd55cebbdeb0995b),
C(5be4f11dbd12b512), C(97069199e30381b), C(919c497f89819090), C(d401c56351009d73)},
{C(c15d366b98d92986), C(55bf21d6d14af29c), C(5dc43a61cfcdab12), C(a0f0e3426fd64509), C(c83035f88a0b77ef), C(2763e28e263f5dff), C(c3ec793cc63dd03b),
C(a0f0e3426fd64509), C(c83035f88a0b77ef), C(2763e28e263f5dff), C(c3ec793cc63dd03b),
C(1e229cb421996b55), C(b68c01cd16195350), C(23dea84286d682f), C(6d5b29ae0007cb17)},
{C(9addb551a523df05), C(e60921a3a4aceffd), C(80019fdf97a7e7af), C(45b80999148290a8), C(a28692320b415d8e), C(2bb4caa23d8fc335), C(886ba29327b7d888),
C(45b80999148290a8), C(a28692320b415d8e), C(2bb4caa23d8fc335), C(886ba29327b7d888),
C(ca789b84ded5ebcb), C(e118c6fe9d46d9b6), C(816c55b87f81240b), C(46c88dee2eae333d)},
{C(bd0a37a2ad2465b9), C(27f0829d19cd8cf2), C(f13fdd1a783dea5b), C(d865088b9158ab0d), C(8dce24a60fd399d), C(3c1c449046cf6093), C(c868ae36645aa748),
C(d865088b9158ab0d), C(8dce24a60fd399d), C(3c1c449046cf6093), C(c868ae36645aa748),
C(81631dfae2756bd), C(91d52f25379c3f39), C(9a5ba7a5d5351300), C(96afee16d063cc7c)},
{C(e7a7162d930c5056), C(cca8d5eac07d1880), C(98d7152acf6aead9), C(d120fb01533ea3db), C(87ba57b866606483), C(d1ab19f981be7f77), C(575d2739539d89da),
C(d120fb01533ea3db), C(87ba57b866606483), C(d1ab19f981be7f77), C(575d2739539d89da),
C(df85c5fc71528b29), C(b17b7b6dde134721), C(1529708be58423bb), C(de15779ba404dd6e)},
{C(b9982c5395b09406), C(ae9457448497d448), C(ccb79281b0518ea2), C(750ad5690f9e2ea4), C(480ea42b71c98703), C(183a69ca58d0194b), C(38d9e1c3079391c3),
C(750ad5690f9e2ea4), C(480ea42b71c98703), C(183a69ca58d0194b), C(38d9e1c3079391c3),
C(a2c6a015e19f2784), C(198f72af6da4e94d), C(25f60149191ea280), C(76c3d95e77e15)},
{C(e41766d004eef8fd), C(aadcffc4079be65), C(1762a40c971c5256), C(6c0dca72f7dfd702), C(b72cd78f733b3838), C(98fd59dc49501bb7), C(d64b0dfa56080086),
C(6c0dca72f7dfd702), C(b72cd78f733b3838), C(98fd59dc49501bb7), C(d64b0dfa56080086),
C(788cefe1ddd43774), C(f04b73f0360928f3), C(abd47c7a8b98008), C(e84366d094a829bc)},
{C(a3074a96c88c47de), C(9dbfa7ee06c6d629), C(7a3010c27a54bf5b), C(1a646f82ee865034), C(1cd75ba2ba2e18db), C(7886c70238c13919), C(a375790fbc0d4075),
C(1a646f82ee865034), C(1cd75ba2ba2e18db), C(7886c70238c13919), C(a375790fbc0d4075),
C(d6110e4f72210885), C(5f785bc6e1be1073), C(70e597dcc6bd0f2a), C(cadd313cdd0895b8)},
{C(881caa3913271394), C(914f7f9497f7ba94), C(76ac64366b6f5660), C(61f8efb455b1df84), C(1c7d6d187eaa165b), C(f4037a9c1b15a760), C(af6952534fbb79b4),
C(61f8efb455b1df84), C(1c7d6d187eaa165b), C(f4037a9c1b15a760), C(af6952534fbb79b4),
C(9fd0df72e32d0a3b), C(bf93abf0ffe97d1d), C(9a388a58d6df3651), C(41a071e47a61f284)},
{C(77d95a600f824230), C(e5a39f40ea41304f), C(7547f8e9b3d7d3bc), C(2ae7e1bb1086a0c9), C(9db29c86d65743b9), C(3499bfb7b01c552), C(44bc45426b974a12),
C(2ae7e1bb1086a0c9), C(9db29c86d65743b9), C(3499bfb7b01c552), C(44bc45426b974a12),
C(15a9572e8fb11973), C(b4a9867437743c91), C(2d8887964ba315c2), C(2e4c0e78edd6a336)},
{C(1984adb7bcfec495), C(b07d8992ffb8817e), C(c4d5f5af08537d31), C(ffb697b8cfa03374), C(416e215f645d8cb), C(f7a9783188157a80), C(fe0671c77addf1fa),
C(ffb697b8cfa03374), C(416e215f645d8cb), C(f7a9783188157a80), C(fe0671c77addf1fa),
C(f21b9a779ef9f2e2), C(6cf329bcb0e8d867), C(f962e0d8a11a57f2), C(293058a121eb76d6)},
{C(66f613698d2263a7), C(2d5e27113b032421), C(50723eb3c45bba59), C(19fa48781ce2b326), C(f34fc200e9ca457c), C(aa074b3b25a4f79), C(b030f76a048d3c4e),
C(19fa48781ce2b326), C(f34fc200e9ca457c), C(aa074b3b25a4f79), C(b030f76a048d3c4e),
C(a8dfdad7b8d5c09e), C(c33ea522c687fd0a), C(6bf90444ddd8c94), C(79c65a303f46e176)},
{C(50cf2a1c284f5a5a), C(2caca2361a3ba2e0), C(3762f19bdf869c75), C(fc34a738dc6b2cf), C(1a72ebef9f3084a1), C(e80baa530c593464), C(745f6c0e1f21e62f),
C(abecd6c216d1805c), C(67e7a4ccc0e1b612), C(105003e6c4c24aca), C(e6d31a2f74a24541),
C(813b21623b28adb3), C(83ef2a6a2e8a7c22), C(abecd6c216d1805c), C(67e7a4ccc0e1b612)},
};

void Check(uint64 expected, uint64 actual) {
  if (expected != actual) {
    cerr << "ERROR: expected 0x" << hex << expected << ", but got 0x" << actual << "\n";
    ++errors;
  }
}

void Test(const uint64* expected, int offset, int len) {
  const uint128 u = CityHash128(data + offset, len);
  const uint128 v = CityHash128WithSeed(data + offset, len, kSeed128);
  Check(expected[0], CityHash64(data + offset, len));
  Check(expected[1], CityHash64WithSeed(data + offset, len, kSeed0));
  Check(expected[2], CityHash64WithSeeds(data + offset, len, kSeed0, kSeed1));
  Check(expected[3], Uint128Low64(u));
  Check(expected[4], Uint128High64(u));
  Check(expected[5], Uint128Low64(v));
  Check(expected[6], Uint128High64(v));
#ifdef __SSE4_2__
  const uint128 y = CityHashCrc128(data + offset, len);
  const uint128 z = CityHashCrc128WithSeed(data + offset, len, kSeed128);
  uint64 crc256_results[4];
  CityHashCrc256(data + offset, len, crc256_results);
  Check(expected[7], Uint128Low64(y));
  Check(expected[8], Uint128High64(y));
  Check(expected[9], Uint128Low64(z));
  Check(expected[10], Uint128High64(z));
  for (int i = 0; i < 4; i++) {
    Check(expected[11 + i], crc256_results[i]);
  }
#endif
}

#else

#define Test(a, b, c) Dump((b), (c))
void Dump(int offset, int len) {
  const uint128 u = CityHash128(data + offset, len);
  const uint128 v = CityHash128WithSeed(data + offset, len, kSeed128);
  const uint128 y = CityHashCrc128(data + offset, len);
  const uint128 z = CityHashCrc128WithSeed(data + offset, len, kSeed128);
  uint64 crc256_results[4];
  CityHashCrc256(data + offset, len, crc256_results);
  cout << hex
       << "{C(" << CityHash64(data + offset, len) << "), "
       << "C(" << CityHash64WithSeed(data + offset, len, kSeed0) << "), "
       << "C(" << CityHash64WithSeeds(data + offset, len, kSeed0, kSeed1) << "), "
       << "C(" << Uint128Low64(u) << "), "
       << "C(" << Uint128High64(u) << "), "
       << "C(" << Uint128Low64(v) << "), "
       << "C(" << Uint128High64(v) << "),\n"
       << "C(" << Uint128Low64(y) << "), "
       << "C(" << Uint128High64(y) << "), "
       << "C(" << Uint128Low64(z) << "), "
       << "C(" << Uint128High64(z) << "),\n";
  for (int i = 0; i < 4; i++) {
    cout << hex << "C(" << crc256_results[i] << (i == 3 ? ")},\n" : "), ");
  }
}

#endif

int main(int argc, char** argv) {
  setup();
  int i = 0;
  for ( ; i < kTestSize - 1; i++) {
    Test(testdata[i], i * i, i);
  }
  Test(testdata[i], 0, kDataSize);
  return errors > 0;
}
