;;; sysdep.lisp -- system-dependent parts of pg-dot-lisp
;;;
;;; Author: Eric Marsden <eric.marsden@free.fr>
;;; Time-stamp: <2005-07-17 emarsden>
;;
;;

(in-package :postgresql)

(eval-when (:compile-toplevel :load-toplevel :execute)
  #+allegro (require :socket)
  #+lispworks (require "comm")
  #+cormanlisp (require :sockets)
  #+sbcl (require :sb-bsd-sockets)
  #+(and mcl (not openmcl)) (require "OPENTRANSPORT"))


(defmacro %sysdep (desc &rest forms)
  (when (null forms)
    (error "No system dependent code to ~A" desc))
  (car forms))


#+(and cmu glibc2)
(eval-when (:compile-toplevel :load-toplevel)
  (format t ";; Loading libcrypt~%")
  ;; (ext:load-foreign "/lib/libcrypt.so.1")
  (sys::load-object-file "/usr/lib/libcrypt.so"))

#+(and cmu glibc2)
(defun crypt (key salt)
  (declare (type string key salt))
  (alien:alien-funcall
   (alien:extern-alien "crypt"
          (function c-call:c-string c-call:c-string c-call:c-string))
   key salt))

#-(and cmu glibc2)
(defun crypt (key salt)
  (declare (ignore salt))
  key)


(defun md5-digest (string &rest strings)
  (declare (type simple-string string))
  (let ((vec (md5sum-sequence
              (map '(vector (unsigned-byte 8)) #'char-code 
                   (apply #'concatenate 'string string strings)))))
    (format nil "~(~{~2,'0X~}~)" (coerce vec 'list))))

(defun md5-encode-password (user password salt)
  (concatenate 'string "md5"
               (md5-digest (md5-digest password user) salt)))



;; this is a little fiddly, because CLISP can be built without support
;; for the Linux package
;; #+CLISP
;; (defun crypt (key salt)
;;   (linux::crypt key salt))


;; bug in WRITE-SEQUENCE in CMUCL
#+(or cmu18c cmu18d)
(defun write-sequence (seq stream &key start end)
  (declare (ignore start end))
  (loop :for element :across seq
        :do (write-byte element stream)))



;; work around bug in FASL fop dumping 
#+cmu (setf c::top-level-lambda-max 0)


#+(and cmu ssl)
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
      (let ((fd (ext:connect-to-inet-socket host port)))
        (ssl:make-ssl-client-stream fd))
    (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))


#+cmu
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
   (let ((fd (if host
                 (ext:connect-to-inet-socket host port)
                 (ext:connect-to-unix-socket
                  (format nil "/var/run/postgresql/.s.PGSQL.~D" port)))))
     (sys:make-fd-stream fd :input t :output t
                         :element-type '(unsigned-byte 8)))
   (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))

;; this doesn't currently work, because WRITE-SEQUENCE is not
;; implemented
#+(and cmu simple-streams broken)
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
      (make-instance 'stream:socket-simple-stream
                     :remote-host host
                     :remote-port port
                     :direction :io)
    (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))

#+clisp
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
   (#+lisp=cl socket:socket-connect
    #-lisp=cl lisp:socket-connect
    port host :element-type '(unsigned-byte 8))
   (error (e)
      (declare (ignore e))
      (error 'connection-failure :host host :port port))))


#+(and db-sockets broken)
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
   (let ((s (sockets:make-inet-socket :stream :tcp))
         (num (car (sockets:host-ent-addresses
                    (sockets:get-host-by-name host)))))
     (sockets:socket-connect s num port)
     (sockets:socket-make-stream s :element-type '(unsigned-byte 8)
                                 :input t :output t :buffering :none))
   (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))

#+sbcl
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
      (sb-bsd-sockets:socket-make-stream
       (if host
	   (let ((s (make-instance 'sb-bsd-sockets:inet-socket
                                   :type :stream :protocol :tcp))
		 (num (car (sb-bsd-sockets:host-ent-addresses
			    (sb-bsd-sockets:get-host-by-name host)))))
	     (sb-bsd-sockets:socket-connect s num port)
             s)
	   (let ((s (make-instance 'sb-bsd-sockets:local-socket :type :stream)))
	     (sb-bsd-sockets:socket-connect
	      s (format nil "/var/run/postgresql/.s.PGSQL.~D" port))
	     s))
       :element-type '(unsigned-byte 8)
       :input t
       :output t
       :buffering :none)
    (error (e)
      (error 'connection-failure :host host :port port :transport-error e))))

#+allegro
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
   (socket:make-socket :remote-host host
                       :remote-port port
                       :format :binary)
   (error (e)
      (error 'connection-failure
              :host host
              :port port
              :transport-error e))))

;; Lispworks 4.2 doesn't seem to implement WRITE-SEQUENCE on binary
;; streams. Fixed in version 4.3. 
#+lispworks
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
      (comm:open-tcp-stream host port
                            :element-type '(unsigned-byte 8)
                            :direction :io)
    (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))

;; this doesn't work, since the Corman sockets module doesn't support
;; binary I/O on socket streams.
#+cormanlisp
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
      (progn
        (sockets:start-sockets)
        (let ((sock (sockets:make-client-socket :host host :port port)))
          (sockets:make-socket-stream sock)))
    (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))

#+openmcl
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case
      (if host
          (make-socket :address-family :internet
                       :type :stream
                       :connect :active
                       :format :binary
                       :remote-host host
                       :remote-port port)
          (make-socket :address-family :file
                       :type :stream
                       :connect :active
                       :format :binary
                       :remote-filename (format nil "/var/run/postgresql/.s.PGSQL.~D" port)))
    (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))

;; from John DeSoi
#+(and mcl (not openmcl))
(defun socket-connect (port host)
  (declare (type integer port))
  (ccl::open-tcp-stream host port :element-type '(unsigned-byte 8)))

;; There is a bug in MCL (4.3.1 tested) where read-sequence and
;; write-sequence fail with binary tcp streams. These two methods
;; provide a work-around.
#+(and mcl (not openmcl))
(defmethod ccl:stream-write-sequence ((s ccl::opentransport-binary-tcp-stream)
                                      (sequence ccl::simple-unsigned-byte-vector)
                                      &key (start 0) end)
  (ccl::stream-write-vector s sequence start (or end (length sequence)))
  s)

#+(and mcl (not openmcl))
(defmethod ccl:stream-read-sequence ((s ccl::opentransport-binary-tcp-stream)
                                     (sequence ccl::simple-unsigned-byte-vector)
                                     &key (start 0) (end (length sequence)))
  (ccl::io-buffer-read-bytes-to-vector (ccl::stream-io-buffer s)
                                       sequence (- end start) start)
  end)


#+ecl
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case 
      (si:open-client-stream host port)
    (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))



;; as of version 2.6 GCL is way too broken to run this: DEFPACKAGE doesn't
;; work, DEFINE-CONDITION not implemented, ...
#+gcl
(defun socket-connect (port host)
  (declare (type integer port))
  (si::socket port :host host))



#+armedbear
(eval-when (:load-toplevel :execute :compile-toplevel)
  (require 'format))

;; MAKE-SOCKET with :element-type as per 2004-03-09
#+armedbear
(defun socket-connect (port host)
  (declare (type integer port))
  (handler-case 
      (ext:make-socket host port :element-type '(unsigned-byte 8))
    (error (e)
      (error 'connection-failure
             :host host
             :port port
             :transport-error e))))



;;; character encoding support

(defvar *pg-client-encoding*)

#+(and :sbcl :sb-unicode)
(defun sbcl-ext-form-from-client-encoding (encoding)
  (cond
   ((string= encoding "SQL_ASCII") :ascii)
   ((string= encoding  "LATIN1") :latin1)
   ((string= encoding "LATIN9") :latin9)
   ((string= encoding "UNICODE") :utf8)
   (t (error "unkown encoding ~A" encoding))))
  
(defun convert-string-to-bytes (string &optional (encoding *pg-client-encoding*))
  (declare (type string string))
  (%sysdep "convert string to bytes"
     #+(and clisp unicode)
     (ext:convert-string-to-bytes string encoding)
     #+(and allegro ics)
     (excl:string-to-octets string :null-terminate nil
			    :external-format encoding)
     #+(and :sbcl :sb-unicode)
     (sb-ext:string-to-octets string :external-format (sbcl-ext-form-from-client-encoding encoding))
     #+(or cmu sbcl gcl ecl)
     (let ((octets (make-array (length string) :element-type '(unsigned-byte 8))))
       (map-into octets #'char-code string))))

(defun convert-string-from-bytes (bytes &optional (encoding *pg-client-encoding*))
  (declare (type (vector (unsigned-byte 8)) bytes))
  (%sysdep "convert octet-array to string"
    #+(and clisp unicode)
    (ext:convert-string-from-bytes bytes encoding)
    #+(and allegro ics)
    (excl:octets-to-string bytes :external-format encoding)
    #+(and :sbcl :sb-unicode)
    (sb-ext:octets-to-string bytes :external-format encoding)
    ;; for implementations that have no support for character
    ;; encoding, we assume that the encoding is an octet-for-octet
    ;; encoding, and convert directly
    #+(or cmu (and sbcl (not :sb-unicode)) gcl ecl)
    (let ((string (make-string (length bytes))))
      (map-into string #'code-char bytes))))


;; EOF
