#!/usr/bin/env python

########################################################################
# Chris's Lame Filebrowser 4 (Mimetype Handler)
# Copyright 2004, Gabe Ginorio <gabe@zevallos.com.br>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################################################

import commands, mimetypes, os, os.path, re, shutil, string, filecmp, sys, db
import fstabreader, create, mimeeditor

MIME_PATH = os.path.abspath(string.replace(sys.argv[0], "claw4.py", "") + "/Mimetypes")

#####################################################	
# Configure, update and set mime types
#####################################################		
def initialize():

	types = ["application","audio","image","text","video","types"]

	# For any mimetype files that don't exist in .claw/, copy them from the program mimetypes directory	
	for a_type in types:
	 	if not os.path.exists(os.environ['HOME'] + "/.claw/mime." + a_type):
			shutil.copy(MIME_PATH +  "/mime." + a_type, os.environ['HOME'] + "/.claw/")

	# Update any installed mime files
	__updateMimetypes__()

	# Now add the file types in mime.types to the python path
	mimetypes.init([os.environ['HOME'] + "/.claw/mime.types"])
	#mimetypes.knownfiles[0:0] = [os.environ['HOME'] + "/.claw/mime.types"]

	# Lastly initilize the database
	db.initialize("Claw4")


########################################################################
# This function updates any installed mime files in ~/.claw
########################################################################		
def __updateMimetypes__():

	# Create a list of the mime files we expect to find
	mimes = ["application","audio","image","text", "video"]

	# Open each file in sequence
	for mime in mimes:

		if not filecmp.cmp(MIME_PATH + "/mime." + mime, os.environ['HOME'] + "/.claw/mime." + mime):

			# Read the default and user mime files
			default_mime_file = file(MIME_PATH + "/mime." + mime, 'r').readlines()
			user_mime_file = file(os.environ['HOME'] + "/.claw/mime." + mime, 'r').readlines()

			# Make a dictionary of the mimetypes and the associated apps 
			default = {}
			for a_line in default_mime_file:
				if len(string.strip(a_line)) > 1 and not a_line.startswith("#"):
					split_line = a_line.split("::") 
					extention = string.strip(split_line[0])
					applications = split_line[1].split(",")
					default[extention] = []
					for app in applications:
						default[extention].append(app.strip())
					
			user = {}
			for a_line in user_mime_file:
				if len(string.strip(a_line)) > 1:
					split_line = a_line.split("::") 
					extention = string.strip(split_line[0])
					applications = split_line[1].split(",")
					user[extention] = []
					for app in applications:
						user[extention].append(app.strip())

			# Check for lines to write to the file
			mime_file = file(os.environ['HOME'] + "/.claw/mime." + mime, 'w')
			for d_type in default:
				# First, check for missing lines
				if d_type not in user:
					apps = default[d_type][0]
					for application in default[d_type][1:]:
						apps += "," + application
					mime_file.write(d_type + "\t\t::\t" + apps + "\n")
				else:
					# If the line exists, have apps been added
					for application in default[d_type]:
						if string.strip(application) not in user[d_type]:
							user[d_type].append(string.strip(application))
					
					apps = user[d_type][0]
					for application in user[d_type][1:]:
						apps += "," + application
					mime_file.write(d_type + "\t\t::\t" + apps + "\n")					

			mime_file.close()

#########################################	
# This function returns the mimetypes for files in a directory
########################################
def getMimeTypes(directory):
	return db.read(directory)


#########################################	
# This function finds the mime types of files at a location
#########################################
def getTypes(directory, single_file=None):
	'''The function returns a dictionary, indexed by location, that
	contains the mime type infotmaion i.e. (location : ['text', 'plain']).'''

	# This gets the mountpoints in  /etc/fstab
	mountpoints = fstabreader.read()

	# Associate a location with a mimetype (this is the new method)
	types = {}

	# Determine if this is a file or directory
	if single_file != None:
		locations = [single_file]
	else:
		locations = os.listdir(directory)

	for location in locations:
		location = os.path.abspath(location)

		# First sort out dead links
		if not os.path.exists(location):
			types[location] = ["dead_link","dead_link"]

		# Sort out the mounted partitions
		elif os.path.ismount(location):
			if location.startswith("cdrom"):
				group, mime = "mounted","cdrom_mounted"
			elif location.startswith("dvd"):
				group, mime =  "mounted","dvd_mounted"
			else:
				group, mime =  "mounted","harddrive_mounted"		
			
			types[location] = [group, mime]
	
		# Sort out the unmounted partitions
		elif location in mountpoints:
			if re.findall("cdrom",	location):
				group, mime = "unmounted","cdrom_unmounted"
			elif re.findall("dvd", location):
				group, mime = "unmounted","dvd_unmounted"
			else:
				group, mime = "unmounted","harddrive_unmounted"	

			types[location] = [group, mime]

		# Sort out the folders
		elif os.path.isdir(location):
			types[location] = ["folder", "folder"]

		# Get the mime type of anything remaining
		else:

			# Use the built in python to get the mime type
			mime = mimetypes.guess_type(location)[0]
			if mime != None and mime != '' and mime != "text/plain":
				if mime == 'audio/mpeg':
					mime='audio/mp3'
				if mime == "video/disk_image" or mime == "application/octet-stream":
					second_mime = commands.getoutput("file -ibL " + __returnClean__(location))
					second_mime = re.split("[;,]",second_mime)[0]
					if second_mime == "application/x-shellscript":
						mime = second_mime
			else:
				# Use the 'file' command
				mime = commands.getoutput("file -ibL " + __returnClean__(location))
				mime = re.split("[;,]",mime)[0]

			# See if the file command returned anything useful
			try:	
				group, mime = string.split(mime, "/")
			except:
				group, mime = "text","plain"

			# If the file command can't guess it, then it is unknown or we don't have permission
			if group not in ["application","audio","image","text","video"]:
				 group, mime = "text","plain"

			types[location] = [group, mime]

	return types


########################################################################
# This function calls the mimetype editor (for associating applications with filetypes)
########################################################################		
def editor(location, group, extention, image_path):

	# Use the mimeeditor module for sake of readability
	mimeeditor.editor(location, group, extention, image_path)
	

#####################################################
# This function gets the applications associated with an icon
#####################################################	
def get_associated_applications(mime):

	# Get the group and filetype from the mime information
	group, extention = mime[0], mime[1]

	# Open the appropriate mime-file
	mimefile = file(os.environ['HOME'] + "/.claw/mime." + group).readlines()
	
	# Get the appropriate line
	line = filter(lambda line: re.findall("^" + extention, string.strip(string.split(line, "::")[0])), mimefile)
	
	# Check to see if there was a match
	if line != []:
		# Get the apps fron the line (one big mess, since there are tabs and spaces to deal with)
		applications = string.strip(string.split(line[0], "::")[1])
		applications = string.split(applications, ",")
		applications = map(string.strip, applications)
			
	else:
		# If there wasn't a match, then send back nothing and let the user know
		applications = []
			
		# This is mainly for my own use, so I can add mimetypes
		print "-------------------------------------"
		print mime, "is not is the mime files"
		print "-------------------------------------"	

	# Then we need to see if the apps actually exists
	applications = filter(lambda app: commands.getstatusoutput("which " + string.split(app)[0])[0] == 0, applications)

	# Return the appropriate information
	return applications	


########################################################################
# This function returns BASH friendly text
########################################################################		
def __returnClean__(text):
	
	# Here are the bad characters
	badChars = [" ","\'","&","(",")"]
		
	# Here are the replacement characters
	goodChars=["\ ","\\'","\&","\(","\)"]
		
	# Replace the bad with the good
	for x in range(len(badChars)):
		text = string.replace(text, badChars[x], goodChars[x])		
		
	# Return the "cleaned", BASH friendly text
	return text
		
		
		
