/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */
 
using Gtk;
 
namespace Clinica {

    public class SettingsManager : Gtk.Window {
    
        /**
         * @brief Settings to set and read from.
         */
        private Settings settings;
        
        private ResourceManager resource_manager;
        
        public SettingsManager (ResourceManager resources) {
            /* Init the GtkWindow object */
            GLib.Object (type: Gtk.WindowType.TOPLEVEL);
            resource_manager = resources;
            
            /* Load the settings of clinica */
            this.settings = resources.settings;
            
            /* Build up the interface */
            build_ui ();
            set_title (_("Clinica settings"));
        }
        
        /**
         * @brief This functions take care of constructing
         * all the pieces of the interface.
         */
        private void build_ui () {
            var structure_vbox = new VBox (false, 6);
            var notebook = new Notebook ();
            
            /* Create the use-plugins checkbox */
            var plugins_switch = new SettingsSwitch (_("Use plugins (need restart)"));
            settings.bind ("use-plugins", 
                plugins_switch.switcher,
                "active",
                SettingsBindFlags.DEFAULT);
            structure_vbox.pack_start (plugins_switch, false);
            
            /* Create the selector for the search engine */
            var mse_selector = new MedicineSearchEngineSelector (resource_manager);
            structure_vbox.pack_start (mse_selector, false);
            
            /* Add structure vbox to the first page of the Notebook */
            var sv_alignment = new Alignment (0.5F, 0.5F, 1.0F, 1.0F);
            sv_alignment.set_padding (6,6,6,6);
            sv_alignment.add (structure_vbox);
            notebook.append_page (sv_alignment, new Label (_("General")));
            
            /* Plugins window in the Notebook */
            notebook.append_page (new PluginManager (resource_manager), 
                new Label (_("Plugins")));
            
            /* Ask for a window sufficiently large */
            set_size_request (240, 380  );
            
            /* Create an alignment to set the right padding */
            var alignment = new Alignment (0.5F, 0.5F, 1.0F, 1.0F);
            alignment.set_padding (6, 6, 6, 6);
            alignment.add (notebook);
            
            add (alignment);
        }
    }
    
    public class SettingsSwitch : HBox {
    
        public Switch switcher;
        
        public SettingsSwitch (string label) {
            GLib.Object (homogeneous: false, spacing: 6);
            
            switcher = new Switch ();
            var my_label = new Label (label);
            my_label.set_alignment (0.0F, 0.5F);
            
            pack_start (my_label,  true,  true);
            pack_start (switcher, false, true);
        }
    }
    
    public class MedicineSearchEngineSelector : HBox {
        
        private ResourceManager resource_manager { get; set; }
        private ComboBoxText    medicine_se_selector;
        
        public MedicineSearchEngineSelector (ResourceManager resources) {
            GLib.Object (homogeneous: false, spacing: 6);
            resource_manager = resources;
            var label = new Label (_("Select the search engine\nused to find medicines"));
            label.set_alignment (0.0F, 0.5F);
            pack_start (label);
            
            /* Create the combobox and keep a track of its position in the array. */
            medicine_se_selector = new ComboBoxText ();
            refresh_engines_list ();
            resource_manager.medicine_search_engines_added.connect ((engine) => refresh_engines_list ());
            resource_manager.medicine_search_engines_removed.connect ((engine) => refresh_engines_list ());
            pack_start (medicine_se_selector, false, true);
            medicine_se_selector.changed.connect (on_medicine_se_selector_changed);
        }
        
        private void on_medicine_se_selector_changed () {
            string? active_id = medicine_se_selector.get_active_id ();
            if (active_id == null)
                return;
            var index = int.parse (active_id);
            resource_manager.settings.selected_search_engine = 
                resource_manager.medicine_search_engines.nth_data (index);
            resource_manager.settings.set_string ("medicine-search-engine", 
                resource_manager.settings.selected_search_engine.get_name ());
        }
        
        private void refresh_engines_list () {
            int i = 0;
            medicine_se_selector.remove_all ();
            
            /* If no preference is set set the first engine as selected */
            if (resource_manager.settings.get_string ("medicine-search-engine") == "") {
                if (resource_manager.medicine_search_engines.length () != 0) {
                    resource_manager.settings.selected_search_engine = 
                        resource_manager.medicine_search_engines.nth_data (0);
                    resource_manager.settings.set_string ("medicine-search-engine", 
                        resource_manager.settings.selected_search_engine.get_name ());
                }
            }
            
            /* Load engines in the liststore */
            foreach (var engine in resource_manager.medicine_search_engines) {
                medicine_se_selector.append ("%d".printf (i++), engine.get_name ());
                if (resource_manager.settings.get_string ("medicine-search-engine") == engine.get_name ()) {
                    debug ("Loading search engine: %s", engine.get_name ());
                    resource_manager.settings.selected_search_engine = engine;
                }
                if (engine == resource_manager.settings.selected_search_engine) {
                    medicine_se_selector.set_active (i - 1);
                }
            }
        }
    }
}
