/*
 * ClutterCairo.
 *
 * An simple Clutter Cairo 'Drawable'.
 *
 * Authored By Matthew Allum  <mallum@openedhand.com>
 *
 * Copyright (C) 2006 OpenedHand
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
 * SECTION:clutter-cairo
 * @short_description: Actor for displaying 
 *
 * #ClutterCairo is a #ClutterTexture that displays text.
 */
#include "clutter-cairo.h"
#include <string.h>

G_DEFINE_TYPE (ClutterCairo, clutter_cairo, CLUTTER_TYPE_TEXTURE);

enum
{
  PROP_0,
  PROP_SURFACE_WIDTH,
  PROP_SURFACE_HEIGHT,
};

#define CLUTTER_CAIRO_GET_PRIVATE(obj) \
(G_TYPE_INSTANCE_GET_PRIVATE ((obj), CLUTTER_TYPE_CAIRO, ClutterCairoPrivate))

struct _ClutterCairoPrivate
{
  cairo_surface_t *cr_surface;
  guchar          *cr_surface_data;
  gint             width, height;

};

static const cairo_user_data_key_t clutter_cairo_surface_key;
static const cairo_user_data_key_t clutter_cairo_context_key;

static void
clutter_cairo_surface_destroy (void *data)
{
  ClutterCairo *cairo = data;

  cairo->priv->cr_surface = NULL;
}

static void
clutter_cairo_set_property (GObject      *object, 
			    guint         prop_id,
			    const GValue *value, 
			    GParamSpec   *pspec)
{
  ClutterCairo        *cairo;
  ClutterCairoPrivate *priv;

  cairo = CLUTTER_CAIRO(object);
  priv = cairo->priv;

  switch (prop_id) 
    {
    case PROP_SURFACE_WIDTH:
      priv->width = g_value_get_int (value);
      break;
    case PROP_SURFACE_HEIGHT:
      priv->height = g_value_get_int (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
clutter_cairo_get_property (GObject    *object, 
			    guint       prop_id,
			    GValue     *value, 
			    GParamSpec *pspec)
{
  ClutterCairo        *cairo;
  ClutterCairoPrivate *priv;

  cairo = CLUTTER_CAIRO(object);
  priv = cairo->priv;

  switch (prop_id) 
    {
    case PROP_SURFACE_WIDTH:
      g_value_set_int (value, priv->width);
      break;
    case PROP_SURFACE_HEIGHT:
      g_value_set_int (value, priv->height);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    } 
}

static void 
clutter_cairo_dispose (GObject *object)
{
  ClutterCairo         *self = CLUTTER_CAIRO(object);
  ClutterCairoPrivate  *priv;  

  priv = self->priv;

  if (priv->cr_surface)
    {
      cairo_surface_t *surface = priv->cr_surface;

      cairo_surface_finish (priv->cr_surface);
      cairo_surface_set_user_data (priv->cr_surface,
                                   &clutter_cairo_surface_key,
                                   NULL, NULL);
      cairo_surface_destroy (surface);

      priv->cr_surface = NULL;
    }

  if (priv->cr_surface_data)
    {
      g_slice_free1 (priv->width * priv->height * 4, priv->cr_surface_data);
      priv->cr_surface_data = NULL;
    }
  
  G_OBJECT_CLASS (clutter_cairo_parent_class)->dispose (object);
}

static void 
clutter_cairo_finalize (GObject *object)
{
  G_OBJECT_CLASS (clutter_cairo_parent_class)->finalize (object);
}


static GObject*
clutter_cairo_constructor (GType type, guint n_construct_properties,
			   GObjectConstructParam *construct_properties)
{
  ClutterCairo        *cairo;
  ClutterCairoPrivate *priv;
  GObject             *obj;
 
  obj = G_OBJECT_CLASS (clutter_cairo_parent_class)->constructor
    (type, n_construct_properties, construct_properties);

  /* Now all of the object properties are set */

  cairo = CLUTTER_CAIRO(obj);
  priv = CLUTTER_CAIRO_GET_PRIVATE (cairo);

  g_return_val_if_fail(priv->width && priv->height, NULL);

  priv->cr_surface_data = g_slice_alloc0 (priv->width * priv->height * 4);
  priv->cr_surface 
    = cairo_image_surface_create_for_data (priv->cr_surface_data,
					   CAIRO_FORMAT_ARGB32, 
					   priv->width,
					   priv->height, 
					   priv->width * 4);
  
  cairo_surface_set_user_data (priv->cr_surface, &clutter_cairo_surface_key,
                               cairo, clutter_cairo_surface_destroy);

  return obj;
}  

static void
clutter_cairo_class_init (ClutterCairoClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->finalize     = clutter_cairo_finalize;
  gobject_class->dispose      = clutter_cairo_dispose;
  gobject_class->set_property = clutter_cairo_set_property;
  gobject_class->get_property = clutter_cairo_get_property;
  gobject_class->constructor  = clutter_cairo_constructor;

  g_type_class_add_private (gobject_class, sizeof (ClutterCairoPrivate));

#define PARAM_FLAGS (G_PARAM_CONSTRUCT_ONLY |                   \
                     G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |\
                     G_PARAM_STATIC_BLURB |                     \
                     G_PARAM_READABLE | G_PARAM_WRITABLE)

  g_object_class_install_property (gobject_class,
                                   PROP_SURFACE_WIDTH,
                                   g_param_spec_int ("surface-width",
                                                     "Surface-Width",
                                                     "Surface Width",
                                                     0, G_MAXINT,
                                                     0,
                                                     PARAM_FLAGS));

  g_object_class_install_property (gobject_class,
                                   PROP_SURFACE_HEIGHT,
                                   g_param_spec_int ("surface-height",
                                                     "Surface-Height",
                                                     "Surface Height",
                                                     0, G_MAXINT,
                                                     0,
                                                     PARAM_FLAGS));
#undef PARAM_FLAGS
}

static void
clutter_cairo_init (ClutterCairo *self)
{
  ClutterCairoPrivate *priv;

  self->priv = priv = CLUTTER_CAIRO_GET_PRIVATE (self);
}

/**
 * clutter_cairo_new
 * @width: clutter cairo surface width
 * @height: clutter cairo surface height
 *
 * Creates a new #ClutterCairo texture.
 *
 * Return value: a #ClutterCairo texture
 */
ClutterActor*
clutter_cairo_new (guint width, guint height)
{
  return g_object_new (CLUTTER_TYPE_CAIRO, 
		       "surface-width", width,
		       "surface-height", height,
		       NULL);
}

static void
clutter_cairo_context_destroy (void *data)
{
  ClutterCairo        *cairo = CLUTTER_CAIRO (data);
  ClutterCairoPrivate *priv;
  GdkPixbuf           *pixbuf;

  int            cairo_width, cairo_height, cairo_rowstride;
  unsigned char *pixbuf_data, *dst, *cairo_data;
  int            pixbuf_rowstride, pixbuf_n_channels;
  unsigned int  *src;
  int            x, y;

  priv = CLUTTER_CAIRO_GET_PRIVATE (cairo);

  if (!priv->cr_surface)
    return;

  cairo_width     = cairo_image_surface_get_width (priv->cr_surface);
  cairo_height    = cairo_image_surface_get_height (priv->cr_surface);
  cairo_rowstride = cairo_width * 4;
  cairo_data      = priv->cr_surface_data;

  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, 
			   TRUE, 
			   8,
			   cairo_width,
			   cairo_height);

  pixbuf_data       = gdk_pixbuf_get_pixels (pixbuf);
  pixbuf_rowstride  = gdk_pixbuf_get_rowstride (pixbuf);
  pixbuf_n_channels = gdk_pixbuf_get_n_channels (pixbuf);

  /* BAH BAH BAH ! un-pre-multiply alpha... 
   * FIXME: Need to figure out if GL has a premult texture 
   *        format... or go back to battling glitz
  */
  for (y = 0; y < cairo_height; y++)
    {
      src = (unsigned int *) (cairo_data + y * cairo_rowstride);
      dst = pixbuf_data + y * pixbuf_rowstride;

      for (x = 0; x < cairo_width; x++) 
	{
	  guchar alpha = (*src >> 24) & 0xff;

	  /* FIXME: Handle big endian ? */
	  if (alpha == 0)
	    {
	      dst[0] = dst[1] = dst[2] = dst[3] = alpha;
	    }
	  else
	    {
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
	      dst[0] = (((*src >> 16) & 0xff) * 255 ) / alpha;
	      dst[1] = (((*src >> 8) & 0xff) * 255 ) / alpha; 
	      dst[2] = (((*src >> 0) & 0xff) * 255 ) / alpha;
	      dst[3] = alpha;
#elif G_BYTE_ORDER == G_BIG_ENDIAN
              dst[0] = alpha;
	      dst[1] = (((*src >> 0) & 0xff) * 255 ) / alpha;
	      dst[2] = (((*src >> 8) & 0xff) * 255 ) / alpha; 
	      dst[3] = (((*src >> 16) & 0xff) * 255 ) / alpha;
#else /* !G_LITTLE_ENDIAN && !G_BIG_ENDIAN */
#error unknown ENDIAN type
#endif /* !G_LITTLE_ENDIAN && !G_BIG_ENDIAN */
	    }
	  dst += pixbuf_n_channels;
	  src++;
	}
    }

  clutter_texture_set_pixbuf (CLUTTER_TEXTURE (cairo), pixbuf, NULL);
  
  g_object_unref (pixbuf); 
  
  if (CLUTTER_ACTOR_IS_VISIBLE (CLUTTER_ACTOR(cairo)))
    clutter_actor_queue_redraw (CLUTTER_ACTOR(cairo));
}

/**
 * clutter_cairo_create
 * @cairo:  A #ClutterCairo texture.
 *
 * Creates a new cairo context #ClutterCairo texture. 
  *
 * Return value: a newly created cairo context. Free with cairo_destroy() 
 * when you are done drawing.  
 */
cairo_t*
clutter_cairo_create (ClutterCairo *cairo)
{
  cairo_t *cr;

  cr = cairo_create (cairo->priv->cr_surface);

  cairo_set_user_data (cr, &clutter_cairo_context_key,
                       cairo, clutter_cairo_context_destroy);
  return cr;
}

/**
 * clutter_cairo_set_source_color:
 * @cr: a #cairo_t
 * @color: a #ClutterColor
 *
 * Sets @color as the source color for the cairo context.
 */
void
clutter_cairo_set_source_color (cairo_t            *cr,
                                const ClutterColor *color)
{
  g_return_if_fail (cr != NULL);
  g_return_if_fail (color != NULL);

  if (color->alpha == 0xff)
    cairo_set_source_rgb (cr,
                          color->red / 255,
                          color->green / 255,
                          color->blue / 255);
  else
    cairo_set_source_rgba (cr,
                           color->red / 255,
                           color->green / 255,
                           color->blue / 255,
                           color->alpha / 255);
}
