/*  Clzip - LZMA lossless data compressor
    Copyright (C) 2010, 2011, 2012, 2013, 2014 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

enum { rd_buffer_size = 16384 };

struct Range_decoder
  {
  unsigned long long partial_member_pos;
  uint8_t * buffer;		/* input buffer */
  int pos;			/* current pos in buffer */
  int stream_pos;		/* when reached, a new block must be read */
  uint32_t code;
  uint32_t range;
  int infd;			/* input file descriptor */
  bool at_stream_end;
  };

bool Rd_read_block( struct Range_decoder * const rdec );

static inline bool Rd_init( struct Range_decoder * const rdec, const int ifd )
  {
  rdec->partial_member_pos = 0;
  rdec->buffer = (uint8_t *)malloc( rd_buffer_size );
  if( !rdec->buffer ) return false;
  rdec->pos = 0;
  rdec->stream_pos = 0;
  rdec->code = 0;
  rdec->range = 0xFFFFFFFFU;
  rdec->infd = ifd;
  rdec->at_stream_end = false;
  return true;
  }

static inline void Rd_free( struct Range_decoder * const rdec )
  { free( rdec->buffer ); }

static inline bool Rd_finished( struct Range_decoder * const rdec )
  { return rdec->pos >= rdec->stream_pos && !Rd_read_block( rdec ); }

static inline unsigned long long
Rd_member_position( const struct Range_decoder * const rdec )
  { return rdec->partial_member_pos + rdec->pos; }

static inline void Rd_reset_member_position( struct Range_decoder * const rdec )
  { rdec->partial_member_pos = -rdec->pos; }

static inline uint8_t Rd_get_byte( struct Range_decoder * const rdec )
  {
  if( Rd_finished( rdec ) ) return 0xAA;	/* make code != 0 */
  return rdec->buffer[rdec->pos++];
  }

static inline int Rd_read_data( struct Range_decoder * const rdec,
                                uint8_t * const outbuf, const int size )
  {
  int rest = size;
  while( rest > 0 && !Rd_finished( rdec ) )
    {
    const int rd = min( rest, rdec->stream_pos - rdec->pos );
    memcpy( outbuf + size - rest, rdec->buffer + rdec->pos, rd );
    rdec->pos += rd;
    rest -= rd;
    }
  return size - rest;
  }

static inline void Rd_load( struct Range_decoder * const rdec )
  {
  int i;
  rdec->code = 0;
  for( i = 0; i < 5; ++i )
    rdec->code = (rdec->code << 8) | Rd_get_byte( rdec );
  rdec->range = 0xFFFFFFFFU;
  rdec->code &= rdec->range;	/* make sure that first byte is discarded */
  }

static inline void Rd_normalize( struct Range_decoder * const rdec )
  {
  if( rdec->range <= 0x00FFFFFFU )
    {
    rdec->range <<= 8;
    rdec->code = (rdec->code << 8) | Rd_get_byte( rdec );
    }
  }

static inline int Rd_decode( struct Range_decoder * const rdec,
                             const int num_bits )
  {
  int symbol = 0;
  int i;
  for( i = num_bits; i > 0; --i )
    {
    uint32_t mask;
    Rd_normalize( rdec );
    rdec->range >>= 1;
/*    symbol <<= 1; */
/*    if( rdec->code >= rdec->range ) { rdec->code -= rdec->range; symbol |= 1; } */
    mask = 0U - (rdec->code < rdec->range);
    rdec->code -= rdec->range;
    rdec->code += rdec->range & mask;
    symbol = (symbol << 1) + (mask + 1);
    }
  return symbol;
  }

static inline int Rd_decode_bit( struct Range_decoder * const rdec,
                                 Bit_model * const probability )
  {
  uint32_t bound;
  Rd_normalize( rdec );
  bound = ( rdec->range >> bit_model_total_bits ) * *probability;
  if( rdec->code < bound )
    {
    rdec->range = bound;
    *probability += (bit_model_total - *probability) >> bit_model_move_bits;
    return 0;
    }
  else
    {
    rdec->range -= bound;
    rdec->code -= bound;
    *probability -= *probability >> bit_model_move_bits;
    return 1;
    }
  }

static inline int Rd_decode_tree( struct Range_decoder * const rdec,
                                  Bit_model bm[], const int num_bits )
  {
  int symbol = 1;
  int i;
  for( i = num_bits; i > 0; --i )
    symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  return symbol - (1 << num_bits);
  }

static inline int Rd_decode_tree6( struct Range_decoder * const rdec,
                                   Bit_model bm[] )
  {
  int symbol = 1;
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  return symbol & 0x3F;
  }

static inline int Rd_decode_tree_reversed( struct Range_decoder * const rdec,
                                           Bit_model bm[], const int num_bits )
  {
  int model = 1;
  int symbol = 0;
  int i;
  for( i = 0; i < num_bits; ++i )
    {
    const bool bit = Rd_decode_bit( rdec, &bm[model] );
    model <<= 1;
    if( bit ) { ++model; symbol |= (1 << i); }
    }
  return symbol;
  }

static inline int Rd_decode_tree_reversed4( struct Range_decoder * const rdec,
                                            Bit_model bm[] )
  {
  int model = 1;
  int symbol = Rd_decode_bit( rdec, &bm[model] );
  int bit;
  model = (model << 1) + symbol;
  bit = Rd_decode_bit( rdec, &bm[model] );
  model = (model << 1) + bit; symbol |= (bit << 1);
  bit = Rd_decode_bit( rdec, &bm[model] );
  model = (model << 1) + bit; symbol |= (bit << 2);
  if( Rd_decode_bit( rdec, &bm[model] ) ) symbol |= 8;
  return symbol;
  }

static inline int Rd_decode_matched( struct Range_decoder * const rdec,
                                     Bit_model bm[], int match_byte )
  {
  Bit_model * const bm1 = bm + 0x100;
  int symbol = 1;
  while( symbol < 0x100 )
    {
    int match_bit, bit;
    match_byte <<= 1;
    match_bit = match_byte & 0x100;
    bit = Rd_decode_bit( rdec, &bm1[match_bit+symbol] );
    symbol = ( symbol << 1 ) | bit;
    if( match_bit != bit << 8 )
      {
      while( symbol < 0x100 )
        symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
      break;
      }
    }
  return symbol & 0xFF;
  }

static inline int Rd_decode_len( struct Range_decoder * const rdec,
                                 struct Len_model * const lm,
                                 const int pos_state )
  {
  if( Rd_decode_bit( rdec, &lm->choice1 ) == 0 )
    return Rd_decode_tree( rdec, lm->bm_low[pos_state], len_low_bits );
  if( Rd_decode_bit( rdec, &lm->choice2 ) == 0 )
    return len_low_symbols +
           Rd_decode_tree( rdec, lm->bm_mid[pos_state], len_mid_bits );
  return len_low_symbols + len_mid_symbols +
         Rd_decode_tree( rdec, lm->bm_high, len_high_bits );
  }


struct LZ_decoder
  {
  unsigned long long partial_data_pos;
  struct Range_decoder * rdec;
  unsigned dictionary_size;
  int buffer_size;
  uint8_t * buffer;		/* output buffer */
  int pos;			/* current pos in buffer */
  int stream_pos;		/* first byte not yet written to file */
  uint32_t crc;
  int outfd;			/* output file descriptor */

  Bit_model bm_literal[1<<literal_context_bits][0x300];
  Bit_model bm_match[states][pos_states];
  Bit_model bm_rep[states];
  Bit_model bm_rep0[states];
  Bit_model bm_rep1[states];
  Bit_model bm_rep2[states];
  Bit_model bm_len[states][pos_states];
  Bit_model bm_dis_slot[len_states][1<<dis_slot_bits];
  Bit_model bm_dis[modeled_distances-end_dis_model];
  Bit_model bm_align[dis_align_size];

  struct Len_model match_len_model;
  struct Len_model rep_len_model;
  };

void LZd_flush_data( struct LZ_decoder * const d );

static inline uint8_t LZd_get_prev_byte( const struct LZ_decoder * const d )
  {
  const int i = ( ( d->pos > 0 ) ? d->pos : d->buffer_size ) - 1;
  return d->buffer[i];
  }

static inline uint8_t LZd_get_byte( const struct LZ_decoder * const d,
                                    const int distance )
  {
  int i = d->pos - distance - 1;
  if( i < 0 ) i += d->buffer_size;
  return d->buffer[i];
  }

static inline void LZd_put_byte( struct LZ_decoder * const d, const uint8_t b )
  {
  d->buffer[d->pos] = b;
  if( ++d->pos >= d->buffer_size ) LZd_flush_data( d );
  }

static inline void LZd_copy_block( struct LZ_decoder * const d,
                                   const int distance, int len )
  {
  int i = d->pos - distance - 1;
  if( i < 0 ) i += d->buffer_size;
  if( len < d->buffer_size - max( d->pos, i ) && len <= abs( d->pos - i ) )
    {
    memcpy( d->buffer + d->pos, d->buffer + i, len );	/* no wrap, no overlap */
    d->pos += len;
    }
  else for( ; len > 0; --len )
    {
    d->buffer[d->pos] = d->buffer[i];
    if( ++d->pos >= d->buffer_size ) LZd_flush_data( d );
    if( ++i >= d->buffer_size ) i = 0;
    }
  }

static inline bool LZd_init( struct LZ_decoder * const d,
                             struct Range_decoder * const rde,
                             const int dict_size, const int ofd )
  {
  d->partial_data_pos = 0;
  d->rdec = rde;
  d->dictionary_size = dict_size;
  d->buffer_size = max( 65536, dict_size );
  d->buffer = (uint8_t *)malloc( d->buffer_size );
  if( !d->buffer ) return false;
  d->pos = 0;
  d->stream_pos = 0;
  d->crc = 0xFFFFFFFFU;
  d->outfd = ofd;

  Bm_array_init( d->bm_literal[0], (1 << literal_context_bits) * 0x300 );
  Bm_array_init( d->bm_match[0], states * pos_states );
  Bm_array_init( d->bm_rep, states );
  Bm_array_init( d->bm_rep0, states );
  Bm_array_init( d->bm_rep1, states );
  Bm_array_init( d->bm_rep2, states );
  Bm_array_init( d->bm_len[0], states * pos_states );
  Bm_array_init( d->bm_dis_slot[0], len_states * (1 << dis_slot_bits) );
  Bm_array_init( d->bm_dis, modeled_distances - end_dis_model );
  Bm_array_init( d->bm_align, dis_align_size );

  Lm_init( &d->match_len_model );
  Lm_init( &d->rep_len_model );
  d->buffer[d->buffer_size-1] = 0;		/* prev_byte of first byte */
  return true;
  }

static inline void LZd_free( struct LZ_decoder * const d )
  { free( d->buffer ); }

static inline unsigned LZd_crc( const struct LZ_decoder * const d )
  { return d->crc ^ 0xFFFFFFFFU; }

static inline unsigned long long
LZd_data_position( const struct LZ_decoder * const d )
  { return d->partial_data_pos + d->pos; }

int LZd_decode_member( struct LZ_decoder * const d,
                       struct Pretty_print * const pp );
