/*
//
//  Copyright 1997-2009 Torsten Rohlfing
//
//  Copyright 2004-2012 SRI International
//
//  This file is part of the Computational Morphometry Toolkit.
//
//  http://www.nitrc.org/projects/cmtk/
//
//  The Computational Morphometry Toolkit is free software: you can
//  redistribute it and/or modify it under the terms of the GNU General Public
//  License as published by the Free Software Foundation, either version 3 of
//  the License, or (at your option) any later version.
//
//  The Computational Morphometry Toolkit is distributed in the hope that it
//  will be useful, but WITHOUT ANY WARRANTY; without even the implied
//  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with the Computational Morphometry Toolkit.  If not, see
//  <http://www.gnu.org/licenses/>.
//
//  $Revision: 4539 $
//
//  $LastChangedDate: 2012-10-02 15:28:44 -0700 (Tue, 02 Oct 2012) $
//
//  $LastChangedBy: torstenrohlfing $
//
*/

#include "cmtkStudy.h"

#include <IO/cmtkFileFormat.h>
#include <IO/cmtkVolumeIO.h>

#include <string>
#include <limits.h>

namespace
cmtk
{

/** \addtogroup IO */
//@{

Study::Study()
  : m_FileSystemPath( NULL ),
    m_Name( NULL ),
    m_Modality( NULL ),
    m_Volume( NULL ),
    m_MinimumValue( 0.0 ),
    m_MaximumValue( 0.0 ),
    m_Padding( false ),
    m_HaveUserColorMap( false ),
    m_Black( 0.0 ),
    m_White( 0.0 )
{
}

Study::Study( const char* fileSystemPath, const char *name )
  : m_FileSystemPath( NULL ),
    m_Name( NULL ),
    m_Modality( NULL ),
    m_Volume( NULL ),
    m_MinimumValue( 0.0 ),
    m_MaximumValue( 0.0 ),
    m_Padding( false ),
    m_HaveUserColorMap( false ),
    m_Black( 0.0 ),
    m_White( 0.0 )
{
  if ( fileSystemPath ) 
    {
    this->m_FileSystemPath = strdup( fileSystemPath );
    this->m_Description = FileFormat::Describe( this->m_FileSystemPath );

    // cut trailing '/'s off the study path.
    char *endp = this->m_FileSystemPath + strlen( this->m_FileSystemPath ) - 1;
    while ( endp > this->m_FileSystemPath ) 
      {
      if ( *endp == '/' ) 
	*endp = 0;
      else
	break;
      }
    
    this->SetMakeName( name );
  }
}

const char*
Study::SetMakeName( const char* name, const int suffix )
{
  if ( name )
    {
    if ( suffix )
      {
      char fullname[PATH_MAX];
      snprintf( fullname, sizeof( fullname ), "%s <%d>", name, suffix );
      this->SetName( fullname );
      }
    else
      {
      this->SetName( name );
      }
    }
  else
    {
    char buffer[PATH_MAX+1];
    strncpy( buffer, this->m_FileSystemPath, PATH_MAX );
    buffer[PATH_MAX] = 0; // safety - add null at end of buffer

    char* lastChar = buffer + strlen( buffer ) - 1;
    while ( (lastChar != buffer) && // not yet empty string
	    (*lastChar =='/') ) 
      {
      *lastChar = 0;
      --lastChar;
      }
    
    const char *slash = strrchr( buffer, '/' );
    if ( slash )
      strcpy( buffer, slash+1 );
    else 
      strcpy( buffer, this->m_FileSystemPath );
    
    char* dot = strchr( buffer, '.' );
    if ( dot )
      *dot = 0;
    else
      dot = buffer + strlen(buffer);
    
    if ( suffix ) 
      {
      snprintf( dot, sizeof( buffer ) - (dot-buffer), "<%d>", suffix );
      }

    this->SetName( buffer );
    }

  return this->m_Name;
}

Study::~Study() 
{
  free( this->m_FileSystemPath );
  free( this->m_Name );
}

bool 
Study::ReadVolume( const bool reRead, const char* orientation )
{
  UniformVolume::SmartPtr oldVolume( NULL );

  if ( this->m_Volume && reRead ) 
    {
    oldVolume = this->m_Volume;
    this->m_Volume = UniformVolume::SmartPtr( NULL );
    }
  
  if ( !this->m_Volume ) 
    {
    if ( orientation )
      this->m_Volume = UniformVolume::SmartPtr( VolumeIO::ReadOriented( this->m_FileSystemPath, orientation ) );
    else
      this->m_Volume = UniformVolume::SmartPtr( VolumeIO::Read( this->m_FileSystemPath ) );
    
    if ( this->m_Volume ) 
      {
      this->m_Dims = this->m_Volume->GetDims();
      this->m_DisplayedImageIndex = this->m_Dims[AXIS_Z] / 2 ;
      this->m_ZoomFactor = 1;
      const TypedArray *dataArray = this->m_Volume->GetData();
      if ( dataArray ) 
	{
	const Types::DataItemRange range = dataArray->GetRange();
	this->m_MinimumValue = range.m_LowerBound;
	this->m_MaximumValue = range.m_UpperBound;

	this->m_Black = dataArray->GetPercentile( 0.01, 1024 );
	this->m_White = dataArray->GetPercentile( 0.99, 1024 );

	this->m_StandardColormap = 0;
	this->m_ReverseColormap = false;
	}
      }
    }
  
  if ( this->m_Volume && this->m_Volume->GetData() ) 
    {
    return true;
    }
  
  this->m_Volume = oldVolume;
  return false;
}

Study* 
Study::Read( const char* path )
{
  return new Study( path );
}

void
Study::CopyColormap( const Study* other )
{
  this->m_MinimumValue = other->m_MinimumValue;
  this->m_MaximumValue = other->m_MaximumValue;
  this->m_StandardColormap = other->m_StandardColormap;
  this->m_ReverseColormap = other->m_ReverseColormap;
  this->m_Black = other->m_Black;
  this->m_White = other->m_White;
  this->m_Gamma = other->m_Gamma;
}

} // namespace cmtk
