/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *
 * License: GPL-2+
 *
 * Initial version: 2010-02-24
 *
 */


#ifndef LIBCN_SOURCES_H
#define LIBCN_SOURCES_H

#include <string>
#include <vector>

#include "gsl/gsl_rng.h"
#include "gsl/gsl_randist.h"

#include "config.h"

using namespace std;

namespace CNRun {


typedef enum { SRC_NULL, SRC_TAPE, SRC_PERIODIC, SRC_FUNCTION, SRC_NOISE } TSourceType;
extern const char * const __SourceTypes[];

class C_BaseSource {
    public:
	string name;
	TSourceType type;
	C_BaseSource( const char *id, TSourceType intype = SRC_NULL)
	      : name (id), type (intype)
		{}
	virtual ~C_BaseSource()
		{}

	virtual double operator() ( double)
		{  return 0.;  }
	virtual bool is_periodic()
		{  return false;  }
	bool operator== ( const C_BaseSource &rv)
		{  return name == rv.name; }
	bool operator== ( const char *rv)
		{  return name == name; }
	virtual void dump( FILE *strm = stdout)
		{
			fprintf( strm, "%s (%s)\n", name.c_str(), __SourceTypes[type]);
		}
};



class CSourceTape : public C_BaseSource {
    private:
	CSourceTape();
    public:
	string fname;
	vector< pair<double, double> > values;
	bool is_looping;

	CSourceTape( const char *id, const char *infname, bool is_looping = false);

	double operator() ( double at);

	void dump( FILE *strm = stdout)
		{
			fprintf( strm, "%s (%s) %zu values from %s%s\n",
				 name.c_str(), __SourceTypes[type],
				 values.size(), fname.c_str(), is_looping ? "" : " (looping)");
		}
    private:
	vector< pair<double, double> >::iterator I;
};


class CSourcePeriodic : public C_BaseSource {
    private:
	CSourcePeriodic();
    public:
	string fname;
	vector<double> values;
	double period;
	bool is_looping;

	CSourcePeriodic( const char *id, const char *fname, bool is_looping = false, double period = 0.);

	double operator() ( double at)
		{
			size_t	i_abs = (size_t)(at / period),
				i_eff = is_looping
					? i_abs % values.size()
					: min (i_abs, values.size() - 1);
			return values[i_eff];
		}

	void dump( FILE *strm = stdout)
		{
			fprintf( strm, "%s (%s) %zu values at %g from %s%s\n",
				 name.c_str(), __SourceTypes[type],
				 values.size(), period, fname.c_str(), is_looping ? "" : " (looping)");
		}

	bool is_periodic()
		{  return true;  }
};


class CSourceFunction : public C_BaseSource {
    private:
	CSourceFunction();
    public:
	double (*function)( double at);

	CSourceFunction( const char *id, double (*f)(double))
	      : C_BaseSource (id, SRC_FUNCTION), function (f)
		{}

	double operator() ( double at)
		{
			return function( at);
		}
};


extern  const char * const distribution_names[];

class CSourceNoise : public C_BaseSource {
    private:
	CSourceNoise();
    public:
	double _min, _max, _sigma;
	enum TDistribution {
		SOURCE_RANDDIST_UNIFORM,
		SOURCE_RANDDIST_GAUSSIAN,
	};
	TDistribution dist_type;
	gsl_rng	*rng;

	CSourceNoise( const char *id, double in_min = 0., double in_max = 1.,
		      TDistribution type = SOURCE_RANDDIST_UNIFORM,
		      int seed = 0);
       ~CSourceNoise();

	double operator() ( double unused) const
		{
			switch ( dist_type ) {
			case SOURCE_RANDDIST_UNIFORM:	return gsl_rng_uniform( rng) * (_max - _min) + _min;
			case SOURCE_RANDDIST_GAUSSIAN:	return gsl_ran_gaussian( rng, _sigma) + (_max - _min)/2;
			default:			return 42.;
			}
		}

	void dump( FILE *strm = stdout)
		{
			fprintf( strm, "%s (%s) %s in range %g:%g (sigma = %g)\n",
				 name.c_str(), __SourceTypes[type],
				 distribution_names[dist_type], _min, _max, _sigma);
		}
};

}

#endif

// EOF
