/*
*
* put.c - COBEX - Send files to a device.
*
* Copyright (c) 2003,2004,2005 Fredrik Srensson
*
* History:
* v0.5 - fsn - 04-02-22 - First version
* v0.6 - fsn - 05-07-19 - Using general signal handling and general path handling
*
* Source:
*
*/

#include <stdio.h>
#include <ezV24/ezV24.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "cobex_core.h"
#include "cobex_tools.h"
#include "cobex_serial.h"

v24_port_t *UsedPort=NULL;

// Send the Buffer

int sendBuffer (char *inBuffer, int inBuffLgt, obex_packet *inPacket) {

	int sent=0;
	while (sent < inBuffLgt) {
		obex_opcode_put ( inPacket );
		sent+=obex_hi_body ( inPacket, inBuffer, inBuffLgt, sent );
		cobex_packlgt ( inPacket );

		if (cobex_packet_send(inPacket, UsedPort) != COBEX_OK) {
			cobex_closePort(UsedPort);
			return COBEX_ERR;
		}
		if ( cobex_packet_recieve(inPacket, UsedPort) != COBEX_OK) {
			cobex_closePort(UsedPort);
			return COBEX_ERR;
		}
	}
	return COBEX_OK;
}

// Send a file

int sendFile (char *fileName, char *nameOnDevice, obex_packet *inPacket ) {
	FILE *fh;
	char fileBuffer[4001];
	int total=0;
	int buffLgt;
	int rc;
	struct stat buf;

	// Open file...

	if (stat(fileName,&buf) != 0) { return COBEX_ERR; }

	fh=fopen(fileName, "r");
	if (fh==NULL) { return COBEX_ERR; }

	// Send put start

	obex_opcode_put ( inPacket );
	obex_hi_length ( inPacket, buf.st_size );
	obex_hi_name ( inPacket, nameOnDevice, strlen(nameOnDevice)+1 );
	cobex_packlgt ( inPacket );
	
	// dump_packet_headers (inPacket);


	if ( cobex_packet_send(inPacket, UsedPort) != COBEX_OK) {
		fclose (fh);
		return COBEX_ERR;
	}
	if ( cobex_packet_recieve(inPacket, UsedPort) != COBEX_OK) {
		fclose (fh);
		return COBEX_ERR;
	}
	// dump_packet_headers (inPacket);

	if ( inPacket->buffer[0] != (OBEX_RESPONSE_CONTINUE | OBEX_FINAL_BIT) ) {
		fclose (fh);
		return COBEX_ERR;
	}

	// Send file

	while ((buffLgt=fread (fileBuffer,1,4000,fh)) > 0 ) {
		rc=sendBuffer ( fileBuffer, buffLgt, inPacket);
		if (rc != 0) { fclose(fh); return rc; }
		if (cobex_response_code(inPacket) != (OBEX_RESPONSE_CONTINUE | OBEX_FINAL_BIT)) {
			fclose(fh);
			return COBEX_ERR;
		}
		total+=buffLgt;
		printf("Sent %d bytes.\r",total);
		fflush(stdout);
	}

	printf ("\n");

	fclose (fh);

	// Send end_of_body

	obex_opcode_put ( inPacket );
	obex_hi_endofbody ( inPacket );
	cobex_packlgt ( inPacket );
	cobex_set_final_bit( inPacket );

	if (cobex_packet_send(inPacket, UsedPort) != COBEX_OK) {
		return COBEX_ERR;
	}
	if ( cobex_packet_recieve(inPacket, UsedPort) != COBEX_OK) {
		return COBEX_ERR;
	}
	if (inPacket->buffer[0] != (OBEX_RESPONSE_OK | OBEX_FINAL_BIT) ) {
		return COBEX_ERR;
	}

	return COBEX_OK;
}

void help () {
	printf ("obex_put v0.5\n\n");
	printf ("Usage: obex_put <device> <infile> <name_on_device>\n\n");
	printf ("All arguments are compulsory and order is important. Returncodes are used to\n");
	printf ("indicate errors. Use the perl frontend if you want more userfriendlyness.\n\n");
}

// Main routine

int main (int argc, char *argv[]) {
	int rc;
	int endStatus;

	char outBuffer[513];
	obex_packet outPacket;

	outPacket.max=512;
	outPacket.buffer=outBuffer;

	if (argc != 4) {
		help() ;
		return COBEX_ERR;
	}

	// Set up the port
	UsedPort = cobex_openPort(argv[1]);
	ctools_installSignalhandler();
	ctools_setup_state(UsedPort);

	// Do some testing
	if (cobex_testComm(UsedPort)) {
		fputs ("ERR: Comms testing failed.\n",stderr);
		cobex_closePort(UsedPort);
		return COBEX_ERR;
	}	

	// Go obex
	rc = ctools_connect(UsedPort);
	if (rc != COBEX_OK) {
		fputs ("ERR: OBEX connect failed.\n",stderr);
		cobex_closePort(UsedPort);
		return COBEX_ERR;
	}	

	// Do an obex connect

	obex_opcode_connect( &outPacket );
	cobex_packlgt( &outPacket );

	if (cobex_packet_send(&outPacket, UsedPort) != COBEX_OK) {
		cobex_closePort(UsedPort);
		return COBEX_ERR;
	}
	if ( cobex_packet_recieve(&outPacket, UsedPort) != COBEX_OK) {
		cobex_closePort(UsedPort);
		return COBEX_ERR;
	}

	// Now we now the max packet size the device is sending to us, so we should malloc
	// ourselves a new input buffer. Should. It's on the TODO list.

	// Here we GO!

	endStatus=sendFile(argv[2],argv[3],&outPacket);
	if (cobex_response_code(&outPacket) != (OBEX_RESPONSE_OK|OBEX_FINAL_BIT)) {
		fprintf (stderr, "ERR: %s \n",
			cobex_respstring(cobex_response_code(&outPacket)) ) ;
	}

	// Bye, y'all!

	rc = ctools_disconnect( &outPacket, UsedPort );	
	cobex_closePort(UsedPort);

	return rc;

}
