# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gobject
from gettext import gettext as _

from cohoba.contacts.AbstractContact import AbstractContact
from cohoba.contacts.ui.ContactListWindowManager import get_contact_list_window_manager

class ContactGroup(gobject.GObject, AbstractContact):
	__gsignals__ = {"changed": (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE, [])}
	
	def __init__(self, name=None, groups=None):
		gobject.GObject.__init__(self)
		AbstractContact.__init__(self)
		
		# Dynamic
		self.groups = groups
		self.name = name
		
		# Never Changes
		self.alias = None
		self.favorite = True
		self.is_group = True
	
	def save(self):
		self.emit('changed')
	
	def get_class_serialize_id(self):
		return "group"
		
	def serialize(self):
		return (self.get_class_serialize_id(), self.name, self.groups)
	
	def unserialize(self, data):
		id, self.name, self.groups = data
		self.emit("changed")
		
	def is_in_group(self, contact):
		if contact == None or contact == self or contact.groups == None:
			return False
		
		if self.groups == None:
			return False
			
		if len(self.groups) == 0:
			return (len(contact.groups) == 0)
			
		for i in self.groups:
			if i not in contact.groups:
				return False
		
		return True
		
	def on_filter_row(self, model, iter):
		contact = model[iter]
		return self.is_in_group(contact)
	
	def get_description(self):
		# WARNING: This is a complete hack, there should be a proper way to know
		# the number of contacts in a group without reading the whole list.
		# Besides, it appears that using python iterators for treemodel
		# causes unexplained GtkWarning assetion failures and causes havoc.
		# ----------- Start Here ------------------
		from cohoba.contacts.MergedContactList import get_contact_list
		flst = get_group_filter(self, get_contact_list())
		
		class ItemCount:
			def __init__(self):
				self.people = 0
				self.groups = 0

		def count_items(model, path, iter, item_count):
			if model[path][0].is_group:
				item_count.groups += 1
			else:
				item_count.people += 1
		
		item_count = ItemCount()
		flst.foreach(count_items, item_count)
		# ------------------ Ends Here ---------------------
		
		string = '<b>%s</b> (%s)\n<span size="small"><i>%r</i></span>' % (
			self.name,
			_("%d contacts, %d groups") % (item_count.people, item_count.groups),
			self.groups)
		return string
	
	def is_favorite(self):
		return False
	
	def __repr__(self):
		return "Group[%r](%r, %r, %r)" % (self.__class__, self.name, self.groups, self.favorite)

	def action(self):
		get_contact_list_window_manager().show(self)

class FavoriteContactGroup(ContactGroup):
	def __init__(self):
		ContactGroup.__init__(self, _("Favorites"), None)
	
	def get_class_serialize_id(self):
		return "favoritegroup"
		
	def is_in_group(self, contact):
		if contact == None or contact == self:
			return False

		if contact.favorite :#and not contact.is_group:
			return True
		else:
			return False
		
	def is_favorite(self):
		return True

class OverviewContactGroup(ContactGroup):
	def __init__(self):
		ContactGroup.__init__(self, _("Overview"), None)
	
	def get_class_serialize_id(self):
		return "overviewgroup"
		
	def is_in_group(self, contact):
		if contact == None or contact == self:
			return False
		
		if contact.favorite or contact.is_group or contact.is_uncategorized():
			return True
		else:
			return False

def group_for_serialized_data(data):
	group = None
	if data[0] == "group":
		group = ContactGroup()
	elif data[0] == "favoritegroup":
		group = FavoriteContactGroup()
	elif data[0] == "overviewgroup":
		group = OverviewContactGroup()
	else:
		return None
		
	group.unserialize(data)
	return group
	
def get_group_filter(group, model):
	# FIXME: Apparently this cause problem for some people
	# Try to comment the two following lines for a *workaround*..
	filter_model = model.filter_new()
	filter_model.set_visible_func(group.on_filter_row)
	return filter_model
