# Cohoba - a GNOME client for Telepathy
# Derived from telepathy-python
#
# Copyright (C) 2005 Collabora Limited
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import re
import dbus, dbus.glib
import dbus.service
import gobject

from telepathy.interfaces import CONN_INTERFACE
from telepathy.constants import CONNECTION_STATUS_REASON_REQUESTED

from cohoba.common.TelepathyConstants import (MISSION_CONTROL_INTERFACE,
	MISSION_CONTROL_BUS_NAME, MISSION_CONTROL_OBJECT_PATH,
	MISSION_CONTROL_UI_INTERFACE, CONNECTION_PRESENCE_OFFLINE)

from cohoba.common.Utils import singleton, global_error_handler, get_logger, ignore_error_handler, conn_reason_to_string
from cohoba.common.DBusProxyProvider import get_proxy_provider
from cohoba.common.Store import Store
from cohoba.common.ui.Notification import NotificationManager

from cohoba.houston.AccountList import AccountList
from cohoba.houston.ChannelHandler import ChannelHandler

LOG = get_logger("Houston")

NET_MONITOR_BUS_NAME = "org.freedesktop.NetworkManager"
NET_MONITOR_OBJECT_PATH = "/org/freedesktop/NetworkManager"
NET_MONITOR_IFACE = "org.freedesktop.NetworkManager"

NM_STATE_UNKNOWN = 0
NM_STATE_ASLEEP = 1
NM_STATE_CONNECTING = 2
NM_STATE_CONNECTED = 3
NM_STATE_DISCONNECTED = 4


CONNECTION_MATCH = re.compile(r'^org\.freedesktop\.Telepathy\.Connection\.([^.]+)\.([^.]+)(\..*)$')
def telepathy_service_to_obj_path(service):
	return "/%s" % re.sub(r"\.", "/", service)

class PresenceStore(Store):
	def __init__(self):
		Store.__init__(self, "houston-presence")
PresenceStore = singleton(PresenceStore)

class Houston(dbus.service.Object):	
	def __init__(self):
		bus_name = MISSION_CONTROL_BUS_NAME
		object_path = MISSION_CONTROL_OBJECT_PATH
		dbus.service.Object.__init__(self, dbus.service.BusName(bus_name), object_path)
		LOG.info('Houston listening')
		
		# Read the saved presence data (presence status, status message)
		self.presence = PresenceStore().load((CONNECTION_PRESENCE_OFFLINE, ""))
		
		# Create the stored account list
		self.accounts = AccountList()
		
		# Since channel handler will listen to 'account-added' signal, we create it before
		# loading the accounts from gconf
		self.channel_handler = ChannelHandler(self.accounts)
		
		# Load the accounts data from gconf
		self.accounts.load()
		
		# Detect any existing connection on dbus that are handled by MC (known
		# by gconf). We know the connection path on first successful connection to it
		# because the path is cached in gconf
		self.hook_active_connections()
		
		# Finally, listen for new accounts being created (via the UI or other)
		self.accounts.connect('account-added', self.on_account_added)
		
		# Now we can initialise the states of the various accounts
		for account in self.accounts:
			# When an account is added we initialise our various callbacks
			self.on_account_added(self.accounts, account, do_set_presence=False)
		
		NotificationManager().connect("action", self.on_notification_action)
	
		self.nm_iface = None
		self.setup_net_monitor()
		if self.nm_iface == None:
			self.on_net_monitor_reconnect()
	
	def setup_net_monitor(self):
		try:
			proxy = dbus.SystemBus().get_object(NET_MONITOR_BUS_NAME, NET_MONITOR_OBJECT_PATH)
			self.nm_iface = dbus.Interface(proxy, NET_MONITOR_IFACE)
			
			self.nm_iface.connect_to_signal('StateChange', self.on_net_monitor_state)
			self.nm_iface.state(reply_handler=self.on_net_monitor_state, error_handler=ignore_error_handler)
		except dbus.DBusException, e:
			LOG.info("Unable to startup network-manager integration: %s", e)
	
	def on_net_monitor_state(self, state):
		LOG.info("NetworkMonitor says: %s", state)
		if state == NM_STATE_CONNECTED:
			self.on_net_monitor_reconnect()
		else:
			self.disconnect()
	
	def on_net_monitor_reconnect(self):
		all_connected = True
		for account in self.accounts:
			self.set_presence(account)
			if account.enabled and not account.is_connected():
				all_connected = False
		
		if all_connected:
			NotificationManager().show("Accounts Connected", "All enabled accounts are connected", actions=["show-accounts", "Show Accounts"])
	
	def on_notification_action(self, manager, action):
		if action == "show-accounts":
			self.ShowAccounts()
			
	def hook_active_connections(self):
		get_proxy_provider().create_proxy(
				"org.freedesktop.DBus",
				"/org/freedesktop/DBus")

		dbus_iface = get_proxy_provider().get_iface(
			"/org/freedesktop/DBus",
			"org.freedesktop.DBus")
		
		try:
			LOG.info('Hooking-up to existing connections on the bus')
			self.got_names(dbus_iface.ListNames())
			LOG.info('--Finished hooking-up to existing connections on the bus')
		except dbus.DBusException, e:
			LOG.exception("Couldn't retreive existing DBus names")
	
	def got_names(self, names):
		for name in names:
			match = CONNECTION_MATCH.match(name)
			if match != None:
				conn_name, conn_path = name, telepathy_service_to_obj_path(name)
				manager, protocol, connmgrinfo = match.group(1), match.group(2), match.group(3)
				# Try to find an existing cached connection in gconf matching the conn_path
				account = self.accounts.find(conn_path)
				if account != None:
					LOG.info('Found existing connection when starting houston. Hooking-up: %s', account)
					get_proxy_provider().create_proxy(conn_name, conn_path)
					account.connect_account_to_connection(conn_name, conn_path, do_connect=False)
	
	def on_account_added(self, accountlist, account, do_set_presence=True):
		account.connect('connected', self.on_account_connected)
		account.connect('enabled', self.on_account_enabled)
		if do_set_presence:
			self.set_presence(account)
	
	def on_account_connected(self, account, connected):
		if connected:
			self.AccountConnected(account.connection[0], account.connection[1])
		elif not connected and account.status[1] > CONNECTION_STATUS_REASON_REQUESTED:
			# Tell the user we have a situation if reason != User Request
			NotificationManager().show(
					"Disconnected",
					"This account has been disconnected:\n<b>%s</b> (%s)\n<b><i>%s</i></b>" % (
						account.get_username(),
						account.profile.get_description(),
						conn_reason_to_string(account.status[1])),
					actions=["show-accounts", "Show Accounts"])
					
			self.set_presence(account)
		
	def on_account_enabled(self, account, enabled):
		# When the account gets enabled or disabled, check if we should connect/disconnect
		self.set_presence(account)	
			
	def set_presence(self, account):
		LOG.debug('Setting account presence: %s - %s (%s:%s-%s)', account, self.presence, account.status, account.is_connected(), account.is_connectable())
		# First try to disconnect if we are disabled
		if not account.enabled:
			LOG.debug('Account disabled. Disconnecting: %s', account)
			account.disconnect_account()
			return
		
		# We want to be offline, bu the account is connected, disconnect it
		if self.presence[0] == CONNECTION_PRESENCE_OFFLINE and account.is_connected():
			LOG.debug('presence == CONNECTION_PRESENCE_OFFLINE and account.is_connected() -> Disconnect: %s', account)
			account.disconnect_account()
			return
			
		# We want to be online, but the account is disconnected, connect it
		if self.presence[0] != CONNECTION_PRESENCE_OFFLINE and \
					not account.is_connected() and \
					account.is_connectable():
			LOG.debug('presence != CONNECTION_PRESENCE_OFFLINE and not connected and connectable -> Connect: %s', account)
			self.try_connecting_account(account)
			
		# Now we want to be online with a specified presence setting, let's do it
		if account.is_connected():
			account.set_presence(*self.presence)
	
	def try_connecting_account(self, account):
		try:
			account.connect_account()
		except dbus.DBusException, e:
			if account.failed_connection_attempts > 3:
				LOG.exception('Couldn\'t connect account. Too many retries, giving up')
				NotificationManager().show(
					"Unable to Connect",
					"Unable to connect to this account:\n<b>%s</b> (%s)" % (account.get_username(), account.profile.get_description()),
					actions=["show-accounts", "Show Accounts"])
			else:
				timeout_sec = 5**account.failed_connection_attempts
				LOG.exception('Couldn\'t connect account. Scheduling reconnection in %d secs', timeout_sec) 
				gobject.timeout_add(timeout_sec*1000, self.set_presence, account)
					
	def disconnect(self):
		LOG.info('Houston is disconnecting every connection.')
		for account in self.accounts:
			try:
				account.disconnect_account()
			except Exception, e:
				LOG.exception('--Unable to disconnect account: %s', account)
	
	# --- Methods ---
	@dbus.service.method(MISSION_CONTROL_INTERFACE, in_signature='ss', out_signature='')
	def SetPresence(self, name, message):
		self.presence = (name, message)
		PresenceStore().save(self.presence)
		
		for account in self.accounts:
			self.set_presence(account)
	
	@dbus.service.method(MISSION_CONTROL_INTERFACE, in_signature='', out_signature='ss')
	def GetPresence(self):
		return self.presence
			
	@dbus.service.method(MISSION_CONTROL_INTERFACE, in_signature='', out_signature='a(sosss)')
	def GetConnectedAccounts(self):
		result = []
		for account in self.accounts:
			if not account.is_connected():
				continue

			result += [(account.connection[0],
				account.connection[1],
				account.profile.name,
				account.profile.get_description(),
				account.get_username())]

		return result
	
	@dbus.service.method(MISSION_CONTROL_INTERFACE, in_signature='sosuu', out_signature='')
	def RequestChannel(self, conn_name, conn_obj, chan_type, handle_type, handle):
		conn_iface = get_proxy_provider().get_iface(conn_obj, CONN_INTERFACE)
		conn_iface.RequestChannel(chan_type, handle_type, handle, True,
			reply_handler=lambda chan_obj: self.channel_handler.handle_channel(conn_name, conn_obj, chan_obj, chan_type, handle_type, handle),
			error_handler=global_error_handler)
	
	@dbus.service.method(MISSION_CONTROL_UI_INTERFACE, in_signature='', out_signature='')
	def ShowAccounts(self):
		from cohoba.houston.ui.AccountsWindow import AccountsWindow
		AccountsWindow().show()
	
#	@dbus.service.method(MISSION_CONTROL_UI_INTERFACE, in_signature='', out_signature='')
#	def Exit(self):
#		self.disconnect()
#		gtk.main_quit()
	
	# --- Signals ---
	@dbus.service.signal(MISSION_CONTROL_INTERFACE, signature='so')
	def AccountConnected(self, bus_name, object_path):
		self.channel_handler.handle_connection(bus_name, object_path)

Houston = singleton(Houston)
