# Cohoba - a GNOME client for Telepathy
#
# Copyright (C) 2006 Collabora Limited
#
# This package is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This package is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this package; if not, write to the Free Software
# Foundation, 51 Franklin Street, Fifth Floor, Boston, MA, 02110-1301 USA.

import gtk, gtk.glade, gobject
from os.path import join
import cohoba
from gettext import gettext as _

from cohoba.common.ui.ProfileIcons import ProfileIcons
from cohoba.common.Utils import conn_reason_to_string, conn_status_to_string, get_logger
from cohoba.houston.AccountList import AccountList
from cohoba.houston.ui.AccountEditor import AccountEditor, AccountEditorManager

LOG = get_logger("Houston")

class AccountsListView(gtk.TreeView):	
	def __init__(self):
		gtk.TreeView.__init__(self)		

		props = {
			'headers-visible': False,
			"enable-search": False,
			"rules-hint": True,
		}
		for prop, val in props.items():
			self.set_property(prop, val)
			
		self.set_model(AccountList())
		
		# -- Columns setup
		cell = gtk.CellRendererPixbuf()
		column = gtk.TreeViewColumn(_("Icon"), cell)
		column.set_cell_data_func(cell, self.get_account_icon)
		self.append_column(column)

		cell = gtk.CellRendererToggle()
		cell.connect('toggled', self.on_account_toggled)
		column = gtk.TreeViewColumn(_("Auto Connect"), cell)
		column.set_cell_data_func(cell, self.get_account_enabled)
		self.append_column(column)
				
		cell = gtk.CellRendererText()
		column = gtk.TreeViewColumn(_("Account"), cell)
		column.set_cell_data_func(cell, self.get_account_description)
		self.append_column(column)
		# ---
		
		self.connect('row-activated', self.on_row_activated)
				
	def get_account_icon(self, column, cell, model, iter):
		account = model[iter]
		if account.profile != None:
			cell.set_property("pixbuf", ProfileIcons().get_icon(account.profile.name, account.is_connected()))
	
	def get_account_enabled(self, column, cell, model, iter):
		account = model[iter]
		cell.set_property("active", account.enabled)
				
	def get_account_description(self, column, cell, model, iter):
		account = model[iter]
		if account.enabled:
			autoconnect_string = _("This account connects automatically")
		else:
			autoconnect_string = _("This account stays disconnected")
		
		reconnect_string = ""
		if account.reconnect_required:
			reconnect_string = "\n"+ _("Some changes will take effect on next connection.")
		
		if not account.is_valid():
			cell.set_property("markup", "<b>%s</b>" % _("New Account"))
		else:
			cell.set_property("markup",
				"<b>%s</b>\n<span size='small'>%s</span>\n<span size='small'>%s <span foreground='gray'>(%s)</span>%s</span>" % (
					account.get_username(),
					autoconnect_string,
					_("Currently %s") % conn_status_to_string(account.status[0]),
					conn_reason_to_string(account.status[1]),
					reconnect_string))

			
	def show_selected_account_editor(self):
		model, iter = self.get_selection().get_selected()
		if iter == None:
			return
		
		AccountEditorManager().show_account_editor(self.get_model()[iter])
				
	def remove_selected(self):
		model, iter = self.get_selection().get_selected()
		if iter == None:
			return

		AccountList().remove_account(model[iter])
	
	def connect_selected(self):
		model, iter = self.get_selection().get_selected()
		if iter == None:
			return

		try:
			model[iter].connect_account()
		except:
			LOG.exception("Unable to connect account: %s", model[iter])
		
	def disconnect_selected(self):
		model, iter = self.get_selection().get_selected()
		if iter == None:
			return

		model[iter].disconnect_account()
		
	def on_row_activated(self, treeview, path, view_column):
		self.show_selected_account_editor()
			
	def on_account_toggled(self, cell, path):
		account = self.get_model()[path]
		account.set_enabled(not cell.get_active())
