/**
 * 
 * dock.c:
 * author: david mueller, 2006
 * e-mail: d.mueller@e-aktuell.at OR davidlukas.mueller@reflex.at
 * version: 0.2.0_alpha3
 * released: 07/28/2006 02:01 UTC
 * feel free to reuse anything you may find interesting.
 * this is a fork of miniwin as miniwin is intended to be a testbed for new ideas only.
 *
**/

/*
Last Changes:
  - Attention: load plugins in the following order: gconf decoration dbus miniwin wobbly ...

  - use dbus to control mindows (needs the dbus plugin)
  - example: create a mindow, rotate it, move it, scale it and destroy it.

  -  - dbus-send --type=method_call --dest=org.freedesktop.compiz  \
       /org/freedesktop/compiz/miniwin/allscreens/create org.freedesktop.compiz.activate \
      :parent int32:{window-id, use xwininfo}

  -  - dbus-send --type=method_call --dest=org.freedesktop.compiz \
       /org/freedesktop/compiz/miniwin/allscreens/rotate org.freedesktop.compiz.activate \
       :mindow int32:{mindow-id} :angle double:180

  -  - dbus-send --type=method_call --dest=org.freedesktop.compiz \
       /org/freedesktop/compiz/miniwin/allscreens/move org.freedesktop.compiz.activate \
       mindow int32:{mindow-id} :x int32:400 :y int32:400

  -  - dbus-send --type=method_call --dest=org.freedesktop.compiz \
       /org/freedesktop/compiz/miniwin/allscreens/grow org.freedesktop.compiz.activate \
       :mindow int32:{mindow-id} :angle factor:4.0

  -  - dbus-send --type=method_call --dest=org.freedesktop.compiz \
       /org/freedesktop/compiz/miniwin/allscreens/release org.freedesktop.compiz.activate \
       :mindow int32:{mindow-id}

  -  - be aware: normally "create" would send back the Window-id of the mindow, but compiz 
       cannot (at the moment) answer to method_calls. however, I'm using a private patch which
       allows it, but I don't like hacking around in the compiz source, so it will remain private ;)
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <assert.h> //don't blame me, blame the weatherman!
#include <unistd.h>

#include <X11/Xatom.h>
#include <X11/extensions/Xrender.h>

#include <compiz.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

#ifndef DBUS_PATCH
   static void *dbus_connection = 0;
   static void *dbus_message_reply = 0;
#endif

#ifndef N_
#define N_(x) x
#endif

static char *winType[] = {
    "Splash",
    "Normal",
    "Dialog",
    "ModalDialog",
};
#define N_WIN_TYPE (sizeof (winType) / sizeof (winType[0]))

static int displayPrivateIndex;


#define GET_MINIWIN_DISPLAY(d) \
         ((MiniwinDisplay *) (d)->privates[displayPrivateIndex].ptr)
         
#define MINIWIN_DISPLAY(d) \
         MiniwinDisplay* md = GET_MINIWIN_DISPLAY(d)
         
#define GET_MINIWIN_SCREEN(s, fd) \
   ((MiniwinScreen*) (s)->privates[(fd)->screenPrivateIndex].ptr)
   
#define MINIWIN_SCREEN(s) \
   MiniwinScreen* ms = GET_MINIWIN_SCREEN(s, GET_MINIWIN_DISPLAY(s->display))
   
#define GET_MINIWIN_WINDOW(w, ss)					  \
    ((MiniwinWindow *) (w)->privates[(ss)->windowPrivateIndex].ptr)

#define MINIWIN_WINDOW(w)					       \
    MiniwinWindow *mw = GET_MINIWIN_WINDOW  (w,		       \
		      GET_MINIWIN_SCREEN  (w->screen,	       \
		      GET_MINIWIN_DISPLAY (w->screen->display)))

#define _NDEBUG
#ifndef _NDEBUG
#define DEBUG(X) fprintf (stderr, X)
#define DEBUG1(X,Y) fprintf (stderr, X, Y)
#define DEBUG2(X,Y,Z) fprintf (stderr, X, Y, Z)
#else
#define DEBUG(X) 
#define DEBUG1(X,Y)
#define DEBUG2(X,Y,Z)
#endif

#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))

#define MINIWIN_MINDOW_WIDTH_DEFAULT 64
#define MINIWIN_MINDOW_WIDTH_MIN 20
#define MINIWIN_MINDOW_WIDTH_MAX 512

#define MINIWIN_RUNS_ON_AIGLX_DEFAULT FALSE

#define MINIWIN_DEMINDOW_BUTTON_DEFAULT 	Button1
#define MINIWIN_DEMINDOW_MODIFIERS_DEFAULT 0

#define MINIWIN_CLOSE_WINDOW_BUTTON_DEFAULT 	Button3
#define MINIWIN_CLOSE_WINDOW_MODIFIERS_DEFAULT 0

enum
{
   //General mindow settings
   MINIWIN_SCREEN_OPTION_MINDOW_WIDTH,
   //Internal settings
   MINIWIN_SCREEN_OPTION_WINDOW_TYPE,
   MINIWIN_SCREEN_OPTION_RUNS_ON_AIGLX,

   MINIWIN_SCREEN_OPTION_NUM
};

enum
{
   //basic actions
   MINIWIN_DISPLAY_OPTION_CREATE_MINDOW,
   MINIWIN_DISPLAY_OPTION_RELEASE_MINDOW,
   MINIWIN_DISPLAY_OPTION_GET_PARENT,
   MINIWIN_DISPLAY_OPTION_HIDE,
   MINIWIN_DISPLAY_OPTION_SHOW,
   //geometry actions
   MINIWIN_DISPLAY_OPTION_MOVE,
   MINIWIN_DISPLAY_OPTION_GROW,
   MINIWIN_DISPLAY_OPTION_ROTATE,
   //animation actions
   MINIWIN_DISPLAY_OPTION_RING,
   MINIWIN_DISPLAY_OPTION_PULSE,

   MINIWIN_DISPLAY_OPTION_NUM
};

typedef enum _MiniwinWindowState
{
   MINIWIN_WINDOW_STATE_UNKNOWN_INITIALIZING,
 
   MINIWIN_WINDOW_STATE_NORMAL_HIDDEN,
   MINIWIN_WINDOW_STATE_NORMAL_NONE,
 
   MINIWIN_WINDOW_STATE_MINDOW_HIDDEN,
   MINIWIN_WINDOW_STATE_MINDOW_NONE,
} MiniwinWindowState;

#define MwmHintsDecorations (1L << 1)

typedef struct {
    unsigned long flags;
    unsigned long functions;
    unsigned long decorations;
} MwmHints;

typedef struct _MiniwinDisplay
{
   int screenPrivateIndex;

   CompOption opt[MINIWIN_DISPLAY_OPTION_NUM];

   HandleEventProc handleEvent;
} MiniwinDisplay;

typedef struct _MiniwinScreen
{
   int windowPrivateIndex;

   unsigned int mindowWidth;

   //Internal settings
   Bool runsOnAiglx;    
   unsigned long transformedWindows;
   
   CompOption opt[MINIWIN_SCREEN_OPTION_NUM];
   int wMask;

   DrawWindowTextureProc drawWindowTexture;
   DrawWindowGeometryProc drawWindowGeometry;
   PreparePaintScreenProc preparePaintScreen;
   DamageWindowRectProc   damageWindowRect;
   PaintScreenProc paintScreen;
   PaintWindowProc paintWindow;
   
} MiniwinScreen;

#define MINDOW_TRANSFORMED_SIZE 1 << 1
#define MINDOW_TRANSFORMED_ANGLE 1 << 2

typedef struct _MiniwinWindow {
   MiniwinWindowState state;   
    
   CompWindow *parent;
   CompWindow **icons;
   int max_icons;
       
   //hide data
   int originalX;
   int originalY;

   //animation data
   int destinationX;
   int destinationY;
   double angle;

   double ring_max;
   double ring_step;
   int ring_time;

   double pulse_max;
   double pulse_step;
   int pulse_time;
   
   unsigned long transformed;

} MiniwinWindow;

//Basic functions
static Bool isMindow (CompScreen *screen, Window id);
static CompWindow *getParentWindow (CompWindow *mindow);
static Window getParent (CompScreen *screen, Window id);

static Window miniwinCreateMindow (CompWindow *from_parent);
static void miniwinInitializeMindow (CompWindow *mindow);
static void miniwinReleaseMindow (CompWindow *mindow);
static void miniwinDestroyMindow (CompWindow *mindow);
static void miniwinHide (CompWindow *window);
static void miniwinShow (CompWindow *window);

static void miniwinMove (CompWindow *mindow, int x, int y);
static void miniwinGrow (CompWindow *mindow, double factor);
static void miniwinRotate (CompWindow *mindow, double angle);

static void miniwinPulse (CompWindow *mindow, double max_size, double step, int time);
static void miniwinRing (CompWindow *mindow, double max_angle, double step, int time);


//DBus interface
static Bool miniwinActionCreate (CompDisplay	*d, CompAction *action, 
                                 CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionRelease (CompDisplay	*d, CompAction *action, 
                                 CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionHide (CompDisplay *d, CompAction *action, 
                               CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionShow (CompDisplay *d, CompAction *action, 
                               CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionMove (CompDisplay *d, CompAction *action, 
                               CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionParent (CompDisplay *d, CompAction *action, 
                                 CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionGrow (CompDisplay *d, CompAction *action, 
                               CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionRotate (CompDisplay *d, CompAction *action, 
                                 CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionPulse (CompDisplay *d, CompAction *action, 
                                CompActionState state, CompOption *option, int nOption);
static Bool miniwinActionRing (CompDisplay *d, CompAction *action, 
                               CompActionState state, CompOption *option, int nOption);




static Bool
isMindow (CompScreen *screen, Window id)
{
   Atom typeAtom = XInternAtom (screen->display->display, "_WINDOW_TYPE_ATOM", FALSE);
   Atom mindowAtom = XInternAtom (screen->display->display, "_WINDOW_TYPE_MINDOW_ATOM", FALSE);

   Atom windowType; 
   int format;
   unsigned long nitems;
   unsigned long bytes;
   unsigned char *prop;

   XGetWindowProperty (screen->display->display, id, typeAtom, 0, 0, FALSE, AnyPropertyType, &windowType,
                       &format, &nitems, &bytes, &prop);

   if (windowType == mindowAtom) return TRUE;
   else return FALSE;
}

static Window
getParent (CompScreen *screen, Window id)
{
   Atom windowType;
   int format;
   unsigned long nitems;
   unsigned long remaining;
   unsigned char *tmp;

   assert(isMindow (screen, id));

   XGetWindowProperty (screen->display->display, id, 
                       XInternAtom (screen->display->display, "_WINDOW_TYPE_ATOM", FALSE),
                       0, 1, FALSE,
                       XInternAtom (screen->display->display, "_WINDOW_TYPE_MINDOW_ATOM", FALSE),
                       &windowType, &format, &nitems, &remaining, &tmp);

   Window parent_id = *(Window*) tmp;
   DEBUG1("parent_id: %lu\n", parent_id);
   return parent_id;
}

static CompWindow *
getParentWindow (CompWindow *mindow)
{
   Window parent = getParent (mindow->screen, mindow->id);
   return findWindowAtScreen (mindow->screen, parent);
}

static Window 
miniwinCreateMindow (CompWindow *from_parent)
{  
   CompWindow *orig = from_parent;
   CompScreen *screen = orig->screen;

   MINIWIN_SCREEN (screen);
   MINIWIN_WINDOW (from_parent);
   
   Atom type;   
 	Atom mwmHintsAtom;
	MwmHints mwmHints;
   
   Display *dpy = screen->display->display;
  
   int whiteColor = WhitePixel(dpy, DefaultScreen(dpy));
   int width = 0, height = 0;
   
   width = ms->mindowWidth * 2;              
   height = (width * orig->height) / orig->width;

   //don't let the mindow get too long.
   if (height > ms->mindowWidth * 2)
   {
      height = ms->mindowWidth * 2;
      width = (height * orig->width) / orig->height;
   }
  
   DEBUG ("Trying to create a simple mindow\n"); 
   Window win = XCreateSimpleWindow(dpy, DefaultRootWindow(dpy), 0, 0,
                                    width, height, 1, whiteColor, 0xFF000000);

   
   mwmHintsAtom = XInternAtom (dpy, "_MOTIF_WM_HINTS", 0);
	memset (&mwmHints, 0, sizeof (mwmHints));  
   mwmHints.flags	= MwmHintsDecorations;
	mwmHints.decorations = 0;

	XChangeProperty (dpy, win, mwmHintsAtom, mwmHintsAtom,
			 8, PropModeReplace, (unsigned char *) &mwmHints,
			 sizeof (mwmHints));

   type = XInternAtom (dpy, "_NET_WM_WINDOW_TYPE_SPLASH", 0);
   XChangeProperty (dpy, win, XInternAtom (dpy, "_NET_WM_WINDOW_TYPE", 0),
                    XA_ATOM, 32, PropModeReplace, (unsigned char *) &type, 1);
                      

   Window id = from_parent->id;
   DEBUG1("Created mindow from: %lu\n", id);
   XChangeProperty (dpy, win, XInternAtom (dpy, "_WINDOW_TYPE_ATOM", 0),
                    XInternAtom (dpy, "_WINDOW_TYPE_MINDOW_ATOM", 0),
                    32, PropModeReplace, (unsigned char*)&id, 1);
                    
   XMapWindow (screen->display->display, win);
   XSelectInput (dpy, win, StructureNotifyMask | PointerMotionMask);
 
   mw->state = MINIWIN_WINDOW_STATE_NORMAL_NONE;
   mw->originalX = orig->serverX;
   mw->originalY = orig->serverY;
   
   return win;
}

static void
miniwinInitializeMindow (CompWindow *mini)
{
   assert (isMindow(mini->screen, mini->id));
   CompScreen *screen = mini->screen;
   
   Atom mwmHintsAtom = XInternAtom (screen->display->display, "_MOTIF_WM_HINTS", 0);
         
   MwmHints mwmHints;
   memset (&mwmHints, 0, sizeof (mwmHints));  
   mwmHints.flags	= MwmHintsDecorations;
   mwmHints.decorations = 0;

   XChangeProperty (screen->display->display, mini->id, mwmHintsAtom, mwmHintsAtom,
                    8, PropModeReplace, (unsigned char *) &mwmHints, sizeof (mwmHints));
      	                 
   Atom type = XInternAtom (screen->display->display, "_NET_WM_WINDOW_TYPE_SPLASH", 0);
   XChangeProperty (screen->display->display, mini->id, 
                    XInternAtom (screen->display->display, "_NET_WM_WINDOW_TYPE", 0),
                    XA_ATOM, 32, PropModeReplace, (unsigned char *) &type, 1);
                          
   XSelectInput (screen->display->display, mini->id, 
                 EnterWindowMask | LeaveWindowMask | FocusChangeMask | StructureNotifyMask
                 | VisibilityChangeMask | PointerMotionMask);
   
   
   MINIWIN_WINDOW (mini);   
   MINIWIN_SCREEN (mini->screen);
   CompWindow *parent = getParentWindow (mini);

   assert(parent);

   mw->originalX = -1;
   mw->originalY = -1;
   mw->parent = parent;
   
   //miniwinHide (mini);   

   {
      MINIWIN_WINDOW(parent);

      int i;
      for (i = 0; i < mw->max_icons; ++i)
      {
         if (!mw->icons[i])
         {
            mw->icons[i] = mini;
            break;
         }
      }

      if (i == mw->max_icons)
      {
         DEBUG ("reached max. number of allowed mindows for this window!\n");
         miniwinDestroyMindow (mini);
      }
   }   

   mini->state |= CompWindowStateAboveMask;
   mini->state |= CompWindowStateHiddenMask;
   
   glXDestroyGLXPixmap (screen->display->display, mini->pixmap);
   disableTexture (mini->screen, mini->texture);
   finiTexture (mini->screen, mini->texture);
}

static void
miniwinChangeState (CompWindow *window, MiniwinWindowState state)
{
   MINIWIN_WINDOW (window);
   mw->state = state;
}

static void
miniwinHide (CompWindow *window)
{
   MINIWIN_WINDOW (window);
   mw->originalX = window->serverX;
   mw->originalY = window->serverY;

   XMoveWindow (window->screen->display->display, window->id, window->screen->width/2, window->screen->height*4);   
   addWindowDamage (window);

   MINIWIN_SCREEN (window->screen);
}

static void
miniwinShow (CompWindow *window)
{
   MINIWIN_WINDOW (window);
   XMoveWindow (window->screen->display->display, window->id, mw->originalX, mw->originalY);   
   raiseWindow (window);
   addWindowDamage (window);
   updateWindowAttributes (window,TRUE);   
   moveInputFocusToWindow (window);
}

static void
miniwinReleaseMindow (CompWindow *mindow)
{
   assert (isMindow(mindow->screen, mindow->id));
   CompWindow *parent;
   {
      MINIWIN_WINDOW (mindow);
      parent = mw->parent;
   }
   if (parent->destroyed) return;

   MINIWIN_WINDOW (parent);
   int i;
   for (i = 0; i < mw->max_icons; ++i)
   {
      if (mw->icons[i] == mindow)
      {
         mw->icons[i] = 0;
         miniwinDestroyMindow (mindow);
      }
   }
}

static void
miniwinDestroyMindow (CompWindow *mindow)
{
   MINIWIN_SCREEN (mindow->screen);
   DEBUG (sprintf("destroying a mindow! %d\n", mindow->id));
   //XDestroyWindow (mindow->screen->display->display, mindow->id);
}

static void
miniwinMove (CompWindow *window, int x, int y)
{
   MINIWIN_WINDOW (window);
   mw->destinationX = x;
   mw->destinationY = y;
   addWindowDamage (window);
}

static void
miniwinGrow (CompWindow *target, double factor)
{
   MINIWIN_SCREEN (target->screen);
   MINIWIN_WINDOW (target);

   if (factor < 1.001f)
   {
      factor = 1.0;
      mw->transformed ^= MINDOW_TRANSFORMED_SIZE;
      ms->transformedWindows--;
      int diffX = target->attrib.width * (target->paint.xScale - 1);
      int diffY = target->attrib.height * (target->paint.yScale - 1);
      moveWindow (target, diffX/2, diffY/2, FALSE, TRUE);
      syncWindowPosition (target);
   }
   else
   {
      mw->transformed |= MINDOW_TRANSFORMED_SIZE;
      ms->transformedWindows++;
      int diffX = target->attrib.width * (factor - target->paint.xScale);
      int diffY = target->attrib.height * (factor - target->paint.yScale);
      moveWindow (target, -diffX/2, -diffY/2, FALSE, TRUE);
      syncWindowPosition (target);
   }
   (*target->screen->setWindowScale)(target, factor, factor);

   addWindowDamage (target);
}

static void
miniwinPulse (CompWindow *target, double max_size, double step, int time)
{
   if (max_size < 1.01f || time < 50 || step < 0.0001f) return;

   MINIWIN_WINDOW (target);

   if (mw->pulse_time == 0)
   {
      MINIWIN_SCREEN (target->screen);
      ms->transformedWindows++;
   }

   mw->pulse_max = max_size;
   mw->pulse_time = time;
   mw->pulse_step = step;
   mw->transformed |= MINDOW_TRANSFORMED_SIZE;
   addWindowDamage (target);
}

static void
miniwinRotate (CompWindow *window, double angle)
{
   MINIWIN_WINDOW (window);
   MINIWIN_SCREEN (window->screen);

   if (angle < 0.001 && angle > -0.001)
   {
      angle = 0.0;
      mw->transformed ^= MINDOW_TRANSFORMED_ANGLE;
      ms->transformedWindows--;
   }
   else
   {
      mw->transformed |= MINDOW_TRANSFORMED_ANGLE;
      ms->transformedWindows++;
   }
   mw->angle = angle;

   addWindowDamage (window);
}

static void
miniwinRing (CompWindow *window, double max_angle, double step, int time)
{
   if (max_angle < 1.0f || time < 50 || step < 0.01f) return;   

   MINIWIN_WINDOW (window);
   
   if (mw->ring_time == 0)
   {
      MINIWIN_SCREEN (window->screen);
      ms->transformedWindows++;
   }

   mw->ring_max = max_angle;
   mw->ring_time = time;
   mw->ring_step = step;
   mw->transformed |= MINDOW_TRANSFORMED_ANGLE;
   addWindowDamage (window);

   
}

static CompWindow *extractTargetWindow (CompDisplay *d, CompOption *option, int nOption)
{
   Window id = getIntOptionNamed (option, nOption, "mindow", 0);
   if (id == 0)
      id = getIntOptionNamed (option, nOption, "window", 0);
   if (id == 0)
      id = getIntOptionNamed (option, nOption, "parent", 0);

   return findWindowAtDisplay (d, id);
}

static Bool
miniwinActionCreate (CompDisplay *d, CompAction *action,
                     CompActionState state, CompOption *option, int nOption)
{
   CompWindow *window = extractTargetWindow (d, option, nOption);
   if (window)
   {
      Window mini = miniwinCreateMindow (window);

      if (!dbus_connection) return TRUE; //compiz.h, src/main.c and plugins/dbus.c not patched!
      dbus_message_append_args (dbus_message_reply, DBUS_TYPE_INT32, &mini, DBUS_TYPE_INVALID);
      dbus_connection_send (dbus_connection, dbus_message_reply, NULL);
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionRelease (CompDisplay *d, CompAction *action, 
                      CompActionState state, CompOption *option, int nOption)
{
   CompWindow *mindow = extractTargetWindow (d, option, nOption);
   if (mindow && isMindow(mindow->screen, mindow->id))
   {
      miniwinReleaseMindow (mindow);
      return TRUE;
   }
   return FALSE;
}


static Bool
miniwinActionParent (CompDisplay *d, CompAction *action, 
                     CompActionState state, CompOption *option, int nOption)
{
   assert (dbus_connection);
   CompWindow *mindow = extractTargetWindow (d, option, nOption);
   if (mindow && isMindow(mindow->screen, mindow->id))
   {
      CompWindow *parent = getParentWindow (mindow);

      if (!dbus_connection) return TRUE; //compiz.h, src/main.c and plugins/dbus.c not patched!
      dbus_message_append_args (dbus_message_reply, DBUS_TYPE_INT32, &parent->id, DBUS_TYPE_INVALID);
      dbus_connection_send (dbus_connection, dbus_message_reply, NULL);
   
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionHide (CompDisplay *d, CompAction *action, 
                   CompActionState state, CompOption *option, int nOption)
{
   CompWindow *target = extractTargetWindow (d, option, nOption);
   if (target) 
   {
      miniwinHide (target);
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionShow (CompDisplay *d, CompAction *action, 
                   CompActionState state, CompOption *option, int nOption)
{
   CompWindow *target = extractTargetWindow (d, option, nOption);
   if (target) 
   {
      miniwinShow (target);
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionMove (CompDisplay *d, CompAction *action, 
                   CompActionState state, CompOption *option, int nOption)
{
   CompWindow *target = extractTargetWindow (d, option, nOption);
   if (target) 
   {
      int x, y;
      x = getIntOptionNamed (option, nOption, "x", 0);
      y = getIntOptionNamed (option, nOption, "y", 0);
      miniwinMove (target, x, y);
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionGrow (CompDisplay *d, CompAction *action, 
                   CompActionState state, CompOption *option, int nOption)
{
   CompWindow *target = extractTargetWindow (d, option, nOption);
   if (target && isMindow(target->screen, target->id)) 
   {
      double factor;
      factor = getFloatOptionNamed (option, nOption, "factor", 1.0f);
      miniwinGrow (target, factor);
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionPulse (CompDisplay *d, CompAction *action, 
                    CompActionState state, CompOption *option, int nOption)
{
   CompWindow *target = extractTargetWindow (d, option, nOption);
   if (target && isMindow(target->screen, target->id)) 
   {
      double max_size, step;
      int time;
      max_size = getFloatOptionNamed (option, nOption, "max_factor", 1.0f);
      step = getFloatOptionNamed (option, nOption, "step", 0.0f);
      time = getIntOptionNamed (option, nOption, "timeout", 0);
      miniwinPulse (target, max_size, step, time);
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionRotate (CompDisplay *d, CompAction *action, 
                     CompActionState state, CompOption *option, int nOption)
{
   CompWindow *target = extractTargetWindow (d, option, nOption);
   if (target && isMindow(target->screen, target->id)) 
   {
      double angle;
      angle = getFloatOptionNamed (option, nOption, "angle", 1.0f);
      DEBUG1 ("rotating a mindow (%f degrees)\n", angle);
      miniwinRotate (target, angle);
      return TRUE;
   }
   return FALSE;
}

static Bool
miniwinActionRing (CompDisplay *d, CompAction *action, 
                   CompActionState state, CompOption *option, int nOption)
{
   CompWindow *target = extractTargetWindow (d, option, nOption);
   if (target && isMindow(target->screen, target->id)) 
   {
      double max_angle, step;
      int time;
      max_angle = getFloatOptionNamed (option, nOption, "max_angle", 0.0f);
      step = getFloatOptionNamed (option, nOption, "step", 0.0f);
      time = getIntOptionNamed (option, nOption, "timeout", 0);
      miniwinRing (target, max_angle, step, time);
   }
   return FALSE;
}


//now, this is getting interesting, because we are doing a lot of initialization stuff in here.
static Bool
miniwinPaintWindow (CompWindow *w, const WindowPaintAttrib *attrib, Region region, unsigned int mask)
{
   MINIWIN_WINDOW (w);
   Bool status;
   CompScreen *screen = w->screen;
   MINIWIN_SCREEN (screen); 
   
   int i;
   for (i = 0; i < mw->max_icons; ++i)
   {
      CompWindow *icon = mw->icons[i];   
      if (icon && !icon->destroyed) 
      {
         addWindowDamage(icon);
      }
   }

   if (mw->transformed)
   {
      mask |= PAINT_WINDOW_TRANSFORMED_MASK;
   }
/*   if (mw->transformed)
   {
      //mask |= PAINT_WINDOW_TRANSFORMED_MASK;
      Region tmp = XCreateRegion ();
      XUnionRegion (region, tmp, tmp);

      UNWRAP (ms, screen, paintWindow);
      status = (*screen->paintWindow) (w, attrib, tmp, mask);
      WRAP (ms, screen, paintWindow, miniwinPaintWindow);

      damageScreenRegion (screen, tmp);

      XDestroyRegion (tmp);
   }
   else
   {*/
      UNWRAP (ms, screen, paintWindow);
      status = (*screen->paintWindow) (w, attrib, region, mask);
      WRAP (ms, screen, paintWindow, miniwinPaintWindow);
   //}

   //for each newly created window there will be a mindow: it's "icon"
   //which is available until the window gets destroyed.
   if (mw->state == MINIWIN_WINDOW_STATE_UNKNOWN_INITIALIZING)
   {
      if (isMindow(screen, w->id))
      {
         miniwinInitializeMindow (w);
         miniwinChangeState (w, MINIWIN_WINDOW_STATE_MINDOW_NONE);
      }
      else //it's a window
      {
         miniwinChangeState (w, MINIWIN_WINDOW_STATE_NORMAL_NONE);
      }
   }
   
   return status;
}

static void 
miniwinPreparePaintScreen (CompScreen *screen, int msSinceLastPaint)
{
   MINIWIN_SCREEN (screen);   

   CompWindow *w;
   for (w = screen->windows; w; w = w->next)
   {
      MINIWIN_WINDOW (w);
      if (!mw) continue;

      if (mw->destinationX != -1 && mw->destinationY != -1)
      {
         int moveX=0, moveY=0;
         if (mw->destinationX != w->attrib.x)
         {
            moveX = (mw->destinationX - w->attrib.x) / msSinceLastPaint * (100 / 50.0f);
            if (mw->destinationX > w->attrib.x) moveX+=1;
            else if (mw->destinationX <w->attrib.x) moveX-=1;
         }
         if (mw->destinationY != w->attrib.y)
         {
            moveY = (mw->destinationY - w->attrib.y) / msSinceLastPaint * (100 / 50.0f);
            if (mw->destinationY > w->attrib.y) moveY+=1;
            else if (mw->destinationY < w->attrib.y) moveY-=1;
         }
         if (moveX || moveY)
         {
            moveWindow (w, moveX, moveY, TRUE, TRUE);
            syncWindowPosition (w);
         }
         else
            mw->destinationX = mw->destinationY = -1;
      }

      if (mw->pulse_time)
      {
         mw->pulse_time -= msSinceLastPaint;
         if (mw->pulse_time < 0)
         {
            mw->pulse_time = 0;
            miniwinGrow (w, 1.0f);
            mw->pulse_max = 1.0f;
            mw->pulse_step = 0.0f;
            mw->transformed ^= MINDOW_TRANSFORMED_SIZE;
            ms->transformedWindows--;
         }
         else
         {
            double scale = w->paint.xScale;
            scale += mw->pulse_step;
            if (scale > mw->pulse_max || scale <= 1.0f)
            {
               mw->pulse_step *= -1;
            }
            miniwinGrow (w, scale);
         }
      }
   
      if (mw->ring_time)
      {
         mw->ring_time -= msSinceLastPaint;
         if (mw->ring_time < 0)
         {
            mw->ring_time = 0;
            mw->angle = 0.0f;
            mw->ring_max = 0.0f;
            mw->ring_step = 0.0f;
            mw->transformed ^= MINDOW_TRANSFORMED_ANGLE;
            ms->transformedWindows--;
         }
         else
         {
            mw->angle += mw->ring_step;
            if (mw->angle > mw->ring_max || mw->angle < -mw->ring_max)
            {
               mw->ring_step *= -1;
            }
         }
      }
   }


   UNWRAP (ms, screen, preparePaintScreen);
   (*screen->preparePaintScreen) (screen, msSinceLastPaint);
   WRAP (ms, screen, preparePaintScreen, miniwinPreparePaintScreen);
}

static Bool
miniwinPaintScreen (CompScreen              *screen,
                      const ScreenPaintAttrib *sAttrib,
                      Region                  region,
				    	int			 output,
                      unsigned int            mask)
{
   Bool status;

   /*CompWindow *w;
   for (w = screen->windows; w; w = w->next)
   {
      MINIWIN_WINDOW (w);
      if (mw->transformed & MINDOW_TRANSFORMED_SIZE)
      {
         mask |= PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK;
         break;
      }
   }*/

  
   MINIWIN_SCREEN (screen); 
   if (ms->transformedWindows)
   {
      mask |= PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK;
   }
  /* if (FALSE)//ms->transformedWindows)
   {   
      static Region tmpRegion = NULL;

      if (mask & PAINT_SCREEN_REGION_MASK)
      {
         if (mask & PAINT_SCREEN_TRANSFORMED_MASK)
	      { 
            if (mask & PAINT_SCREEN_FULL_MASK)
	         {
               (*screen->paintTransformedScreen) (screen, sAttrib, mask);
         		return TRUE;
	         }

	         return FALSE;
	      }

      }
      else if (mask & PAINT_SCREEN_FULL_MASK)
      {
         (*screen->paintTransformedScreen) (screen, sAttrib, mask);
         return TRUE;
      }
      else
         return FALSE;

      screenLighting (screen, FALSE);

      glPushMatrix ();

      glTranslatef (-0.5f, -0.5f, -DEFAULT_Z_CAMERA);
      glScalef (1.0f / screen->width, -1.0f / screen->height, 1.0f);
      glTranslatef (0.0f, -screen->height, 0.0f);

      (*screen->paintBackground) (screen, region, 0);
      
      if (mask & PAINT_SCREEN_WITH_TRANSFORMED_WINDOWS_MASK)
      {
         (*screen->paintBackground) (screen, region, 0);

      	for (w = screen->windows; w; w = w->next)
	      {
	         if (w->destroyed)
         		continue;

	         if (!w->shaded)
            {
         		if (w->attrib.map_state != IsViewable || !w->damaged)
		         continue;
            }

            (*screen->paintWindow) (w, &w->paint, region, 0);
	      }
      }
      else
      {
         int cnt = 0;

         if (!tmpRegion)
         {
            tmpRegion = XCreateRegion ();
            if (!tmpRegion)
            return FALSE;
         }

         XSubtractRegion (region, &emptyRegion, tmpRegion);
   
   	   for (w = screen->reverseWindows; w; w = w->prev)
         {
            if (w->destroyed)
               continue;

            if (!w->shaded)
   	      {
               if (w->attrib.map_state != IsViewable || !w->damaged)
            	  continue;
            }

            MINIWIN_WINDOW (w);
            if (mw->transformed) 
            {
               (*screen->paintWindow) (w, &w->paint, region, 0);
              
               XPoint points[4];
               double mx, my;
               double ax, ay;
               ax = w->attrib.width / 2.0f;
               ay = w->attrib.height / 2.0f;
               mx = w->attrib.x + ax;
               my = w->attrib.y + ay;
               ax *= w->paint.xScale;
               ay *= w->paint.yScale;
               double a = mw->angle / 57.29577951;

               points[2].x = mx - cos (a) * ax + sin (a) * ay;
               points[2].y = my - cos (a) * ay - sin (a) * ax;

               points[1].x = mx - cos (a) * ax - sin (a) * ay;
               points[1].y = my + cos (a) * ay - sin (a) * ax;

               points[3].x = mx + cos (a) * ax + sin (a) * ay;
               points[3].y = my - cos (a) * ay + sin (a) * ax;

               points[0].x = mx + cos (a) * ax - sin (a) * ay;
               points[0].y = my + cos (a) * ay + sin (a) * ax;

               Region transformed = XPolygonRegion (points, 4, EvenOddRule); 

               XSubtractRegion (tmpRegion, transformed, tmpRegion);
               XDestroyRegion (transformed);
            }
            else if ((*screen->paintWindow) (w, &w->paint, tmpRegion, PAINT_WINDOW_SOLID_MASK))
            {
               XSubtractRegion (tmpRegion, w->region, tmpRegion);


               if (cnt == 0 && !REGION_NOT_EMPTY (tmpRegion) && 
                   screen->opt[COMP_SCREEN_OPTION_UNREDIRECT_FS].value.b &&
                   XEqualRegion (w->region, &screen->region))
               {
                  unredirectWindow (w);
               }
            }

   	      XSubtractRegion (tmpRegion, &emptyRegion, w->clip);
   
   	      cnt++;
         }

   	   if (tmpRegion->numRects)
            (*screen->paintBackground) (screen, tmpRegion, 0);

         for (w = screen->windows; w; w = w->next)
         {
            if (w->destroyed)
               continue;

            if (!w->shaded)
            {
               if (w->attrib.map_state != IsViewable || !w->damaged)
                  continue;
            }

            (*screen->paintWindow) (w, &w->paint, w->clip, PAINT_WINDOW_TRANSLUCENT_MASK);
         }
      }

      glPopMatrix ();
   }
   else
   {*/
      UNWRAP (ms, screen, paintScreen);
      status = (*screen->paintScreen) (screen, sAttrib, region, output, mask);
      WRAP (ms, screen, paintScreen, miniwinPaintScreen);
   /*}

   return TRUE;*/

   return status;
}

static Bool
miniwinBindPixmap (CompWindow *mindow, CompWindow *window)
{
   CompScreen *screen = window->screen;

   if (!mindow->texture->pixmap || !window->texture->pixmap)
   {
      DEBUG("rebinding mindow\n");
      finiTexture (screen, mindow->texture);
      if (!bindPixmapToTexture (screen, mindow->texture, window->pixmap,
                                mindow->width, mindow->height, mindow->attrib.depth))
      {
         return FALSE;
      }
   }

   return TRUE;
}

static Bool
miniwinDamageWindowRect (CompWindow *w, Bool initial, BoxPtr rect)
{
   Bool status;
   MINIWIN_WINDOW (w);
   MINIWIN_SCREEN (w->screen);

   UNWRAP (ms, w->screen, damageWindowRect);
   status = (*w->screen->damageWindowRect) (w, initial, rect);
   WRAP (ms, w->screen, damageWindowRect, miniwinDamageWindowRect);

   int i;

   if (mw->parent)
   {
      if (!mw->parent->destroyed && !w->destroyed)
      {
         miniwinBindPixmap (w, mw->parent);
      }
   }
   else
   {
      for (i = 0; i < mw->max_icons; ++i)
      {
         CompWindow *icon = mw->icons[i];
         if (icon && !icon->destroyed && !w->destroyed)
         {
            miniwinBindPixmap (icon, w);
         }
      }
   }

   return status;
}

static void
miniwinDrawWindowTexture (CompWindow *w, CompTexture *texture, const WindowPaintAttrib *attrib, unsigned int mask)
{
   MINIWIN_SCREEN (w->screen);
   MINIWIN_WINDOW (w);
   
   Bool damage = FALSE;
		
   glPushMatrix ();        

   if (mw->angle != 0.0)
   {
      int filter;
      
      glTranslatef (w->attrib.x, w->attrib.y, 0.0f);
      glTranslatef (w->attrib.width*attrib->xScale/2.0f, w->attrib.height*attrib->yScale/2.0f, 0.0f);
      glRotatef (mw->angle, 0.0f, 0.0f, 1.0f);
      glTranslatef (-w->attrib.width*attrib->xScale/2.0f, -w->attrib.height*attrib->yScale/2.0f, 0.0f);
      glScalef (attrib->xScale, attrib->yScale, 0.0f);
      glTranslatef (-w->attrib.x, -w->attrib.y, 0.0f);

      filter = w->screen->filter[WINDOW_TRANS_FILTER];

      enableTexture (w->screen, texture, filter);
         
      glEnable (GL_BLEND);
      if (attrib->opacity != OPAQUE || attrib->brightness != BRIGHT)
      {
         GLushort color;

         color = (attrib->opacity * attrib->brightness) >> 16;

         screenTexEnvMode (w->screen, GL_MODULATE);
         glColor4us (color, color, color, attrib->opacity);

         (*w->screen->drawWindowGeometry) (w);

         glColor4usv (defaultColor);
         screenTexEnvMode (w->screen, GL_REPLACE);
      }
      else
      {
         (*w->screen->drawWindowGeometry) (w);
      }

      glDisable (GL_BLEND);
         
      disableTexture (w->screen, texture);

      damage = TRUE;
   }
   else
   {      
      UNWRAP (ms, w->screen, drawWindowTexture);
      (*w->screen->drawWindowTexture) (w, texture, attrib, mask);
      WRAP (ms, w->screen, drawWindowTexture, miniwinDrawWindowTexture);
   }
   
   glPopMatrix ();
   
   if (damage) damageScreen (w->screen);   
}

static void
miniwinDrawWindowGeometry (CompWindow *w)
{
   MINIWIN_WINDOW (w);
   MINIWIN_SCREEN (w->screen);
   
	UNWRAP (ms, w->screen, drawWindowGeometry);
	(*w->screen->drawWindowGeometry) (w);
	WRAP (ms, w->screen, drawWindowGeometry, miniwinDrawWindowGeometry);
}

//TODO: add some code in order to be able to move iconified windows more easily.
static void
miniwinHandleEvent (CompDisplay* d, XEvent* event)
{
   CompWindow* w;
   MINIWIN_DISPLAY (d);
   
   switch (event->type) {
   case DestroyNotify:
      w = findWindowAtDisplay (d, event->xdestroywindow.window);
      if (w)
      {
         MINIWIN_WINDOW (w);
         int i;
         for (i = 0; i < mw->max_icons; ++i)
         {
            CompWindow *icon = mw->icons[i];
            if (icon)
            {
               DEBUG ("destroying a parent window...\n");
               miniwinDestroyMindow (icon);
            }
         }
      }   
      break;
   case ButtonPress:
   case ButtonRelease:       
      w = findWindowAtDisplay (d, event->xkey.window);
      if (w)
      {         
         CompScreen *s = w->screen;      
         MINIWIN_SCREEN (s);           
         MINIWIN_WINDOW (w);
            
         if (!(ms->wMask & w->type)) break;
      }
   case KeyPress:
   case KeyRelease:
      w = findWindowAtDisplay (d, event->xkey.window);
      if (w)
      {  
         CompScreen *s = w->screen;
         MINIWIN_SCREEN (s);       
         MINIWIN_WINDOW (w);
                    
         if (!(ms->wMask & w->type)) break;
      }
      break;
   default:
      break;
   }
   
   UNWRAP (md, d, handleEvent);
   (*d->handleEvent) (d, event);
   WRAP (md, d, handleEvent, miniwinHandleEvent);
}

//From here on, it will get boring. I don't mind if you just close the file now.
//What follows is only compiz initialization stuff...
 
static CompOption*
miniwinGetScreenOptions (CompScreen* screen, int* count)
{
   DEBUG ("Getting a screen option\n");
   MINIWIN_SCREEN (screen); //hate this way of getting a miniwin object. have to change that.
   *count = NUM_OPTIONS (ms);
   
   DEBUG ("Got the screen option\n");
   assert (ms->opt);
   return ms->opt;
}

static Bool
miniwinSetScreenOption (CompScreen* screen, char* name, CompOptionValue* value)
{
   DEBUG ("Setting a screen option\n");
   CompOption* o;
   int index;
   
   MINIWIN_SCREEN (screen);
   
   o = compFindOption (ms->opt, NUM_OPTIONS (ms), name, &index);
   if (!o) return FALSE;
   
   switch (index) {
   case MINIWIN_SCREEN_OPTION_MINDOW_WIDTH:
      if (compSetIntOption (o, value))
      {
         ms->mindowWidth = o->value.i;
      }
      break;
   case MINIWIN_SCREEN_OPTION_RUNS_ON_AIGLX:
      if (compSetBoolOption (o, value))
      {
         ms->runsOnAiglx = o->value.b;
         return TRUE;
      }
      break;
   case MINIWIN_SCREEN_OPTION_WINDOW_TYPE:
      if (compSetOptionList (o, value))
      {
         ms->wMask = compWindowTypeMaskFromStringList (&o->value);
         ms->wMask &= ~CompWindowTypeDesktopMask;
         return TRUE;
      }
   default: 
      break;
   }
   
   return FALSE;
}

static void
miniwinScreenInitOptions (MiniwinScreen* ms)
{
   CompOption* o;
   int i;
   
   DEBUG ("Initializing options...\n");
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_MINDOW_WIDTH];
   o->name = "icon_width";
   o->shortDesc = "Standard width of icons";
   o->longDesc = "Standard width of icons";
   o->type = CompOptionTypeInt;
   o->value.i = MINIWIN_MINDOW_WIDTH_DEFAULT;
   o->rest.i.min = MINIWIN_MINDOW_WIDTH_MIN;
   o->rest.i.max = MINIWIN_MINDOW_WIDTH_MAX;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_RUNS_ON_AIGLX];
   o->name = "runs_on_aiglx";
   o->shortDesc = "Is it AIGLX?";
   o->longDesc = "Is it AIGLX?";
   o->type = CompOptionTypeBool;
   o->value.b = MINIWIN_RUNS_ON_AIGLX_DEFAULT;
   
   o = &ms->opt[MINIWIN_SCREEN_OPTION_WINDOW_TYPE];
   o->name = "window_type";
   o->shortDesc = "Window Types";
   o->longDesc = "Window types that can be iconified (be rather conservative with that)";
   o->type = CompOptionTypeList;
   o->value.list.type = CompOptionTypeString;
   o->value.list.nValue = N_WIN_TYPE;
   o->value.list.value = malloc (sizeof (CompOptionValue) * N_WIN_TYPE);
   for (i = 0; i < N_WIN_TYPE; ++i)
      o->value.list.value[i].s = strdup (winType[i]);
   o->rest.s.string = windowTypeString;
   o->rest.s.nString = nWindowTypeString;
   
   ms->wMask = compWindowTypeMaskFromStringList (&o->value);      
}

static Bool
miniwinInit (CompPlugin* p)
{
   DEBUG("init\n");
   displayPrivateIndex = allocateDisplayPrivateIndex ();
   if (displayPrivateIndex < 0)
   {
      return FALSE;
   }
   return TRUE;
}

static void
miniwinFini (CompPlugin *p)
{
   if (displayPrivateIndex >= 0)
      freeDisplayPrivateIndex (displayPrivateIndex);
}


static CompOption *
miniwinGetDisplayOptions (CompDisplay *display, int *count)
{
   MINIWIN_DISPLAY (display);
   *count = NUM_OPTIONS (md);
   return md->opt;
}

static Bool
miniwinSetDisplayOption (CompDisplay *display, char *name, CompOptionValue *value)
{
   CompOption *o;
   int index;

   MINIWIN_DISPLAY (display);

   o = compFindOption (md->opt, NUM_OPTIONS (md), name, &index);
   if (!o) return FALSE;

   switch (index)
   {
   /*case ZOOM_DISPLAY_OPTION_INITIATE:
   case ZOOM_DISPLAY_OPTION_IN:
	  if (setDisplayAction (display, o, value))
	    return TRUE;
	break;
    case ZOOM_DISPLAY_OPTION_OUT:
	if (compSetActionOption (o, value))
	    return TRUE;*/
    default:
	break;
    }

    return FALSE;
}

static void
miniwinDisplayInitOptions (MiniwinDisplay *md, Display *display)
{
   CompOption *o;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_CREATE_MINDOW];
   o->name = "create";
   o->shortDesc = N_("Create a mindow");
   o->longDesc	 = N_("Create a new mindow from a window");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionCreate;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_RELEASE_MINDOW];
   o->name = "release";
   o->shortDesc = N_("Release a mindow");
   o->longDesc	 = N_("Release (destroy) an existing mindow");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionRelease;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_RELEASE_MINDOW];
   o->name = "release";
   o->shortDesc = N_("Release a mindow");
   o->longDesc	 = N_("Release (destroy) an existing mindow");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionRelease;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_HIDE];
   o->name = "hide";
   o->shortDesc = N_("Hide a mindow or a window");
   o->longDesc	 = N_("Hide a mindow or a window");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionHide;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_SHOW];
   o->name = "show";
   o->shortDesc = N_("Show a mindow or a window");
   o->longDesc	 = N_("Show a (hidden) mindow or a (hidden) window");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionShow;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_MOVE];
   o->name = "move";
   o->shortDesc = N_("Move a mindow or a window");
   o->longDesc	 = N_("Do a animated move of a mindow or a window");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionMove;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_GROW];
   o->name = "grow";
   o->shortDesc = N_("Grow a mindow");
   o->longDesc	 = N_("Grow a mindow by a factor");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionGrow;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_ROTATE];
   o->name = "rotate";
   o->shortDesc = N_("Rotate a mindow");
   o->longDesc	 = N_("Rotate a mindow by an angle");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionRotate;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_GET_PARENT];
   o->name = "parent";
   o->shortDesc = N_("Get the parent of a mindow");
   o->longDesc	 = N_("Get the parent of a mindow");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionParent;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_RING];
   o->name = "ring";
   o->shortDesc = N_("Make a mindow ring");
   o->longDesc	 = N_("Make a mindow ring");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionRing;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;

   o = &md->opt[MINIWIN_DISPLAY_OPTION_PULSE];
   o->name = "pulse";
   o->shortDesc = N_("Make a mindow pulse");
   o->longDesc	 = N_("Make a mindow pulse");
   o->type = CompOptionTypeAction;
   o->value.action.initiate = miniwinActionPulse;
   o->value.action.terminate = 0;
   o->value.action.bell	= FALSE;
   o->value.action.edgeMask = 0;
   o->value.action.state = 0;
}

static Bool
miniwinInitDisplay (CompPlugin* p, CompDisplay* d)
{
   MiniwinDisplay *md;
   md = malloc (sizeof (MiniwinDisplay));
   if (!md) return FALSE;
   
   md->screenPrivateIndex = allocateScreenPrivateIndex (d);
   if (md->screenPrivateIndex < 0)
   {
      free (md);
      return FALSE;
   }
   
   miniwinDisplayInitOptions (md, d->display);

   WRAP (md, d, handleEvent, miniwinHandleEvent);
   d->privates[displayPrivateIndex].ptr = md;
   
   return TRUE;
}

static void
miniwinFiniDisplay (CompPlugin* p, CompDisplay* d)
{
   MINIWIN_DISPLAY(d);
   
   freeScreenPrivateIndex (d, md->screenPrivateIndex);
   
   UNWRAP (md, d, handleEvent);
   
   free (md);
}



static Bool
miniwinInitScreen (CompPlugin* p, CompScreen* s)
{
   MiniwinScreen* ms;
   MINIWIN_DISPLAY (s->display);
   
   ms = malloc (sizeof (MiniwinScreen));
   if (!ms) return FALSE;
   
   ms->windowPrivateIndex = allocateWindowPrivateIndex (s);
   if (ms->windowPrivateIndex < 0)
   {
      free (ms);
      return FALSE;
   }   
   
   DEBUG("trying to initialize screen...\n");

   ms->mindowWidth = MINIWIN_MINDOW_WIDTH_DEFAULT;
   ms->runsOnAiglx = MINIWIN_RUNS_ON_AIGLX_DEFAULT;
   ms->transformedWindows = FALSE;
   
   miniwinScreenInitOptions (ms);
   s->privates[md->screenPrivateIndex].ptr = ms;

   WRAP (ms, s, paintWindow, miniwinPaintWindow);
   WRAP (ms, s, preparePaintScreen, miniwinPreparePaintScreen);
   WRAP (ms, s, paintScreen, miniwinPaintScreen);
   WRAP (ms, s, drawWindowTexture, miniwinDrawWindowTexture);
   WRAP (ms, s, drawWindowGeometry, miniwinDrawWindowGeometry);
   WRAP (ms, s, damageWindowRect, miniwinDamageWindowRect);
      
   return TRUE;
}

static void
miniwinFiniScreen (CompPlugin* p, CompScreen* s)
{
   MINIWIN_SCREEN(s);

   freeWindowPrivateIndex (s, ms->windowPrivateIndex);
   
   UNWRAP (ms, s, paintWindow);
   UNWRAP (ms, s, preparePaintScreen);
   UNWRAP (ms, s, paintScreen);
   UNWRAP (ms, s, drawWindowTexture);
   UNWRAP (ms, s, drawWindowGeometry);
   UNWRAP (ms, s, damageWindowRect);
   
   free (ms);
}

static Bool
miniwinInitWindow (CompPlugin *p, CompWindow *w)
{
   DEBUG ("Created a window!\n");
   
   MiniwinWindow *mw;

   MINIWIN_SCREEN (w->screen);

   mw = malloc (sizeof (MiniwinWindow));
   if (!mw) return FALSE;

   mw->state = MINIWIN_WINDOW_STATE_UNKNOWN_INITIALIZING;      
   mw->max_icons = 8;
   mw->icons = malloc (sizeof(CompWindow*) * mw->max_icons);
   int i;
   for (i = 0; i < mw->max_icons; ++i)
   {
      mw->icons[i] = 0;
   }
   mw->parent = 0;
   mw->destinationX = -1;
   mw->destinationY = -1;
   mw->transformed = FALSE;

   mw->angle = 0.0;
   mw->ring_max = 0.0;
   mw->ring_step = 0.0;
   mw->ring_time = 0;

   mw->pulse_max = 1.0;
   mw->pulse_step = 0.0;
   mw->pulse_time = 0;

   w->privates[ms->windowPrivateIndex].ptr = mw;
   return TRUE;
}

static void
miniwinFiniWindow (CompPlugin *p, CompWindow *w)
{
    MINIWIN_WINDOW (w);
    free (mw->icons);
    free (mw);
}
static int
miniwinGetVersion (CompPlugin *plugin,
		 int	    version)
{
     return ABIVERSION;
}

CompPluginDep dockDeps[] = {
    { CompPluginRuleAfter, "decoration" }
};

static CompPluginVTable miniwinVTable = {
   "miniwin",
   "Miniwin",
   "Miniwin",
   miniwinGetVersion,
   miniwinInit,
   miniwinFini,
   miniwinInitDisplay,
   miniwinFiniDisplay,
   miniwinInitScreen,
   miniwinFiniScreen,
   miniwinInitWindow,
   miniwinFiniWindow,
   miniwinGetDisplayOptions,
   miniwinSetDisplayOption,
   /*miniwinGetScreenOptions*/0,
   miniwinSetScreenOption,
   dockDeps,
   sizeof (dockDeps) / sizeof (dockDeps[0]),
   0,
   0
};

CompPluginVTable *
getCompPluginInfo (void)
{
   return &miniwinVTable;
}

