/**
 * Beryl Opacify 
 *
 * Copyright (c) 2006 Kristian Lyngstøl <kristian@beryl-project.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *
 * Opacify increases opacity on targeted windows and reduces it on
 * blocking windows, makeing whatever window you are targeting easily
 * visible. 
 *
 * TODO: 
 * - Possibly do a "total opacity reduction thing. If there's one 
 *   window blocking a section: reduce it to X, if there's two windows
 *   blocking that section: reduce their opacity to X/2, and so on.
 * 
 * - What do we do when we switch apps? Option to clear_passive and 
 *   reset active? 
 * 
 * - Detect a raised window.
 *
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <X11/Xatom.h>
#include <X11/extensions/Xrender.h>
#include <compiz.h>

#define GET_OPACIFY_DISPLAY(d)                            \
	((OpacifyDisplay *) (d)->privates[displayPrivateIndex].ptr)
#define OPACIFY_DISPLAY(d)                                \
    OpacifyDisplay *od = GET_OPACIFY_DISPLAY (d)
#define GET_OPACIFY_SCREEN(s, od)                         \
	((OpacifyScreen *) (s)->privates[(od)->screenPrivateIndex].ptr)
#define OPACIFY_SCREEN(s)                                 \
	OpacifyScreen *os = GET_OPACIFY_SCREEN (s, GET_OPACIFY_DISPLAY (s->display))

#define OPACIFY_SCREEN_OPTION_ACTIVE_OP 0
#define OPACIFY_SCREEN_OPTION_PASSIVE_OP 1
#define OPACIFY_SCREEN_OPTION_WINDOW_TYPE 2
#define OPACIFY_SCREEN_OPTION_ONLY_IF_BLOCK 3
#define OPACIFY_SCREEN_OPTION_NUM 4
#define NUM_OPTIONS(s) (sizeof ((s)->opt) / sizeof (CompOption))
#define N_WIN_TYPE (sizeof (winType) / sizeof (winType[0]))

/* Size of the Window array storing passive windows. */
#define MAX_WINDOWS 64

static int displayPrivateIndex = 0;

typedef struct _OpacifyDisplay {
	int screenPrivateIndex;
	Bool active;
	HandleEventProc handleEvent;
} OpacifyDisplay;

static char *winType[] = {
	N_("Toolbar"),
	N_("Utility"),
	N_("Dialog"),
	N_("ModalDialog"),
	N_("Fullscreen"),
	N_("Normal")
};

typedef struct _OpacifyScreen {
	Window active;
	Window passive[MAX_WINDOWS];
	Region intersect;
	unsigned short int passive_num;
	unsigned int active_op;
	unsigned int passive_op;
	int wMask;
	Bool only_if_block;
	PaintWindowProc paintWindow;
	CompOption opt[OPACIFY_SCREEN_OPTION_NUM];
} OpacifyScreen;

/* Core opacify functions. These do the real work. ---------------------*/

/* Finds the corrosponding CompWindow for Window id. Returns NULL if
 * there is no such window. 
 */
static CompWindow *find_window(CompScreen *s, Window id) 
{
	CompWindow *w;
	for(w = s->windows; w; w = w->next) 
		if(w->id == id)
			return w; 
	return NULL;
}

/* Sets the real opacity and damages the window if actual opacity and 
 * requested opacity differs. */
static void set_opacity(CompWindow *w, int opacity)
{
	if (!w) 
		return;
	if (w->paint.opacity == opacity) 
		return;
	w->paint.opacity = opacity;
	addWindowDamage(w);
}

/* Resets the Window to the original opacity if it still exists.
 * This is based on w->opacity which should represent the Opacity
 * atom of the window. 
 */
static void reset_opacity(CompScreen *s, Window ow) 
{ 
	CompWindow *w;
	w = find_window(s, ow);
	if(!w) 
		return;
	set_opacity(w,w->opacity);
}

/* Resets the opacity of windows on the passive list, if they still
 * exist.  
 */
static void clear_passive(CompScreen *s)
{
	OPACIFY_SCREEN(s);
	int i;
	for (i = 0; i < os->passive_num; i++ ) 
		reset_opacity(s, os->passive[i]);
	os->passive_num = 0;
}

/* Dim an (inactive) window. Place it's id, opacity and desktop on the
 * passive list and update passive_num. Then change the opacity.
 */
static void dim_window(CompScreen *s, CompWindow *w) 
{
	OPACIFY_SCREEN(s);
	if(os->passive_num >= MAX_WINDOWS - 1) {
		fprintf(stderr, "opacify: Trying to store information "
			"about too many windows, or you hit a bug.\nIf "
			"you don't have around %d windows blocking the "
			"currently targeted window, please report this.\n",
			MAX_WINDOWS);
		return;
	} 
	os->passive[os->passive_num++] = w->id;
	set_opacity(w, MIN(os->passive_op,w->paint.opacity));
}

/* Walk through all windows, skip until we've passed the active
 * window, skip if it's invisible, hidden or minimized, skip if
 * it's not a window type we're looking for. 
 * Dim it if it intersects. 
 *
 * Returns number of changed windows.
 */
static int passive_windows(CompScreen *s, Region a_region)
{ 
	CompWindow *w; 
	OPACIFY_SCREEN(s);
	Bool flag = FALSE;
	int i = 0;
	for(w=s->windows; w; w = w->next) {
		if (w->id==os->active) { 
			flag = TRUE;
			continue;
		}
		if(!flag)
			continue;
		if (!(w->type & os->wMask))  
			continue;
		if (w->invisible || w->hidden || w->minimized)  
			continue;
		XIntersectRegion(w->region, a_region, os->intersect);
		if(!XEmptyRegion(os->intersect)) {
			dim_window(s,w); 
			i++; 
		}
	}
	return i;
}

/* Check if we switched active window, reset the old passive windows
 * if we did. If we have an active window and switched: reset that too.
 * If we have a window (w is true), either update the active id and
 * passive list, or check if we manually changed the opacity on the
 * active window and if so: Flag it to not get reset.
 *
 * The opacity-change flag is not fool proof: it relies on receiving
 * a mouse-event ON THE ACTIVE WINDOW when we change opacity. This
* should be improved.
 */
static void opacifyHandleMotion(CompScreen *s, CompWindow *w)
{
	OPACIFY_SCREEN(s);
	int num;
	if (otherScreenGrabExist(s,"rotate","move","scale",0)) {
		clear_passive(s);
		return;
	}
	if (!w || os->active != w->id) {
		clear_passive(s);
		if (os->active) {
			reset_opacity(s, os->active); 
			os->active = 0;
		}
	}
	if (!w)
		return;
	if (w->id != os->active && (w->type & os->wMask)) {
		os->active = w->id;
		num = passive_windows(s,w->region); 
		if(num || !os->only_if_block)
			set_opacity(w,MAX(os->active_op,w->paint.opacity));
	} 
}

/* Takes the inital event. If it's interesting, pass it on. */
static void opacifyHandleEvent(CompDisplay *d, XEvent *event)
{
	CompScreen *s;
	CompWindow *w;
	OPACIFY_DISPLAY(d);
	switch(event->type) { 
	case EnterNotify:
		s = findScreenAtDisplay(d, event->xcrossing.root);
		if (s) {
			w = 
			findTopLevelWindowAtScreen(s, 
						    event->xcrossing.window);
			opacifyHandleMotion(s, w);
		}
		break;
	default:
		break;
	}
	UNWRAP(od, d, handleEvent);
	(*d->handleEvent) (d, event);
	WRAP(od, d, handleEvent, opacifyHandleEvent);
}


/* Configuration, initliazation, boring stuff. ----------------------- */
static void opacifyFiniScreen(CompPlugin * p, CompScreen * s)
{
	OPACIFY_SCREEN(s);
	if(os->opt[OPACIFY_SCREEN_OPTION_WINDOW_TYPE].value.list.value)
		free(os->opt[OPACIFY_SCREEN_OPTION_WINDOW_TYPE].value.list.value);
	XDestroyRegion(os->intersect);
	free(os);
}

static Bool
opacifySetScreenOptions(CompScreen * screen, char *name,
		     CompOptionValue * value)
{
	CompOption *o;
	int index;
	OPACIFY_SCREEN(screen);
	o = compFindOption(os->opt, NUM_OPTIONS(os), name, &index);
	if (!o)
		return FALSE;

	switch (index) {
		case OPACIFY_SCREEN_OPTION_ACTIVE_OP:
			if (compSetIntOption(o, value)) {
				os->active_op = (o->value.i * 0xffff) / 100;
				return TRUE;
			}
			break;
		case OPACIFY_SCREEN_OPTION_PASSIVE_OP:
			if (compSetIntOption(o, value)) {
				os->passive_op = (o->value.i * 0xffff) / 100;
				return TRUE;
			}
			break;
		case OPACIFY_SCREEN_OPTION_WINDOW_TYPE:
			if (compSetOptionList(o, value)) {
				os->wMask = 
				    compWindowTypeMaskFromStringList(&o->value);
				return TRUE;
			}
			break;
		case OPACIFY_SCREEN_OPTION_ONLY_IF_BLOCK:
			if (compSetBoolOption(o, value)) { 
				os->only_if_block = o->value.b;
				return TRUE;
			}
			break; 
		default:
			break;
	}

	return FALSE;
}

static void opacifyScreenInitOptions(OpacifyScreen * os) 
{
	CompOption *o;
	int i;
	o = &os->opt[OPACIFY_SCREEN_OPTION_ACTIVE_OP];
	o->name = "active_op";
	o->shortDesc = N_("Active Opacity");
	o->longDesc = N_("The minimum opacity to ensure a targeted window has. A target window will have either this opacity or the pre-set opacity, whichever is higher.");
	o->type = CompOptionTypeInt;
	o->value.i = 100; 
	o->rest.i.min = 1;
	o->rest.i.max = 100;
	os->active_op = OPAQUE;

	o = &os->opt[OPACIFY_SCREEN_OPTION_PASSIVE_OP];
	o->name = "passive_op";
	o->shortDesc = N_("Passive Opacity");
	o->longDesc = N_("The maximum opacity a window blocking the current targeted window can have. A blocking window will have either tihs opacity or pre-set opacity, whichever is lower.");
	o->type = CompOptionTypeInt;
	o->value.i = 10; 
	o->rest.i.min = 1;
	o->rest.i.max = 100;
	os->passive_op = 0xffff / 10;

	o = &os->opt[OPACIFY_SCREEN_OPTION_WINDOW_TYPE];
	o->name = "window_types";
	o->shortDesc = N_("Window Types");
	o->longDesc =
	    N_("Window types that should be opacified");
	o->type = CompOptionTypeList;
	o->value.list.type = CompOptionTypeString;
	o->value.list.nValue = N_WIN_TYPE;
	o->value.list.value = malloc(sizeof(CompOptionValue) * N_WIN_TYPE);
	for (i = 0; i < N_WIN_TYPE; i++)
		o->value.list.value[i].s = strdup(winType[i]);
	o->rest.s.string = (char **) windowTypeString;
	o->rest.s.nString = nWindowTypeString;
	os->wMask = compWindowTypeMaskFromStringList(&o->value);

	o = &os->opt[OPACIFY_SCREEN_OPTION_ONLY_IF_BLOCK];
	o->name = "only_if_block";
	o->shortDesc = N_("Only increase opacity if a window is blocking");
	o->longDesc =
	    N_("Only increase the opacity on the targeted window if it has one or more windows "
	       "blocking it from view.");
	o->type = CompOptionTypeBool;
	o->value.b = FALSE;
	os->only_if_block = FALSE;
}

static Bool opacifyInitScreen(CompPlugin * p, CompScreen * s)
{
	OPACIFY_DISPLAY(s->display);

	OpacifyScreen *os = (OpacifyScreen *) calloc(1, sizeof(OpacifyScreen));
	s->privates[od->screenPrivateIndex].ptr = os;
	os->intersect = XCreateRegion();

	opacifyScreenInitOptions(os);

	return TRUE;
}

static CompOption *opacifyGetScreenOptions(CompScreen *screen, int *count)
{
	if (screen) {
		OPACIFY_SCREEN(screen);

		*count = NUM_OPTIONS(os);
		return os->opt;
	} else {
		OpacifyScreen *os = malloc(sizeof(OpacifyScreen));
		opacifyScreenInitOptions(os);
		*count = NUM_OPTIONS(os);
		return os->opt;
	}
}

static Bool opacifyInitDisplay(CompPlugin * p, CompDisplay * d)
{
	OpacifyDisplay *od = (OpacifyDisplay *) malloc(sizeof(OpacifyDisplay));
	od->screenPrivateIndex = allocateScreenPrivateIndex(d);

	if (od->screenPrivateIndex < 0) {
		free(od);
		return FALSE;
	}

	WRAP(od, d, handleEvent, opacifyHandleEvent);

	d->privates[displayPrivateIndex].ptr = od;

	return TRUE;
}

static void opacifyFiniDisplay(CompPlugin * p, CompDisplay * d)
{
	OPACIFY_DISPLAY(d);
	UNWRAP(od, d, handleEvent);
	freeScreenPrivateIndex(d, od->screenPrivateIndex);
	free(od);
}

static Bool opacifyInit(CompPlugin * p)
{
	displayPrivateIndex = allocateDisplayPrivateIndex();
	if (displayPrivateIndex < 0)
		return FALSE;

	return TRUE;
}

static void opacifyFini(CompPlugin * p)
{
	if (displayPrivateIndex >= 0)
		freeDisplayPrivateIndex(displayPrivateIndex);
}

static int
opacifyGetVersion (CompPlugin *plugin,
		int	   version)
{
    return ABIVERSION;
}

CompPluginVTable opacifyVTable = {
	"opacify",
	N_("Opacify"),
	N_("Make windows easily visible by hovering the mouse over them"),
	opacifyGetVersion,
	opacifyInit,
	opacifyFini,
	opacifyInitDisplay,
	opacifyFiniDisplay,
	opacifyInitScreen,
	opacifyFiniScreen,
	0,
	0,
	0,
	0,
	opacifyGetScreenOptions,
	opacifySetScreenOptions,
	0,
	0,
	0,
	0
};

CompPluginVTable *getCompPluginInfo(void)
{
	return &opacifyVTable;
}
