#!/usr/bin/env python

#****************************************************************************
# finddlg.py, provides dialog interface for unit find dialog
#
# ConvertAll, a units conversion program
# Copyright (C) 2004, Douglas W. Bell
#
# This is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License, Version 2.  This program is
# distributed in the hope that it will be useful, but WITTHOUT ANY WARRANTY.
#*****************************************************************************

import sys, os.path
import convertdlg
from qt import *

stdFlags = Qt.WStyle_Customize | Qt.WStyle_NormalBorder | Qt.WStyle_Title | \
           Qt.WStyle_SysMenu

class FindDlg(QDialog):
    """Dialog for filtering and searching for units"""
    def __init__(self, mainDlg, parent=None, name=None, \
                 modal=0, flags=stdFlags):
        QDialog.__init__(self, parent, name, modal, flags)
        self.mainDlg = mainDlg
        self.setIcon(QPixmap(convertdlg.ConvertDlg.convert_xpm))
        if sys.platform == 'win32':
            self.setCaption('Unit Finder  (PyQt)')
        else:
            self.setCaption('Unit Finder')
        self.currentType = ''
        self.currentSearch = ''

        mainLayout = QVBoxLayout(self, 10)
        upperLayout = QHBoxLayout(mainLayout, 10)
        filterBox = QHGroupBox('&Filter Unit Types', self)
        upperLayout.addWidget(filterBox, 1)
        self.filterCombo = QComboBox(0, filterBox)

        searchBox = QHGroupBox('&Search String', self)
        mainLayout.addWidget(searchBox)
        self.searchEdit = QLineEdit(searchBox)
        clearButton = QPushButton('C&lear', searchBox)
        clearButton.setFocusPolicy(QWidget.NoFocus)

        self.unitListView = QListView(self)
        mainLayout.addWidget(self.unitListView)
        self.unitListView.addColumn('Unit Name', 170)
        self.unitListView.addColumn('Unit Type', 100)
        self.unitListView.addColumn('Comments', 200)
        self.unitListView.setAllColumnsShowFocus(1)
        self.unitListView.setShowSortIndicator(1)

        lowerLayout = QHBoxLayout(mainLayout, 10)
        self.fromBox = QVButtonGroup('From Unit', self)
        lowerLayout.addWidget(self.fromBox)
        fromReplaceButton = QPushButton('&Replace', self.fromBox)
        self.connect(fromReplaceButton, SIGNAL('clicked()'), self.fromRepl)
        fromInsertButton = QPushButton('&Insert', self.fromBox)
        self.connect(fromInsertButton, SIGNAL('clicked()'), self.fromIns)
        self.toBox = QVButtonGroup('To Unit', self)
        lowerLayout.addWidget(self.toBox)
        toReplaceButton = QPushButton('Replac&e', self.toBox)
        self.connect(toReplaceButton, SIGNAL('clicked()'), self.toRepl)
        toInsertButton = QPushButton('Inser&t', self.toBox)
        self.connect(toInsertButton, SIGNAL('clicked()'), self.toIns)

        closeButton = QPushButton('&Close', self)
        upperLayout.addWidget(closeButton)
        self.connect(closeButton, SIGNAL('clicked()'), self, SLOT('close()'))

        option = convertdlg.ConvertDlg.option
        xSize = option.intData('FinderXSize', 0, 10000)
        ySize = option.intData('FinderYSize', 0, 10000)
        if xSize and ySize:
            self.resize(xSize, ySize)
        self.move(option.intData('FinderXPos', 0, 10000), \
                  option.intData('FinderYPos', 0, 10000))
        self.loadUnits()
        self.loadTypes()
        self.connect(self.filterCombo, SIGNAL('activated(const QString&)'), \
                     self.changeType)
        self.connect(self.searchEdit, SIGNAL('textChanged(const QString&)'), \
                     self.changeSearch)
        self.connect(clearButton, SIGNAL('clicked()'), self.searchEdit, \
                     SLOT('clear()'))
        self.connect(self.unitListView, SIGNAL('selectionChanged()'), \
                     self.updateCtrls)

    def loadUnits(self):
        """Load unit items"""
        self.unitListView.clear()
        for unit in convertdlg.ConvertDlg.unitData.\
                               filteredList(self.currentType, \
                                            self.currentSearch):
            FindUnitListItem(unit, self.unitListView)
        if self.unitListView.childCount() == 1:
            self.unitListView.setSelected(self.unitListView.firstChild(), 1)
        self.updateCtrls()

    def loadTypes(self):
        """Load combobox with type names"""
        types = convertdlg.ConvertDlg.unitData.typeList[:]
        types.sort()
        self.filterCombo.clear()
        self.filterCombo.insertItem('[All]')
        prevName = ''
        for name in types:
            if name != prevName:
                self.filterCombo.insertItem(name)
            prevName = name

    def updateCtrls(self):
        """Change active status of unit set buttons"""
        item = self.unitListView.selectedItem()
        if item:
            self.fromBox.setEnabled(1)
            self.toBox.setEnabled(1)
        else:
            self.fromBox.setEnabled(0)
            self.toBox.setEnabled(0)

    def changeType(self, newType):
        """Change current unit type setting"""
        self.currentType = str(newType)
        if self.currentType == '[All]':
            self.currentType = ''
        self.loadUnits()

    def changeSearch(self, newStr):
        self.currentSearch = str(newStr)
        self.loadUnits()

    def fromRepl(self):
        """Replace from unit with selected unit"""
        item = self.unitListView.currentItem()
        if item:
            unit = item.unit
            self.mainDlg.fromGroup.clearUnit()
            self.mainDlg.fromGroup.replaceCurrent(unit)
            self.mainDlg.fromUnitEdit.unitUpdate()
            self.mainDlg.fromUnitListView.updateSelection()

    def fromIns(self):
        """Insert selected unit into from unit"""
        item = self.unitListView.currentItem()
        if item:
            unit = item.unit
            self.mainDlg.fromGroup.replaceCurrent(unit)
            self.mainDlg.fromUnitEdit.unitUpdate()
            self.mainDlg.fromUnitListView.updateSelection()

    def toRepl(self):
        """Replace to unit with selected unit"""
        item = self.unitListView.currentItem()
        if item:
            unit = item.unit
            self.mainDlg.toGroup.clearUnit()
            self.mainDlg.toGroup.replaceCurrent(unit)
            self.mainDlg.toUnitEdit.unitUpdate()
            self.mainDlg.toUnitListView.updateSelection()

    def toIns(self):
        """Insert selected unit into to unit"""
        item = self.unitListView.currentItem()
        if item:
            unit = item.unit
            self.mainDlg.toGroup.replaceCurrent(unit)
            self.mainDlg.toUnitEdit.unitUpdate()
            self.mainDlg.toUnitListView.updateSelection()


class FindUnitListItem(QListViewItem):
    """Item in the find unit list view"""
    def __init__(self, unit, parent=None):
        QListViewItem.__init__(self, parent)
        self.unit = unit
        self.setText(0, unit.description())
        self.setText(1, unit.typeName)
        self.setText(2, unit.comments[1])
