#!/usr/bin/env python
#
#   ConVirt   -  Copyright (c) 2008 Jd & Hap Hazard
#   ======
#
# ConVirt is a Xen management tool with a GTK based graphical interface
# that allows for performing the standard set of domain operations
# (start, stop, pause, kill, shutdown, reboot, snapshot, etc...). It
# also attempts to simplify certain aspects such as the creation of
# domains, as well as making the consoles available directly within the
# tool's user interface.
#
#
# This software is subject to the GNU Lesser General Public License (LGPL)
# and for details, please consult it at:
#
#    http://www.fsf.org/licensing/licenses/lgpl.txt
#
# author : Jd <jd_jedi@users.sourceforge.net>
#


# All classes in thse files are required Interfaces for managing
# machines/host/nodes with virtualization technology.

from NodeProxy import Node, NodeWrapper
from utils import XMConfig, LVMProxy
from constants import *
from phelper import HostValidationException,\
     AuthenticationException, CommunicationException

import sys,os,re,types, platform
import threading


class NodeException(Exception):
    def __init__(self):
	Exception.__init__(self)



class ManagedNode:
    """
    Interface that represents a node being managed.It defines useful APIs
    for clients to be able to do Management for a virtualized Node
    """

    def __init__(self,
                 hostname = None,
                 ssh_port = 22,
                 username=Node.DEFAULT_USER,
                 password=None,
                 isRemote=False,
                 helper = None,
                 use_keys = False,
                 address = None):

        self.hostname = hostname
        self.ssh_port = ssh_port
        self.username = username
        self.password = password
        self.isRemote = isRemote
        self.helper   = helper
        self.use_keys = use_keys
        if not address:
            self.address  = hostname
        else:
            self.address = address
        
        
        self._node_proxy_lock = threading.RLock()
        
        self._node_proxy = None    # lazy-init self.node_proxy 
        self._lvm_proxy = None     # lazy-init self.lvm_proxy 
        self._isLVMEnabled = None  # lazy-init self.isLVMEnabled 
        self._config = None        # lazy-init self.config
        self._environ = None       # lazy-init self.environ
        self._exec_path = None     # lazy init self.exec_path

        self.metrics = None
        
    # implement lazy initialization. 
    def __getattr__(self, name):
        if name == 'node_proxy':
            return self._init_node_proxy()
        if name == 'lvm_proxy': 
            return self._init_lvm_proxy()
        if name == 'config':
            return self._init_config()
        if name == 'isLVMEnabled':
            return self._init_isLVMEnabled()
        if name == 'environ':
            return self._init_environ()
        if name == 'exec_path':
            return self._init_exec_path()

    def is_in_error(self):
        if self.is_authenticated():
            # The following return does not seem to work!!
            #return  self.node_proxy.get_last_error() is not None
            return  self._node_proxy._last_error is not None
        else:
            return True
       

    def disconnect(self):
        if self._node_proxy is not None:
            self._node_proxy.disconnect()

    def connect(self):
        if self._node_proxy is not None:
            self._node_proxy.connect(self.hostname,
                                     self.ssh_port,
                                     self.username,
                                     self.password,
                                     self.isRemote,
                                     self.use_keys)

        else:
            self._init_node_proxy()

    # for now simple check. Later need to go deep
    def is_authenticated(self):
        if self._node_proxy is not None:
            return self._node_proxy.n_p is not None
        return self._node_proxy
    
    # set credentials, allow credentials to be set later.
    def set_credentials(self, username, password):
        self.username = username
        self.password = password
        
    def is_remote(self):
        return self.isRemote


    def get_address(self):
        return self.address

    def set_address(self, address):
        self.address = address


    def get_metrics(self, refresh=False):
        if not refresh: return self.metrics

    def get_VM_count(self):
        pass
    
    def get_console(self):
        """
        get the console for the dom
        API migght need revision...
        """
        pass

    def get_terminal(self, username, password):
        """
        return tty terminal (telnet, ssh session ?)
        """
        pass
    
    def get_vnc(self):
        """
        get VNC session for this dom. VNC would popup username/password.
        """
        pass    
    
    def _init_node_proxy(self):        
        self._node_proxy_lock.acquire()
        try:
            if self._node_proxy is None:
                while True:
                    creds = None
                    try:
                        self._node_proxy = NodeWrapper( \
                            hostname = self.hostname,
                            ssh_port = self.ssh_port,
                            username = self.username,
                            password = self.password,
                            isRemote = self.isRemote,
                            use_keys = self.use_keys)
                        
                    except AuthenticationException, e:
                        creds = None
                        if self.helper and not use_keys:
                            creds = self.helper.get_credentials(self.hostname,
                                                                self.username)
                            if creds is None:
                                raise Exception("Server not Authenticated")
                            else:
                                self.username = creds.username
                                self.password = creds.password
                        else:
                            raise e
                    else:
                        break
        finally:
            self._node_proxy_lock.release()
        
        
        return self._node_proxy
        

    def _init_config(self):
        if self._config is None:
            self._config = XMConfig(self.node_proxy)
        return self._config

    def _init_lvm_proxy(self):
        if self._lvm_proxy is None and self.isLVMEnabled:            
                self._lvm_proxy = LVMProxy(self.node_proxy,
                                           self.exec_path)
        return self._lvm_proxy

    def _init_isLVMEnabled(self):
        if self._isLVMEnabled is None:
            conf_val = self.config.get(XMConfig.ENV, prop_lvm)
            if conf_val is None:
                self._isLVMEnabled = LVMProxy.isLVMEnabled(self.node_proxy,
                                                           self.exec_path)
                self.config.set(XMConfig.ENV,prop_lvm,str(self._isLVMEnabled))
            else:
                self._isLVMEnabled = eval(conf_val)                
        return self._isLVMEnabled
    
    def _init_environ(self):
        if self._environ is None:
            self._environ = NodeEnvHelper(self.node_proxy)
        return self._environ

    def _init_exec_path(self):
        if self._exec_path is None:
            self._exec_path = self.config.get(XMConfig.PATHS,
                                              prop_exec_path)
        return self._exec_path
    



class NodeEnvHelper:
    """A helper for ManagedNode for retrieving and storing
    Environment/Config information for a host."""

    def __init__(self, node_proxy):
        self.node_proxy = node_proxy
        self.__dict = None
        self.__populateDictionary()

    def __iter__(self):
        return self.__dict.iterkeys()

    def keys(self):
        return self.__dict.keys()
    
    def __getitem__(self, item):
        if not item: return None
        if self.__dict.has_key(item):
            return self.__dict[item]
        else:
            return None

    def refresh(self):
        self.__dict = None
        self.__populateDictionary()


    def __populateDictionary(self):
        """ retrieve environment information for the
        node and store it in the local dictionary. """

        if self.__dict is not None:
            self.__dict = None
        
        attributes  = [prop_env_SYSTEM,
                       prop_env_NODE_NAME,
                       prop_env_RELEASE,
                       prop_env_VER,
                       prop_env_MACHINE_TYPE,
                       prop_env_PROCESSOR]
        
        if self.node_proxy.isRemote:
            # remote host
            values = self.node_proxy.exec_cmd( \
                'uname -s; uname -n; uname -r;'+
                'uname -v; uname -m; uname -p'\
                )[0].split('\n')[:-1]
        else:
            # local host
            values = platform.uname()

        self.__dict = dict((attributes[x],values[x]) \
                           for x in range(6))




#########################
# SELF TEST
#########################

if __name__ == '__main__':

    REMOTE_HOST = '192.168.123.155'
    REMOTE_USER = 'root'
    REMOTE_PASSWD = ''

    REMOTE = False 
    
    local_node = ManagedNode(hostname=LOCALHOST)
    if not REMOTE:
        remote_node = local_node  # for local-only testing
    else:        
        remote_node = ManagedNode(hostname=REMOTE_HOST,
                           username=REMOTE_USER,
                           password = REMOTE_PASSWD,
                           isRemote = True)    

    #
    # lvm_proxy  tests
    #

    print '\nManagedHost.LVMProxy interface test STARTING'

    print 'Local Node LVM enabled?:' , local_node.isLVMEnabled
    print 'Remote Node LVM enabled?:' , remote_node.isLVMEnabled
    
    for lvm in (local_node.lvm_proxy,remote_node.lvm_proxy):
        if lvm:
            vgs =  lvm.listVolumeGroups()
            for g in vgs:
                print g
                print lvm.listLogicalVolumes(g)
                print '\t Creating test LV'
                lvm.createLogicalVolume('selfTest',0.1,g)
                print '\t Deleting test LV'
                lvm.removeLogicalVolume('selfTest',g)            
    print 'ManagedHost.LVMPRoxy interface test COMPLETED\n'
   

    #
    # environment tests
    #
    print '\nManagedHost.Environment access  test STARTING'

    for nd in (local_node, remote_node):
        print "\n%s's environment:" % nd.hostname
        print 'Available attributes:', nd.environ.keys()
        for attr in nd.environ:
            print attr, ':',nd.environ[attr]

    print 'ManagedHost.Environment access  test COMPLETED\n'

    sys.exit(0)
    




    
    
    


