/*
 * This file is part of the FORS Data Reduction Pipeline
 * Copyright (C) 2002-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * fors_response.cpp
 *
 *  Created on: 2014 4 2
 *      Author: cgarcia
 */

#include <stdexcept>
#include <vector>
#include <fors_response.h>
#include "extinction.h"
#include "spec_std_star.h"
#include "response.h"

cpl_table *fors_compute_response
(cpl_image *spectra, double startwave, double dispersion, double gain,
 double exptime, cpl_table *ext_table, double airmass, cpl_table *flux_table,
 const std::vector<double>& ignored_waves,
 const std::vector<std::pair<double, double> >& ignored_wave_ranges,
 int order)
{

    cpl_image *spectrum       = NULL; // Extracted standard star spectrum
    cpl_image *flux           = NULL; // Standard star flux binned as "spectrum"
    cpl_image *efficiency     = NULL; // Raw efficiency curve
    cpl_image *smo_efficiency = NULL; // Smoothed efficiency curve
    cpl_image *response       = NULL; // Raw response curve
    cpl_table *response_table;
    int        nx, ny;


    if (spectra == NULL || ext_table == NULL || flux_table == NULL) 
        throw std::invalid_argument("Empty spectra, ext_table or flux_table");

    if (!cpl_table_has_column(ext_table, "WAVE")) 
        throw std::invalid_argument("Column WAVE in atmospheric extinction table");

    if (!cpl_table_has_column(ext_table, "EXTINCTION")) 
        throw std::invalid_argument("Column EXTINCTION in atmospheric extinction table");

    if (!cpl_table_has_column(flux_table, "WAVE")) 
        throw std::invalid_argument("Column WAVE in standard star flux table");

    if (!cpl_table_has_column(flux_table, "FLUX")) 
        throw std::invalid_argument("Column FLUX in standard star flux table");

    if (gain < 0.1) 
        throw std::invalid_argument("Invalid gain factor (<0.1)");

    if (exptime < 0.001) 
        throw std::invalid_argument("Invalid exposure time (<0.001)");

    if (dispersion < 0.001) 
        throw std::invalid_argument("Invalid dispersion (<0.001)");

    if (order < 2) 
        throw std::invalid_argument("Order of the polynomial fitting the " 
                                 "instrument response must be at least 2");

    nx = cpl_image_get_size_x(spectra);
    ny = cpl_image_get_size_y(spectra);

    /*
     * Find brightest spectrum and duplicate it.
     */

    if (ny == 1) {
        spectrum = cpl_image_duplicate(spectra);
    }
    else {
        cpl_size        x, y;
        cpl_image *brights = cpl_image_collapse_create(spectra, 1);

        cpl_image_get_maxpos(brights, &x, &y);
        cpl_image_delete(brights);
        spectrum = cpl_image_extract(spectra, 1, y, nx, y);
    }


    /*
     * Convert standard star spectrum in electrons per second per Angstrom.
     */

    cpl_image_multiply_scalar(spectrum, gain / exptime / dispersion);

    mosca::spectrum std_obs_spectrum(spectrum, startwave,  dispersion);
    
    mosca::extinction atm_extinction(ext_table);

    mosca::spectrum std_extcorrect = 
            atm_extinction.correct_spectrum(std_obs_spectrum, airmass);

    mosca::spec_std_star std_star(flux_table);

    mosca::response resp;

    resp.compute_response(std_extcorrect, std_star);

    resp.fit_response(order, ignored_waves, ignored_wave_ranges);


    /*
     * Assemble the product spectrophotometric response_table.
     */

    response_table = cpl_table_new(resp.wave().size());

    cpl_table_new_column(response_table, "WAVE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "WAVE", "Angstrom");
    cpl_table_copy_data_double(response_table, "WAVE", &(resp.wave()[0]));

    cpl_table_new_column(response_table, "STD_FLUX", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "STD_FLUX", 
                              "10^(-16) erg/(cm^2 s Angstrom)");
    cpl_table_copy_data_double(response_table, "STD_FLUX", &(resp.tabulated_flux()[0]));
    cpl_image_delete(flux); flux = NULL;

    cpl_table_new_column(response_table, "OBS_FLUX", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "OBS_FLUX", "electron/(s Angstrom)");
    cpl_table_copy_data_double(response_table, "OBS_FLUX", &(resp.observed_flux()[0]));
    cpl_image_delete(spectrum); spectrum = NULL;

    cpl_table_new_column(response_table, "RAW_EFFICIENCY", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "RAW_EFFICIENCY", "electron/photon");
    cpl_table_copy_data_double(response_table, "RAW_EFFICIENCY", &(resp.efficiency_raw()[0]));
    cpl_image_delete(efficiency); efficiency = NULL;

    cpl_table_new_column(response_table, "EFFICIENCY", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "EFFICIENCY", "electron/photon");
    cpl_table_copy_data_double(response_table, "EFFICIENCY", &(resp.efficiency_fit()[0]));
    cpl_image_delete(smo_efficiency); smo_efficiency = NULL;

    cpl_table_new_column(response_table, "RAW_RESPONSE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, "RAW_RESPONSE", 
                              "10^(-16) erg/(cm^2 electron)");
    cpl_table_copy_data_double(response_table, "RAW_RESPONSE", &(resp.response_raw()[0]));
    cpl_image_delete(response); response = NULL;

    cpl_table_new_column(response_table, "RESPONSE", CPL_TYPE_DOUBLE);
    cpl_table_set_column_unit(response_table, 
                              "RESPONSE", "10^(-16) erg/(cm^2 electron)");
    cpl_table_copy_data_double(response_table, "RESPONSE", &(resp.response_fit()[0]));

    return response_table;
}

