/* Crafted: a pud editor for the FreeCraft project
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * LIBGIMP - The GIMP Library
 * Copyright (C) 1995-1997 Peter Mattis and Spencer Kimball
 *
 * gimpfileselection.c
 * Copyright (C) 1999-2000 Michael Natterer <mitch@gimp.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>

#include <gtk/gtk.h>

#include "craftedfileselection.h"
#include "craftedintl.h"

#include "pixmaps/yes.xpm"
#include "pixmaps/no.xpm"

/*  callbacks  */
static void crafted_file_selection_realize                 (GtkWidget *widget);
static void crafted_file_selection_entry_callback          (GtkWidget *widget,
                                                            gpointer   data);
static gint crafted_file_selection_entry_focus_out_callback(GtkWidget *widget,
                                                            GdkEvent  *event,
                                                            gpointer   data);
static void crafted_file_selection_browse_callback         (GtkWidget *widget,
                                                            gpointer   data);

/*  private functions  */
static void crafted_file_selection_check_filename(CraftedFileSelection *gfs);

enum
{
  FILENAME_CHANGED,
  LAST_SIGNAL
};

static guint crafted_file_selection_signals[LAST_SIGNAL] = { 0 };

static GtkHBoxClass *parent_class = NULL;

static void crafted_file_selection_destroy(GtkObject *object)
{
  CraftedFileSelection *cfs;

  g_return_if_fail(object != NULL);
  g_return_if_fail(CRAFTED_IS_FILE_SELECTION(object));

  cfs = CRAFTED_FILE_SELECTION(object);

  if (cfs->file_selection)
    gtk_widget_destroy(cfs->file_selection);

  if (cfs->title)
    g_free(cfs->title);

  if (cfs->yes_pixmap)
    gdk_pixmap_unref(cfs->yes_pixmap);
  if (cfs->yes_mask)
    gdk_bitmap_unref(cfs->yes_mask);

  if (cfs->no_pixmap)
    gdk_pixmap_unref(cfs->no_pixmap);
  if (cfs->no_mask)
    gdk_bitmap_unref(cfs->no_mask);

  if (GTK_OBJECT_CLASS(parent_class)->destroy)
    GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static void crafted_file_selection_class_init(CraftedFileSelectionClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass *)class;
  widget_class = (GtkWidgetClass *)class;

  parent_class = gtk_type_class(gtk_hbox_get_type());

  crafted_file_selection_signals[FILENAME_CHANGED] =
    gtk_signal_new("filename_changed",
                   GTK_RUN_FIRST,
                   object_class->type,
                   GTK_SIGNAL_OFFSET(CraftedFileSelectionClass,
                                     filename_changed),
                   gtk_signal_default_marshaller, GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, crafted_file_selection_signals, LAST_SIGNAL);

  class->filename_changed = NULL;

  object_class->destroy = crafted_file_selection_destroy;
  widget_class->realize = crafted_file_selection_realize;
}

static void crafted_file_selection_init(CraftedFileSelection *cfs)
{
  cfs->title          = NULL;
  cfs->file_selection = NULL;
  cfs->check_valid    = FALSE;

  cfs->file_exists    = NULL;
  cfs->yes_pixmap     = NULL;
  cfs->yes_mask       = NULL;
  cfs->no_pixmap      = NULL;
  cfs->no_mask        = NULL;

  gtk_box_set_spacing(GTK_BOX(cfs), 2);
  gtk_box_set_homogeneous(GTK_BOX(cfs), FALSE);

  cfs->browse_button = gtk_button_new_with_label(" ... ");
  gtk_box_pack_end(GTK_BOX(cfs), cfs->browse_button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(cfs->browse_button), "clicked",
                     (GtkSignalFunc)crafted_file_selection_browse_callback,
                     cfs);
  gtk_widget_show(cfs->browse_button);

  cfs->entry = gtk_entry_new();
  gtk_box_pack_end(GTK_BOX(cfs), cfs->entry, TRUE, TRUE, 0);
  gtk_signal_connect(GTK_OBJECT(cfs->entry), "activate",
                     (GtkSignalFunc)crafted_file_selection_entry_callback, cfs);
  gtk_signal_connect(GTK_OBJECT(cfs->entry), "focus_out_event",
                     (GtkSignalFunc)crafted_file_selection_entry_focus_out_callback, cfs);
  gtk_widget_show(cfs->entry);
}

GtkType crafted_file_selection_get_type(void)
{
  static GtkType cfs_type = 0;

  if (!cfs_type)
  {
    GtkTypeInfo cfs_info =
    {
      "CraftedFileSelection",
      sizeof(CraftedFileSelection),
      sizeof(CraftedFileSelectionClass),
      (GtkClassInitFunc)crafted_file_selection_class_init,
      (GtkObjectInitFunc)crafted_file_selection_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc)NULL
    };

    cfs_type = gtk_type_unique(gtk_hbox_get_type(), &cfs_info);
  }

  return cfs_type;
}

/**
 * crafted_file_selection_new:
 * @title: The title of the #GtkFileSelection dialog.
 * @filename: The initial filename.
 * @dir_only: #TRUE if the file selection should accept directories only.
 * @check_valid: #TRUE if the widget should check if the entered file
 *               really exists.
 *
 * Creates a new #CraftedFileSelection widget.
 *
 * Returns: A pointer to the new #CraftedFileSelection widget.
 **/
GtkWidget *crafted_file_selection_new(const gchar *title,
                                      const gchar *filename,
                                      gboolean     dir_only,
                                      gboolean     check_valid)
{
  CraftedFileSelection *cfs;

  cfs = gtk_type_new(crafted_file_selection_get_type());

  cfs->title       = g_strdup(title);
  cfs->dir_only    = dir_only;
  cfs->check_valid = check_valid;

  crafted_file_selection_set_filename(cfs, filename);

  return GTK_WIDGET(cfs);
}

/**
 * crafted_file_selection_get_filename:
 * @cfs: The file selection you want to know the filename from.
 *
 * Note that you have to g_free() the returned string.
 *
 * Returns: The file or directory the user has entered.
 **/
gchar *crafted_file_selection_get_filename(CraftedFileSelection *cfs)
{
  g_return_val_if_fail(cfs != NULL, g_strdup(""));
  g_return_val_if_fail(CRAFTED_IS_FILE_SELECTION(cfs), g_strdup(""));

  return gtk_editable_get_chars(GTK_EDITABLE(cfs->entry), 0, -1);
}

/**
 * crafted_file_selection_set_filename:
 * @cfs: The file selection you want to set the filename for.
 * @filename: The new filename.
 *
 * If you specified @check_valid as #TRUE in crafted_file_selection_new()
 * the #CraftedFileSelection will immediately check the validity of the file
 * name.
 *
 */
void crafted_file_selection_set_filename(CraftedFileSelection *cfs,
                                         const gchar          *filename)
{
  g_return_if_fail(cfs != NULL);
  g_return_if_fail(CRAFTED_IS_FILE_SELECTION(cfs));

  gtk_entry_set_text(GTK_ENTRY(cfs->entry), filename ? filename : "");

  /*  update everything */
  crafted_file_selection_entry_callback(cfs->entry, (gpointer)cfs);
}

static void crafted_file_selection_realize(GtkWidget *widget)
{
  CraftedFileSelection *cfs;
  GtkStyle             *style;

  cfs = CRAFTED_FILE_SELECTION(widget);
  if (!cfs->check_valid)
    return;

  if (GTK_WIDGET_CLASS(parent_class)->realize)
    (* GTK_WIDGET_CLASS(parent_class)->realize)(widget);

  style = gtk_widget_get_style(widget);

  cfs->yes_pixmap = gdk_pixmap_create_from_xpm_d(widget->window,
                                                 &cfs->yes_mask,
                                                 &style->bg[GTK_STATE_NORMAL],
                                                 yes_xpm);
  cfs->no_pixmap = gdk_pixmap_create_from_xpm_d(widget->window,
                                                &cfs->no_mask,
                                                &style->bg[GTK_STATE_NORMAL],
                                                no_xpm);

  cfs->file_exists = gtk_pixmap_new(cfs->no_pixmap, cfs->no_mask);
  gtk_box_pack_start(GTK_BOX(cfs), cfs->file_exists, FALSE, FALSE, 0);

  crafted_file_selection_check_filename(cfs);
  gtk_widget_show(cfs->file_exists);
}

static void crafted_file_selection_entry_callback(GtkWidget *widget,
                                                  gpointer   data)
{
  CraftedFileSelection *cfs;
  gchar                *filename;
  gint                  len;

  cfs = CRAFTED_FILE_SELECTION(data);

  /*  filenames still need more sanity checking
   *  (erase double G_DIR_SEPARATORS, ...)
   */
  filename = gtk_editable_get_chars(GTK_EDITABLE(widget), 0, -1);
  filename = g_strstrip(filename);

  while (((len = strlen(filename)) > 1) &&
          (filename[len - 1] == G_DIR_SEPARATOR))
    filename[len - 1] = '\0';

  gtk_signal_handler_block_by_data(GTK_OBJECT(cfs->entry), cfs);
  gtk_entry_set_text(GTK_ENTRY(cfs->entry), filename);
  gtk_signal_handler_unblock_by_data(GTK_OBJECT(cfs->entry), cfs);

  if (cfs->file_selection)
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(cfs->file_selection),
                                    filename);
  g_free(filename);

  crafted_file_selection_check_filename(cfs);

  gtk_entry_set_position(GTK_ENTRY(cfs->entry), -1);

  gtk_signal_emit(GTK_OBJECT(cfs),
                  crafted_file_selection_signals[FILENAME_CHANGED]);
}

static gboolean crafted_file_selection_entry_focus_out_callback(GtkWidget *widget,
                                                                GdkEvent  *event,
                                                                gpointer   data)
{
  crafted_file_selection_entry_callback(widget, data);

  return TRUE;
}

/*  local callbacks of crafted_file_selection_browse_callback()  */
static void crafted_file_selection_filesel_ok_callback(GtkWidget *widget,
                                                       gpointer   data)
{
  CraftedFileSelection *cfs;
  gchar                *filename;

  cfs = CRAFTED_FILE_SELECTION(data);
  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(cfs->file_selection));

  gtk_entry_set_text(GTK_ENTRY(cfs->entry), filename);

  gtk_widget_hide(cfs->file_selection);

  /*  update everything  */
  crafted_file_selection_entry_callback(cfs->entry, data);
}

static void crafted_file_selection_browse_callback(GtkWidget *widget,
                                                   gpointer   data)
{
  CraftedFileSelection *cfs;
  gchar                *filename;

  cfs = CRAFTED_FILE_SELECTION(data);
  filename = gtk_editable_get_chars(GTK_EDITABLE(cfs->entry), 0, -1);

  if (cfs->file_selection == NULL)
  {
    if (cfs->dir_only)
    {
      cfs->file_selection = gtk_file_selection_new(cfs->title);

      /*  hiding these widgets uses internal gtk+ knowledge, but it's
       *  easier than creating my own directory browser -- michael
       */
      gtk_widget_hide(GTK_FILE_SELECTION(cfs->file_selection)->fileop_del_file);
      gtk_widget_hide(GTK_FILE_SELECTION(cfs->file_selection)->file_list->parent);
    } else
    {
      cfs->file_selection = gtk_file_selection_new(_("Select File"));
    }

    gtk_window_set_position(GTK_WINDOW(cfs->file_selection),
                            GTK_WIN_POS_MOUSE);
    gtk_window_set_wmclass(GTK_WINDOW(cfs->file_selection),
                           "file_select", "Crafted");

    /* slightly compress the dialog */
    gtk_container_set_border_width(GTK_CONTAINER(cfs->file_selection), 2);
    gtk_container_set_border_width(GTK_CONTAINER(GTK_FILE_SELECTION(cfs->file_selection)->button_area), 2);

    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(cfs->file_selection)->ok_button),
                       "clicked",
                       (GtkSignalFunc)crafted_file_selection_filesel_ok_callback, cfs);
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(cfs->file_selection)->selection_entry),
                       "activate",
                       (GtkSignalFunc)crafted_file_selection_filesel_ok_callback, cfs);
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(cfs->file_selection)->cancel_button),
                              "clicked",
                              (GtkSignalFunc)gtk_widget_hide,
                              GTK_OBJECT(cfs->file_selection));
    gtk_signal_connect_object(GTK_OBJECT(cfs), "unmap",
                              (GtkSignalFunc)gtk_widget_hide,
                              GTK_OBJECT(cfs->file_selection));
    gtk_signal_connect_object(GTK_OBJECT(cfs->file_selection),
                              "delete_event",
                              (GtkSignalFunc)gtk_widget_hide,
                              GTK_OBJECT(cfs->file_selection));
  }

  gtk_file_selection_set_filename(GTK_FILE_SELECTION(cfs->file_selection),
                                  filename);

  if (!GTK_WIDGET_VISIBLE(cfs->file_selection))
    gtk_widget_show(cfs->file_selection);
  else
    gdk_window_raise(cfs->file_selection->window);
}

static void crafted_file_selection_check_filename(CraftedFileSelection *cfs)
{
  static struct stat  statbuf;
  gchar              *filename;

  if (!cfs->check_valid)
    return;

  if (cfs->file_exists == NULL)
    return;

  filename = gtk_editable_get_chars(GTK_EDITABLE(cfs->entry), 0, -1);
  if ((stat(filename, &statbuf) == 0) &&
      (cfs->dir_only ? S_ISDIR(statbuf.st_mode) : S_ISREG(statbuf.st_mode)))
  {
    gtk_pixmap_set(GTK_PIXMAP(cfs->file_exists),
                   cfs->yes_pixmap, cfs->yes_mask);
  } else
  {
    gtk_pixmap_set(GTK_PIXMAP(cfs->file_exists),
                   cfs->no_pixmap, cfs->no_mask);
  }
  g_free(filename);
}

