/* ppc64.c -- core analysis suite
 *
 * Copyright (C) 2004 David Anderson
 * Copyright (C) 2004 Red Hat, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * CVS: $Revision: 1.4 $ $Date: 2004/05/11 16:02:57 $
 */
#ifdef PPC64
#include "defs.h"

static int ppc64_kvtop(struct task_context *, ulong, physaddr_t *, int);
static int ppc64_uvtop(struct task_context *, ulong, physaddr_t *, int);
static ulong ppc64_vmalloc_start(void);
static int ppc64_is_task_addr(ulong);
static int ppc64_verify_symbol(const char *, ulong, char);
static ulong ppc64_get_task_pgd(ulong);
static int ppc64_translate_pte(ulong, void *, ulonglong);
static ulong ppc64_processor_speed(void);
static int ppc64_eframe_search(struct bt_info *);
static void ppc64_back_trace_cmd(struct bt_info *);
static void ppc64_back_trace(struct gnu_request *, struct bt_info *);
static void get_ppc64_frame(struct bt_info *, ulong *, ulong *);
static void ppc64_print_stack_entry(int,struct gnu_request *,
	ulong, char *, struct bt_info *);
static void ppc64_exception_frame(ulong, struct bt_info *, struct gnu_request *);
static void ppc64_dump_irq(int);
static ulong ppc64_get_pc(struct bt_info *);
static ulong ppc64_get_sp(struct bt_info *);
static void ppc64_get_stack_frame(struct bt_info *, ulong *, ulong *);
static int ppc64_dis_filter(ulong, char *);
static void ppc64_cmd_mach(void);
static int ppc64_get_smp_cpus(void);
static void ppc64_display_machine_stats(void);
static void ppc64_dump_line_number(ulong);
static struct line_number_hook ppc64_line_number_hooks[];
static int ppc64_is_kvaddr(ulong);
static int ppc64_is_uvaddr(ulong, struct task_context *);
void ppc64_compiler_warning_stub(void);

/*
 *  Do all necessary machine-specific setup here.  This is called several
 *  times during initialization.
 */
void
ppc64_init(int when)
{
	switch (when)
	{
	case PRE_SYMTAB:
		machdep->verify_symbol = ppc64_verify_symbol;
                if (pc->flags & KERNEL_DEBUG_QUERY)
                        return;
                machdep->pagesize = memory_page_size();
                machdep->pageshift = ffs(machdep->pagesize) - 1;
                machdep->pageoffset = machdep->pagesize - 1;
                machdep->pagemask = ~((ulonglong)machdep->pageoffset);
		machdep->stacksize = machdep->pagesize * 2;
                if ((machdep->pgd = (char *)malloc(PAGESIZE())) == NULL)
                        error(FATAL, "cannot malloc pgd space.");
                if ((machdep->pmd = (char *)malloc(PAGESIZE())) == NULL)
                        error(FATAL, "cannot malloc pmd space.");
                if ((machdep->ptbl = (char *)malloc(PAGESIZE())) == NULL)
                        error(FATAL, "cannot malloc ptbl space.");
                machdep->last_pgd_read = 0;
                machdep->last_pmd_read = 0;
                machdep->last_ptbl_read = 0;
		machdep->verify_paddr = generic_verify_paddr;
		machdep->ptrs_per_pgd = PTRS_PER_PGD;
		break;

	case PRE_GDB:
	        machdep->kvbase = 0;
		machdep->identity_map_base = 0;
                machdep->is_kvaddr = ppc64_is_kvaddr;
                machdep->is_uvaddr = ppc64_is_uvaddr;
	        machdep->eframe_search = ppc64_eframe_search;
	        machdep->back_trace = ppc64_back_trace_cmd;
	        machdep->processor_speed = ppc64_processor_speed;
	        machdep->uvtop = ppc64_uvtop;
	        machdep->kvtop = ppc64_kvtop;
	        machdep->get_task_pgd = ppc64_get_task_pgd;
		machdep->get_stack_frame = ppc64_get_stack_frame;
		machdep->get_stackbase = generic_get_stackbase;
		machdep->get_stacktop = generic_get_stacktop;
		machdep->translate_pte = ppc64_translate_pte;
		machdep->memory_size = generic_memory_size;
		machdep->is_task_addr = ppc64_is_task_addr;
		machdep->dis_filter = ppc64_dis_filter;
		machdep->cmd_mach = ppc64_cmd_mach;
		machdep->get_smp_cpus = ppc64_get_smp_cpus;
		machdep->line_number_hooks = ppc64_line_number_hooks;
		machdep->value_to_symbol = generic_machdep_value_to_symbol;
		machdep->init_kernel_pgd = NULL;
		break;

	case POST_GDB:
		machdep->nr_irqs = 0;  /* TBD */
		machdep->vmalloc_start = ppc64_vmalloc_start;
		machdep->dump_irq = ppc64_dump_irq;
		machdep->hz = HZ;
		break;

	case POST_INIT:
		break;
	}
}

#ifndef KSYMS_START
#define KSYMS_START 1
#endif

void
ppc64_dump_machdep_table(ulong arg)
{
        int others; 
 
        others = 0;
        fprintf(fp, "              flags: %lx (", machdep->flags);
	if (machdep->flags & KSYMS_START)
		fprintf(fp, "%sKSYMS_START", others++ ? "|" : "");
	if (machdep->flags & SYSRQ)
		fprintf(fp, "%sSYSRQ", others++ ? "|" : "");
        fprintf(fp, ")\n");

	fprintf(fp, "             kvbase: %lx\n", machdep->kvbase);
	fprintf(fp, "  identity_map_base: %lx\n", machdep->kvbase);
        fprintf(fp, "           pagesize: %d\n", machdep->pagesize);
        fprintf(fp, "          pageshift: %d\n", machdep->pageshift);
        fprintf(fp, "           pagemask: %llx\n", machdep->pagemask);
        fprintf(fp, "         pageoffset: %lx\n", machdep->pageoffset);
	fprintf(fp, "          stacksize: %ld\n", machdep->stacksize);
        fprintf(fp, "                 hz: %d\n", machdep->hz);
        fprintf(fp, "                mhz: %ld\n", machdep->mhz);
        fprintf(fp, "            memsize: %lld (0x%llx)\n", 
		machdep->memsize, machdep->memsize);
	fprintf(fp, "               bits: %d\n", machdep->bits);
	fprintf(fp, "            nr_irqs: %d\n", machdep->nr_irqs);
        fprintf(fp, "      eframe_search: ppc64_eframe_search()\n");
        fprintf(fp, "         back_trace: ppc64_back_trace_cmd()\n");
        fprintf(fp, "    processor_speed: ppc64_processor_speed()\n");
        fprintf(fp, "              uvtop: ppc64_uvtop()\n");
        fprintf(fp, "              kvtop: ppc64_kvtop()\n");
        fprintf(fp, "       get_task_pgd: ppc64_get_task_pgd()\n");
	fprintf(fp, "           dump_irq: ppc64_dump_irq()\n");
        fprintf(fp, "    get_stack_frame: ppc64_get_stack_frame()\n");
        fprintf(fp, "      get_stackbase: generic_get_stackbase()\n");
        fprintf(fp, "       get_stacktop: generic_get_stacktop()\n");
        fprintf(fp, "      translate_pte: ppc64_translate_pte()\n");
	fprintf(fp, "        memory_size: generic_memory_size()\n");
	fprintf(fp, "      vmalloc_start: ppc64_vmalloc_start()\n");
	fprintf(fp, "       is_task_addr: ppc64_is_task_addr()\n");
	fprintf(fp, "      verify_symbol: ppc64_verify_symbol()\n");
	fprintf(fp, "         dis_filter: ppc64_dis_filter()\n");
	fprintf(fp, "           cmd_mach: ppc64_cmd_mach()\n");
	fprintf(fp, "       get_smp_cpus: ppc64_get_smp_cpus()\n");
        fprintf(fp, "          is_kvaddr: ppc64_is_kvaddr()\n");
        fprintf(fp, "          is_uvaddr: ppc64_is_uvaddr()\n");
        fprintf(fp, "       verify_paddr: generic_verify_paddr()\n");
        fprintf(fp, "    init_kernel_pgd: NULL\n");
        fprintf(fp, "  line_number_hooks: ppc64_line_number_hooks\n");
	fprintf(fp, "    value_to_symbol: generic_machdep_value_to_symbol()\n");
        fprintf(fp, "      last_pgd_read: %lx\n", machdep->last_pgd_read);
        fprintf(fp, "      last_pmd_read: %lx\n", machdep->last_pmd_read);
        fprintf(fp, "     last_ptbl_read: %lx\n", machdep->last_ptbl_read);
        fprintf(fp, "                pgd: %lx\n", (ulong)machdep->pgd);
        fprintf(fp, "                pmd: %lx\n", (ulong)machdep->pmd);
        fprintf(fp, "               ptbl: %lx\n", (ulong)machdep->ptbl);
	fprintf(fp, "       ptrs_per_pgd: %d\n", machdep->ptrs_per_pgd);
	fprintf(fp, "           machspec: %lx\n", (ulong)machdep->machspec);
}

/*
 *  Deal with non-traditional overlaying of kernel/user address spaces.
 */
static int 
ppc64_is_kvaddr(ulong addr)
{
        return (error(FATAL, "ppc64_is_kvaddr: TBD\n"));
}

static int 
ppc64_is_uvaddr(ulong addr, struct task_context *tc)
{
        return (error(FATAL, "ppc64_is_uvaddr: TBD\n"));
}


/*
 *  Translates a user virtual address to its physical address.  cmd_vtop()
 *  sets the verbose flag so that the pte translation gets displayed; all
 *  other callers quietly accept the translation.
 *
 *  This routine can also take mapped kernel virtual addresses if the -u flag
 *  was passed to cmd_vtop().  If so, it makes the translation using the
 *  kernel-memory PGD entry instead of swapper_pg_dir.
 */

static int
ppc64_uvtop(struct task_context *tc, ulong vaddr, physaddr_t *paddr, int verbose)
{
	return (error(FATAL, "ppc64_uvtop: TBD\n"));
}

/*
 *  Translates a kernel virtual address to its physical address.  cmd_vtop()
 *  sets the verbose flag so that the pte translation gets displayed; all
 *  other callers quietly accept the translation.
 */
static int
ppc64_kvtop(struct task_context *tc, ulong kvaddr, physaddr_t *paddr, int verbose)
{
        ulong *pgd;
        ulong *page_dir;
        ulong page_table;
	ulong pgd_index;
        ulong pgd_pte;
	ulong pte_offset;
        ulong pte;

        if (!IS_KVADDR(kvaddr))
                return FALSE;

	return (error(FATAL, "ppc64_kvtop: TBD\n"));

no_kpage:
        return FALSE;
}

/*
 *  Determine where vmalloc'd memory starts.
 */
static ulong
ppc64_vmalloc_start(void)
{
	return (first_vmalloc_address());
}

/*
 * 
 */
static int
ppc64_is_task_addr(ulong task)
{
	return (error(FATAL, "ppc64_is_task_addr: TBD\n"));
}


/*
 * 
 */
static ulong
ppc64_processor_speed(void)
{
	return (error(FATAL, "ppc64_processor_speed: TBD\n"));
}


/*
 *  Accept or reject a symbol from the kernel namelist.
 */
static int
ppc64_verify_symbol(const char *name, ulong value, char type)
{
	return (error(FATAL, "ppc64_verify_symbol: TBD\n"));
}


/*
 *  Get the relevant page directory pointer from a task structure.
 */
static ulong
ppc64_get_task_pgd(ulong task)
{
	return (error(FATAL, "ppc64_get_task_pgd: TBD\n"));
}


/*
 *  Translate a PTE, returning TRUE if the page is present.
 *  If a physaddr pointer is passed in, don't print anything.
 */
static int
ppc64_translate_pte(ulong pte, void *physaddr, ulonglong unused)
{
	return (error(FATAL, "ppc64_translate_pte: TBD\n"));
}


/*
 *  Look for likely exception frames in a stack.
 */

static int 
ppc64_eframe_search(struct bt_info *bt)
{
	return (error(FATAL, "ppc64_eframe_search: TBD\n"));
}

/*
 *  Unroll a kernel stack.
 */
static void
ppc64_back_trace_cmd(struct bt_info *bt)
{
	error(FATAL, "ppc64_back_trace_cmd: TBD\n");
}

/*
 *  Unroll the kernel stack using a minimal amount of gdb services.
 */
static void
ppc64_back_trace(struct gnu_request *req, struct bt_info *bt)
{
	error(FATAL, "ppc64_back_trace: TBD\n");
}

/*
 *  print one entry of a stack trace
 */
static void 
ppc64_print_stack_entry(int frame, 
		      struct gnu_request *req, 
		      ulong callpc, 	
		      char *name, 
		      struct bt_info *bt)
{
	error(FATAL, "ppc64_print_stack_entry: TBD\n");
}

/*
 *  Print exception frame information for ppc64
 */
static void
ppc64_exception_frame(ulong addr, struct bt_info *bt, struct gnu_request *req)
{
	error(FATAL, "ppc64_exception_frame: TBD\n");
}


/*
 *  Get a stack frame combination of pc and ra from the most relevent spot.
 */
static void
ppc64_get_stack_frame(struct bt_info *bt, ulong *pcp, ulong *spp)
{
	error(FATAL, "ppc64_get_stack_frame: TBD\n");
}


/*
 *  Get the saved ESP from a user-space copy of the kernel stack.
 */
static ulong
ppc64_get_sp(struct bt_info *bt)
{
	return (error(FATAL, "ppc64_get_sp: TBD\n"));
}

/*
 *  Get the saved PC from a user-space copy of the kernel stack.
 */
static ulong
ppc64_get_pc(struct bt_info *bt)
{
	return (error(FATAL, "ppc64_get_pc: TBD\n"));
}


/*
 *  Do the work for ppc64_get_sp() and ppc64_get_pc().
 */
static void
get_ppc64_frame(struct bt_info *bt, ulong *getpc, ulong *getsp)
{
	error(FATAL, "get_ppc64_frame: TBD\n");
}

/*
 *  Do the work for cmd_irq().
 */
static void 
ppc64_dump_irq(int irq)
{
	error(FATAL, "ppc64_dump_irq: TBD\n");
}

/*
 *  Filter disassembly output if the output radix is not gdb's default 10
 */
static int 
ppc64_dis_filter(ulong vaddr, char *inbuf)
{
        char buf1[BUFSIZE];
        char buf2[BUFSIZE];
        char *colon, *p1;
        int argc;
        char *argv[MAXARGS];
        ulong value;

	if (!inbuf) 
		return TRUE;
/*
 *  For some reason gdb can go off into the weeds translating text addresses,
 *  (on alpha -- not necessarily seen on ppc64) so this routine both fixes the 
 *  references as well as imposing the current output radix on the translations.
 */
	console("IN: %s", inbuf);

	colon = strstr(inbuf, ":");

	if (colon) {
		sprintf(buf1, "0x%lx <%s>", vaddr,
			value_to_symstr(vaddr, buf2, pc->output_radix));
		sprintf(buf2, "%s%s", buf1, colon);
		strcpy(inbuf, buf2);
	}

	strcpy(buf1, inbuf);
	argc = parse_line(buf1, argv);

	if ((FIRSTCHAR(argv[argc-1]) == '<') && 
	    (LASTCHAR(argv[argc-1]) == '>')) {
		p1 = rindex(inbuf, '<');
		while ((p1 > inbuf) && !STRNEQ(p1, " 0x")) 
			p1--;

		if (!STRNEQ(p1, " 0x"))
			return FALSE;
		p1++;

		if (!extract_hex(p1, &value, NULLCHAR, TRUE))
			return FALSE;

		sprintf(buf1, "0x%lx <%s>\n", value,	
			value_to_symstr(value, buf2, pc->output_radix));

		sprintf(p1, buf1);
	}

	console("    %s", inbuf);

	return TRUE;
}


/*
 *   Override smp_num_cpus if possible and necessary.
 */
int
ppc64_get_smp_cpus(void)
{
	return (error(FATAL, "ppc64_get_smp_cpus: TBD\n"));
}

/*
 *  Machine dependent command.
 */
void
ppc64_cmd_mach(void)
{
        int c;

        while ((c = getopt(argcnt, args, "")) != EOF) {
                switch(c)
                {
                default:
                        argerrs++;
                        break;
                }
        }

        if (argerrs)
                cmd_usage(pc->curcmd, SYNOPSIS);

        ppc64_display_machine_stats();
}

/*
 *  "mach" command output.
 */
static void
ppc64_display_machine_stats(void)
{
	error(FATAL, "ppc64_display_machine_stats: TBD\n");
}

static const char *hook_files[] = {
        "arch/ppc64/kernel/entry.S",
        "arch/ppc64/kernel/head.S",
        "arch/ppc64/kernel/semaphore.c"
};

#define ENTRY_S      ((char **)&hook_files[0])
#define HEAD_S       ((char **)&hook_files[1])
#define SEMAPHORE_C  ((char **)&hook_files[2])

static struct line_number_hook ppc64_line_number_hooks[] = {

       {NULL, NULL}    /* list must be NULL-terminated */
};

static void
ppc64_dump_line_number(ulong callpc)
{
	error(FATAL, "ppc64_dump_line_number: TBD\n");
}

void
ppc64_compiler_warning_stub(void)
{
        struct line_number_hook *lhp;
        char **p;

        lhp = &ppc64_line_number_hooks[0]; lhp++;
        p = ENTRY_S;
	ppc64_back_trace(NULL, NULL);
	get_ppc64_frame(NULL, NULL, NULL);
	ppc64_print_stack_entry(0, NULL, 0, NULL, NULL);
	ppc64_exception_frame(0, NULL, NULL);
	ppc64_get_pc(NULL);
	ppc64_get_sp(NULL);
	ppc64_dump_line_number(0);
}
#endif 
