"=====================================================================
" cream-statusline.vim
"
" Cream -- An easy-to-use configuration of the famous Vim text  editor
" [ http://cream.sourceforge.net ] Copyright (C) 2002-2004  Steve Hall
"
" License:
" This program is free software; you can redistribute it and/or modify
" it under the terms of the GNU General Public License as published by
" the Free Software Foundation; either version 2 of  the  License,  or
" (at your option) any later version.
" [ http://www.gnu.org/licenses/gpl.html ]
"
" This program is distributed in the hope that it will be useful,  but
" WITHOUT  ANY  WARRANTY;  without  even  the  implied   warranty   of
" MERCHANTABILITY or FITNESS FOR A PARTICULAR  PURPOSE.  See  the  GNU
" General Public License for more details.
"
" You should have received a copy of the GNU  General  Public  License
" along with  this  program;  if  not,  write  to  the  Free  Software
" Foundation,  Inc.,  59  Temple  Place  -  Suite  330,   Boston,   MA
" 02111-1307, USA.
"
" Description:
" o WARNING!! Statuslines prior to version 6.2.071 are unable to
"   handle more than 50 items. (An items is an "%" object, defined by
"   begin group, enclosee [, close group]). In fact, prior to some
"   version of 6.0.? this limit was even lower.
" o Close of a highlighting group (%*) is not necessary--Vim always
"   assumes the beginning of the next ends the previous.
" o Cream's statusline colors are dependent on colors definitions
"   elsewhere for User highlight groups 1-4, represented as "%N*" ...
"   "%*" corresponding to the highlight group "UserN"
" o Changing highlighting groups dynamically in the status bar has far
"   too much overhead and is too slow. So we:
"   * Preset all the highlight groups (elsewhere)
"   * Pair two calls to two corresponding ON/OFF functions for *each*
"     evaluation
"   * Return a value for the actual condition (displayed)
"   * Return empty for reverse (isn't displayed)
" o :set statusline= requires a very specific syntax and can not
"   accept a function since it is specially evaluated by Vim, probably
"   because it happens so often. Examples of illegal/non-functioning:
"
"     set statusline=Cream_statusline()
"     execute "set statusline=" . Cream_statusline()
"   
"   It's the internal components of the statusline that continually
"   get re-evaluated, not the entire statusline. (&stl is not
"   continually executed, only the components *within*.)

" show status line for all buffers
set laststatus=2

" evaluation functions
" path/file {{{1
function! Cream_statusline_path()
    call Cream_buffer_pathfile()
	" ignore path in gui (it's in titlebar)
	if has("gui_running")
		return " "
	endif
	" strip filename
	let path = fnamemodify(b:cream_pathfilename, ":h")
	" ensure trailing slash
	let path = Cream_path_addtrailingslash(path)
	return path
endfunction

" file state conditions
function! Cream_statusline_filestate()
	let state = ""

	" test read-only state once
	if !exists("b:cream_readonly")
		let b:cream_readonly = filewritable(b:cream_pathfilename)
	endif

	" help file
	if &buftype == "help"
		return 'H'
	" writable
	elseif b:cream_readonly == 0
		\ || &buftype == "nowrite"
		return '-'
	" modified
	elseif &modified != 0
		return '*'
	" unmodified
	else
		return ' '
	endif
endfunction

function! Cream_statusline_filename()
	if b:cream_pathfilename == ""
		return "(Untitled)"
	endif
	return fnamemodify(b:cream_pathfilename, ":t")
endfunction

" file properties {{{1
" grouped to preserve group count

" fileformat (three characters only)
function! Cream_statusline_fileformat()
	if &fileformat == ""
		return "--"
	else
		return &fileformat
	endif
		
endfunction

" fileencoding (three characters only)
function! Cream_statusline_fileencoding()
	if &fileencoding == ""
		if &encoding != ""
			return &encoding
		else
			return "--"
		endif
	else
		return &fileencoding
	endif
endfunction

" file type
function! Cream_statusline_filetype()
	if &filetype == ""
		return "--"
	else
		return &filetype
	endif
endfunction

function! Cream_statusline_fileinfo()

	return Cream_statusline_fileformat() . ":" .
		\  Cream_statusline_fileencoding() . ":" .
		\  Cream_statusline_filetype() 

endfunction


" specials {{{1

" indicate expert mode
function! Cream_statusline_expert()
	if exists("g:CREAM_EXPERTMODE") && g:CREAM_EXPERTMODE == 1
		return "expert "
	endif
	return ""
endfunction

" diff mode
function! Cream_statusline_diffmode()
	if exists("b:cream_diffmode")
		return "diff "
	endif
	return ""
endfunction

function! Cream_statusline_specials()
" this function collects multiple special states together so that we
" are able to minimize the number of items in the final statusline

	let myspecials = ""
	let myspecials = myspecials . Cream_statusline_expert()
	let myspecials = myspecials . Cream_statusline_diffmode()
	return myspecials

endfunction

" right side {{{1

" show invisibles
function! Cream_statusline_showON()
	if g:LIST == 1
		if     &encoding == "latin1"
			"return ""
			return nr2char(182)
		elseif &encoding == "utf-8"
		\ && v:version >= 602
		\ || v:version == 601
		\ && has("patch469")
			" decimal 182
			return nr2char(182)
		else
			return "$"
		endif
	else
		return ""
	endif
endfunction
function! Cream_statusline_showOFF()
	if g:LIST == 1
		return ""
	else
		if &encoding == "latin1"
			"return ""
			return nr2char(182)
		elseif &encoding == "utf-8"
		\ && v:version >= 602
		\ || v:version == 601
		\ && has("patch469")
			" decimal 182
			return nr2char(182)
		else
			return "$"
		endif
	endif
endfunction

" wrap
function! Cream_statusline_wrapON()
	if exists("g:CREAM_WRAP")
		if g:CREAM_WRAP == 1
			return "wrap"
		endif
	endif
	" wrap is on
	return ""
endfunction
function! Cream_statusline_wrapOFF()
	if exists("g:CREAM_WRAP")
		if g:CREAM_WRAP != 1
			return "wrap"
		endif
	endif
	" wrap is on
	return ""
endfunction

" Auto Wrap
function! Cream_statusline_autowrapON()
	if exists("g:CREAM_AUTOWRAP")
		if g:CREAM_AUTOWRAP == 1
			" let's just use the number
			return "auto"
		endif
	endif
	" wrap is off or uninitialized
	return ""
endfunction
function! Cream_statusline_autowrapOFF()
	if exists("g:CREAM_AUTOWRAP")
		if g:CREAM_AUTOWRAP != 1
			return "auto"
		endif
	endif
	" wrap is on or uninitialized
	return ""
endfunction

" OBSOLETE
"" return (3 digit) autowidth setting
"" * g:CREAM_AUTOWRAP (1 == on, 0 == off)
"function! Cream_statusline_autowrap_widthON()
"    if exists("g:CREAM_AUTOWRAP")
"        if g:CREAM_AUTOWRAP == 1
"            if exists("g:CREAM_AUTOWRAP_WIDTH")
"                if g:CREAM_AUTOWRAP_WIDTH < 1000
"                    " let's skip the equal mark
"                    let myreturn = g:CREAM_AUTOWRAP_WIDTH
"                    "while strlen(myreturn) < 4
"                    "    let myreturn = ' ' . myreturn
"                    "endwhile
"                    return myreturn
"                else
"                    return 'max'
"                endif
"            endif
"        else
"                " CREAM_AUTOWRAP width unset yet by user (show in "off" colors)
"                return ""
"            endif
"        else
"            return ""
"        endif
"    else
"        " error
"    endif
"endfunction
"function! Cream_statusline_autowrap_widthOFF()
"    if exists("g:CREAM_AUTOWRAP")
"        if g:CREAM_AUTOWRAP == 0
"            if exists("g:CREAM_AUTOWRAP_WIDTH")
"                if g:CREAM_AUTOWRAP_WIDTH < 1000
"                    let myreturn = g:CREAM_AUTOWRAP_WIDTH
"                    "while strlen(myreturn) < 4
"                    "    let myreturn = ' ' . myreturn
"                    "endwhile
"                    return myreturn
"                else
"                    " let's skip the equal mark
"                    return 'max'
"                endif
"            else
"                " CREAM_AUTOWRAP width unset yet by user (show in "off" colors)
"                return ""
"            endif
"        else
"            return ""
"        endif
"    else
"        " error
"    endif
"endfunction

" wrap width
function! Cream_statusline_wrap_width()
" return wrap width setting
    " (don't check existance of g:CREAM_AUTOWRAP)
    if exists("g:CREAM_AUTOWRAP_WIDTH")
        return g:CREAM_AUTOWRAP_WIDTH
    endif
endfunction

" justification
function! Cream_statusline_wrap_justifyON()
" return justification mode if not "left"
	if !exists("g:cream_justify")
		return ""
	endif
	if     g:cream_justify == "center"
		return "cntr"
	elseif g:cream_justify == "right"
		return "rght"
	elseif g:cream_justify == "full"
		return "full"
	else
		return ""
	endif
endfunction
function! Cream_statusline_wrap_justifyOFF()
" return justification mode if not "left"
	if !exists("g:cream_justify")
		return "left"
	endif
	if     g:cream_justify == "left"
		return "left"
	else
		return ""
	endif
endfunction

" tabs
" &expandtab
function! Cream_statusline_expandtabON()
	if &expandtab == 0
		return "tabs"
	else
		return ""
	endif
endfunction
function! Cream_statusline_expandtabOFF()
	if &expandtab == 0
		return ""
	else
		return "tabs"
	endif
endfunction

" tabstop and softtabstop
function! Cream_statusline_tabstop()
	let str = ""
	if exists("g:CREAM_TABSTOP")
		let str = str . g:CREAM_TABSTOP
	endif
	if exists("g:CREAM_SOFTTABSTOP")
		let str = str . ":" . g:CREAM_SOFTTABSTOP
	endif
	return str
endfunction

" autoindent
function! Cream_statusline_autoindentON()
	if exists("g:CREAM_AUTOINDENT")
		if g:CREAM_AUTOINDENT == "1"
			return "indt"
		else
			return ""
		endif
	else
		" wrap is on if never initialized
		return "wrap"
	endif
endfunction
function! Cream_statusline_autoindentOFF()
	if exists("g:CREAM_AUTOINDENT")
		if g:CREAM_AUTOINDENT == "1"
			return ""
		else
			return "indt"
		endif
	else
		" autoindent is on if never initialized
		return ""
	endif
endfunction

" mode (Insert/Visual/Select/Replace v. Normal)
" ( see :help mode() for return strings)
function! Cream_statusline_modeNO()
	let mymode = mode()
	if mymode ==? "i"
		return ""
	elseif mymode ==? "v"
		return ""
	elseif mymode ==? "s"
		return ""
	elseif mymode ==? "R"
		return ""
	elseif mymode == ""
		return ""
	elseif mymode ==? "n"
		return "N"
	else
		return "  " . mymode . "  "
	endif
endfunction
"function! Cream_statusline_modeCOL()
"    let mymode = mode()
"    else
"        return ""
"    endif
"endfunction
function! Cream_statusline_modeOK()
	let mymode = mode()
	if     mymode ==? "i"
		return "I"
	elseif mymode ==? "v"
		return "V"
	elseif mymode ==? "s"
		return "S"
	elseif mymode ==? "R"
		return "R"
	elseif mymode == ""
		return "C"
	else
		return ""
	endif
endfunction

function! Cream_statusline_bufsize()
	let bufsize = line2byte(line("$") + 1) - 1
    " prevent negative numbers (non-existant buffers)
    if bufsize < 0
        let bufsize = 0
    endif
	" add commas
	let remain = bufsize
	let bufsize = ""
	while strlen(remain) > 3
		let bufsize = "," . strpart(remain, strlen(remain) - 3) . bufsize
		let remain = strpart(remain, 0, strlen(remain) - 3)
	endwhile
	let bufsize = remain . bufsize
    " too bad we can't use "" (nr2char(1068)) :)
    let char = "b"
	return bufsize . char
endfunction

" 1}}}

" utility function ("real time") {{{1

function! Cream_cursor_pos(mode, cursor)
    let b:cursor_{a:mode} = a:cursor
    return ""
endfunction

" 1}}}

" set statusline {{{1

set statusline=
    \%{Cream_cursor_pos(mode(),virtcol(\".\"))}
	\%1*%{Cream_statusline_path()}
	\%2*%{Cream_statusline_filename()}
	\\ %3*%{Cream_statusline_filestate()}
	\%1*\|%{Cream_statusline_fileinfo()}\|
    \%{Cream_statusline_bufsize()}\ %=
	\%3*%{Cream_statusline_specials()}
	\%1*\|
	\%2*%{Cream_statusline_showON()}
	\%1*%{Cream_statusline_showOFF()}\|
	\%2*%{Cream_statusline_wrapON()}
	\%1*%{Cream_statusline_wrapOFF()}:
	\%2*%{Cream_statusline_autowrapON()}
	\%1*%{Cream_statusline_autowrapOFF()}:
	\%3*%{Cream_statusline_wrap_justifyON()}
	\%1*%{Cream_statusline_wrap_justifyOFF()}:
	\%{Cream_statusline_wrap_width()}\|
	\%2*%{Cream_statusline_expandtabON()}
	\%1*%{Cream_statusline_expandtabOFF()}:
	\%{Cream_statusline_tabstop()}:
	\%2*%{Cream_statusline_autoindentON()}
	\%1*%{Cream_statusline_autoindentOFF()}\|
	\%4*%{Cream_statusline_modeNO()}
	\%1*%{Cream_statusline_modeOK()}\|
	\%05(%l%),%03(%v%)
	\%2*\ %P



" we don't really need to express column mode
    "\%3*%{Cream_statusline_modeCOL()}

" set statusline 2 {{{1
" (can handle more objects since only half are returned)

"function! Cream_statusline_set()
"" returns a valid string for the statusline
"
"    let statusline = ""
"
"    let statusline = statusline . '%1*%{Cream_statusline_path()}'
"
"    let statusline = statusline . '%2*%{Cream_statusline_filename()}'
"    let statusline = statusline . '\ %3*%{Cream_statusline_filestate()}'
"    let statusline = statusline . '%1*[%{Cream_statusline_fileformat()}:'
"    let statusline = statusline . '%{Cream_statusline_fileencoding()}:'
"    let statusline = statusline . '%{Cream_statusline_filetype()}]\ %='
"
"    if Cream_statusline_expert() != ""
"        let statusline = statusline . '%3*%{Cream_statusline_expert()}'
"    endif
"    let statusline = statusline . '\ '
"    if Cream_statusline_diffmode() != ""
"        let statusline = statusline . '%3*%{Cream_statusline_diffmode()}'
"    endif
"
"    let statusline = statusline . '%1*\|'
"
"    if Cream_statusline_showON() != ""
"        let statusline = statusline . '%2*%{Cream_statusline_showON()}'
"        let statusline = statusline . '%1*\|'
"    else
"        let statusline = statusline . '%{Cream_statusline_showOFF()}\|'
"    endif
"
"    if Cream_statusline_wrapON() != ""
"        let statusline = statusline . '%2*%{Cream_statusline_wrapON()}'
"        let statusline = statusline . '%1*:'
"    else
"        let statusline = statusline . '%{Cream_statusline_wrapOFF()}:'
"    endif
"
"    if Cream_statusline_autowrapON() != ""
"        let statusline = statusline . '%2*%{Cream_statusline_autowrapON()}'
"        let statusline = statusline . '%1*:'
"    else
"        let statusline = statusline . '%{Cream_statusline_autowrapOFF()}:'
"    endif
"
"    if Cream_statusline_wrap_justifyON() != ""
"        let statusline = statusline . '%3*%{Cream_statusline_wrap_justifyON()}'
"        let statusline = statusline . '%1*:'
"    else
"        let statusline = statusline . '%{Cream_statusline_wrap_justifyOFF()}:'
"    endif
"    
"    let statusline = statusline . '%{Cream_statusline_wrap_width()}\|'
"    
"    if Cream_statusline_autoindentON() != ""
"        let statusline = statusline . '%2*%{Cream_statusline_autoindentON()}'
"        let statusline = statusline . '%1*\|'
"    else
"        let statusline = statusline . '%{Cream_statusline_autoindentOFF()}\|'
"    endif
"
"    if Cream_statusline_modeNO() != ''
"        let statusline = statusline . '%4*%{Cream_statusline_modeNO()}'
"    endif
"    if Cream_statusline_modeCOL() != ''
"        let statusline = statusline . '%3*%{Cream_statusline_modeCOL()}'
"    endif
"    if Cream_statusline_modeOK() != ''
"        let statusline = statusline . '%1*%{Cream_statusline_modeOK()}'
"    endif
"
"    let statusline = statusline . '%1*%{Cream_statusline_bufsize()}b\|'
"    let statusline = statusline . '%05(%l%),%03(%c%)'
"    let statusline = statusline . '%2*\ %P'
"
"    return statusline
"
"endfunction
"execute "set statusline=" . Cream_statusline_set()


" set statusline 3 {{{1
" attempt at bottom line statusline with "ruler"
"execute "set rulerformat=%" . winwidth(0) . "(" . 
"    \"%1*%{expand('%:p:h')}/" .
"    \"%2*%{Cream_statusline_filename()}\\ " .
"    \"%3*%{Cream_statusline_filestate()}" .
"    \"%1*\\ [%{Cream_statusline_fileformat()}]\\ %=" .
"    \"%3*%{Cream_statusline_expert()}" .
"    \"%{Cream_statusline_diffmode()}" .
"    \"%1*\\|" .
"    \"%2*%{Cream_statusline_showON()}" .
"    \"%1*%{Cream_statusline_showOFF()}\\|" .
"    \"%2*%{Cream_statusline_wrapON()}" .
"    \"%1*%{Cream_statusline_wrapOFF()}:" .
"    \"%2*%{Cream_statusline_autowrapON()}" .
"    \"%1*%{Cream_statusline_autowrapOFF()}:" .
"    \"%{Cream_statusline_wrap_width()}\\|" .
"    \"%4*%{Cream_statusline_modeNO()}" .
"    \"%3*%{Cream_statusline_modeCOL()}" .
"    \"%1*%{Cream_statusline_modeOK()}\\|" .
"    \"%1*%{Cream_statusline_bufsize()}\\ bytes\\|" .
"    \"%1*%05(%l%),%03(%c%)" .
"    \"%2*\\ %P" .
"    \"%)"
"set ruler
"set statusline=
"set laststatus=0

" set statusline 4 {{{1
"function! Cream_statusline()
"" wrapper for statusline so string is evaluated, not taken literally
"    return Cream_statusline_state()
"endfunction

function! Cream_statusline()
" returns a valid statusline statement

	let sl = ""

	" path
	let sl = sl . '%1*%{Cream_statusline_path()}'
	" filename
	let sl = sl . '%2*%{Cream_statusline_filename()}'
	" filestate (read only, non-existing)
	let sl = sl . '\ %3*%{Cream_statusline_filestate()}'
	" fileformat
	let sl = sl . '%1*[%{Cream_statusline_fileformat()}:'
	" encoding
	let sl = sl . '%{Cream_statusline_fileencoding()}:'
	" filetype
	let sl = sl . '%{Cream_statusline_filetype()}]'

	" right justified
	let sl = sl . '\ %='
	" special states (diff, expert mode, etc.)
	let sl = sl . '%3*%{Cream_statusline_specials()}'

	" show invisibles
	"if Cream_statusline_showON() != ""
	"    let sl = sl . '%1*\|%2*%{Cream_statusline_showON()}%1*\|'
	"else
	"    let sl = sl . '%1*\|%{Cream_statusline_showOFF()}\|'
	"endif
	let sl = sl . Cream_statusline_show_eval()

	" wrap
	let sl = sl . '%2*%{Cream_statusline_wrapON()}'
	let sl = sl . '%1*%{Cream_statusline_wrapOFF()}:'
	" auto wrap
	let sl = sl . '%2*%{Cream_statusline_autowrapON()}'
	let sl = sl . '%1*%{Cream_statusline_autowrapOFF()}:'
	" justification
	let sl = sl . '%3*%{Cream_statusline_wrap_justifyON()}'
	let sl = sl . '%1*%{Cream_statusline_wrap_justifyOFF()}:'
	" wrap width
	let sl = sl . '%{Cream_statusline_wrap_width()}\|'

	" expand tabs
	let sl = sl . '%2*%{Cream_statusline_expandtabON()}'
	let sl = sl . '%1*%{Cream_statusline_expandtabOFF()}\|'

	" autoindent
	let sl = sl . '%2*%{Cream_statusline_autoindentON()}'
	let sl = sl . '%1*%{Cream_statusline_autoindentOFF()}\|'

	" mode indication
	let sl = sl . '%4*%{Cream_statusline_modeNO()}'
	let sl = sl . '%1*%{Cream_statusline_modeOK()}'

	" file size
	let sl = sl . '%{Cream_statusline_bufsize()}b\|'

	" position, line and column 
	let sl = sl . '%05(%l%),%03(%v%)'
	" position percentage 
	let sl = sl . '%2*\ %P'

	return sl

endfunction
"set statusline=%{Cream_statusline()}
"execute "set statusline=" . Cream_statusline()

" 1}}}
" vim:foldmethod=marker
