// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// path.cpp -- pathfinding functions
////////////////////////////////////////////////////////////////////////

#include "path.h"

////////////////////////////////////////////////////////////////////////
// NAME       : BasicPath::BasicPath
// DESCRIPTION: Create a new path object.
// PARAMETERS : map    - map to use for pathfinding
//              buffer - buffer to store path in; if non-NULL must be
//                       large enough to hold at least (map height x
//                       map width) bytes. If NULL, a new buffer will
//                       be allocated and freed when the path object is
//                       destroyed.
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

BasicPath::BasicPath( Map *map, signed char *buffer ) {
  this->map = map;
  buf_private = false;

  SetBuffer( buffer );
  Clear();
}

////////////////////////////////////////////////////////////////////////
// NAME       : BasicPath::~BasicPath
// DESCRIPTION: Destroy the Path object.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

BasicPath::~BasicPath( void ) {
  if ( buf_private ) delete [] path;
}

////////////////////////////////////////////////////////////////////////
// NAME       : BasicPath::Clear
// DESCRIPTION: Clear the path buffer.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void BasicPath::Clear( void ) {
  unsigned short size = map->Width() * map->Height();
  for ( int i = 0; i < size; ++i ) path[i] = -1;
}

////////////////////////////////////////////////////////////////////////
// NAME       : BasicPath::SetBuffer
// DESCRIPTION: Set a new path buffer.
// PARAMETERS : buffer - new buffer. If NULL a new buffer is allocated.
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void BasicPath::SetBuffer( signed char *buffer ) {
  if ( buf_private ) delete [] path;

  if ( !buffer ) {
    path = new signed char [map->Width() * map->Height()];
    buf_private = true;
  } else {
    path = buffer;
    buf_private = false;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : BasicPath::Find
// DESCRIPTION: Search a path on the map.
// PARAMETERS : u     - unit to search a path for; this may be NULL
//                      only in some special cases (see subclasses)
//              start - hex to start from (can be different from the
//                      unit's current position)
//              end   - destination hex
//              qual  - path quality/speed trade-off (1 is best, 10 is
//                      worst but fastest); Default PATH_BEST.
//              off   - sometimes you don't want to reach the target
//                      hex proper but only get close to it. This is
//                      the maximum distance to the destination for a
//                      path to be considered valid. Default 0.
// RETURNS    : approximate number of turns to reach the destination
//              hex, or -1 if no valid path was found
////////////////////////////////////////////////////////////////////////

short BasicPath::Find( const Unit *u, const Point &start, const Point &end,
                       unsigned char qual, unsigned char off ) {
  PathNode *pnode;
  List openlist, donelist;
  this->start = start;
  this->end = end;

  // to begin, insert the starting node in openlist
  pnode = new PathNode;
  if ( !pnode ) return -1;

  // starting from the destination would make it a bit easier to record the actual path
  // but this way we can support going towards the target until we have reached a certain
  // distance which is rather useful for AI routines...
  pnode->pos = start;
  pnode->eta = Distance( start, end ) * qual;
  pnode->cost = 0;
  pnode->switched = false;
  openlist.AddHead( pnode );

  while( !openlist.IsEmpty() ) {
    pnode = GetBestNode( openlist );
    pnode->Remove();
    donelist.AddHead( pnode );

    // check for destination
    if ( StopSearch( pnode, off ) ) break;

    // not there yet, so let's look at the hexes around it
    for ( short dir = NORTH; dir <= NORTHWEST; ++dir ) {
      Point p;

      // get the hex in that direction
      if ( !map->Dir2Hex( pnode->pos, (Direction)dir, p ) ) {
        PathNode *pnode2 = GetCost( u, pnode, p );
        if ( pnode2 ) {
          pnode2->eta = Distance( p, end ) * qual;
          MarkBuffer( pnode2, (Direction)dir, openlist );
        }
      }
    }
  }

  // check if we reached our destination
  pnode = static_cast<PathNode *>( donelist.Head() );
  return (pnode ? RecordPath( u, pnode, off ) : -1 );
}

////////////////////////////////////////////////////////////////////////
// NAME       : BasicPath::GetStep
// DESCRIPTION: Get the path buffer content at a hex position. Usually,
//              it will contain the direction of the next step on the
//              path.
// PARAMETERS : current - hex position to look at
// RETURNS    : content of path buffer at the position of 'current'
////////////////////////////////////////////////////////////////////////

signed char BasicPath::GetStep( const Point &current ) const {
  return path[map->Hex2Index(current)];
}

////////////////////////////////////////////////////////////////////////
// NAME       : BasicPath::StepsToDest
// DESCRIPTION: Get the number of steps on the path from a given hex to
//              the destination.
// PARAMETERS : pos - hex position to start from
// RETURNS    : number of steps until destination is reached, or 0 if
//              no path available or already there
////////////////////////////////////////////////////////////////////////

unsigned short BasicPath::StepsToDest( const Point &pos ) const {
  Point p( pos );
  short index = map->Hex2Index( p );
  Direction dir = (Direction)path[index];
  unsigned short steps = 0;

  while ( dir != -1 ) {
    map->Dir2Hex( p, dir, p );
    index = map->Hex2Index( p );
    dir = (Direction)path[index];
    ++steps;
  }
  return steps;
}


// PATH - a Path is the object used for normal unit pathfinding,
//        trying to move a unit from A to B
//
//
////////////////////////////////////////////////////////////////////////
// NAME       : Path::GetBestNode
// DESCRIPTION: The best node for standard pathfinding is the one with
//              the smallest sum of cost and eta.
// PARAMETERS : open - list of open nodes
// RETURNS    : best node
////////////////////////////////////////////////////////////////////////

PathNode *Path::GetBestNode( const List &open ) const {
  PathNode *p1, *p2;

  p1 = static_cast<PathNode *>( open.Head() );
  p2 = static_cast<PathNode *>( p1->Next() );
  while ( p2 ) {
    if ( (p2->cost + p2->eta) < (p1->cost + p1->eta) ) p1 = p2;
    p2 = static_cast<PathNode *>( p2->Next() );
  }
  return p1;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Path::StopSearch
// DESCRIPTION: For the normal pathfinder, the search ends when the
//              destination is reached (or we're sufficiently close).
// PARAMETERS : next - path node to be checked next if search is not
//                     cancelled
//              off  - minimum distance to end to abort
// RETURNS    : TRUE if search aborted, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool Path::StopSearch( const PathNode *next, unsigned char off ) const {
  return Distance( next->pos, end ) <= off;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Path::GetCost
// DESCRIPTION: Calculate the cost for the unit to move from one hex to
//              another.
// PARAMETERS : u       - unit
//              pn_from - path node for source hex
//              to      - destination hex (adjacent to from)
// RETURNS    : new PathNode or NULL if step is not allowed
////////////////////////////////////////////////////////////////////////

PathNode *Path::GetCost( const Unit *u, const PathNode *pn_from, const Point &to ) const {
  unsigned short sofar = pn_from->cost, state;
  short cost = map->MoveCost( u, pn_from->pos, to, state );

  if ( state != 0 ) {
    if ( !map->GetMapObject( to ) || (to == end) )
      cost = MAX( u->Moves() - sofar, cost );
    else cost = -1;
  }

  if ( cost >= 0 ) {
    PathNode *ret = new PathNode;
    if ( ret ) {
      ret->pos = to;
      ret->cost = sofar + cost;
      return ret;
    }
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Path::MarkBuffer
// DESCRIPTION: When a hex has been checked and was found ok to move
//              onto, this hex must be properly marked in the path
//              buffer and possibly in the list of nodes, too.
// PARAMETERS : next - node which is to be added to the open list
//              dir  - direction of the step to reach next node
//              open - list of open nodes
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void Path::MarkBuffer( PathNode *next, Direction dir, List &open ) const {
  unsigned short index = map->Hex2Index( next->pos );
  PathNode *p;

  // if the new hex is not marked in path yet, mark it and put it in openlist
  if ( path[index] == -1 ) {
    path[index] = ReverseDir( dir );
    open.AddTail( next );
  } else {
    // try to find the node in openlist and replace it if new cost is smaller
    p = static_cast<PathNode *>( open.Head() );
    while ( p ) {
      if ( p->pos == next->pos ) {   // found the node. shall we replace it?
        if ( p->cost > next->cost ) {
          p->Remove();
          delete p;
          path[index] = ReverseDir( dir );
          open.AddTail( next );
          next = NULL;
        }
        break;
      } else p = static_cast<PathNode *>( p->Next() );
    }
    delete next;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Path::RecordPath
// DESCRIPTION: Check whether a path has been found, prepare the buffer
//              and set the return code of the pathfinder.
// PARAMETERS : u    - unit searching the path
//              last - final path node
//              off  - maximum allowed distance to destination
// RETURNS    : cost of the path in turns for the unit (approximately),
//              or -1 if no valid path was found
////////////////////////////////////////////////////////////////////////

short Path::RecordPath( const Unit *u, const PathNode *last,
                        unsigned char off ) const {
  short turns = -1;
  Point p = last->pos;
  if ( Distance( p, end ) <= off ) {
    // correct the path
    // move back to beginning
    short index = map->Hex2Index( p );
    Direction dir = (Direction)path[index], dir2;
    path[index] = -1;

    while ( p != start ) {
      map->Dir2Hex( p, dir, p );
      index = map->Hex2Index( p );
      dir2 = (Direction)path[index];
      path[index] = ReverseDir( dir );
      dir = dir2;
    }

    if ( u->Type()->Speed() == 0 ) turns = 100;
    else turns = (last->cost + u->Type()->Speed() - 1) / u->Type()->Speed();
  }
  return turns;
}



// MOVE SHADER - this is used to create the buffer for shading of
//               illegal moves when selecting a unit
//
//
////////////////////////////////////////////////////////////////////////
// NAME       : MoveShader::GetBestNode
// DESCRIPTION: For the shader the order of nodes doesn't matter. Nodes
//              with a too high cost are not added to the list, and all
//              others must be processed anyway. Therefore we always
//              use the list head.
// PARAMETERS : open - list of open nodes
// RETURNS    : best node
////////////////////////////////////////////////////////////////////////

PathNode *MoveShader::GetBestNode( const List &open ) const {
  PathNode *p1, *p2;

  p1 = static_cast<PathNode *>( open.Head() );
  p2 = static_cast<PathNode *>( p1->Next() );
  while ( p2 ) {
    if ( p2->cost < p1->cost ) p1 = p2;
    p2 = static_cast<PathNode *>( p2->Next() );
  }
  return p1;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MoveShader::StopSearch
// DESCRIPTION: The shader never stops the serach when there are nodes
//              left in the open list.
// PARAMETERS : next - path node to be checked next if search is not
//                     cancelled
//              off  - minimum distance to end to abort
// RETURNS    : TRUE if search aborted, FALSE otherwise
////////////////////////////////////////////////////////////////////////

bool MoveShader::StopSearch( const PathNode *next, unsigned char off ) const {
  return false;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MoveShader::GetCost
// DESCRIPTION: Calculate the cost for the unit to move from one hex to
//              another.
// PARAMETERS : u       - unit
//              pn_from - path node for source hex
//              to      - destination hex (adjacent to from)
// RETURNS    : next PathNode or NULL on illegal path
////////////////////////////////////////////////////////////////////////

PathNode *MoveShader::GetCost( const Unit *u, const PathNode *pn_from, const Point &to ) const {
  unsigned short sofar = pn_from->cost;
  unsigned short state;
  short cost = map->MoveCost( u, pn_from->pos, to, state );

  if ( cost > u->Moves() - sofar ) cost = -1;
  else if ( state != 0 ) cost = u->Moves() - sofar;

  if ( cost >= 0 ) {
    PathNode *ret = new PathNode;
    if ( ret ) {
      ret->pos = to;
      ret->cost = pn_from->cost + cost;
      return ret;
    }
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MoveShader::MarkBuffer
// DESCRIPTION: When a hex has been checked and was found ok to move
//              onto, this hex must be properly marked in the path
//              buffer and possibly in the list of nodes, too.
// PARAMETERS : next - node which is to be added to the open list
//              dir  - direction of the step to reach next node
//              open - list of open nodes
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MoveShader::MarkBuffer( PathNode *next, Direction dir, List &open ) const {
  unsigned short index = map->Hex2Index( next->pos );
  PathNode *p;

  // if the new hex is not marked in path yet, mark it and put it in openlist
  if ( path[index] == -1 ) {
    path[index] = 1;
    open.AddTail( next );
  } else {
    // try to find the node in openlist and replace it if new cost is smaller
    p = static_cast<PathNode *>( open.Head() );
    while ( p ) {
      if ( p->pos == next->pos ) {   // found the node. shall we replace it?
        if ( p->cost > next->cost ) {
          p->Remove();
          delete p;
          open.AddTail( next );
          next = NULL;
        }
        break;
      } else p = static_cast<PathNode *>( p->Next() );
    }
    delete next;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : MoveShader::RecordPath
// DESCRIPTION: The shader doesn't have a path to prepare, but we must
//              check for valid targets which must not be shaded.
// PARAMETERS : u    - unit searching the path
//              last - final path node
//              off  - maximum allowed distance to destination
// RETURNS    : always 1
////////////////////////////////////////////////////////////////////////

short MoveShader::RecordPath( const Unit *u, const PathNode *last,
                              unsigned char off ) const {
  for ( Unit *target = static_cast<Unit *>(units.Head());
        target; target = static_cast<Unit *>(target->Next()) ) {
    if ( u->CanHit( target ) )
      path[map->Hex2Index( target->Position() )] = 1;
  }
  path[map->Hex2Index( u->Position() )] = 1;

  return 1;
}


////////////////////////////////////////////////////////////////////////
// NAME       : MinesweeperShader::GetCost
// DESCRIPTION: See whether there's a mine to be swept on a hex next to
//              the mine-sweeper unit. Enemy mines can only be cleared
//              if there is no other enemy unit (excluding mines) on a
//              field adjacent to the mine.
// PARAMETERS : u       - mine-sweeper unit
//              pn_from - path node for source hex
//              to      - destination hex (adjacent to from)
// RETURNS    : next PathNode or NULL on illegal path (no mine)
////////////////////////////////////////////////////////////////////////

PathNode *MinesweeperShader::GetCost( const Unit *u, const PathNode *pn_from, const Point &to ) const {
  if ( NextTo( u->Position(), to ) && (u->Terrain() & map->TerrainTypes(to)) ) {
    Unit *m = map->GetUnit( to );

    if ( m && m->IsMine() ) {
      bool enemy = false;

      if ( m->Owner() != u->Owner() ) {
        Point adj[6];
        map->GetNeighbors( to, adj );
        for ( int i = NORTH; (i <= NORTHWEST) && !enemy; ++i ) {
          if ( adj[i].x != -1 ) {
            Unit *e = map->GetUnit( adj[i] );
            if ( e && (e->Owner() == m->Owner()) && !e->IsMine() ) enemy = true;
          }
        }
      }

      if ( !enemy ) {
        PathNode *ret = new PathNode;
        if ( ret ) {
          ret->pos = to;
          ret->cost = 0;
          return ret;
        }
      }
    }
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MinesweeperShader::MarkBuffer
// DESCRIPTION: When a hex has been checked and was found ok to move
//              onto, this hex must be properly marked in the path
//              buffer.
// PARAMETERS : next - node which is to be added to the open list
//              dir  - direction of the step to reach next node
//              open - list of open nodes
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void MinesweeperShader::MarkBuffer( PathNode *next, Direction dir, List &open ) const {
  unsigned short index = map->Hex2Index( next->pos );

  // if the new hex is not marked in path yet, mark it and put it in openlist
  if ( path[index] == -1 ) {
    path[index] = 1;
    open.AddTail( next );
  } else delete next;
}

////////////////////////////////////////////////////////////////////////
// NAME       : MinesweeperShader::RecordPath
// DESCRIPTION: The shader doesn't have a path to prepare.
// PARAMETERS : u    - unit searching the path
//              last - final path node
//              off  - maximum allowed distance to destination
// RETURNS    : always 1
////////////////////////////////////////////////////////////////////////

short MinesweeperShader::RecordPath( const Unit *u, const PathNode *last,
                              unsigned char off ) const {
  path[map->Hex2Index( u->Position() )] = 1;
  return 1;
}


////////////////////////////////////////////////////////////////////////
// NAME       : TransPath::TransPath
// DESCRIPTION: Create a new path object for finding a path from a
//              source to a destination via a transport. A single
//              TransPath does only half of the work, either the unit
//              getting to the transport or the transport containing the
//              unit getting to the destination. Therefore you always
//              need two TransPaths to really service a transportation
//              request. Always call TransPath::Find( <unit>, <transport
//              position>, <unit position or destination>, ... )!
// PARAMETERS : map    - map to use for pathfinding
//              trans  - transport which will carry the unit
//              buffer - buffer to store path in; if non-NULL must be
//                       large enough to hold at least (map height x
//                       map width) bytes. If NULL, a new buffer will
//                       be allocated and freed when the path object is
//                       destroyed.
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

TransPath::TransPath( Map *map, const Transport *trans, signed char *buffer ) :
           Path( map, buffer ) {
  t = trans;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TransPath::GetCost
// DESCRIPTION: Calculate the cost for the unit to move from one hex to
//              another.
// PARAMETERS : u       - unit
//              pn_from - path node for source hex
//              to      - destination hex (adjacent to from)
// RETURNS    : next PathNode or NULL on illegal move
////////////////////////////////////////////////////////////////////////

PathNode *TransPath::GetCost( const Unit *u, const PathNode *pn_from, const Point &to ) const {
  const Point &from = pn_from->pos;
  unsigned short sofar = pn_from->cost;
  const Unit *cur = (pn_from->switched ? u : t);

  // important for all checks: on a TransPath the path is checked in the
  // direction Transport->Destination where destination may be the unit
  // or the final destination. This means that the unit must have made at
  // least one step before reaching the destination (switched == true)
  // for any path to be legal.
  short cost = -1;
  const TerrainType *type = map->HexType( to );
  bool hexok = (type->tt_type & cur->Terrain()) != 0;

  Unit *block = map->GetUnit( to );
  if ( (to != end) || pn_from->switched ) {
    if ( block ) {
      if ( block->IsTransport() && (to == end) && pn_from->switched &&
           !cur->IsSheltered() && static_cast<Transport *>(block)->Allow(cur) )
        cost = MAX( cur->Moves() - sofar, MCOST_MIN );
      else if ( hexok ) cost = MCOST_UNIT;
    } else {
      unsigned short state;
      cost = map->MoveCost( cur, from, to, state );
      if ( state != 0 ) {
        if ( sofar >= cur->Moves() ) cost = cur->Moves();
        else cost = MAX( cur->Moves() - sofar, cost );
      }
    }
  }

  PathNode *pn_to = NULL;
  if ( cost != -1 ) {
    pn_to = new PathNode();
    pn_to->pos = to;
    pn_to->cost = sofar + cost;
    pn_to->switched = pn_from->switched;
  } else if ( !pn_from->switched ) {
    // let's see if we can continue here when switching to the other unit
    PathNode pn_tmp( *pn_from );
    pn_tmp.switched = true;
    pn_to = GetCost( u, &pn_tmp, to );
  }

  return pn_to;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TransPath::Reverse
// DESCRIPTION: Rerecord a path in the opposite direction.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void TransPath::Reverse( void ) {
  Point p( start );
  short index = map->Hex2Index( p );
  Direction dir = (Direction)path[index], dir2;
  path[index] = -1;

  while ( p != end ) {
    map->Dir2Hex( p, dir, p );
    index = map->Hex2Index( p );
    dir2 = (Direction)path[index];
    path[index] = ReverseDir( dir );
    dir = dir2;
  }

  end = start;
  start = p;
}

