/*
 *  cryptmount - a utility for user-level mounting of encrypted filesystems
 *  $Revision: 158 $, $Date: 2007-04-10 07:27:01 +0100 (Tue, 10 Apr 2007) $
 *  Copyright 2005-2007 RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exemption, permission is granted to link cryptmount
    with the OpenSSL project's "OpenSSL" library and distribute
    the linked code without invoking clause 2(b) of the GNU GPL version 2.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with cryptmount; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>

#include <ctype.h>
#include <fcntl.h>
#include <getopt.h>
#include <inttypes.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/types.h>
#ifdef HAVE_SYSLOG
#  include <syslog.h>
#endif
#include <linux/fs.h>       /* beware ordering conflict with sys/mount.h */


#include "armour.h"
#include "cryptmount.h"
#include "delegates.h"
#include "dmutils.h"
#include "fsutils.h"
#include "looputils.h"
#include "tables.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif


typedef struct targelt
{
    const cment_t *entry;

    struct targelt *nx;
} targelt_t;


typedef enum
{
    M_DEFAULT, M_HELP,
    M_PREPARE, M_RELEASE,
    M_MOUNT, M_UNMOUNT,
    M_SWAPON, M_SWAPOFF,
    M_LIST, M_KEYMGRS,
    M_PASSWORD, M_KEYGEN, M_VERSION
} cmmode_t;


static int64_t getblkcount(const char *device);
static int execute_list(cmmode_t mode, const char *params,
                const targelt_t *eltlist);
static int do_list(const cment_t *cment);
static int do_devsetup(const cment_t *cment, char **mntdev);
static int do_devshutdown(const cment_t *cment);
static int do_mount(const cment_t *cment);
static int do_unmount(const cment_t *cment);
static int do_swapon(const cment_t *cment);
static int do_swapoff(const cment_t *cment);
static int do_passwd(const cment_t *cment);
static int do_keygen(const cment_t *cment, const char *params);


static const char *USAGE_STRING = N_("\
usage: cryptmount [OPTION [target ...]]\n\
\n\
  available options are as follows:\n\
\n\
    -h | --help\n\
    -a | --all\n\
    -c | --change-password <target>\n\
    -k | --key-managers\n\
    -l | --list\n\
    -m | --mount <target>\n\
    -u | --unmount <target>\n\
    --generate-key <key-size> <target>\n\
    --prepare <target>\n\
    --release <target>\n\
    --config-fd <num>\n\
    --swapon <target>\n\
    --swapoff <target>\n\
    --version\n\
\n\
  please report bugs to cryptmount@rwpenney.org.uk\n\
");


#ifdef TESTING

cm_testinfo_t test_context;
cm_testinfo_t *test_ctxtptr=&test_context;

int fs_test_blkgetsz()
    /* check that 32bit & 64bit device size calculations agree */
{
#ifdef BLKGETSIZE64
    int fd, n_open, seclen, s0, s1, s2;
    long len;
    uint64_t len64;
    const char **dev;
    const char *devices[] = {
        "/dev/hda", "/dev/hda1", "/dev/hda2", "/dev/hda3",
        "/dev/hdb", "/dev/hdb1", "/dev/hdb2", "/dev/hdb3",
        "/dev/hdc", "/dev/hdc1", "/dev/hdc2", "/dev/hdc3",
        "/dev/sda", "/dev/sda1", "/dev/sda2", "/dev/sda3",
        "/dev/scd", "/dev/scd1", "/dev/scd2", "/dev/scd3",
        NULL };
#endif

    CM_TEST_START("BLKGETSIZE ioctl calls");
#ifndef BLKGETSIZE64
    /* assume that there is no ambiguity with BLKGETSIZE */
    CM_TEST_PASS();
#else
    dev = devices;
    n_open = 0;
    while (*dev != NULL) {
        fd = open(*dev, O_RDONLY);
        if (fd >= 0) {
            ++n_open;
            if (ioctl(fd, BLKSSZGET, &seclen) != 0
              || ioctl(fd, BLKGETSIZE, &len) != 0
              || ioctl(fd, BLKGETSIZE64, &len64) != 0) {
                CM_TEST_FAIL();
            }
            close(fd);
            if (len64 < (1<<31)) {
                CM_ASSERT_EQUAL(len64, ((int64_t)len * (int64_t)seclen));
            }
            /* fprintf(stderr, "%s: %d  %ld  %lld\n", *dev, seclen, len, len64 / seclen); */
        }
        ++dev;
    }

    if (n_open > 0) {
        CM_TEST_OK();
    } else {
        CM_TEST_ABORT();
    }
#endif  /* BLKGETSIZE64 */
}

#endif  /* TESTING */


static int64_t getblkcount(const char *device)
    /* find size of raw device in blocks */
{   int64_t count=-1;
    int fd;
#ifdef BLKGETSIZE64
    int blklen;
#else
    long len;
#endif

    fd = open(device, O_RDONLY);
    if (fd < 0) return (int64_t)-1;

#ifdef BLKGETSIZE64
    if (ioctl(fd, BLKGETSIZE64, &count) == 0
        && ioctl(fd, BLKSSZGET, &blklen) == 0) {
        count /= (int64_t)blklen;
    } else {
        count = -1;
    }
#else
    if (ioctl(fd, BLKGETSIZE, &len) == 0) {
        count = (int64_t)len;
    }
#endif

    (void)close(fd);
    return count;
}


static int do_list(const cment_t *cment)
    /* print list of available filing-system targets */
{
    printf(_("%-16s  [to mount on \"%s\" as \"%s\"]\n"),
            cment->ident, cment->dir, cment->fstype);

    return ERR_NOERROR;
}


static int do_devsetup(const cment_t *cment, char **mntdev)
    /* setup all devices needed to access encrypted target */
{   enum { BUFFMIN=1024 };
    unsigned char *key=NULL;
    int i, killloop=0, keylen=0, eflag=ERR_NOERROR;
    int64_t devlen=0, fslen=0;
    struct stat sbuff;
    size_t dpsize;
    char *dmparams=NULL, *tgtdev=NULL, *loopdev=NULL;

    /* get crypto-key for filing system: */
    eflag = get_key(cment->ident, &cment->key, &key, &keylen);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("failed to extract cipher key\n"));
        goto bail_out;
    }

    if (cment->dev == NULL || stat(cment->dev, &sbuff) != 0) {
        fprintf(stderr, _("cannot stat device \"%s\" for target \"%s\"\n"),
                (cment->dev != NULL ? cment->dev : "(NULL)"), cment->ident);
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* check whether using raw device or loopback to file: */
    if (S_ISBLK(sbuff.st_mode)) {
        tgtdev = cment->dev;
    } else if (S_ISREG(sbuff.st_mode)) {
        if (cment->loopdev != NULL && strcmp(cment->loopdev, "auto") != 0) {
            loopdev = (char*)malloc((size_t)(strlen(cment->loopdev) + 1));
            strcpy(loopdev, cment->loopdev);
        } else {
            loopdev = (char*)malloc((size_t)BUFFMIN);
            if (loop_findfree(loopdev, (size_t)BUFFMIN) != 0) {
                fprintf(stderr, _("no available loopback devices\n"));
                eflag = ERR_BADDEVICE;
                goto bail_out;
            }
        }
        eflag = loop_setup(loopdev, cment->dev,
                        (is_readonlyfs(cment->dev) ? O_RDONLY : O_RDWR));
        tgtdev = loopdev;
    } else {
        fprintf(stderr,
            _("bad device type (%x) for \"%s\" (need block/file)\n"),
            (unsigned)sbuff.st_mode, cment->dev);
        tgtdev = NULL;
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* get size in blocks of target device: */
    devlen = getblkcount(tgtdev);
    if (devlen < 0) {
        fprintf(stderr, _("failed to get size of \"%s\"\n"), tgtdev);
        eflag = ERR_BADIOCTL;
        goto bail_out;
    }
    if (cment->length < 0 || (cment->start + cment->length) > devlen) {
        fslen = devlen - cment->start;
    } else {
        fslen = cment->length;
    }
    if (cment->start < 0 || fslen <= 0) {
        fprintf(stderr,_("bad device-mapper start/length"));
        fprintf(stderr, " (%" PRId64 ",%" PRId64 ")\n",
                cment->start, cment->length);
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* setup device-mapper crypt table (CIPHER KEY IV_OFFSET DEV START): */
    dpsize = 2 * keylen + BUFFMIN;
    dmparams = sec_realloc(dmparams, dpsize);
    i = snprintf(dmparams, dpsize, "%s ",
                (cment->cipher != NULL ? cment->cipher : DFLT_CIPHER));
    i += mk_key_string(key, (size_t)keylen, dmparams + i);
    snprintf(dmparams + i, (dpsize - i), " %" PRId64 " %s %" PRId64,
                cment->ivoffset, tgtdev, cment->start);

    /* setup device-mapper target: */
    eflag = devmap_create(cment->ident,
            (uint64_t)0, (uint64_t)fslen, "crypt", dmparams);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr,
            _("device-mapper target-creation failed for \"%s\"\n"),
            cment->ident);
        killloop = 1;
        goto bail_out;
    }
    if (mntdev != NULL) {
        devmap_path(mntdev, cment->ident);
    }


  bail_out:

    if (loopdev != NULL) {
        if (killloop) loop_destroy(loopdev);    /* mounting failed? */
        free((void*)loopdev);
    }
    sec_free(dmparams);
    sec_free(key);

    return eflag;
}   /* do_devsetup() */


int do_devshutdown(const cment_t *cment)
    /* remove all devices attached to encrypted target */
{   struct stat sbuff;
    unsigned i,devcnt=0;
    int eflag=ERR_NOERROR;
    dev_t *devids=NULL;
    char buff[256];

    /* check if filing system has been configured at all: */
    if (!is_configured(cment->ident, NULL)) {
        fprintf(stderr, _("target \"%s\" does not appear to be configured\n"),
                        cment->ident);
        eflag = WRN_UNCONFIG;
        goto bail_out;
    }

    /* find any underlying (e.g. loopback) devices for device-mapper target: */
    (void)devmap_dependencies(cment->ident, &devcnt, &devids);
#ifdef DEBUG
    fprintf(stderr, "shutting down %s [%u dependencies]\n",
            cment->ident, devcnt);
#endif

    if (stat(cment->dev, &sbuff) != 0) {
        fprintf(stderr, _("cannot stat \"%s\"\n"), cment->dev);
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    /* remove demice-mapper target: */
    eflag = devmap_remove(cment->ident);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("failed to remove device-mapper target \"%s\"\n"),
            cment->ident);
        goto bail_out;
    }

    /* tidy-up any associated loopback devices: */
    if (S_ISREG(sbuff.st_mode) && devids != NULL) {
        for (i=0; i<devcnt; ++i) {
            if (loop_ident(major(devids[i]),minor(devids[i]),buff,sizeof(buff)) || (loop_destroy(buff) != ERR_NOERROR)) {
                fprintf(stderr, _("failed to free device (%d,%d)\n"),
                        major(devids[i]), minor(devids[i]));
            }
        }
    }

  bail_out:

    if (devids != NULL) free((void*)devids);

    return eflag;
}


static int do_mount(const cment_t *cment)
{   int freedev=0,eflag=ERR_NOERROR;
    char *mntdev=NULL;
    tgtstat_t *tstat;

    if (is_mounted(cment)) {
        fprintf(stderr, _("target \"%s\" is already mounted\n"), cment->ident);
        eflag = ERR_BADMOUNT;
        goto bail_out;
    }

    eflag = do_devsetup(cment, &mntdev);
    if (eflag != ERR_NOERROR) goto bail_out;

#if WITH_FSCK
    if ((cment->flags & FLG_FSCK) != 0) {
        if (fs_check(mntdev, cment) != ERR_NOERROR) {
            freedev = 1; eflag = ERR_BADMOUNT;
            goto bail_out;
        }
    }
#endif

    if (fs_mount(mntdev, cment) != ERR_NOERROR) {
        freedev = 1; eflag = ERR_BADMOUNT;
        goto bail_out;
    }

    tstat = alloc_tgtstatus(cment);
    tstat->uid = (unsigned long)getuid();
    put_tgtstatus(cment, tstat);
    free_tgtstatus(tstat);

  bail_out:

    if (freedev) {
        /* tidy-up debris if mount failed */
        do_devshutdown(cment);
    }
    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_unmount(const cment_t *cment)
{   int eflag=ERR_NOERROR;
    struct passwd *pwent;
    char *mntdev=NULL;
    tgtstat_t *tstat;

    /* check if filing system has been configured at all: */
    if (!is_mounted(cment) || (tstat = get_tgtstatus(cment)) == NULL) {
        fprintf(stderr, _("target \"%s\" does not appear to be mounted\n"),
                        cment->ident);
        eflag = WRN_UNCONFIG;
        goto bail_out;
    }

    /* check if filing system has been mounted & locked by another user: */
    if (getuid() != 0 && (uid_t)tstat->uid != getuid()) {
        pwent = getpwuid((uid_t)tstat->uid);
        if (pwent != NULL) {
            fprintf(stderr, _("only \"%s\" can unmount \"%s\"\n"),
                pwent->pw_name, cment->ident);
        } else {
            fprintf(stderr, _("only user-%lu can unmount \"%s\"\n"),
                tstat->uid, cment->ident);
        }
        eflag = ERR_BADPRIV;
        goto bail_out;
    }

    /* unmount filing system: */
    if (fs_unmount(cment) != ERR_NOERROR) {
        eflag = ERR_BADMOUNT;
        goto bail_out;
    }
    put_tgtstatus(cment, NULL);

    /* remove supporting device-mapper target etc */
    if (do_devshutdown(cment) != ERR_NOERROR) {
        eflag = ERR_BADDEVICE;
    }


  bail_out:

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_swapon(const cment_t *cment)
{   int freedev=0,eflag=ERR_NOERROR;
    char *mntdev=NULL;
    tgtstat_t *tstat;

#if WITH_CSWAP

    eflag = do_devsetup(cment, &mntdev);
    if (eflag != ERR_NOERROR) goto bail_out;

    if (fs_swapon(mntdev, cment) != ERR_NOERROR) {
        freedev = 1;
        eflag = ERR_BADSWAP;
        goto bail_out;
    }

    tstat = alloc_tgtstatus(cment);
    tstat->uid = (unsigned long)getuid();
    put_tgtstatus(cment, tstat);
    free_tgtstatus(tstat);

#else   /* !WITH_CSWAP */

    fprintf(stderr, _("crypto-swap is not supported by this installation of cryptmount\n"));
    eflag = ERR_BADSWAP;

#endif

  bail_out:

    if (freedev) {
        /* tidy-up debris if swapon failed */
        do_devshutdown(cment);
    }

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_swapoff(const cment_t *cment)
{   int eflag=ERR_NOERROR;
    char *mntdev=NULL;
    tgtstat_t *tstat;

#if WITH_CSWAP

    /* check if device has been configured at all: */
    if ((tstat = get_tgtstatus(cment)) == NULL) {
        fprintf(stderr, _("target \"%s\" does not appear to be configured\n"),
                        cment->ident);
        eflag = WRN_UNCONFIG;
        goto bail_out;
    }

    /* remove swap-partition: */
    if (fs_swapoff(cment) != ERR_NOERROR) {
        eflag = ERR_BADSWAP;
        goto bail_out;
    }
    put_tgtstatus(cment, NULL);

    /* remove supporting device-mapper target etc */
    if (do_devshutdown(cment) != ERR_NOERROR) {
        eflag = ERR_BADDEVICE;
    }

#else   /* !WITH_CSWAP */

    fprintf(stderr, _("crypto-swap is not supported by this installation of cryptmount\n"));
    eflag = ERR_BADSWAP;

#endif

  bail_out:

    if (mntdev != NULL) free((void*)mntdev);

    return eflag;
}


static int do_passwd(const cment_t *cment)
{   unsigned char *key=NULL;
    int keylen=0,eflag=ERR_NOERROR;
    char *newfname=NULL,*oldfname=NULL;
    struct stat sbuff;
    size_t sz;
    FILE *fp;

    if (!is_keysafe(&cment->key)) {
        fprintf(stderr, _("key-file for \"%s\" isn't password-protected\n"),
                cment->ident);
        eflag = WRN_NOPASSWD;
        goto bail_out;
    }

    eflag = get_key(cment->ident, &cment->key, &key, &keylen);
    if (eflag != ERR_NOERROR) goto bail_out;

    sz = strlen(cment->key.filename) + 16;
    oldfname = (char*)malloc(2 * sz);
    newfname = oldfname + sz;
    snprintf(oldfname, sz, "%s-old", cment->key.filename);
    snprintf(newfname, sz, "%s-new", cment->key.filename);

    fp = fopen(newfname, "wb");
    if (fp == NULL) {
        fprintf(stderr, _("cannot open \"%s\" for writing\n"), newfname);
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    eflag = put_key(cment->ident, &cment->key, key, keylen, fp);
    if (fclose(fp) != 0) eflag = ERR_BADFILE;

    if (eflag == ERR_NOERROR) {
        /* FIXME - check return codes */
        (void)stat(cment->key.filename, &sbuff);
        (void)rename(cment->key.filename, oldfname);
        (void)chown(oldfname, 0, 0);
        (void)chmod(oldfname, S_IRUSR | S_IWUSR);

        (void)rename(newfname, cment->key.filename);
        (void)chown(cment->key.filename, sbuff.st_uid, sbuff.st_gid);
        (void)chmod(cment->key.filename, sbuff.st_mode);

        newfname = NULL;

        fprintf(stderr, _("backup of previous key is in \"%s\"\n"), oldfname);
    }

  bail_out:

    if (newfname != NULL) {
        unlink(newfname);
    }
    if (oldfname != NULL) free((void*)oldfname);

    return eflag;
}


static int do_keygen(const cment_t *cment, const char *params)
{   unsigned char *key=NULL;
    int keylen=0, eflag=ERR_NOERROR;
    char *fname=NULL;
    size_t sz;
    struct stat sbuff;
    FILE *fp;

    if (params == NULL || sscanf(params, "%i", &keylen) != 1 || keylen < 0) {
        fprintf(stderr, _("bad key-length parameter"));
        eflag = ERR_BADPARAM;
        goto bail_out;
    }

    if (stat(cment->key.filename, &sbuff) == 0) {
        fprintf(stderr,_("key-file \"%s\" already exists for target \"%s\"\n"),
            cment->key.filename, cment->ident);
        eflag = ERR_BADFILE;
        goto bail_out;
    }

    fprintf(stderr, _("generating random key; please be patient...\n"));
    key = (unsigned char*)sec_realloc(NULL, (size_t)keylen);
    eflag = get_randkey(key, (size_t)keylen);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("failed to generate new key\n"));
        goto bail_out;
    }

    sz = strlen(cment->key.filename) + 16;
    fname = (char*)malloc(sz);
    snprintf(fname, sz, "%s-new", cment->key.filename);

    fp = fopen(fname, "wb");
    if (fp == NULL) {
        fprintf(stderr, _("cannot open \"%s\" for writing\n"), fname);
        eflag = ERR_BADFILE;
        goto bail_out;
    }
    eflag = put_key(cment->ident, &cment->key, key, keylen, fp);
    fclose(fp);

    if (eflag == ERR_NOERROR) {
        /* FIXME - check return codes */
        (void)rename(fname, cment->key.filename);
        (void)chown(cment->key.filename, 0, 0);
        (void)chmod(cment->key.filename, S_IRUSR | S_IWUSR);
    }

  bail_out:

    if (fname != NULL) free((void*)fname);
    if (key != NULL) sec_free((void*)key);

    return eflag;
}


static void check_priv_opt(const char *opt)
    /* check if ordinary user is allowed to perform privileged actions */
{

    if (getuid() != 0) {
        fprintf(stderr, _("only root can use option \"%s\"\n"), opt);
        exit(EXIT_PRIV);
    }

    /* remove effect of any setuid flags, reverting to real user-id: */
    seteuid(getuid());
}


static int check_priv_tgt(const cment_t *cment)
    /* check if ordinary user is allowed to perform privileged actions */
{
    if ((cment->flags & FLG_USER) == 0 && getuid() != 0) {
        fprintf(stderr, _("only root can configure \"%s\"\n"), cment->ident);
        return ERR_BADPRIV;
    }

    return ERR_NOERROR;
}


static int check_mode(cmmode_t mode, const char *params,
                    const targelt_t *eltlist)
    /* check that arguments supplied & privilege match selected mode */
{   enum {
        F_NORMALUSER =  0x001,
        F_NEEDS_TGT =   0x010,
        F_NEEDS_PRM =   0x020
    };
    struct tblent {
        cmmode_t mode;
        unsigned flags; } *mdent=NULL;
    struct tblent mdtbl[] = {
        { M_HELP,       F_NORMALUSER },
        { M_PREPARE,    F_NEEDS_TGT },
        { M_RELEASE,    F_NEEDS_TGT },
        { M_MOUNT,      F_NORMALUSER | F_NEEDS_TGT },
        { M_UNMOUNT,    F_NORMALUSER | F_NEEDS_TGT },
        { M_SWAPON,     F_NEEDS_TGT },
        { M_SWAPOFF,    F_NEEDS_TGT },
        { M_LIST,       F_NORMALUSER },
        { M_KEYMGRS,    F_NORMALUSER },
        { M_PASSWORD,   F_NORMALUSER | F_NEEDS_TGT },
        { M_KEYGEN,     F_NEEDS_TGT | F_NEEDS_PRM},
        { M_VERSION,    F_NORMALUSER },
    };
    unsigned idx,nmds;

    nmds = sizeof(mdtbl) / sizeof(mdtbl[0]);
    for (mdent=mdtbl,idx=0; idx<nmds && mdent->mode!=mode; ++mdent,++idx);
    if (idx >= nmds) {
        fprintf(stderr, "bad mode [%u]\n", mode);
        abort();
    }

    if ((mdent->flags & F_NORMALUSER) != F_NORMALUSER) {
        /* turn off any setuid effects for privileged modes: */
        seteuid(getuid());
    }

    if ((mdent->flags & F_NEEDS_TGT) == F_NEEDS_TGT && eltlist == NULL) {
        fprintf(stderr, _("no targets specified\n"));
        exit(EXIT_BADTGT);
    }

    if ((mdent->flags & F_NEEDS_PRM) == F_NEEDS_PRM && params == NULL) {
        fprintf(stderr, _("missing parameter\n"));
        exit(EXIT_BADOPT);
    }

    return 0;
}


static int execute_list(cmmode_t mode, const char *params,
                 const targelt_t *eltlist)
{   const targelt_t *elt;
    int i, prio=0, eflag=ERR_NOERROR;
    struct passwd *pwent=NULL;
    const char *username, **keymgrs, *syslogmsg=NULL;

    pwent = getpwuid(getuid());
    username = (pwent != NULL ? pwent->pw_name : "UNKNOWN");

#if defined(HAVE_SYSLOG) && !defined(TESTING)
    openlog(PACKAGE, LOG_PID, LOG_AUTHPRIV);
#endif

    /* check that we have suitable privileges + parameters: */
    (void)check_mode(mode, params, eltlist);

    /* execute user-selected task: */
    if (mode == M_VERSION) {
        fprintf(stderr, "%s-%s\n", PACKAGE_NAME, PACKAGE_VERSION);
    } else if (mode == M_KEYMGRS) {
        keymgrs = get_keymgr_list();
        for (i=0; keymgrs[i]!=NULL; ) {
            printf("%s", keymgrs[i]);
            if (keymgrs[++i] != NULL) printf(", "); else printf("\n");
        }
        free((void*)keymgrs);
    } else {
        for (elt=eltlist; elt!=NULL && eflag<ERR_threshold; elt=elt->nx) {
            syslogmsg = NULL;
            prio = LOG_AUTHPRIV | LOG_NOTICE;
            switch (mode) {
                case M_LIST:
                    eflag = do_list(elt->entry);
                    break;
                case M_PREPARE:
                    syslogmsg = "prepare of \"%s\" by %s %s";
                    eflag = do_devsetup(elt->entry, NULL);
                    break;
                case M_RELEASE:
                    syslogmsg = "release of \"%s\" by %s %s";
                    prio = LOG_AUTHPRIV | LOG_NOTICE;
                    eflag = do_devshutdown(elt->entry);
                    if (eflag == WRN_UNCONFIG) syslogmsg = NULL;
                    break;
                case M_MOUNT:
                    if ((eflag = check_priv_tgt(elt->entry)) != ERR_NOERROR) break;
                    syslogmsg = "mount of \"%s\" by %s %s";
                    eflag = do_mount(elt->entry);
                    break;
                case M_UNMOUNT:
                    if ((eflag = check_priv_tgt(elt->entry)) != ERR_NOERROR) break;
                    syslogmsg = "unmount of \"%s\" by %s %s";
                    eflag = do_unmount(elt->entry);
                    if (eflag == WRN_UNCONFIG) syslogmsg = NULL;
                    break;
                case M_SWAPON:
                    syslogmsg = "swapon \"%s\" by %s %s";
                    eflag = do_swapon(elt->entry);
                    break;
                case M_SWAPOFF:
                    syslogmsg = "swapoff \"%s\" by %s %s";
                    eflag = do_swapoff(elt->entry);
                    if (eflag == WRN_UNCONFIG) syslogmsg = NULL;
                    break;
                case M_PASSWORD:
                    if ((eflag = check_priv_tgt(elt->entry)) != ERR_NOERROR) break;
                    syslogmsg = "changing password for \"%s\" by %s %s";
                    eflag = do_passwd(elt->entry);
                    break;
                case M_KEYGEN:
                    if ((eflag = check_priv_tgt(elt->entry)) != ERR_NOERROR) break;
                    syslogmsg = "key generation for \"%s\" by %s %s";
                    eflag = do_keygen(elt->entry, params);
                    break;
                default:
                    break;
            }
#if defined(HAVE_SYSLOG) && !defined(TESTING)
            if (syslogmsg != NULL) {
                syslog(prio, syslogmsg, elt->entry->ident, username,
                    (eflag == ERR_NOERROR ? "succeeded" : "failed"));
            }
#endif
        }
    }

#ifdef HAVE_SYSLOG
    closelog();
#endif

    return eflag;
}


static cmmode_t get_defaultmode(int argc, char *argv[])
    /* translate program-name into default action (or just assume M_MOUNT) */
{   cmmode_t mode = M_MOUNT;

#ifdef WITH_ARGV0
    struct modename {
        cmmode_t mode;
        const char *name; } modetable[] = {
            { M_MOUNT,      "cryptmount" },
            { M_UNMOUNT,    "cryptumount" },
            { M_UNMOUNT,    "cryptunmount" },
#if WITH_CSWAP
            { M_SWAPON,     "cryptswapon" },
            { M_SWAPOFF,    "cryptswapoff", },
#endif
            { M_PREPARE,    "cryptprepare" },
            { M_RELEASE,    "cryptrelease" },
            { M_DEFAULT, NULL } },
        *mp;
    const char *base;

    if (argc >= 1) {
        base = strrchr(argv[0], '/');
        if (base != NULL) ++base; else base = argv[0];

        for (mp=modetable; mp->name!=NULL; ++mp) {
            if (strcmp(base, mp->name) == 0) {
                mode = mp->mode;
                break;
            }
        }
    }

#endif

    return mode;
}


int main(int argc, char *argv[])
{   cmmode_t mode=M_DEFAULT;
    enum {
        F_ALL=0x1 }
        flags=0;
#ifdef _GNU_SOURCE
    struct option opttable[] = {
        { "all",                no_argument,        NULL, (int)'a' },
        { "change-password",    no_argument,        NULL, (int)'c' },
        { "config-fd",          required_argument,  NULL, (int)'f' },
        { "generate-key",       required_argument,  NULL, (int)'g' },
        { "help",               no_argument,        NULL, (int)'h' },
        { "key-managers",       no_argument,        NULL, (int)'k' },
        { "list",               no_argument,        NULL, (int)'l' },
        { "mount",              no_argument,        NULL, (int)'m' },
        { "prepare",            no_argument,        NULL, (int)'p' },
        { "release",            no_argument,        NULL, (int)'r' },
        { "swapon",             no_argument,        NULL, (int)'s' },
        { "swapoff",            no_argument,        NULL, (int)'x' },
        { "unmount",            no_argument,        NULL, (int)'u' },
        { "version",            no_argument,        NULL, (int)'v' },
#  ifdef TESTING
        { "config-dir",         required_argument,  NULL, (int)'D' },
        { "password",           required_argument,  NULL, (int)'W' },
        { "newpassword",        required_argument,  NULL, (int)'N' },
        { "self-test",          no_argument,        NULL, (int)'T' },
#  endif
        { NULL, 0, NULL, 0 } };
#endif
    const char *params=NULL, *optlist="acf:g:klmprsuvx";
    char *cmtab=NULL;
    int val,idx,config_fd=-1,eflag=ERR_NOERROR;
    cment_t *cmtable=NULL;
    const cment_t *ent=NULL;
    targelt_t *eltlist=NULL,**eltptr=&eltlist;

#ifdef HAVE_GETTEXT
    /* setup internationalization of message-strings via gettext(): */
    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
#endif

#ifdef TESTING
    fprintf(stderr, "WARNING!!! cryptmount has been compiled for TESTING only - DO NOT INSTALL\n");
    test_context.argconfigdir = test_context.argpassword[0] = test_context.argpassword[1] = NULL;
#endif

    if (mlockall(MCL_CURRENT | MCL_FUTURE) != 0) {
        fprintf(stderr, _("memory-locking failed...\n"));
    }

    /* parse command-line options: */
    for (;;) {
#ifdef _GNU_SOURCE
        val = getopt_long(argc, argv, optlist, opttable, &idx);
#else
        val = getopt(argc, argv, optlist);
#endif
        if (val == -1) break;
        switch (val) {
            case 'a':
                flags |= F_ALL; break;
            case 'c':
                mode = M_PASSWORD; break;
            case 'f':                       check_priv_opt("--config-fd");
                sscanf(optarg, "%d", &config_fd); break;
            case 'g':                       check_priv_opt("--generate-key");
                mode = M_KEYGEN; params = optarg; break;
            case 'h':
                mode = M_HELP; break;
            case 'k':
                mode = M_KEYMGRS; break;
            case 'l':
                mode = M_LIST;  break;
            case 'm':
                mode = M_MOUNT; break;
            case 'p':                       check_priv_opt("--prepare");
                mode = M_PREPARE; break;
            case 'r':                       check_priv_opt("--release");
                mode = M_RELEASE; break;
            case 's':                       check_priv_opt("--swapon");
                mode = M_SWAPON; break;
            case 'x':                       check_priv_opt("--swapoff");
                mode = M_SWAPOFF; break;
            case 'u':
                mode = M_UNMOUNT; break;
            case 'v':
                mode = M_VERSION; break;
#ifdef TESTING
            case 'D':
                test_context.argconfigdir = optarg; break;
            case 'W':
                test_context.argpassword[0] = optarg; break;
            case 'N':
                test_context.argpassword[1] = optarg; break;
            case 'T':
                return cm_run_tests(); break;
#endif
            case '?':
                fprintf(stderr, "%s", _(USAGE_STRING));
                exit(EXIT_BADOPT);
                break;
            default:
                break;
        }
    }

    (void)cm_path(&cmtab, "cmtab");

    if (mode == M_DEFAULT) mode = get_defaultmode(argc, argv);

    if (mode == M_HELP) {
        fprintf(stderr, "%s", _(USAGE_STRING));
        exit(EXIT_OK);
    }

    /* (check &) read-in configuration file: */
#ifndef TESTING
    if (sycheck_cmtab(cmtab) != ERR_NOERROR
      || sycheck_directory(CM_MODULE_DIR) != ERR_NOERROR) {
        fprintf(stderr, _("security failure\n"));
        exit(EXIT_INSECURE);
    }
#endif
    if (config_fd >= 0) {
        cmtable = parse_config_fd(config_fd);
    } else {
        cmtable = parse_config(cmtab);
    }
    free((void*)cmtab);

    if (mode == M_LIST && optind >= argc) flags |= F_ALL;

    /* if '--all' given, assemble list of targets from entire config-file */
    if ((flags & F_ALL) != 0) {
        if (optind < argc) {
            fprintf(stderr, _("trailing command-line arguments given with '--all' option\n"));
            exit(EXIT_BADOPT);
        }
        for (ent=cmtable; ent!=NULL; ent=ent->nx) {
            *eltptr = (targelt_t*)malloc(sizeof(targelt_t));
            (*eltptr)->entry = ent;
            (*eltptr)->nx = NULL;
            eltptr = &((*eltptr)->nx);
        }
    }

    /* assemble list of targets from remaining command-line arguments: */
    while (optind < argc) {
        ent = get_cment(cmtable, argv[optind]);
        if (ent != NULL) {
            *eltptr = (targelt_t*)malloc(sizeof(targelt_t));
            (*eltptr)->entry = ent;
            (*eltptr)->nx = NULL;
            eltptr = &((*eltptr)->nx);
        } else {
            fprintf(stderr, _("target name \"%s\" is not recognized\n"),
                    argv[optind]);
            exit(EXIT_BADTGT);
        }
        ++optind;
    }


    /* check security of all targets being processed: */
    for (eltptr=&eltlist; *eltptr!=NULL; eltptr=&((*eltptr)->nx)) {
        if (sycheck_target((*eltptr)->entry) != ERR_NOERROR) {
            fprintf(stderr, _("target security failure\n"));
            exit(EXIT_INSECURE);
        }
    }


    /* execute user-selected task: */
    eflag = execute_list(mode, params, eltlist);
    free_keymanagers();


    /* tidy-up: */
    while (eltlist != NULL) {
        eltptr = &eltlist;
        eltlist = eltlist->nx;
        free((void*)*eltptr);
    }
    free_config(&cmtable);
    munlockall();

    return eflag;
}

/*
 *  cryptmount.c
 *  (C)Copyright 2005-2007, RW Penney
 */
