/*
 *  methods for LUKS-related key-management for cryptmount
 *  (C)Copyright 2005-2009, RW Penney
 */

/*
    This file is part of cryptmount

    cryptmount is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    cryptmount is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <ctype.h>
#if HAVE_DLFCN && USE_MODULES
#  include <dlfcn.h>
#endif
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "armour.h"
#include "cryptmount.h"
#include "looputils.h"
#include "utils.h"
#ifdef TESTING
#  include "cmtesting.h"
#endif


typedef struct {
    unsigned keyslot;
} luks_overrides_t;


/*
 *  ==== LUKS key-management routines ====
 */

#if USE_LUKSCOMPAT
#  if !USE_MODULES || defined(AS_MODULE)
#    include "luks/luks.h"
#    if HAVE_UUID
#      include <uuid/uuid.h>
#  endif

extern struct setup_backend setup_libdevmapper_backend;

#if defined(TESTING) && defined(AS_MODULE)
cm_testinfo_t *test_ctxtptr;
#endif


static int kmluks_hdrvalid(FILE *fp_key)
    /* Check whether a valid LUKS header is present */
{   const unsigned char luks_magic[] = LUKS_MAGIC;
    char buff[32];
    int flg = 0;

    if (fp_key == NULL) return 0;

    if (cm_fread((void*)buff, (size_t)LUKS_MAGIC_L, fp_key) == 0) {
        fseek(fp_key, -((long)LUKS_MAGIC_L), SEEK_CUR);
        flg = (strncmp(buff, (const char*)luks_magic,
                (size_t)LUKS_MAGIC_L) == 0);
    }

    return flg;
}


static void kmluks_splitmode(const char *fullname, char **cipher, char **mode)
    /* Split fully-qualified cipher name into algorithm + mode */
{   size_t divpos=0, nlen=0;
    const char *pos=fullname;

    if (*cipher != NULL) free((void*)*cipher);
    if (*mode != NULL) free((void*)*mode);
    *cipher = *mode = NULL;

    if (fullname != NULL) {
        /* Split name according to 'ALGO-MODE' pattern: */
        while (*pos != '\0' && *pos != '-') {
            ++pos; ++nlen; }
        divpos = nlen;
        while (*pos != '\0') {
            ++pos; ++nlen; }

        if (divpos > 0) {
            *cipher = (char*)malloc(divpos + 1);
            strncpy(*cipher, fullname, divpos); 
            (*cipher)[divpos] = '\0';
        }

        if (divpos < nlen) {
            *mode = (char*)malloc((nlen - divpos));
            strcpy(*mode, fullname + divpos + 1);
        }
    }

    if (*cipher == NULL) *cipher = cm_strdup("aes");
    if (*mode == NULL) *mode = cm_strdup("cbc-plain");
}


static int kmluks_init_algs()
{
    /* Nothing needed */
    return 0;
}


static int kmluks_free_algs()
{
    /* Nothing needed */
    return 0;
}


static int kmluks_bind(bound_tgtdefn_t *bound, FILE *fp_key)
{   int compat = 0;

    if (bound->tgt->key.format != NULL) {
        compat = (strcmp(bound->tgt->key.format, "luks") == 0);
    } else {
        /* Check header of existing key-file: */
        compat |= kmluks_hdrvalid(fp_key);
    }

    if (compat) {
        tgtdefn_t *tgt = bound->tgt;

        if (tgt->key.filename == NULL && tgt->dev != NULL) {
            tgt->key.filename = cm_strdup(tgt->dev);
        }

        if (tgt->key.digestalg == NULL) {
            tgt->key.digestalg = cm_strdup("sha1");
        }

        if (tgt->key.cipheralg == NULL) {
            tgt->key.cipheralg = cm_strdup("aes128");
        }
    }

    return compat;
}


static unsigned kmluks_get_properties(const bound_tgtdefn_t *boundtgt)
{   unsigned props;
    FILE *fp;

    props = KM_PROP_HASPASSWD | KM_PROP_FIXEDLOC;

    fp = fopen(boundtgt->tgt->key.filename, "rb");
    if (fp != NULL) {
        if (kmluks_hdrvalid(fp)) {
            props |= KM_PROP_FORMATTED;
        }
        fclose(fp);
    }

    return props;
}


static int kmluks_get_key(bound_tgtdefn_t *boundtgt,
            const km_pw_context_t *pw_ctxt,
            unsigned char **key, int *keylen, FILE *fp_key)
    /* Extract key from LUKS header file */
{   tgtdefn_t *tgt = boundtgt->tgt;
    char *passwd=NULL;
    const char *tgtdev=NULL;
    struct luks_phdr lukshdr;
    struct luks_masterkey *lukskey=NULL;
    int killloop=0, slot, eflag=ERR_NOERROR;
    luks_overrides_t *luksor;
    int64_t delta;
    const size_t namesz = LUKS_CIPHERNAME_L + LUKS_CIPHERMODE_L + 8;

    eflag = km_get_passwd(tgt->ident, pw_ctxt, &passwd, 0, 0);
    if (eflag != ERR_NOERROR) goto bail_out;

    if (blockify_file(tgt->key.filename, O_RDONLY, NULL, &tgtdev, &killloop) != ERR_NOERROR) {
        fprintf(stderr, _("Failed to create loop device for LUKS keyfile\n"));
        eflag = ERR_BADDEVICE;
        goto bail_out;
    }

    slot = LUKS_open_any_key(tgtdev, passwd, strlen(passwd),
                    &lukshdr, &lukskey, &setup_libdevmapper_backend);
    if (slot < 0) {
        fprintf(stderr, _("Failed to extract LUKS key for \"%s\"\n"),
                tgt->ident);
        eflag = ERR_BADDECRYPT;
        goto bail_out;
    }

    /* Extract cipher-algorithm parameter from LUKS header: */
    delta = (lukshdr.payloadOffset - tgt->start);
    if (delta >= 0) {
        tgt->start += delta;
        if (tgt->length >= 0) tgt->length -= delta;
    }
    if (tgt->cipher != NULL) free((void*)tgt->cipher);
    tgt->cipher = (char*)malloc(namesz);
    snprintf(tgt->cipher, namesz, "%s-%s",
            lukshdr.cipherName, lukshdr.cipherMode);
    tgt->ivoffset = 0;
    if (boundtgt->km_data != NULL) free((void*)boundtgt->km_data);
    luksor = (luks_overrides_t*)malloc(sizeof(luks_overrides_t));
    luksor->keyslot = slot;
    boundtgt->km_data = (void*)luksor;

    /* Take copy of LUKS master-key: */
    *keylen = lukskey->keyLength;
    *key = (unsigned char*)sec_realloc((void*)*key, (size_t)lukskey->keyLength);
    memcpy((void*)*key, (const void*)lukskey->key, (size_t)*keylen);

  bail_out:

    unblockify_file(&tgtdev, killloop);
    if (passwd != NULL) sec_free((void*)passwd);
    if (lukskey != NULL) LUKS_dealloc_masterkey(lukskey);

    return eflag;
}


static int kmluks_put_key(bound_tgtdefn_t *boundtgt,
            const km_pw_context_t *pw_ctxt,
            const unsigned char *key, const int keylen, FILE *fp_key)
    /* Store or create key in LUKS header */
{
#if HAVE_UUID
    const keyinfo_t *keyinfo = &boundtgt->tgt->key;
    char *passwd=NULL, *ciphername=NULL, *ciphermode=NULL;
    const char *tgtdev=NULL;
    struct luks_phdr lukshdr;
    struct luks_masterkey *lukskey=NULL;
    unsigned keyslot=0;
    luks_overrides_t *luksor=NULL;
    int formatting=0, killloop=0, PBKDF2its, eflag=ERR_NOERROR;

    formatting = (boundtgt->km_data == NULL) && !kmluks_hdrvalid(fp_key);

    if (boundtgt->km_data != NULL) {
        luksor = (luks_overrides_t*)boundtgt->km_data;
    }

    if (formatting) {
#ifndef TESTING
        char msgbuff[1024];
        snprintf(msgbuff, sizeof(msgbuff), _("Formatting \"%s\", will probably destroy all existing data"), keyinfo->filename);
        if (!cm_confirm(msgbuff)) {
            eflag = ERR_ABORT;
            goto bail_out;
        }
#endif  /* !TESTING */
    }

    eflag = km_get_passwd(boundtgt->tgt->ident, pw_ctxt, &passwd, 1, 1);
    if (eflag != ERR_NOERROR) goto bail_out;

    eflag = blockify_file(keyinfo->filename, O_RDWR, NULL, &tgtdev, &killloop);
    if (eflag != ERR_NOERROR) {
        fprintf(stderr, _("Failed to create loop device for LUKS keyfile\n"));
        goto bail_out;
    }

    lukskey = LUKS_alloc_masterkey(keylen);
    memcpy((void*)lukskey->key, (const void*)key, (size_t)keylen);

    if (formatting) {
        /* Format new partition */
        keyslot = 0;

        kmluks_splitmode(boundtgt->tgt->cipher, &ciphername, &ciphermode);
        if (LUKS_generate_phdr(&lukshdr, lukskey,
                            ciphername, ciphermode, LUKS_STRIPES, 0) < 0) {
            fprintf(stderr, _("Failed to create LUKS header for \"%s\"\n"),
                    boundtgt->tgt->ident);
            eflag = ERR_BADDEVICE;
            goto bail_out;
        }
    } else {
        /* Changing password on existing partition */
        keyslot = (luksor != NULL ? luksor->keyslot : 0);

        if (LUKS_read_phdr(tgtdev, &lukshdr) < 0) {
            fprintf(stderr, _("Failed to read LUKS header for \"%s\"\n"),
                    boundtgt->tgt->ident);
            eflag = ERR_BADDEVICE;
            goto bail_out;
        }

        lukshdr.keyblock[keyslot].active = LUKS_KEY_DISABLED;
    }

    printf(_("Setting password on LUKS keyslot-%u\n"), keyslot);
    PBKDF2its = (int)(1.0 * LUKS_benchmarkt_iterations() + 0.5);
    lukshdr.keyblock[keyslot].passwordIterations = (PBKDF2its >= 1 ? PBKDF2its : 1);

    if (LUKS_set_key(tgtdev, keyslot, passwd, strlen(passwd), &lukshdr, lukskey, &setup_libdevmapper_backend) < 0) {
        fprintf(stderr, _("Failed to create LUKS key for \"%s\"\n"),
                boundtgt->tgt->ident);
        fprintf(stderr, "LUKS error: %s\n", get_error());
        eflag = ERR_BADENCRYPT;
        goto bail_out;
    }

  bail_out:

    unblockify_file(&tgtdev, killloop);
    if (lukskey != NULL) LUKS_dealloc_masterkey(lukskey);
    if (passwd != NULL) sec_free((void*)passwd);
    if (ciphername != NULL) free((void*)ciphername);
    if (ciphermode != NULL) free((void*)ciphermode);

    return eflag;

#else   /* !HAVE_UUID */

    fprintf(stderr, _("Writing LUKS keys is not possible unless cryptmount has been built with the uuid-dev package installed\n"));

    return ERR_BADKEYFORMAT;
#endif  /* HAVE_UUID */
}


#  ifdef TESTING

static int kmluks_test_modesplit()
{   struct tcase {
        const char *orig, *cipher, *mode; };
    struct tcase tcases[] = {
        { "",                       "aes", "cbc-plain" },
        { "nothing",                "nothing", "cbc-plain" },
        { "alg-mode",               "alg", "mode" },
        { "blowfish-cfb-essiv",     "blowfish", "cfb-essiv" },
        { "-mode:suffix",           "aes", "mode:suffix" },
        { NULL,                     "aes", "cbc-plain" } };
    char *head=NULL, *tail=NULL;
    unsigned idx, cnt;

    CM_TEST_START("LUKS cipher-mode parsing");

    cnt = sizeof(tcases) / sizeof(struct tcase);
    for (idx=0; idx<cnt; ++idx) {
        kmluks_splitmode(tcases[idx].orig, &head, &tail);

        if (strcmp(head, tcases[idx].cipher) != 0) CM_TEST_FAIL();
        if (strcmp(tail, tcases[idx].mode) != 0) CM_TEST_FAIL();
    }
    if (head != NULL) free((void*)head);
    if (tail != NULL) free((void*)tail);

    CM_TEST_OK();

    return 0;
}

static void kmluks_testctxt(cm_testinfo_t *context)
{
    test_ctxtptr = context;
}

static int kmluks_runtests()
{   int flg = 0;

    flg |= kmluks_test_modesplit();

    return flg;
}

#  endif    /* TESTING */

keymanager_t keymgr_luks = {
    "luks", 0,   kmluks_init_algs, kmluks_free_algs,
                      kmluks_bind, kmluks_get_properties,
                      kmluks_get_key, kmluks_put_key,
    NULL
#ifdef TESTING
    , kmluks_testctxt, kmluks_runtests, (CM_READONLY)
#endif
};

#  endif    /* !USE_MODULES || defined(AS_MODULE) */
#endif  /* USE_LUKSCOMPAT */


#ifndef AS_MODULE

#if defined(TESTING)
#  define MOD_PATH CM_SRCDIR "/cm-luks.so"
#else
#  define MOD_PATH CM_MODULE_DIR "/cm-luks.so"
#endif

keymanager_t *kmluks_gethandle()
{
#if USE_LUKSCOMPAT
#  if USE_MODULES
    KM_GETHANDLE(MOD_PATH, "keymgr_luks");
#  else
    return &keymgr_luks;
#  endif
#else
    return NULL;
#endif
}

#endif  /* !AS_MODULE */

/* vim: set ts=4 sw=4 et: */

/*
 *  (C)Copyright 2005-2009, RW Penney
 */
