/*
 * The Cryptonit security software suite is developped by IDEALX
 * Cryptonit Team (http://IDEALX.org/ and http://cryptonit.org).
 *
 * Copyright 2003-2006 IDEALX
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301, USA. 
 *
 * In addition, as two special exceptions:
 *
 * 1) IDEALX S.A.S gives permission to:
 *  * link the code of portions of his program with the OpenSSL library under
 *    certain conditions described in each source file
 *  * distribute linked combinations including the two, with respect to the
 *    OpenSSL license and with the GPL
 *
 * You must obey the GNU General Public License in all respects for all of the
 * code used other than OpenSSL. If you modify file(s) with this exception,
 * you may extend this exception to your version of the file(s), but you are
 * not obligated to do so. If you do not wish to do so, delete this exception
 * statement from your version, in all files (this very one along with all
 * source files).

 * 2) IDEALX S.A.S acknowledges that portions of his sourcecode uses (by the
 * way of headers inclusion) some work published by 'RSA Security Inc.'. Those
 * portions are "derived from the RSA Security Inc. PKCS #11Cryptographic
 * Token Interface (Cryptoki)" as described in each individual source file.
 */

#include <wx/wx.h>
#include <wx/filename.h>
#include <wx/dir.h>

#include "AddIdentity.hh"
#include "AddCrlDlg.hh"
#include "PasswordDlg.hh"
#include "Pkcs12Dlg.hh"
#include "Common.hh"

#include "../Certificate.hh"
#include "../pkcs10.hh"
#include "../pkcs8.hh"
#include "../pkcs12.hh"
#include "../Utils.hh"

BEGIN_EVENT_TABLE( AddIdentity, CryptonitDlg )
    EVT_BUTTON( wxID_OK, AddIdentity::onOk )
END_EVENT_TABLE()


AddIdentity::AddIdentity( User *u,
			  wxWindow *parent, wxWindowID id,
			  const wxString& title,
			  const wxPoint& pos,
			  const wxSize& size,
			  long style )
    : CryptonitDlg(parent, id, title, pos,  size, style)

{
    user = u;

    wxStaticBox *frame = new wxStaticBox(this , -1 , _("Identity importation method"));
    wxStaticBoxSizer *choiceSizer = new wxStaticBoxSizer(frame, wxVERTICAL);
    
    p12Radio = new wxRadioButton(this, AI_P12_RB_ID, _("Import identity from a PKCS#12 file"), wxDefaultPosition,wxDefaultSize, wxRB_GROUP);
    certRadio = new wxRadioButton(this, AI_CERT_RB_ID, _("Import identity from a certificate (signed certificate request)"), wxDefaultPosition,wxDefaultSize);
    
    p12Radio->SetValue(TRUE);
    certRadio->SetValue(FALSE);
    
    choiceSizer->Add(p12Radio, 0, wxEXPAND | wxCENTER |wxALL, 5);
    choiceSizer->Add(certRadio, 0, wxEXPAND | wxCENTER |wxALL, 5);
  
    mainSizer->Add(choiceSizer, 1, wxEXPAND | wxCENTER |wxALL, 10);
}


void AddIdentity::onOk(wxCommandEvent &WXUNUSED(event))
{
    /* Add from a PKCS12 */
    if( p12Radio->GetValue() == TRUE ) {

	Pkcs12Dlg p12dlg(this, -1 , true, _("Add a new identity to Cryptonit"));
	
	if( p12dlg.showModal() == wxID_OK ) {
	    
	    // If there is no identity name, set if to 'default'
	    if( p12dlg.getIdentity() == wxEmptyString ) {
					identity = _("Default");
			} else {
					identity = p12dlg.getIdentity();
			}

	    int res;
	    wxString errorMsg = wxEmptyString;

	    res = user->addIdentityFromP12( wx2std(identity), 
																			wx2std(p12dlg.getPkcs12Filename()),
																			wx2std(p12dlg.getPkcs12Password()) );

	    switch( res ) {
	    case -1: errorMsg = _("Cannot load PKCS#12.");
		break;
	    case -2: errorMsg = _("This PKCS#12 file already exist for this user.");
		break;
	    case -3: errorMsg = _("This identity is already in the list, please use a different name.");
		break;
	    case -4: errorMsg = _("An error occured while writing certificate on disk.");
		break;
	    case -5: errorMsg = _("Cannot add identity information into the addressbook.");
		break;
	    case -6: errorMsg = _("Cannot write on disk a CA certificate included in the PKCS#12.");
		break;
	    case -7: errorMsg = _("Cannot copy PKCS#12 in your user directory.");
		break;
	    }

	    if( errorMsg != wxEmptyString ) {
					wxMessageBox(errorMsg, _("Error"), wxOK | wxICON_ERROR, this);
					EndModal( wxID_OK );
					return;
	    }

	    p12.load( wx2std(p12dlg.getPkcs12Filename()).c_str(),
								wx2std(p12dlg.getPkcs12Password()).c_str() );
	    cert = p12.getCertificate();
	     //downloading CRL for this p12 certificate and CA
	    //CA
	    std::vector<Certificate> CA;
	    std::vector<Certificate>::iterator it;
	    CA = p12.getCA();
	    for( it = CA.begin(); it != CA.end(); it++ ) {
		std::string uri = it->extractCrlUri();
		if( uri != "" ){
		    wxString newCrlFile = AddCrlDlg::fetchCrl( std2wx(uri) );
		    Crl newCrl;
		    if( newCrl.load( wx2std(newCrlFile).c_str()) == SUCCESS ){
			Certificate newCert( *it );
			AddCrlDlg::saveCrl( std2wx(uri) , &newCrl , newCert , user ); 
		    } 
		}
	    }
	    //certificate
	    std::string uri = cert.extractCrlUri();
	    if( uri != "") {
		wxString newCrlFile = AddCrlDlg::fetchCrl( std2wx(uri) );
		Crl newCrl;
		if( newCrl.load( wx2std(newCrlFile).c_str()) == SUCCESS ){
		    AddCrlDlg::saveCrl( std2wx(uri) , &newCrl , cert , user ); 
		} 
	    }
	    
	}
    }

    /* Add from a certificate */
    else {

	// Prompt the user for the Certificate file
	wxString wildcard;
	wildcard << _("All supported formats") << _T("|*.cer;*.der;*.pem")
		 << _T("|") << _("Certificate") << _T(" (*.cer)|*.cer")
		 << _T("|") << _("DER encoded certificate") << _T(" (*.der)|*.der")
		 << _T("|") << _("PEM encoded certificate") << _T(" (*.pem)|*.pem");
	
	wxFileDialog fileDlg( this, _("Choose a signed certificate request"), wxGetCwd(), _T(""),
			      wildcard, wxOPEN | wxCHANGE_DIR);

	if( fileDlg.ShowModal() == wxID_OK ) {
	    wxFileName certFilename(fileDlg.GetFilename());
	    certFilename=wxMyPrepend(certFilename,fileDlg.GetDirectory());
			

	    Certificate certificate;
	    if (certificate.load( wx2std(certFilename.GetFullPath()).c_str()) != SUCCESS) {
					wxString msg;
					msg << _("Cannot load certificate file name: ") << certFilename.GetFullPath();
					wxMessageBox( msg, _("Error"), wxICON_ERROR | wxOK, this );
					EndModal( wxID_OK );
					return;
			}
		
	    wxFileName requestsDir( std2wx(user->getP12Dir()) );
			
	    wxDir dir;
	    if( dir.Open( requestsDir.GetFullPath() ) == false ) {
					wxString msg;
					msg << _("Cannot open your personnal P12 directory: ") 
							<< requestsDir.GetFullPath();
					wxMessageBox( msg, _("Error"), wxICON_ERROR | wxOK, this );
					
					EndModal( wxID_OK );
					return;
	    }
			
	    pkcs10 request;
	    bool match = false;

	    wxString filename;
	    wxFileName requestFilename;

	    bool cont = dir.GetFirst( &filename, _T("*.p10"), wxDIR_FILES );
			
	    // Search in the P12 user dir, the certificate request
	    // matching the given signed certificate
	    while( cont && ! match ) {
		requestFilename = wxFileName( filename.c_str() );
		requestFilename=wxMyPrepend(requestFilename, std2wx(user->getP12Dir()) );
#ifdef DEBUG			    
		std::cout << filename.c_str() << ": ";
#endif		
			    
		if( request.load( wx2std(requestFilename.GetFullPath()).c_str() ) == 0 ) {
		    if( (match = request.matchCertificate( certificate )) ) {
#ifdef DEBUG			    
			std::cout << "match !";
#endif		    
		    }
		}
#ifdef DEBUG		
		std::cout << std::endl;
#endif		
		cont = dir.GetNext(&filename);
	    }

	    // We find the matching certificate request
	    // Now we try to found the associated private key
	    if( match ) {
					cont = dir.GetFirst( &filename, requestFilename.GetName(), wxDIR_FILES );
					if( cont == false ) {
							wxMessageBox( _("Cannot find any private key associated with this certificate!"),
														_("Error"), wxICON_ERROR | wxOK, this );
							
							EndModal( wxID_OK );
							return;
					}

					while( cont ) {
							wxFileName privateKeyFilename( filename.c_str() );
							privateKeyFilename=wxMyPrepend(privateKeyFilename, std2wx(user->getP12Dir()) );
#ifdef DEBUG
							std::cout << "Key: " << filename.c_str() << std::endl;
#endif							
							// Ask for the private key password
							bool again = true;
							pkcs8 privateKey;
							wxString password;
							
		    while( again ) {
			PasswordDlg pd( this, -1 );

			if( pd.showModal() == wxID_OK ) {
			    password = pd.getPassword();

			    if( privateKey.load( wx2std(privateKeyFilename.GetFullPath()).c_str(), pem_format, wx2std(password).c_str() ) == false ) {
				wxMessageBox( _("Cannot open private key, may be an incorrect passphrase."),
					      _("Error"), wxICON_ERROR | wxOK, this );
				again = true;
			    } else {
				again = false;
			    }
			}
			else { // The user hit the "Cancel" button
			    EndModal( wxID_OK );
			    return;
			}
		    }



		    // Check if the private key match the certificate
		    if( ! request.matchPrivateKey(privateKey) ) {
			wxMessageBox( _("The request does not match the private key."),
				      _("Error"), wxICON_ERROR | wxOK, this );
			
			EndModal( wxID_OK );
			return;
		    }



		    // Ask for the identity, and check if it already exists
		    wxString identityName;
		    again = true;
		    std::vector<std::string> identities;
		    identities = user->getInfos( "Identities" );


		    while( again ) {

			wxTextEntryDialog identityDlg(this, _("Please enter a name for this new identity, you can leave this field blank: "),
						      _("Identity name"), identityName );
		    
			if( identityDlg.ShowModal() == wxID_OK ) {
			    if( identityDlg.GetValue() != wxEmptyString ) {
				identityName = identityDlg.GetValue();
			    } else {
				identityName = _("Default");
			    }
			}
			else {
			    // Aborting certificate importation
			    wxMessageBox( _("Aborting certificate importation."), _("Error"),
					  wxOK | wxICON_ERROR, this );
			    EndModal( wxID_OK );
			    return;
			}
		    
			// Check if the provided identity already exists
			if( find( identities.begin(), identities.end(), wx2std(identityName).c_str() )
			    != identities.end() ) {
			    // the identity is already in the list
			    wxMessageBox( _("This identity already exists, please choose another one."), 
					  _("Error"), wxOK | wxICON_ERROR, this );
			    again = true;
			} else {
			    again = false;
			}
		    }


		    int ret = user->addIdentityFromP10( wx2std(identityName), 
																						wx2std(requestFilename.GetFullPath()),
																						wx2std(certFilename.GetFullPath()),
																						wx2std(privateKeyFilename.GetFullPath()),
																						wx2std(password) );

		    wxString errorMsg = wxEmptyString;
		    switch( ret ) {
		    case 0 : errorMsg = wxEmptyString;
			break;
		    case -1: errorMsg = _("Cannot load the private key generated with the certificate request.");
			break;
		    case -2: errorMsg = _("Cannot load the originating certificate request.");
			break;
		    case -3: errorMsg = _("The private key does not match the certificate request.");
			break;
		    case -4: errorMsg = _("Cannot load the certificate.");
			break;
		    case -5: errorMsg = _("Cannot create the intermediate PKCS#12 object.");
			break;
		    case -6: errorMsg = _("Cannot create temporary file for the PKCS#12 object in your P12 directory.");
			break;
		    case -7: errorMsg = _("Cannot write the temporary PKCS#12 in your P12 directory.");
			break;
			// addIdentityFromP12 errors:
		    case -51: errorMsg = _("Cannot load the generated PKCS#12.");
			break;
		    case -52: errorMsg = _("This PKCS#12 file already exist for this user.");
			break;
		    case -53: errorMsg = _("This identity is already in the list, please use a different name.");
			break;
		    case -54: errorMsg = _("An error occured while writing certificate on disk.");
			break;
		    case -55: errorMsg = _("Cannot add identity information into the addressbook.");
			break;
		    case -56: errorMsg = _("Cannot write on disk a CA certificate included in the PKCS#12.");
			break;
		    case -57: errorMsg = _("Cannot copy PKCS#12 in your user directory.");
			break;
		    default:  errorMsg << _("Unknown error! ") << _T("(") << ret << _T(")");
			break;
		    }
		    
		    if( errorMsg != wxEmptyString ) {
			wxMessageBox(errorMsg, _("Error"), wxOK | wxICON_ERROR, this);
			EndModal( wxID_OK );
			return;
		    }
				
		    // The import was successfull, we can now remove 
		    // the temporary private key and request.
		    
		    // Remove privateKeyFilename
#ifdef DEBUG 
		    std::cout << "Remove " << privateKeyFilename.GetFullPath() << std::endl;
#endif		    
		    wxRemove( privateKeyFilename.GetFullPath() );
		    
		    // Remove requestFilename
#ifdef DEBUG 
		    std::cout << "Remove " << requestFilename.GetFullPath() << std::endl;
#endif		    
		    wxRemove( requestFilename.GetFullPath() );

    
		    // set the global variables 'cert' and 'p12'
		    cert.load(wx2std(certFilename.GetFullPath()).c_str());

		    std::string p12filename;
		    p12filename = appendDir( user->getP12Dir(), cert.getHash() );
		    p12filename = setExt(p12filename, "p12");
		    p12.load(p12filename.c_str(), wx2std(password).c_str());

		    //downloading CRL for this p12 certificate and CA
		    //CA
		    std::vector<Certificate> CA;
		    std::vector<Certificate>::iterator it;
		    CA = p12.getCA();
		    for( it = CA.begin(); it != CA.end(); it++ ) {
			std::string uri = it->extractCrlUri();
			if( uri != "" ){
			    wxString newCrlFile = AddCrlDlg::fetchCrl( std2wx(uri) );
			    Crl newCrl;
			    if( newCrl.load( wx2std(newCrlFile).c_str()) == SUCCESS ){
				Certificate newCert( *it );
				AddCrlDlg::saveCrl( std2wx(uri) , &newCrl , newCert , user ); 
			    } 
			}
		    }
		    //certificate
		    std::string uri = cert.extractCrlUri();
		    if( uri != "") {
						wxString newCrlFile = AddCrlDlg::fetchCrl( std2wx(uri) );
						Crl newCrl;
						if( newCrl.load( wx2std(newCrlFile).c_str()) == SUCCESS ){
								AddCrlDlg::saveCrl( std2wx(uri) , &newCrl , cert , user ); 
						} 
		    }	    
		    
		    EndModal( wxID_OK );
		    return;
		}
				
	    }
	    else {
		wxMessageBox(_("There is no certificate request in your personnal directory matching this certificate."),
			     _("Error"), wxOK | wxICON_ERROR, this);
		EndModal( wxID_OK );
		return;
	    }
	}
    }

    EndModal( wxID_OK );
    return;
}


Certificate& AddIdentity::getCertificate(){
    return cert ;
}



pkcs12& AddIdentity::getKey(){
    return p12 ;
}



wxString& AddIdentity::getIdentity(){
    return identity;
}




