#ifndef SETUP_H
#define SETUP_H

#include <stdint.h>
#include <stdarg.h>

#define SECTOR_SHIFT		9
#define SECTOR_SIZE		(1 << SECTOR_SHIFT)

#define	CRYPT_FLAG_PASSPHRASE	(1 << 0)
#define	CRYPT_FLAG_VERIFY	(1 << 1)

struct crypt_options {
	const char	*name;
	const char	*device;

	const char	*cipher;
	const char	*hash;

	const char	*passphrase;
	int		passphrase_fd;
	const char	*key_file;
	int		key_size;

	unsigned int	flags;

	uint64_t	size;
	uint64_t	offset;
	uint64_t	skip;
};

struct hash_type {
	const char	*name;
	void		*private;
	int		(*fn)(void *data, int size, char *key,
			      const char *passphrase);
};

struct hash_backend {
	const char		*name;
	struct hash_type *	(*get_hashes)(void);
	void			(*free_hashes)(struct hash_type *hashes);
};

struct setup_backend {
	const char	*name;
	int		(*init)(void);
	void		(*exit)(void);
	int		(*create)(int reload, struct crypt_options *options,
			          const char *key);
	int		(*status)(int details, struct crypt_options *options,
			          char **key);
	int		(*remove)(struct crypt_options *options);
};

void set_error_va(const char *fmt, va_list va);
void set_error(const char *fmt, ...);
const char *get_error(void);
void *safe_alloc(size_t size);
void safe_free(void *data);
void *safe_realloc(void *data, size_t size);
char *safe_strdup(const char *s);

struct hash_backend *get_hash_backend(const char *name);
void put_hash_backend(struct hash_backend *backend);
int hash(const char *backend_name, const char *hash_name,
         char *result, int size, const char *passphrase);

struct setup_backend *get_setup_backend(const char *name);
void put_setup_backend(struct setup_backend *backend);

int crypt_create_device(struct crypt_options *options);
int crypt_update_device(struct crypt_options *options);
int crypt_resize_device(struct crypt_options *options);
int crypt_query_device(struct crypt_options *options);
int crypt_remove_device(struct crypt_options *options);

void crypt_get_error(char *buf, size_t size);

#endif /* SETUP_H */
