/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */ 


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#define PLAT_GTK 1

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#define WITH_PLUGINS

#ifdef WIN32
# include <io.h>
# undef WITH_PLUGINS
#endif

#ifdef WITH_TERMINAL
#  include <vte/vte.h>
#endif

#include "cssedwindow.h"
#include "document.h"
#include "callbacks.h"
#include "selectorparser.h"
#include "floating-list.h"
#include "cssdialogs-interface.h"
#include "interface.h"
#include "selector-wizard.h"
#include "color-wizard.h"
#include "box-wizard.h"
#include "cssparser.h"
#include "doc-scanner.h"
#include "support.h"
#include "utils.h"
#include "debug.h"
#include "plugins-dialog.h"

#include <Scintilla.h>
#include <SciLexer.h>
#include <ScintillaWidget.h>
#include <libcroco.h>

#ifdef WITH_TERMINAL
void
on_vte_terminal_child_exited            (VteTerminal *vteterminal,
                                            gpointer user_data)
{
	gchar* shell;
	
	shell = getenv("SHELL");
	if( shell == NULL){
		vte_terminal_reset (vteterminal, TRUE, TRUE);
		vte_terminal_fork_command (vteterminal, "/bin/sh", NULL, NULL, "~/",
				   0, 0, 0);
	}else{
		if( g_file_test(shell, G_FILE_TEST_EXISTS) && g_file_test(shell,G_FILE_TEST_IS_EXECUTABLE) ){
			vte_terminal_reset (vteterminal, TRUE, TRUE);
			vte_terminal_fork_command (vteterminal, shell, NULL, NULL, "~/",
				   0, 0, 0);
		}else{
			vte_terminal_reset (vteterminal, TRUE, TRUE);
			vte_terminal_fork_command (vteterminal, "/bin/sh", NULL, NULL, "~/",
				   0, 0, 0);
		}
	}
}
#endif
void
on_notebook_docs_switch_page (GtkNotebook * notebook, GtkNotebookPage * page,
			      guint page_num, gpointer user_data)
{
	CssedDoc *tmpdoc;
	CssedWindow *window;
	GtkWidget *window_widget;
	gchar* strout;
	gchar* filename;
	gboolean modified;
	
	window = CSSED_WINDOW (user_data);
	DBGMSG ("callbacks.c: on_notebook_docs_switch_page - Page switched to %d, notebook %x, page %x", page_num, notebook, page);

	tmpdoc = document_get_from_notebook_page_num (window, page_num);
	if (tmpdoc == NULL)
	{
		cssed_error_message
			(_("Programming error!!"),
			 _("An invalid document has been passed,\nthings are going to fail from this point.\nPlease tell the developers this has occurred."));
		return;
	}
	else
	{
		filename = document_get_filename( tmpdoc );
		modified = document_get_modified( tmpdoc );
#ifdef DEBUG
		if ( filename != NULL )
			DBGMSG ("callbacks.c: on_notebook_docs_switch_page  - filename %s", filename);
		else
			DBGMSG ("callbacks.c: on_notebook_docs_switch_page() - filename NULL");
#endif
		window_widget = cssed_window_get_window_widget( window );
		strout = g_strdup_printf("cssed: %s",filename ==
				      NULL ? _("Untitled doc") : filename );
		gtk_window_set_title (GTK_WINDOW (window_widget),strout);

		if( document_get_modified( tmpdoc ) ){
			cssed_window_enable_save ( window );
		}else{
			cssed_window_disable_save( window );	
		}
		
		if( document_can_undo( tmpdoc ) ){
			cssed_window_enable_undo ( window );
		}else{
			cssed_window_disable_undo( window );
		}
		
		if( document_can_redo( tmpdoc ) ){
			cssed_window_enable_redo ( window );
		}else{
			cssed_window_disable_redo( window );
		}

		g_free(strout);
		g_free( filename );
	}	
}
void
on_menu_doc_info_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gint len;
	gchar* lenmsg;
	gchar* filename;
	gboolean modified;
	
	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	cssed_window_output_clear( window );

	filename = document_get_filename( tmpdoc );
	modified = document_get_modified( tmpdoc );
	
	if ( filename == NULL)
	{
		cssed_window_output_write ( window, _("Untitled doc"));
	}
	else
	{
		cssed_window_output_write ( window, filename );
	}

	if ( modified )
	{
		cssed_window_output_write ( window, _("Document modified (not saved)"));
	}
	else
	{
		cssed_window_output_write ( window, _("Document not modified (saved)"));
	}
	
	len = document_get_length(tmpdoc);
	lenmsg = g_strdup_printf( _("Document length: %d"), len ); 
	cssed_window_output_write ( window, lenmsg );	
	g_free(lenmsg);	
	
	if( document_get_autocompletion_enabled( tmpdoc ) ){
		cssed_window_output_write ( window,_("Autocompletion enabled for this document"));
	}else{
		cssed_window_output_write ( window,  _("Autocompletion disabled for this document"));
	}	
	
	if( document_get_folding_margin_visible(tmpdoc) ){
		cssed_window_output_write ( window, _("Folding enabled for this document"));
	}else{
		cssed_window_output_write ( window, _("Folding disabled for this document"));
	}	
	
	if( filename ) g_free( filename );
}

/* validation and dump*/
void
on_menu_utils_validate_activate(GtkMenuItem * menuitem, gpointer user_data)
{

	CssedWindow *window;
	CssedDoc *tmpdoc;
	gint doclen;
	gchar* doc_buffer;
	gchar* endbuff;
	
	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	
	if( document_get_filetype( tmpdoc ) == CSSED_FILETYPE_CSS ){	
		doclen = document_get_length (tmpdoc);
		doc_buffer = g_malloc0(++doclen);
		document_get_text (tmpdoc, doclen, doc_buffer);
		endbuff = g_strconcat(doc_buffer, "\n",NULL);	
		document_clear_validator_arrow( tmpdoc );
		cssed_cr_parser_parse_buffer (tmpdoc, endbuff,strlen(endbuff),TRUE);
		g_free( endbuff);
		g_free(doc_buffer);
	}else{
		cssed_error_message(_("I can only validate CSS files!"),_("Unsupported file type"));
	}
}
/* only validation */
void 
on_menu_utils_validate_only_activate	(GtkMenuItem 	  * menuitem,
								gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gint doclen;
	gchar* doc_buffer;
	gchar* endbuff;
	
	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	
	if( document_get_filetype( tmpdoc ) == CSSED_FILETYPE_CSS ){
		doclen = document_get_length (tmpdoc);
		doc_buffer = g_malloc (++doclen);
		document_get_text (tmpdoc, doclen, doc_buffer);
		endbuff = g_strconcat(doc_buffer, "\n",NULL);	
		document_clear_validator_arrow( tmpdoc );
		cssed_cr_parser_parse_buffer (tmpdoc, endbuff,strlen(endbuff),FALSE);
		g_free( endbuff);
		g_free(doc_buffer);	
	}else{
		cssed_error_message(_("I can only validate CSS files!"),_("Unsupported file type"));
	}
}


/* end validation */
#ifdef WITH_PLUGINS
void on_plugins_load_activate(GtkMenuItem * menuitem,
					      gpointer user_data)
{
	GtkWidget* dialog;
	dialog = create_plugins_dialog ( CSSED_WINDOW(user_data) );	
	gtk_widget_show( dialog );
}
#endif
/* this cleans the otuput list store */
void 
on_menu_utils_cleanoutput_activate		
				(GtkMenuItem 	  * menuitem,       gpointer user_data)
{
	cssed_window_output_clear( CSSED_WINDOW(user_data) );	
}

void
on_menu_new_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkNotebook *notebook;
	CssedWindow *window;
	
	window = CSSED_WINDOW (user_data);
	notebook = GTK_NOTEBOOK ( cssed_window_get_document_notebook ( window ) );

	if (!GTK_IS_NOTEBOOK (notebook))
	{
		cssed_error_message
			(_("Programming error"),_("Please see standard output and tell\n\nthe developers what happened!!!"));
		DBGMSG ("callbacks.c:  on_menu_new_activate()- %x Is NOT a GtkNotebook", notebook);
		return;
	}
	create_and_attach_new_doc (window, _("Untitled"));
	gtk_notebook_set_current_page( notebook, -1 );
}
void
on_menu_open_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	GtkWidget *opensel;
#ifdef GTK_ATLEAST_2_4
	gchar* filename;
	gchar* last_open_dir;
#endif

	window = CSSED_WINDOW (user_data);

#ifndef GTK_ATLEAST_2_4
	opensel = create_open_fileselection (window);
	if (opensel != NULL)
	{
		gtk_widget_show (opensel);
	}
#else 
	opensel = gtk_file_chooser_dialog_new(_("Select file"), NULL,
												GTK_FILE_CHOOSER_ACTION_OPEN,
												GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
												GTK_STOCK_OPEN, GTK_RESPONSE_OK, NULL);
	last_open_dir = cssed_window_get_last_open_dir( window );
	if( last_open_dir != NULL ){
		gtk_file_chooser_set_current_folder( GTK_FILE_CHOOSER (opensel), last_open_dir );
		g_free(last_open_dir);
	}

	if( gtk_dialog_run(GTK_DIALOG(opensel)) == GTK_RESPONSE_OK ){
		filename = gtk_file_chooser_get_filename( GTK_FILE_CHOOSER(opensel) );
		gtk_widget_destroy(opensel);
		create_and_attach_new_named_doc(window, filename);
		g_free(filename);
	}else{
		gtk_widget_destroy(opensel);
	}
#endif
}

void 
on_menu_close_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	// call the toolbar close button callback
	on_button_cerrar_clicked (NULL, user_data);
}

void 
on_menu_save_all_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc* doc;
	GtkWidget* confirm;
	gint response;
	GList* doc_list;
	gchar* doc_name;
	
	window = CSSED_WINDOW (user_data);
	doc_list = 	cssed_window_get_document_list_first(window);
	
	while( doc_list ){
		doc = (CssedDoc*) doc_list->data;
		if( document_get_modified(doc) ){
			doc_name = document_get_filename( doc );
			if( doc_name == NULL ){
				document_set_current(doc);
				confirm = create_ok_cancel_dialog (_("This file have no name"), _("I cannot know the location where to store this file.\nDo you want to select the location where this\nfile will be saved?"));
				response = gtk_dialog_run(GTK_DIALOG(confirm));
				if( response == GTK_RESPONSE_OK ){
					document_save(doc);					
				}
				gtk_widget_destroy( confirm );
			}else{
				document_save(doc);
			}				
		}
		doc_list = g_list_next(doc_list);
	}
}

void  
on_menu_close_all_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc* doc;
	GtkWidget* confirm;
	GtkWidget* fileselector;
	gint response;
	GList* doc_list;
	gchar* doc_name;
	gchar* err_msg;
	gchar* last_savedir;
	gchar* full_path;
	G_CONST_RETURN gchar* filename;
	
	window = CSSED_WINDOW (user_data);
	doc_list = 	cssed_window_get_document_list_first(window);
	
	while( doc_list ){
		doc = (CssedDoc*) doc_list->data;
		doc_list = g_list_next(doc_list);
		if( doc != NULL ){ 
			doc_name = document_get_filename( doc );
			if( document_get_modified(doc) ){	
				if( doc_name == NULL ){
					document_set_current(doc);
					confirm = create_yes_no_cancel_dialog (_("This file is not saved"), _("This file is modified and not saved.\nDo you want to select one location \n and save it there?\n\nUse <b>\"Cancel\"</b> to let the file opened.\n"));					
					response = gtk_dialog_run(GTK_DIALOG(confirm));
					gtk_widget_destroy( confirm );
					if( response == GTK_RESPONSE_YES ){
						// add the filename before save the file to avoid other prompts asking
						// for the file's name.
						last_savedir = cssed_window_get_last_save_dir( window );
#ifndef GTK_ATLEAST_2_4
						fileselector = gtk_file_selection_new(_("Save this file"));
#else
						fileselector = gtk_file_chooser_dialog_new(_("Save this file"),
																			NULL,
																			GTK_FILE_CHOOSER_ACTION_SAVE,
																			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
																			GTK_STOCK_SAVE, GTK_RESPONSE_OK,
																			NULL
																		);
#endif
						if( last_savedir != NULL ){
							full_path = g_strdup_printf("%s/", last_savedir );
#ifndef GTK_ATLEAST_2_4
							gtk_file_selection_set_filename(GTK_FILE_SELECTION(fileselector), full_path );
#else
							gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(fileselector), full_path );
#endif
							g_free( last_savedir );
							g_free( full_path );							
						}
						response = gtk_dialog_run( GTK_DIALOG(fileselector) );
						if( response == GTK_RESPONSE_OK ){
#ifndef GTK_ATLEAST_2_4
							filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fileselector));
#else
							filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fileselector));
#endif
							document_set_filename( doc, (gchar*) filename );
							gtk_widget_destroy( fileselector );
							document_save(doc);	
						}else{
							gtk_widget_destroy( fileselector );
						}												
					}else if( response == GTK_RESPONSE_CANCEL ){
							continue;
					}else{
						document_dispose( doc );
					}
				}else{
					document_set_current(doc);
					err_msg = g_strdup_printf(_("The file\n<b>%s</b>\nhas been modified and changes will be lost.\nDo you want to save it?\n\nUse <b>\"Cancel\"</b> to let the file opened.\n"),doc_name ); 
					confirm = create_yes_no_cancel_dialog (_("This file is not saved"), err_msg );
					response = gtk_dialog_run(GTK_DIALOG(confirm));
					gtk_widget_destroy( confirm );
					if( response == GTK_RESPONSE_YES ){
						document_save(doc);					
					}if( response == GTK_RESPONSE_CANCEL ){
						continue;
					}else{
						document_dispose(doc);
					}
				}				
			}else{
				document_dispose(doc);
			}
		}else{
			DBGMSG("on_menu_close_all_activate: NULL document on document list");
		}		
	}
}

void
on_menu_revert_to_saved_activate	 (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	GtkWidget* dialog;
	gint response;
	gchar* filename;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	dialog = create_ok_cancel_dialog (_("Please confirm this action"), _("This file will be closed, all changes will be lost,\nand it will be opened the previously saved version.\n\nDo you want to continue?\n"));
	response = gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);
	if( response == GTK_RESPONSE_OK ){
		filename = document_get_filename(tmpdoc);
		if( filename == NULL ){
			cssed_error_message (_("This file is not saved, unable to revert to previous file"), _("File not saved"));
			return;
		}else{
			if( g_file_test (filename, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR) ){
				document_dispose( tmpdoc );
				create_and_attach_new_named_doc(window, filename);		
			}else{
				cssed_error_message (_("The file is not saved on disk\nand may have been deleted by another process."), _("File does not exist"));
			}
			g_free( filename );
		}
	}
}

void
on_menu_guardar_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	document_save( tmpdoc );
}

void
on_save_fileselection_ok_button_clicked (GtkButton * button,
					 gpointer user_data)
{
	CssedDoc *savedoc;
	GtkWidget *notebook;
	GtkWidget *file_selector;
	CssedFileSelectorData *data;
	CssedWindow *window;	// main window object
	gchar *doc_buffer;
	gint doclen;
	const gchar *selected_filename;
	FILE *f;
	gchar *msg;
#ifdef WIN32
	int nwriten;
#else
	size_t nwriten;
#endif
	
	data = (CssedFileSelectorData *) user_data;
	window = (CssedWindow *) data->main_window;
	notebook = GTK_WIDGET (data->notebook);
	file_selector = GTK_WIDGET (data->fileselector);
	savedoc = document_get_current (window);

	selected_filename =
		gtk_file_selection_get_filename (GTK_FILE_SELECTION
						 (file_selector));
	gtk_widget_destroy (gtk_widget_get_toplevel (file_selector));	// destroy the selector

	f = fopen (selected_filename, "w");
	if (f == NULL)
	{
		msg = g_strdup_printf ("The file %s\ncannot be opened for writing.",
				       selected_filename);
		cssed_error_message ("Cannot open file", msg);
		g_free (msg);
	}
	else
	{
		doclen = document_get_length (savedoc);
		doc_buffer = g_malloc (++doclen);
		document_get_text (savedoc, doclen, doc_buffer);

		nwriten = fwrite (doc_buffer, sizeof (gchar), doclen, f);
		if (nwriten != doclen)
		{
				cssed_error_message
					("Error while writing to file",
					 "There is a mismatch between the file size\nand the bytes written.\nThe file may be damaged.");
		}
		document_set_save_point ( savedoc );
		document_empty_undo_buffer( savedoc );
		g_free (doc_buffer);
		fclose (f);
	}

	g_free (data);
}
// FIXME move this to a document function (ongoing: document_save_as() flaged to be skiped)
void
on_save_as_fileselection_ok_button_clicked (GtkButton * button,
					    gpointer user_data)
{
	CssedDoc *savedoc;
	CssedFileSelectorData *data;
	CssedWindow *window;	// main window object
	GtkWidget *window_widget;
	GtkWidget *notebook;
	GtkWidget *file_selector;
	GtkWidget *editor;
	GtkWidget* new_label;
	const gchar *selected_filename;
	gchar *doc_buffer;
	gchar *filename;
	gint doclen;
	gint current_index;
	FILE *f;
	gchar *msg;
	gchar* dir;
	CssedFileType filetype;
	gchar* basename;
#ifdef WIN32
	int nwriten;
#else
	size_t nwriten;
#endif
	
	
	data = (CssedFileSelectorData *) user_data;
	window = (CssedWindow *) data->main_window;
	notebook = GTK_WIDGET (data->notebook);
	file_selector = GTK_WIDGET (data->fileselector);
	savedoc = document_get_current (window);
	window_widget = cssed_window_get_window_widget( window );
	
	selected_filename =
		gtk_file_selection_get_filename (GTK_FILE_SELECTION(file_selector));
	gtk_widget_destroy (gtk_widget_get_toplevel (file_selector));	// destroy the selector
	
	if( g_file_test( selected_filename , G_FILE_TEST_EXISTS) ){
		GtkWidget* dialog;
		gint response;
	
		dialog = create_ok_cancel_dialog (_("File exists and will be overwritten"),
										  _("Do you want to overwrite the existing file with the current one?") );
		response = gtk_dialog_run( GTK_DIALOG(dialog) );
		gtk_widget_destroy( dialog );
		if( response != GTK_RESPONSE_OK ){
			return;
		}
	}

	f = fopen (selected_filename, "w");
	if (f == NULL)
	{
		msg = g_strdup_printf (_("The file %s\ncannot be opened for writing."),
				       selected_filename);
		cssed_error_message (msg,_("Cannot open file"));
		g_free (msg);
	}
	else
	{
		GList* list;
		GtkWidget* hbox;	
		
		doclen = document_get_length (savedoc);
		doc_buffer = g_malloc (++doclen);
		document_get_text (savedoc, doclen, doc_buffer);

		nwriten = fwrite (doc_buffer, sizeof (gchar), doclen, f);
		if (nwriten != doclen)
		{
			cssed_error_message
			(
			_("There is a mismatch between the file size\nand the bytes written.\nThe file may be damaged."),
			_("Error while writing to file")
			);
		}
		document_empty_undo_buffer(savedoc);
		document_set_save_point ( savedoc );
		current_index =	gtk_notebook_get_current_page (GTK_NOTEBOOK(notebook));
		editor = gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), current_index);
		filename = g_strdup (selected_filename);
		
		hbox = gtk_notebook_get_tab_label(GTK_NOTEBOOK (notebook), editor);
		list = gtk_container_get_children( GTK_CONTAINER(hbox) );
		list = g_list_first( list );
		new_label = GTK_WIDGET( list->data );
		g_list_free( list );
		
		if( new_label != NULL ){
			basename = g_path_get_basename  (filename);
			gtk_label_set_text(GTK_LABEL(new_label),basename);
			g_free(basename);
		}

		gtk_notebook_set_menu_label_text(GTK_NOTEBOOK (notebook), editor, filename );		
		document_set_filename( savedoc, filename );		
		filetype = document_get_filetype_from_filename( filename );
		
		if( filetype != CSSED_FILETYPE_UNKNOW ){
			document_set_filetype( 	savedoc, filetype );
		}

		gtk_window_set_title(GTK_WINDOW(window_widget),filename);

#ifdef WIN32
		dir = g_strconcat( g_dirname( filename ), "\\", NULL );
#else
		dir = g_strconcat( g_dirname( filename ), "/", NULL );
#endif

		cssed_window_set_last_save_dir( window, dir);
		
		g_free ( filename );
		g_free ( dir );
		g_free ( doc_buffer );
		fclose ( f );
	}
	g_free (data);
}
void
on_menu_guardar_como_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *file_selector_dlg;
	GtkWidget* filesel;
	gchar* filename;
	CssedWindow *window;	// main window object
	CssedDoc* doc;
#ifdef GTK_ATLEAST_2_4	
	gchar* last_save_dir;
#endif

	window = CSSED_WINDOW (user_data);
	doc = document_get_current( window );
	filename = document_get_filename( doc );

#ifndef GTK_ATLEAST_2_4
	file_selector_dlg = create_save_as_fileselection (window);
	
	filesel = lookup_widget( GTK_WIDGET(file_selector_dlg),"save_as_fileselection");	
	if( filename != NULL ){
		gtk_file_selection_set_filename( GTK_FILE_SELECTION (filesel), filename );
		g_free( filename );
	}
	gtk_widget_show ( file_selector_dlg );
#else
	file_selector_dlg = gtk_file_chooser_dialog_new(_("Select file"), NULL,
															GTK_FILE_CHOOSER_ACTION_SAVE,
															GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
															GTK_STOCK_SAVE, GTK_RESPONSE_OK,
															NULL );
	if( filename != NULL ){
		gtk_file_chooser_set_filename( GTK_FILE_CHOOSER (file_selector_dlg),	filename );
	}else{
		last_save_dir = cssed_window_get_last_save_dir(window);
		if(last_save_dir != NULL){
			gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(file_selector_dlg), last_save_dir);
			g_free(last_save_dir);
		}
	}
	if( gtk_dialog_run(GTK_DIALOG(file_selector_dlg)) ==  GTK_RESPONSE_OK ){
		filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(file_selector_dlg));
		gtk_widget_destroy(file_selector_dlg);
		document_save_as(doc, filename);		
		g_free(filename);	
	}else{
		gtk_widget_destroy(file_selector_dlg);
	}
#endif
}

// called on window close
void
on_menu_salir_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;	
	window = CSSED_WINDOW (user_data);
	cssed_window_quit( window );	
}

void
on_menu_undo_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	on_button_undo_clicked( NULL, user_data);
}
	
void
on_menu_redo_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	on_button_redo_clicked( NULL, user_data );
}
	
void
on_menu_search_activate (GtkMenuItem *menuitem, gpointer  user_data)
{
	on_button_buscar_clicked( NULL, user_data );
}
	
void
on_menu_replace_activate (GtkMenuItem  *menuitem, gpointer user_data)
{
	on_button_reemplazar_clicked( NULL, user_data );
}

void
on_menu_cortar_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	on_button_cortar_clicked (NULL, user_data);
}
void
on_menu_copiar_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	on_button_copiar_clicked (NULL, user_data);
}
void
on_menu_pegar_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	on_button_pegar_clicked (NULL, user_data);
}


void
on_menu_borrar_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	
	document_clear_sel(tmpdoc);
}

void
on_menu_preferences_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	on_button_preferences_clicked( NULL, user_data );
}

void
on_menu_view_toolbar_toggled(GtkCheckMenuItem      *checkmenuitem,   
	gpointer user_data)
{
	gboolean active;	
	active =  gtk_check_menu_item_get_active( checkmenuitem );
	if( active ){
		gtk_widget_show(GTK_WIDGET(user_data));
	}else{
		gtk_widget_hide(GTK_WIDGET(user_data));
	}
}

void
on_menu_about_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *about = create_about ();
	if (about != NULL)
		gtk_widget_show (about);
}

#ifdef WITH_HELP_MENUS
#ifdef DARWIN
void
on_menu_cssed_website_with_safari_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* cssedwebsiteurl;
	
	cssedwebsiteurl = "http://cssed.sourceforge.net/index.en.php";
	
	command = g_strdup_printf ( "open -a 'Safari' %s", cssedwebsiteurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_website_with_firefox_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* cssedwebsiteurl;
	
	cssedwebsiteurl = "http://cssed.sourceforge.net/index.en.php";
	
	command = g_strdup_printf ( "launch -a 'Firefox' %s", cssedwebsiteurl);
	system(command);
	
	g_free( command );
}
#endif

void
on_menu_cssed_website_with_mozilla_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* cssedwebsiteurl;
	
	cssedwebsiteurl = "http://cssed.sourceforge.net/index.en.php";
	
	command = g_strdup_printf ( "mozilla %s &", cssedwebsiteurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_website_with_galeon_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* cssedwebsiteurl;
	
	cssedwebsiteurl = "http://cssed.sourceforge.net/index.en.php";
	
	command = g_strdup_printf ( "galeon -x %s &", cssedwebsiteurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_website_with_dillo_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* cssedwebsiteurl;
	
	cssedwebsiteurl = "http://cssed.sourceforge.net/index.en.php";
	
	command = g_strdup_printf ( "dillo %s &", cssedwebsiteurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_website_with_gnome_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* cssedwebsiteurl;
	
	cssedwebsiteurl = "http://cssed.sourceforge.net/index.en.php";
	
	command = g_strdup_printf ( "gnome-moz-remote --newwin %s", cssedwebsiteurl);
	system(command);
	
	g_free( command );
}

#ifdef DARWIN
void
on_menu_cssed_doc_online_with_safari_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* csseddoconlineurl;
	
	csseddoconlineurl = "http://cssed.sourceforge.net/manual/index.html";
	
	command = g_strdup_printf ( "open -a 'Safari' %s", csseddoconlineurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_doc_online_with_firefox_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* csseddoconlineurl;
	
	csseddoconlineurl = "http://cssed.sourceforge.net/manual/index.html";
	
	command = g_strdup_printf ( "launch -a 'Firefox' %s", csseddoconlineurl);
	system(command);
	
	g_free( command );
}
#endif

void
on_menu_cssed_doc_online_with_mozilla_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* csseddoconlineurl;
	
	csseddoconlineurl = "http://cssed.sourceforge.net/manual/index.html";
	
	command = g_strdup_printf ( "mozilla %s &", csseddoconlineurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_doc_online_with_galeon_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* csseddoconlineurl;
	
	csseddoconlineurl = "http://cssed.sourceforge.net/manual/index.html";
	
	command = g_strdup_printf ( "galeon -x %s &", csseddoconlineurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_doc_online_with_dillo_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* csseddoconlineurl;
	
	csseddoconlineurl = "http://cssed.sourceforge.net/manual/index.html";
	
	command = g_strdup_printf ( "dillo %s &", csseddoconlineurl);
	system(command);
	
	g_free( command );
}

void
on_menu_cssed_doc_online_with_gnome_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	gchar* command;
	gchar* csseddoconlineurl;
	
	csseddoconlineurl = "http://cssed.sourceforge.net/manual/index.html";
	
	command = g_strdup_printf ( "gnome-moz-remote --newwin %s", csseddoconlineurl);
	system(command);
	
	g_free( command );
}

#endif /* #ifdef WITH_HELP_MENUS */

/*    For future use
void 
on_menu_cssed_help_activate  (GtkMenuItem  *menuitem, gpointer  user_data)
{   // THIS IS OLD TESTING STUFF
	GtkWidget *help_window = create_help_window ();
	if (help_window != NULL){
		gtk_widget_show (help_window);
		help_window_show_file (help_window,"t1.html");
	}
}
*/

/* dynamic menu items */       
void
on_menu_line_numbers_toggled (GtkCheckMenuItem * checkmenuitem,
			      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gboolean visible;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{			// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_line_numbers_toggled  - document NULL");
		return;
	}
	else
	{
		visible = gtk_check_menu_item_get_active (checkmenuitem);
		document_set_line_numbers_visible (tmpdoc, visible);
	}
}

void
on_menu_line_endings_toggled (GtkCheckMenuItem * checkmenuitem,
			      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gboolean visible;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{			// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_line_endings_toggled - document NULL");
		return;
	}
	else
	{
		visible = gtk_check_menu_item_get_active (checkmenuitem);
		document_set_line_endings_visible (tmpdoc, visible);
	}
}

void
on_menu_line_wrapping_toggled (GtkCheckMenuItem * checkmenuitem,
			       gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gboolean wrapped;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{			// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_line_endings_toggled - document NULL");
		return;
	}
	else
	{
		wrapped = gtk_check_menu_item_get_active (checkmenuitem);
		document_set_lines_wrapped (tmpdoc, wrapped);
	}
}

void
on_menu_ver_white_spaces_toggled (GtkCheckMenuItem * checkmenuitem,
				  gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gboolean visible;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{	// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_ver_white_spaces_toggled - document NULL");
		return;
	}
	else
	{
		visible = gtk_check_menu_item_get_active (checkmenuitem);
		document_set_white_spaces_visible (tmpdoc, visible);
	}
}

void
on_menu_item_autocompletion_toggled	(GtkCheckMenuItem      *checkmenuitem,   
	gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gboolean active;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	active =  gtk_check_menu_item_get_active (checkmenuitem);
	document_set_autocompletion_enabled( tmpdoc, active );
}

void
on_menu_item_folding_toggled	(GtkCheckMenuItem      *checkmenuitem,   
	gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gboolean fold;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{			// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_line_numbers_toggled  - document NULL");
		return;
	}
	else
	{
		fold = gtk_check_menu_item_get_active (checkmenuitem);
		if( !fold )
			document_unfold_all(tmpdoc);	
		document_set_folding_margin_visible (tmpdoc, fold);
	}
}

void
on_menu_item_font_activate( GtkMenuItem *menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	CssedConfig* cfg;
	GtkWidget* font_selector;
	gint response;
	gchar* font;
	gchar* family_name;
	gint size;
	PangoFontDescription* pfd; 
	
	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	cfg = cssed_window_get_config(window);
	
	font_selector = gtk_font_selection_dialog_new (_("Select font for current document") );
	
	if( cfg->font != NULL){
		gchar* cfont;
		gchar* font_str;
		cfont = g_strdup(cfg->font);
		g_strdelimit(cfont,"!",' ');
		font_str = g_strdup_printf("%s, %d",cfont,cfg->font_size);
		gtk_font_selection_dialog_set_font_name(
							GTK_FONT_SELECTION_DIALOG(font_selector),
							font_str );
		g_free(font_str);
		g_free(cfont);
	}
	
	response = gtk_dialog_run(GTK_DIALOG(font_selector));
	if( response == GTK_RESPONSE_OK ){
		font =  gtk_font_selection_dialog_get_font_name(
							GTK_FONT_SELECTION_DIALOG(font_selector));
		if( font != NULL){
			pfd = pango_font_description_from_string( font );
			family_name = (gchar*) pango_font_description_get_family ( pfd );
			size = pango_font_description_get_size ( pfd );
			font = g_strdup_printf("!%s",family_name);
			document_set_font_by_name(tmpdoc, font);
			document_set_font_size(tmpdoc, size/PANGO_SCALE);
			document_colourise_all(tmpdoc);	
			g_free(font);
		}
	}
	gtk_widget_destroy(font_selector);
}

void 
on_menuitem_folding_unfoldall_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	document_unfold_all(tmpdoc);
}

void
on_menuitem_fonding_foldall_activate(GtkMenuItem  *menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_fold_all(tmpdoc);	
}

void 
on_menuitem_highlighting_text_activate(GtkMenuItem *menuitem,  
										gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);		
	document_set_filetype( tmpdoc, CSSED_FILETYPE_TEXT );	
}
/* FIXME - this should be moved to document functions */
void
on_menuitem_highlighting_css_activate(GtkMenuItem *menuitem, 
										gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);		
	document_set_filetype( tmpdoc, CSSED_FILETYPE_CSS );	
}

void
on_menuitem_highlighting_html_activate(GtkMenuItem *menuitem, 
										gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);		
	document_set_filetype( tmpdoc, CSSED_FILETYPE_HTML );	
}

void
on_menuitem_highlighting_sh_activate(GtkMenuItem *menuitem, 
										gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	
	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);		
	document_set_filetype( tmpdoc, CSSED_FILETYPE_SH );	
}
void 
on_menuitem_highlighting_c_activate(GtkMenuItem * menuitem,
					      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);		
	document_set_filetype( tmpdoc, CSSED_FILETYPE_C );
}
void 
on_menuitem_highlighting_apache_conf_activate(GtkMenuItem * menuitem,
					      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);		
	document_set_filetype( tmpdoc, CSSED_FILETYPE_APACHE_CONF );		
}
void on_menuitem_highlighting_perl_activate(GtkMenuItem * menuitem,
					      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_filetype( tmpdoc, CSSED_FILETYPE_PERL );
}

void 
on_menuitem_highlighting_python_activate(GtkMenuItem * menuitem,
					      					gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_filetype( tmpdoc, CSSED_FILETYPE_PYTHON );
}
void 
on_menuitem_highlighting_xml_activate(GtkMenuItem * menuitem,
					      				gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_filetype( tmpdoc, CSSED_FILETYPE_XML );
}

void 
on_menuitem_highlighting_diff_activate(GtkMenuItem * menuitem, 
											gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_filetype( tmpdoc, CSSED_FILETYPE_DIFF );
}

void 
on_menuitem_highlighting_make_activate(GtkMenuItem * menuitem, 
											gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_filetype( tmpdoc, CSSED_FILETYPE_MAKE );
}

/* ********************************************************* */
void 
on_menuitem_encoding_default_activate	(GtkMenuItem * menuitem,
					      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_codepage( tmpdoc, CSSED_CODEPAGE_DEFAULT );
}
void 
on_menuitem_encoding_utf8_activate(GtkMenuItem * menuitem,
					      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_codepage( tmpdoc, CSSED_CODEPAGE_UTF8 );
}
void 
on_menuitem_encoding_dbcs_activate	(GtkMenuItem * menuitem,
					      gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	document_set_codepage( tmpdoc, CSSED_CODEPAGE_DBCS );
}
/* ********************************************************* */
void 
on_menuitem_charset_ansi_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_set_character_set(tmpdoc, SC_CHARSET_ANSI);
}
void 
on_menuitem_charset_easteurope_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_set_character_set(tmpdoc, SC_CHARSET_EASTEUROPE);
}
void 
on_menuitem_charset_gb2312_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_set_character_set(tmpdoc, SC_CHARSET_GB2312 );
}
void 
on_menuitem_charset_hangul_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_set_character_set(tmpdoc, SC_CHARSET_HANGUL);
}
void 
on_menuitem_charset_shiftjis_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_set_character_set(tmpdoc, SC_CHARSET_SHIFTJIS);
}
/* ********************************************************* */
void
on_menuitem_selector_wizard_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc* doc;
	GtkWidget* selector_wizard;
	GtkWidget* notebook;
	GtkWidget* treeview;
	GtkWidget* radio;
	GtkTreeModel* model;
	GtkTreeIter iter;
	gboolean one_declaration_for_each;
	gboolean valid;
	gchar *strtoadd;
	gchar *strout;	
	gchar* list_name;
	gint page_active;
	gint response;
	

	window = CSSED_WINDOW (user_data);	
	selector_wizard = create_selector_wizard ();
	response = gtk_dialog_run(GTK_DIALOG(selector_wizard));
	doc = document_get_current(window);
	
	if( response ==  GTK_RESPONSE_OK)
	{
		notebook = lookup_widget(selector_wizard, "notebook_wizard");
		radio = lookup_widget(selector_wizard, "radiobutton_for_each");
		page_active = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
		switch( page_active ){
			case 0: // selector
				list_name = "treeview_selector_selected";
				break;
			case 1: // child
				list_name = "treeview_child_selected";
				break;				
			case 2: //class
				list_name = "treeview_class_selected";
				break;
			case 3: // sibling
				list_name = "treeview_sibling_selected";
				break;
			case 4: // contextual
				list_name = "treeview_contextual_selected";
				break;	
			default:
				DBGMSG("on_menuitem_selector_wizard_activate: notebook page out of range");
				return;
		}
		treeview = lookup_widget(selector_wizard,list_name);
		model = gtk_tree_view_get_model (GTK_TREE_VIEW(treeview));
		one_declaration_for_each =
				gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(radio));			
		valid = gtk_tree_model_get_iter_first(model, &iter);
		
		if( one_declaration_for_each ){
			while(valid){
				gtk_tree_model_get (model, &iter, 0, &strtoadd, -1);
				strout = g_strdup_printf("%s {\n\n}\n\n",strtoadd);
				document_add_text(doc,strout);
				g_free(strout);
				valid = gtk_tree_model_iter_next(model,&iter);
			}				
		}else{
			while(valid){
				gtk_tree_model_get (model, &iter, 0, &strtoadd, -1);
				strout = g_strdup_printf("%s",strtoadd);
				document_add_text(doc,strout);
				g_free(strout);
				valid = gtk_tree_model_iter_next(model,&iter);
				if(valid) document_add_text(doc,",\n");					
			}	
			document_add_text(doc,"	{\n\n}\n\n");
		}
	}
	gtk_widget_destroy(selector_wizard);
}

void
on_menuitem_box_wizard_activate	(GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget* dlg;
	dlg = create_box_wizard (CSSED_WINDOW(user_data));
	gtk_widget_show(dlg);
}

void
on_menuitem_color_wizard_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget* color_wizard;
	CssedWindow *window;
	
	window = CSSED_WINDOW (user_data);
	color_wizard = create_color_wizard_dialog (window);	
	gtk_widget_show(color_wizard);
}

void
on_menu_eol_mode_cr_activate (GtkMenuItem * checkmenuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	CssedEolMode eols;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	
	if (tmpdoc == NULL)
	{			// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_eol_mode_cr_activate - document NULL");
		return;
	}
	else
	{
		eols = document_get_eol_mode(tmpdoc);
		if( eols != EOL_MODE_CR ){
			document_set_eol_mode ( tmpdoc, EOL_MODE_CR );
			document_convert_eols ( tmpdoc, EOL_MODE_CR );
		}
	}
}
// needs fix
void
on_menu_eol_mode_lf_activate (GtkMenuItem * checkmenuitem, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	CssedEolMode eols;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{			// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_eol_mode_lf_activate - document NULL");
		return;
	}
	else
	{
		eols = document_get_eol_mode(tmpdoc);
		if( eols != EOL_MODE_LF ){
			document_set_eol_mode ( tmpdoc, EOL_MODE_LF );
			document_convert_eols ( tmpdoc, EOL_MODE_LF );
		}
	}
}


// needs fix
void
on_menu_eol_mode_crlf_activate (GtkMenuItem * checkmenuitem,
				gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	CssedEolMode eols;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{			// FIXME say something to user
		DBGMSG ("callbacks.c:on_menu_eol_mode_crlf_activate - document NULL");
		return;
	}
	else
	{
		eols = document_get_eol_mode(tmpdoc);
		if( eols != EOL_MODE_CRLF ){
			document_set_eol_mode ( tmpdoc, EOL_MODE_CRLF );
			document_convert_eols ( tmpdoc, EOL_MODE_CRLF );
		}
	}
}

/* set the child menus state */
void
on_menu_view_activate (GtkMenuItem * checkmenuitem, gpointer user_data)
{
	CssedDoc *tmpdoc;
	CssedWindow *window;
	CssedEolMode mode;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	if (tmpdoc == NULL)
	{
		cssed_error_message
			(
			_("Cannot get current document to change menus.\nMenus state will be out of sync with the document\n"),
			_("Menu error")
			);
		return;
	}	
	cssed_window_dynamic_menus_set_state ( 
							window,
							document_get_line_numbers_visible ( tmpdoc ),
							document_get_line_endings_visible ( tmpdoc ),
							document_get_white_spaces_visible ( tmpdoc ),
							document_get_lines_wrapped ( tmpdoc ),
							document_get_autocompletion_enabled ( tmpdoc ),
							document_get_folding_margin_visible ( tmpdoc ) 
							);	
	
	mode = document_get_eol_mode( tmpdoc );
	cssed_window_dynamic_menus_set_eols_state( window, mode ); 
}

/* 
	scan the current selector: extract the buffer and calls 
	parser_css_selector_buffer_to_list()
*/
void
on_menu_scan_selector_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	CssedDoc *tmpdoc;
	CssedWindow *window;
	ScintillaObject *sci;
	gint position;
	gint open_brace;
	gint close_brace;
	gchar *buffer;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);

	// FIXME this will go to a document function
	sci = SCINTILLA( document_get_scintilla_widget(tmpdoc) );
	position = document_get_current_position (tmpdoc);

	DBGMSG ("callbacks.c:on_menu_scan_selector_activate() Position at %d", position);

	// set the anchor
	SSM (sci, SCI_SEARCHANCHOR, 0, 0);
	open_brace = SSM (sci, SCI_SEARCHPREV, 0, (sptr_t) "{");

#ifdef DEBUG
	if (open_brace == -1)
		DBGMSG ("callbacks.c:on_menu_scan_selector_activate() No opening brace before");
	else
		DBGMSG ("callbacks.c:on_menu_scan_selector_activate() Opening brace at %d", open_brace);
#endif

	SSM (sci, SCI_SEARCHANCHOR, 0, 0);
	close_brace = SSM (sci, SCI_SEARCHNEXT, 0, (sptr_t) "}");

#ifdef DEBUG
	if (close_brace == -1)
		DBGMSG ("callbacks.c:on_menu_scan_selector_activate() No closing brace after");
	else
		DBGMSG ("callbacks.c:on_menu_scan_selector_activate() Closing brace at %d", close_brace);
#endif

	if (close_brace < position)
	{
		cssed_error_message
			(
			_("You should place the cursor between the opening\nand closing braces of any declaration to be scanned\n"),
			_("The cursor must be into a declaration.")
			);
	}
	else
	{
		if ((open_brace != -1) && (close_brace != -1))
		{		// we're in a selector then
			SSM (sci, SCI_SETSEL, open_brace + 1, close_brace);
			buffer = g_malloc ((close_brace -
					    open_brace) * sizeof (gchar));
			SSM (sci, SCI_GETSELTEXT, 0, (sptr_t) buffer);

			g_strstrip (buffer);
			parser_css_selector_buffer_to_list ( window, buffer );
		}
	}
}

void
on_show_footer_panel_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(menuitem),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	//pos = gtk_paned_get_position(GTK_PANED(user_data));

	gtk_paned_set_position(GTK_PANED(user_data) ,height-(height/3));
}

void
on_hide_footer_panel_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	//gint pos;
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(menuitem),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	//pos = gtk_paned_get_position(GTK_PANED(user_data));
	gtk_paned_set_position(GTK_PANED(user_data) ,height);
}

void
on_show_side_panel_activate (GtkMenuItem *menuitem, gpointer user_data)
{
	//gint pos;
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(menuitem),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	//pos = gtk_paned_get_position(GTK_PANED(user_data));

	gtk_paned_set_position(GTK_PANED(user_data) ,width-(width/3));
}

void
on_hide_side_panel_activate(GtkMenuItem *menuitem, gpointer user_data)
{
	//gint pos;
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(menuitem),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	//pos = gtk_paned_get_position(GTK_PANED(user_data));
	gtk_paned_set_position(GTK_PANED(user_data) ,width);	
}

void
on_button_abrir_clicked (GtkButton * button, gpointer user_data)
{
	on_menu_open_activate (NULL, user_data);
}
void
on_button_nuevo_clicked (GtkButton * button, gpointer user_data)
{
	on_menu_new_activate (NULL, user_data);
}

void
on_button_cerrar_clicked (GtkButton * button, gpointer user_data)
{
	CssedDoc *doc;
	CssedWindow *window;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current( window );
	document_close( doc );
}

void
on_button_buscar_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *searchdlg;

	searchdlg = create_search_dialog (user_data);
	gtk_widget_show (searchdlg);
}
void
on_button_reemplazar_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *replacedlg;

	replacedlg = create_replace_dialog (user_data);	
	gtk_widget_show (replacedlg);
}
void
on_button_copiar_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_copy (tmpdoc);
}
void
on_button_pegar_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_paste (tmpdoc);
}

void
on_button_cortar_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;

	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);
	document_cut (tmpdoc);
}

void
on_button_color_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *colorselection_dlg;
	gchar* str;
	CssedWindow *window;

	window = CSSED_WINDOW (user_data);

	str = g_strdup("Choose color"); // freed by callbacks
	colorselection_dlg = create_color_selector_dialog (window, str, FALSE, TRUE);
	gtk_widget_show (colorselection_dlg);
}


void
on_button_guardar_clicked (GtkButton * button, gpointer user_data)
{
	on_menu_guardar_activate (NULL, user_data);
}


void
on_button_guardarcomo_clicked (GtkButton * button, gpointer user_data)
{
	on_menu_guardar_como_activate (NULL, user_data);
}

void
on_button_next_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *doc;
	gint line;
	gint position;


	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);
	position = document_get_current_position (doc);
	line = document_get_line_from_position (doc, position);

	if (!document_marker_next (doc, ++line, 1))
	{
		cssed_error_message (
				_("There is no marker between the current\nposition and the end of the document."),
				_("No next marker")
			);
	}else{
		document_ensure_line_is_visible(doc,document_get_line_from_position(doc, document_get_current_position(doc)));
	}
}

void
on_button_prev_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *doc;
	gint line;
	gint position;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);
	position = document_get_current_position (doc);
	line = document_get_line_from_position (doc, position);

	if (!document_marker_prev (doc, --line, 1))
	{
		cssed_error_message (
			_("There is no marker between the current\nposition and the start of the document."),
			_("No previous marker")
		);
	}{
		document_ensure_line_is_visible(doc,document_get_line_from_position(doc, document_get_current_position(doc)));
	}
}

void
on_button_undo_clicked (GtkButton * button, gpointer user_data)
{
	CssedDoc *doc;
	CssedWindow *window;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current( window );
	document_undo (doc);
}

void
on_button_redo_clicked (GtkButton * button, gpointer user_data)
{
	CssedDoc *doc;
	CssedWindow *window;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current( window );
	document_redo (doc);
}

void
on_button_zoom_in_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *doc;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);
	document_zoom_in (doc);
}
void 
on_button_hide_side_clicked (GtkButton * button, gpointer user_data)
{
	gint pos;
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(button),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	pos = gtk_paned_get_position(GTK_PANED(user_data));
	gtk_paned_set_position(GTK_PANED(user_data) ,width);	
}
void
on_button_show_side_clicked (GtkButton * button, gpointer user_data)
{
	gint pos;
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(button),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	pos = gtk_paned_get_position(GTK_PANED(user_data));

	gtk_paned_set_position(GTK_PANED(user_data) ,width-(width/3));
}
void 
on_button_hide_footer_clicked 	(GtkButton * button, gpointer user_data)
{

	gint pos;
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(button),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	pos = gtk_paned_get_position(GTK_PANED(user_data));
	gtk_paned_set_position(GTK_PANED(user_data) ,height);

}
void 
on_button_show_footer_clicked (GtkButton * button, gpointer user_data)
{
	gint pos;
	GtkWidget* win;
	gint width;
	gint height;

	win = lookup_widget(GTK_WIDGET(button),"window");
 	gtk_window_get_size(GTK_WINDOW(win), &width, &height);
	pos = gtk_paned_get_position(GTK_PANED(user_data));

	gtk_paned_set_position(GTK_PANED(user_data) ,height-(height/3));
}
void
on_button_zoom_out_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *doc;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);
	document_zoom_out (doc);
}

void
on_button_zoom_off_clicked (GtkButton * button, gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *doc;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);
	document_zoom_off (doc);
}

void
on_button_preferences_clicked(GtkButton * button, gpointer user_data)
{
	CssedWindow* window;

	window = CSSED_WINDOW( user_data );
	cssed_window_show_preferences_dialog( window );
}

void
on_window_destroy (GtkObject * object, gpointer user_data)
{
	gtk_main_quit();
}

gboolean    on_window_delete_event      	(GtkWidget *widget,
                                            GdkEvent *event,
                                            gpointer user_data)
{
	CssedWindow *window;	
	window = CSSED_WINDOW (user_data);
	cssed_window_quit( window );
	return TRUE;
}
// secondary toolbar callbacks
void
on_button_border_all_clicked (GtkButton  *button, 	gpointer user_data)
{
	GtkWidget* dialog;
	dialog = create_border_properties_dialog (CSSED_WINDOW(user_data),
				 								g_strdup("border"),
												TRUE,
												TRUE);
	gtk_widget_show(dialog);	
}
void 
on_button_margin_all_clicked (GtkButton  *button, 	gpointer user_data)
{
	GtkWidget* dialog;
	dialog = create_margin_property_dialog (CSSED_WINDOW(user_data),
				 								g_strdup("margin"),
												TRUE,
												TRUE);
	gtk_widget_show(dialog);		
}
void 
on_button_padding_all_clicked (GtkButton  *button, 	gpointer user_data)
{
	GtkWidget* dialog;
	dialog = create_padding_property_dialog (CSSED_WINDOW(user_data),
				 								g_strdup("padding"),
												TRUE,
												TRUE);
	gtk_widget_show(dialog);		
}
void
on_button_box_properties_clicked(GtkButton *button,gpointer user_data)
{
	GtkWidget* dialog;
	dialog = create_box_wizard(CSSED_WINDOW(user_data));
	gtk_widget_show(dialog);
}
void 
on_button_sel_wizard_clicked (GtkButton  *button, 	gpointer user_data)
{
	on_menuitem_selector_wizard_activate(NULL, CSSED_WINDOW(user_data));	
}
void 
on_button_col_wizard_clicked (GtkButton  *button, 	gpointer user_data)
{
	GtkWidget* dialog;
	dialog = create_color_wizard_dialog(CSSED_WINDOW(user_data));
	gtk_widget_show(dialog);	
}
void 
on_button_font_clicked	 (GtkButton  *button, 	gpointer user_data)
{
	GtkWidget* dialog;
	dialog = create_font_property_dialog (CSSED_WINDOW(user_data),
				 								g_strdup("font"),
												TRUE,
												TRUE);
	gtk_widget_show(dialog);	
}
void 
on_button_scan_selector_clicked  (GtkButton  *button, 	gpointer user_data)
{
	on_menu_scan_selector_activate(NULL, user_data);
}
void 
on_button_doc_info_clicked (GtkButton  *button, 	gpointer user_data)
{
	on_menu_doc_info_activate(NULL, user_data);
}
void 
on_button_clean_output_clicked  (GtkButton  *button, 	gpointer user_data)
{
	on_menu_utils_cleanoutput_activate(NULL, user_data);
}
void 
on_button_validate_clicked  (GtkButton  *button, 	gpointer user_data)
{
	on_menu_utils_validate_only_activate(NULL, user_data);
}
void 
on_button_validate_dump_clicked  (GtkButton  *button, 	gpointer user_data)
{
	on_menu_utils_validate_activate(NULL, user_data);
}
// end secondary toolbar callbacks
void
on_search_dialog_find_button_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *entry_text_to_find;
	GtkWidget *checkbutton_regex;
	GtkWidget *radiobutton_perl_regex;
	GtkWidget *checkbutton_match_case;
	GtkWidget *checkbutton_whole_word;
	GtkWidget *checkbutton_word_start;
	gboolean isregexp;
	gboolean is_perl_regexp;
	gboolean is_match_case;
	gboolean is_whole_word;
	gboolean is_word_start;
	gint search_flags = 0; // Keep compiler happy
	CssedWindow *window;
	CssedDoc *doc;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);

	entry_text_to_find =
		lookup_widget (GTK_WIDGET (button), "entry_text_to_find");
	checkbutton_regex =
		lookup_widget (GTK_WIDGET (button), "checkbutton_regex");
	radiobutton_perl_regex =
		lookup_widget (GTK_WIDGET (button), "radiobutton_perl_regex");
	checkbutton_match_case =
		lookup_widget (GTK_WIDGET (button), "checkbutton_match_case");
	checkbutton_whole_word =
		lookup_widget (GTK_WIDGET (button), "checkbutton_whole_word");
	checkbutton_word_start =
		lookup_widget (GTK_WIDGET (button), "checkbutton_word_start");

	isregexp =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_regex));
	is_perl_regexp =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (radiobutton_perl_regex));
	is_match_case =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_match_case));
	is_whole_word =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_whole_word));
	is_word_start =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_word_start));

	if (isregexp)
	{
		if (is_perl_regexp)
		{
			search_flags |= SCFIND_REGEXP;
		}
		else
		{
			search_flags |= SCFIND_POSIX;
		}
	}

	if (is_whole_word)
		search_flags |= SCFIND_WHOLEWORD;
	if (is_match_case)
		search_flags |= SCFIND_MATCHCASE;
	if (is_word_start)
		search_flags |= SCFIND_WORDSTART;
	document_search_next (doc,
			      (gchar *)
			      gtk_entry_get_text (GTK_ENTRY
						  (entry_text_to_find)),
			      search_flags);
}

void
on_dialog_replace_search_button_clicked (GtkButton * button,
					 gpointer user_data)
{
	GtkWidget *entry_text_to_find;
	GtkWidget *checkbutton_regex;
	GtkWidget *radiobutton_perl_regex;
	GtkWidget *checkbutton_match_case;
	GtkWidget *checkbutton_whole_word;
	GtkWidget *checkbutton_word_start;
	gboolean isregexp;
	gboolean is_perl_regexp;
	gboolean is_match_case;
	gboolean is_whole_word;
	gboolean is_word_start;
	gint search_flags = 0; // Keep compiler happy
	CssedWindow *window;
	CssedDoc *doc;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);

	entry_text_to_find =
		lookup_widget (GTK_WIDGET (button), "entry_find");
	checkbutton_regex =
		lookup_widget (GTK_WIDGET (button), "checkbutton_regex");
	radiobutton_perl_regex =
		lookup_widget (GTK_WIDGET (button), "radiobutton_perl_regex");
	checkbutton_match_case =
		lookup_widget (GTK_WIDGET (button), "checkbutton_match_case");
	checkbutton_whole_word =
		lookup_widget (GTK_WIDGET (button), "checkbutton_whole_word");
	checkbutton_word_start =
		lookup_widget (GTK_WIDGET (button), "checkbutton_word_start");

	isregexp =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_regex));
	is_perl_regexp =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (radiobutton_perl_regex));
	is_match_case =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_match_case));
	is_whole_word =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_whole_word));
	is_word_start =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_word_start));

	if (isregexp)
	{
		if (is_perl_regexp)
		{
			search_flags |= SCFIND_REGEXP;
		}
		else
		{
			search_flags |= SCFIND_POSIX;
		}
	}

	if (is_whole_word)
		search_flags |= SCFIND_WHOLEWORD;
	if (is_match_case)
		search_flags |= SCFIND_MATCHCASE;
	if (is_word_start)
		search_flags |= SCFIND_WORDSTART;
	document_search_next (doc,
			      (gchar *)
			      gtk_entry_get_text (GTK_ENTRY
						  (entry_text_to_find)),
			      search_flags);
}

void
on_dialog_replace_apply_button_clicked (GtkButton * button,
					gpointer user_data)
{
	GtkWidget *entry_text_to_find;
	GtkWidget *entry_text_to_replace;
	GtkWidget *checkbutton_regex;
	GtkWidget *radiobutton_perl_regex;
	GtkWidget *checkbutton_match_case;
	GtkWidget *checkbutton_whole_word;
	GtkWidget *checkbutton_word_start;
	gboolean isregexp;
	gboolean is_perl_regexp;
	gboolean is_match_case;
	gboolean is_whole_word;
	gboolean is_word_start;
	gint search_flags = 0; // Keep compiler happy
	CssedWindow *window;
	CssedDoc *doc;

	gint selection_start;
	gint selection_end;

	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);

	selection_start = document_get_selection_start (doc);
	selection_end = document_get_selection_end (doc);

	entry_text_to_find =
		lookup_widget (GTK_WIDGET (button), "entry_find");
	entry_text_to_replace =
		lookup_widget (GTK_WIDGET (button), "entry_replace");
	checkbutton_regex =
		lookup_widget (GTK_WIDGET (button), "checkbutton_regex");
	radiobutton_perl_regex =
		lookup_widget (GTK_WIDGET (button), "radiobutton_perl_regex");
	checkbutton_match_case =
		lookup_widget (GTK_WIDGET (button), "checkbutton_match_case");
	checkbutton_whole_word =
		lookup_widget (GTK_WIDGET (button), "checkbutton_whole_word");
	checkbutton_word_start =
		lookup_widget (GTK_WIDGET (button), "checkbutton_word_start");

	if (selection_start >= selection_end)
	{
		cssed_error_message	(_("There must be some text selected\nto be replaced.\nPlease select the text to replace or repeat the search"),_("Nothing to replace") );
		return;
	}
	else
	{
		//document_new_undo_action(doc);
		document_replace_sel (doc,
				      (gchar *)
				      gtk_entry_get_text (GTK_ENTRY
							  (entry_text_to_replace)));
		//document_new_undo_action(doc);
	}
	// search next
	isregexp =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_regex));
	is_perl_regexp =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (radiobutton_perl_regex));
	is_match_case =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_match_case));
	is_whole_word =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_whole_word));
	is_word_start =
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					      (checkbutton_word_start));

	if (isregexp)
	{
		if (is_perl_regexp)
		{
			search_flags |= SCFIND_REGEXP;
		}
		else
		{
			search_flags |= SCFIND_POSIX;
		}
	}
	if (is_whole_word)
		search_flags |= SCFIND_WHOLEWORD;
	if (is_match_case)
		search_flags |= SCFIND_MATCHCASE;
	if (is_word_start)
		search_flags |= SCFIND_WORDSTART;
	document_search_next (doc,
			      (gchar *)
			      gtk_entry_get_text (GTK_ENTRY
						  (entry_text_to_find)),
			      search_flags);
}
void
on_document_scanner_button_clicked(GtkButton * button,
					gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *tmpdoc;
	gint doclen;
	gchar* doc_buffer;
	gchar* endbuff;
	gchar* filename;
	
	window = CSSED_WINDOW (user_data);
	tmpdoc = document_get_current (window);	
	
	if( document_get_filetype( tmpdoc ) != CSSED_FILETYPE_CSS ){
		cssed_error_message(_("I can only parse CSS files!"),_("Unsupported file type"));
		return;
	}
	
	doclen = document_get_length (tmpdoc);
	
	if( doclen == 0)
		return;
	
	doc_buffer = g_malloc0(++doclen);
	document_get_text (tmpdoc, doclen, doc_buffer);
	endbuff = g_strconcat(doc_buffer, "\n",NULL);
	document_clear_validator_arrow( tmpdoc );
	cssed_cr_parser_scan_doc_buffer (tmpdoc, endbuff, strlen(endbuff));
	filename = document_get_filename(tmpdoc);
	cssed_window_set_last_scan_file(window, filename);
	g_free(doc_buffer);	
	g_free( endbuff);
}

void
on_output_list_row_activated(GtkTreeView * treeview,
					  GtkTreePath * path,
					  GtkTreeViewColumn * col,
					  gpointer user_data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *message;
	CssedDoc *tmpdoc;
	CssedWindow *window;
	//gchar line[5];
	gchar number[64];
	gint i,z;

	window = CSSED_WINDOW (user_data);
	//depth = gtk_tree_path_get_depth (path);
	tmpdoc = document_get_current(window);
	
	model = gtk_tree_view_get_model (treeview);
	if (gtk_tree_model_get_iter (model, &iter, path))
	{
		gtk_tree_model_get (model, &iter, 0, &message, -1);
	}else{ return; }
	
	if( message[0] == '<' ){ //is markup
		for( i=0;i< (int) strlen(message);i++){
			if( message[i] == '>')
				break;
		}
		if( i == (int) strlen(message) ){ // bad markup gone to end of line
			g_free( message );
			return;
		}
		if( i + 3 >= (int) strlen( message ) ){ // no room for a line number so not an error message
			g_free( message );
			return;			
		}else{ 
			i++;
		}
		if( message[i] == '[' ){
			if( i+2 >= (int) strlen(message) ){// no room for a line number
				g_free( message );
				return;	
			}else{
				i+=1;z=0;
				memset(number, 0, 64 * sizeof(gchar));
				while( (message[i] != ']') && (z<64) && (i < (int)strlen(message)) ){
					number[z] = message[i];
					i++;z++;
				}
				if( message[i] == ']'){ // got out in the closing brace so it's ok
					document_set_validator_arrow_at_line( tmpdoc, atoi(number) );
				}
			}
		}else{
			g_free( message );
			return;				
		}
	}	
	g_free (message);
}
/* document scanner */
void
on_doc_scanner_row_activated(GtkTreeView * treeview,  GtkTreePath * path,
					  GtkTreeViewColumn * col,		  gpointer user_data)
{
	GtkListStore* current;
	gchar* scanfile;
	GtkWidget* prompt;
	CssedWindow* window;
	
	window = CSSED_WINDOW (user_data);
	current = GTK_LIST_STORE( gtk_tree_view_get_model( treeview ) );
	scanfile = cssed_window_get_last_scan_file(window);
	if( scanfile != NULL ){
		if( cssed_window_is_file_opened(window, scanfile) ){
			on_output_list_row_activated(treeview, path, col, user_data);	
		}else{
			create_and_attach_new_named_doc(window, scanfile);
			if( cssed_window_is_file_opened(window, scanfile) )
				on_output_list_row_activated(treeview, path, col, user_data);			
		}
		g_free(scanfile);
	}else{
		on_output_list_row_activated(treeview, path, col, user_data);
	}
}
void
on_doc_scanner_col_numbers_clicked      (GtkTreeViewColumn *treeviewcolumn,
                                            gpointer user_data)
{
	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(user_data), 0,
										GTK_SORT_ASCENDING);
}
void
on_doc_scanner_col_styles_clicked      (GtkTreeViewColumn *treeviewcolumn,
                                            gpointer user_data)
{
	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(user_data), 1,
										GTK_SORT_ASCENDING);	
}

/* css treview */
void 
on_treeview_css_properties_row_activated (GtkTreeView * treeview,
					  GtkTreePath * path,
					  GtkTreeViewColumn * col,
					  gpointer user_data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *attribute;
	gchar *property;
	CssedDoc *tmpdoc;
	CssedWindow *window;
	//gchar *line_to_insert;
	//gchar* strdialog;
	gint depth;
	//GtkWidget *dialog;

	window = CSSED_WINDOW (user_data);
	depth = gtk_tree_path_get_depth (path);
	tmpdoc = document_get_current(window);
	
	if (depth != 3)// catch only the third col signals
	{			
		DBGMSG ("callbacks.c:on_treeview_css_properties_row_activated, depth %d ignoring", depth);
		return;
	}
	model = gtk_tree_view_get_model (treeview);
	if (gtk_tree_model_get_iter (model, &iter, path))
	{
		gtk_tree_model_get (model, &iter, 2, &attribute, -1);	// columns 0,1,2
		DBGMSG ("callbacks.c:on_treeview_css_properties_row_activated, attr -> %s", attribute);
	}else{
		return;
	}
	gtk_tree_path_up (path);
	if (gtk_tree_model_get_iter (model, &iter, path))
	{
		gtk_tree_model_get (model, &iter, 1, &property, -1);	// columns 0,1,2
		DBGMSG ("callbacks.c:on_treeview_css_properties_row_activated, property -> %s", property);
	}
	else
	{
		g_free (attribute);
		return;
	}
	// call fire dialog on document.c
	fire_dialog_from_strings_or_write_output(window,tmpdoc,attribute,property, TRUE );
	g_free (attribute);
	g_free (property);
}
void 
on_treeview_css_current_row_activated (GtkTreeView *treeview,  GtkTreePath  *path,
                                         GtkTreeViewColumn *col, gpointer     user_data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *attribute;
	gchar *property;
	CssedDoc *tmpdoc;
	CssedWindow *window;
	//gchar *line_to_insert;
	//gchar* strdialog;
	gint depth;
	GList* columns;
	GtkTreeViewColumn* first_col = NULL; // Keep compiler happy
	GtkTreeViewColumn* second_col = NULL;
	GtkTreeViewColumn* third_col = NULL;
	GdkRectangle rect;
	GtkWidget* window_widget;
	GdkWindow* gdw;
	gint root_x, x, new_x;
	gint root_y, y, new_y;
	
	window = CSSED_WINDOW (user_data);
	depth = gtk_tree_path_get_depth (path);
	tmpdoc = document_get_current(window);

	columns =  gtk_tree_view_get_columns(treeview);
	
	columns = g_list_first(columns);
	if( columns != NULL ){
				first_col = columns->data;
	}
	columns = g_list_next(columns);
	if( columns != NULL ){
				second_col = columns->data;
	}	
	columns = g_list_next(columns);
	if( columns != NULL ){
				third_col = columns->data;
	}
	
	if( first_col == NULL ||
		second_col == NULL ||
		third_col == NULL )
	{
			return;			
	}
	g_list_free(columns);

	DBGMSG("callbacks.c: on_treeview_css_current_row_activated() Called from %p , Cols %p %p %p , Match third %s***",
									col, 
									first_col,
									second_col,
									third_col,
									col==third_col?"yes":"no");

	if (col != third_col)// only check third col signals
	{			
		DBGMSG ("callbacks.c: on_treeview_css_current_row_activated() tree Col %p activated, ignoring", col);
		return;
	}
	
	model = gtk_tree_view_get_model (treeview);
	if (gtk_tree_model_get_iter (model, &iter, path))
	{
		gtk_tree_model_get (model, &iter, 0, &property,1, &attribute, -1);	// columns 0,1,2
		gtk_tree_view_set_cursor (treeview,path,col,TRUE);
		
		// get parent col width and height to create a floating list
		gtk_tree_view_get_cell_area(treeview, path, GTK_TREE_VIEW_COLUMN(second_col), &rect);
		gdw = gtk_widget_get_parent_window(GTK_WIDGET(treeview));
		//window_widget = gtk_widget_get_toplevel(GTK_WIDGET(treeview));
		window_widget = cssed_window_get_window_widget( window );
		gtk_widget_translate_coordinates        (GTK_WIDGET(treeview),
												 window_widget,
												 rect.x,
												 rect.y + (rect.height * 2) + 35,
												 &new_x,
												 &new_y); 
		gdk_window_get_root_origin(gdw, &root_x, &root_y);	
		x = root_x + new_x + 5;
		y = root_y + new_y;	
		create_floating_list_from_keyword(	tmpdoc, 
											property, 
											x,
											y,
											rect.width,
											100,
											model,
											gtk_tree_path_to_string(path)
										);		
		g_free(property);
		g_free(attribute);
	}else{
		DBGMSG ("callbacks.c: on_treeview_css_current_row_activated() - Nothing in the list?");
		return;
	}	
}

void 
on_current_css_render_values_edited  (GtkCellRendererText *cell,gchar *path_string,
											gchar *new_text,gpointer user_data)
{
	CssedWindow* window;
	GtkWidget* view;
	
	window = CSSED_WINDOW( user_data );	
	view = cssed_window_get_selector_scanner_treeview( window );
	cssed_window_selector_scanner_edited ( 	window,
											path_string,
											new_text 	);
	on_menu_scan_selector_activate (NULL, window);
	gtk_tree_view_set_cursor_on_cell( GTK_TREE_VIEW(view), 
									  gtk_tree_path_new_from_string(path_string),
									  NULL, NULL, FALSE );
}
/* /////////////// EDITOR CALLBACKS TESTING //////////////////// */
/* This will fire some user list in the future, here for listening to the 
 function keys and others, not used yet */
gboolean
on_editor_key_press_event(GtkWidget *widget,    GdkEventKey *event,
                                            gpointer user_data)
{
	DBGMSG("callbacks.c: on_editor_key_press_event() - editor key pressed");
	return FALSE;
}

/* this will erase all data in the current css list when the user clicks again 
	on the editor widget, or changes something on it. This way, we are sure that
	the current selector list is in sync with the selected text in the editor */
void
on_editor_grab_focus           (GtkWidget *widget,
                                gpointer user_data)
{
	CssedWindow* window;
	GtkWidget* treeview;
	GtkTreeModel* list_store;
	GtkTreeIter iter;
	gboolean valid;
	
	window = CSSED_WINDOW(user_data);	
	treeview =	cssed_window_get_selector_scanner_treeview( window );
	
	DBGMSG("callbacks.c: on_editor_grab_focus() : editor %p grab focus",widget);

	list_store = gtk_tree_view_get_model (GTK_TREE_VIEW (treeview));
	valid = gtk_tree_model_get_iter_first (list_store, &iter);
	
	if( valid ){ // clean all data	
		DBGMSG("callbacks.c: on_editor_grab_focus() - There was data in the css current properties list, cleaning ...");
		gtk_list_store_clear(GTK_LIST_STORE(list_store));
	}
}
	
/* /////////////////// EDITOR CALLBACKS END ////////////////// */
void 
on_menu_autocompletion_activate	
		(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedWindow *window;
	CssedDoc *doc;
	gchar* list;
	
	window = CSSED_WINDOW (user_data);
	doc = document_get_current (window);
	list = cssed_window_get_keyword_list( window );
	document_show_autocompletion_list(doc, 0, list);	
}

void
popmenu_autocomp_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	CssedPopmenuData* data;
	CssedWindow* window;
	CssedDoc* doc;
	gchar* attribute;
	gchar* property; // previous_string
	gboolean add_property_string;	
	
	data = (CssedPopmenuData*) user_data;	
	window = CSSED_WINDOW(data->main_window);
	doc = (CssedDoc*) data->document;

	DBGMSG("callbacks.c: popmenu_autocomp_activate_cb - %s - %s",(gchar*) data->attribute,(gchar*) data->property);

	attribute = g_strdup( (gchar*) data->attribute);
	property = g_strdup( (gchar*) data->property); 
	add_property_string = data->add_property_string;
	// Memory is disposed in parent's "unmap" event ( see below ). This menu is only for one use.
	fire_dialog_from_strings_or_write_output(window,doc,attribute,property,add_property_string);
	g_free(attribute);
	g_free(property);
}

/* When the pop menu is created some memory is dynamically allocated and must
   be disposed. This callback will track the popmenu unmap event and clear all
   memory allocated. The pointers to the data allocated are in a doubly
   linked list passed as user_data  */
void
popmenu_autocomp_event_after_cb(GtkWidget *widget,
                                GdkEvent *event,
                                gpointer user_data)
{
	GList* list;
	CssedPopmenuData* data;
	
	if ( event->type == GDK_UNMAP) {
		list = g_list_first((GList*) user_data);

		DBGMSG("callbacks.c: popmenu_autocomp_event_after_cb() - After menu unmap event let's dispose allocated memory");
	
		if( list != NULL ){
			while( list != NULL ){
				data = (CssedPopmenuData*) list->data;
				DBGMSG("callbacks.c: popmenu_autocomp_event_after_cb() - Cleaning element %p - containing %s %s",
							data, data->attribute,data->property);
				g_free(data->attribute);
				g_free(data->property);
				g_free(data);
				list = g_list_next(list);
			}
			g_list_free (list);
		}
#ifdef DEBUG	
		else
		{
			DBGMSG("callbacks.c: popmenu_autocomp_event_after_cb() - List is empty or NULL");		
		}
#endif	
		g_list_free(user_data);	
		
		if( GTK_IS_WIDGET(widget) ){// ensure the menu is destroyed
			gtk_widget_destroy(widget);
			DBGMSG("callbacks.c: popmenu_autocomp_event_after_cb() - Menu destroyed");
		}
	}
}
/* common to all file selector dialogs */
void file_selector_close (GtkDialog       *dialog,
                          gpointer         user_data)
{
	g_free(user_data);
}
									
void
file_selector_cancel_button_clicked (GtkButton       *button,  
									gpointer         user_data)
{
	GtkWidget* dialog;
	
	dialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
	if(GTK_WIDGET_TOPLEVEL (dialog)){ //ensure it's the toplevel
		gtk_widget_destroy(dialog);
	}
	g_free(user_data);
}

gboolean
on_editor_button_press_event(GtkWidget *widget, GdkEventButton *event,
                                                gpointer user_data)
{
	GtkMenu *menu;
	CssedWindow* window;
	
	window = CSSED_WINDOW( user_data);
	menu = GTK_MENU( cssed_window_get_pop_menu( window ) );
	
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);
	if (event->type == GDK_BUTTON_PRESS)
	{
		if (event->button == 3)
		{
			gtk_menu_popup ( menu, NULL, NULL, NULL, NULL, 
		  					 event->button, event->time );
		 	return TRUE;
		}
	}
 	return FALSE;
}
// editor pop menu callbacks
void 
on_pop_menu_undo_activate(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedDoc* doc;
	
	doc = document_get_current( CSSED_WINDOW(user_data));
	document_undo( doc );
}
	
void 
on_pop_menu_redo_activate(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedDoc* doc;
	
	doc = document_get_current( CSSED_WINDOW(user_data));
	document_redo( doc );
}
	
void 
on_pop_menu_cut_activate(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedDoc* doc;
	
	doc = document_get_current( CSSED_WINDOW(user_data));
	document_cut( doc );
}
	
void 
on_pop_menu_copy_activate(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedDoc* doc;
	
	doc = document_get_current( CSSED_WINDOW(user_data));
	document_copy( doc );
}
	
void
on_pop_menu_paste_activate(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedDoc* doc;
	
	doc = document_get_current( CSSED_WINDOW(user_data));
	document_paste( doc );
}
	
void 
on_pop_menu_delete_activate(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedDoc* doc;
	
	doc = document_get_current( CSSED_WINDOW(user_data));
	document_clear_sel ( doc );
}
	
void
on_pop_menu_selectall_activate(GtkMenuItem	  *menuitem,   		gpointer user_data)
{
	CssedDoc* doc;
	
	doc = document_get_current( CSSED_WINDOW(user_data));
	document_set_selection_range ( doc, 0, document_get_length(doc) );
}
	
/* ***************************** HELP BUTTONS ******************************* */
/*    ---------------------------   TO DO ----------------------------------  */
void
on_color_selection_help_button_clicked (GtkButton * button, gpointer user_data)
{
}

void
on_dialog_search_help_button_clicked (GtkButton * button, gpointer user_data)
{
}


void
on_dialog_replace_help_button_clicked (GtkButton * button, gpointer user_data)
{
}



