# test_parameters.rb: the test suite for the parameter system.
# Copyright (C) 2006 Vincent Fourmond
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA


require 'MetaBuilder/parameters.rb'
require 'test/unit'
require 'optparse'

class TestParameters < Test::Unit::TestCase

  include MetaBuilder

  # Makes a parameter test for the given public type, private
  # class handling it, and a pair of string/corresponding data
  def make_param_type_test(public_type, private_class, string, data, *other)
    cls = ParameterType.get_param_type(public_type)
    assert_equal(cls, private_class)
    instance = cls.new(public_type)
    assert_equal(data, instance.string_to_type(string))
    # This test is removed, as this is not necessary.
    #     assert_equal(string, instance.type_to_string(data))
    unless other.first == :no_reverse_test
      for d in [data] + other
        assert_equal(d, instance.string_to_type(instance.type_to_string(d)))
      end
    end

  end

  # Handy function to test if the OptionParser stuff is working fine.
  def option_parser_test_full(type, data, *args)
    cls = ParameterType.get_param_type(type)
    op = OptionParser.new
    value = nil                 # So that it is not local to the
    # block below.
    cls.new(type).option_parser_option(op, "test", "") do |t|
      value = t
    end
    op.parse!(args)
    assert_equal(value, data)
  end

  # Handy function to test if the OptionParser stuff is working fine.
  def option_parser_test(type, string, data)
    option_parser_test_full(type, data, "--test", string)
  end

  # A simple conversion test, without backwards reconversions
  def ensure_converts(public_type,  string, data)
    cls = ParameterType.get_param_type(public_type)
    instance = cls.new(public_type)
    assert_equal(data, instance.string_to_type(string))
  end

  
  def test_int
    make_param_type_test({:type => :integer}, ParameterTypes::IntegerParameter,
                         "123", 123, -1, 0)
    option_parser_test({:type => :integer}, "123", 123)
  end

  def test_float
    make_param_type_test({:type => :float}, ParameterTypes::FloatParameter,
                         "123.0", 123.0, 1.0, 0.0, 1e-60, 2.0**40)
    option_parser_test({:type => :float}, "1.0", 1.0)
  end

  def test_string
    make_param_type_test({:type => :string}, ParameterTypes::StringParameter,
                         "biniou", "biniou")
    option_parser_test({:type => :string}, "test test", "test test")
  end

  def test_file
    make_param_type_test({:type => :file,
                           :filter => "Biniou (*.txt)"}, 
                         ParameterTypes::FileParameter,
                         "biniou", "biniou")
  end
  def test_bool
    make_param_type_test({:type => :boolean}, ParameterTypes::BooleanParameter,
                         "true", true, false)
    option_parser_test_full({:type => :boolean}, false, "--no-test")
    option_parser_test_full({:type => :boolean}, true, "--test")
  end

  def test_list
    type = { :type => :list, :list => {
        :a => 'a', :b => 'b'}
    }
    make_param_type_test(type, ParameterTypes::ListParameter,
                         "a", :a, :b)
    option_parser_test(type, "b",:b)

    passed = false
    begin
      make_param_type_test(type, ParameterTypes::ListParameter,
                           "incorrect input",:a )
    rescue MetaBuilder::ParameterType::IncorrectInput
      passed = true
    end
    assert(passed)
  end

  def test_range
    make_param_type_test({:type => :float_range}, 
                         ParameterTypes::FloatRangeParameter,
                         "1:2", 1.0..2.0, -1.1..2.0, 2.0..5.0)
    option_parser_test({:type => :float_range}, "1:2", 1.0..2.0)
  end

  def test_type_name
    assert_equal(ParameterTypes::IntegerParameter.new({}).type_name, 
                 'number')
  end


  def test_color
    make_param_type_test({:type => :color}, 
                         ParameterTypes::HTMLColorParameter,
                         "#FF00FF", [255,0,255], 
                         [220,123,0])
    option_parser_test({:type => :color}, 
                       "#FF00FF", [255,0,255])
  end

  module ColorNamespace
    Red = [1.0,0.0,0.0]
    Blue = [0,0,255]
  end

  def test_color_namespace
    make_param_type_test({:type => :color, 
                           :namespaces => [ColorNamespace]}, 
                         ParameterTypes::HTMLColorParameter,
                         "Red", [255,0,0])
    make_param_type_test({:type => :color, 
                           :namespaces => [ColorNamespace]}, 
                         ParameterTypes::HTMLColorParameter,
                         "Blue", [0,0,255])
    # Last stuff, with new syntax
    make_param_type_test({:type => :color, 
                           :namespace => ColorNamespace}, 
                         ParameterTypes::HTMLColorParameter,
                         "Blue", [0,0,255])
  end

  module IntNameSpace1
    Int = 2
  end
  module IntNameSpace2
    Int = 4
    Nice = 2
  end

  def test_int_namespaces
    make_param_type_test({:type => :integer, 
                           :namespace => [IntNameSpace1, IntNameSpace2]}, 
                         ParameterTypes::IntegerParameter,
                         "Int", 2)
    make_param_type_test({:type => :integer, 
                           :namespace => [IntNameSpace1, IntNameSpace2]}, 
                         ParameterTypes::IntegerParameter,
                         "Nice", 2)
    make_param_type_test({:type => :integer, 
                           :namespace => IntNameSpace2}, 
                         ParameterTypes::IntegerParameter,
                         "Int", 4)
  end

  def test_arrays
    make_param_type_test({:type => :array}, 
                         ParameterTypes::ArrayParameter,
                         "1,2", ["1","2"])
    make_param_type_test({
                           :type => :array, 
                           :subtype => {:type => :float}
                         }, 
                         ParameterTypes::ArrayParameter,
                         "1,2", [1.0,2.0])
  end

  def test_stt_hooks
    ensure_converts({:type => :integer, 
                      :stt_hook => proc do |i|
                        i + 2
                      end}, 
                    "2", 4)
  end

  def test_regexps
    make_param_type_test(:string_or_regexp, 
                         ParameterTypes::StringOrRegexpParameter,
                         "a", 'a')
    make_param_type_test(:string_or_regexp, 
                         ParameterTypes::StringOrRegexpParameter,
                         "/ab/", /ab/,  :no_reverse_test) # Won't work.
  end

end

