# styles.rb : style-related codes
# Copyright (C) 2007, 2008  Vincent Fourmond
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

require 'MetaBuilder/parameters'
require 'Tioga/tioga'
require 'CTioga/utils'

module CTioga

  Version::register_svn_info('$Revision: 754 $', '$Date: 2008-02-28 17:05:16 +0100 (Thu, 28 Feb 2008) $')

  # This module holds a lot of constants for styles.
  module Styles
    include Tioga::FigureConstants

    COLORS = { 
      "standard" => 
      [Red, Green, Blue, Cyan, Magenta, Orange],
      "pastel1" => 
      [MediumSeaGreen, RoyalBlue, Pumpkin, DarkChocolate, Lilac, Crimson],
      "colorblind" => 
      [BrightBlue, Goldenrod, Coral, Lilac, FireBrick, RoyalPurple],
    }
      
    # The sets of markers. 
    MARKERS = { 
      "standard" => 
      [Bullet, TriangleUp, Square, Plus, Times],
      "open" => 
      [BulletOpen, TriangleUpOpen, SquareOpen, PlusOpen, TimesOpen],
    }
    
    Line_Type_Dash_Dot_Dot = [[5,2,1,2,1,2],0]
    
    # Linestyles.
    LINES = {
      "standard" => 
      [Line_Type_Solid, 
       Line_Type_Dots, 
       Line_Type_Dashes, 
       Line_Type_Dot_Long_Dash,
       Line_Type_Dash_Dot_Dot
      ],
    }

    # Shortcut for line styles:
    Solid  = Line_Type_Solid
    Dots   = Line_Type_Dots
    Dashes = Line_Type_Dashes
    Dot_Long_Dash = Line_Type_Dot_Long_Dash
    Dash_Dot_Dot = Line_Type_Dash_Dot_Dot

    # Now, a series of ParameterType objects that help parsing
    # a string into a style object

    # A CTioga color
    class CTiogaColorParameter < MetaBuilder::ParameterType
      type_name :color, 'color'
      
      def string_to_type_internal(str)
        return CTioga.get_tioga_color(str)
      end

      def type_to_string_internal(val)
        return val.map {|i| i.to_s}.join(',')
      end
    end
    
    # A CTioga marker
    class CTiogaMarkerParameter < MetaBuilder::ParameterType
      type_name :marker, 'marker'
      
      def string_to_type_internal(str)
        if str =~ /(\d+)\s*,\s*(\d+)(?:\s*,\s*([\d.e+-]+))?/
          ar = [$1.to_i, $2.to_i]
          if $3
            ar << $3.to_f
          end
          return ar
        else
          begin
            return Tioga::MarkerConstants.const_get(str)
          rescue
            raise IncorrectInput, "'#{str}' is not a valid marker"
          end
        end
      end

      def type_to_string_internal(val)
        return val.map {|i| i.to_s}.join(',')
      end
    end

    class CTiogaLineStyleParameter < MetaBuilder::ParameterType
      type_name :line_style, 'linestyle'

      def string_to_type_internal(str)
        if str =~ /(:?\d+\s*,)+/
          ar = str.split(/\s*,\s*/).map{|v| v.to_f}
          if ar.size % 2 == 0
            return [ar, 0]
          else
            last = ar.pop
            return [ar, last]
          end
        elsif Tioga::FigureConstants.const_defined?(str)
          return Tioga::FigureConstants.const_get(str)
        elsif Styles.const_defined?(str)
          return Styles.const_get(str)
        else
          raise IncorrectInput, "'#{str}' is not a valid line type"
        end
      end

      def type_to_string_internal(val)
        return val.flatten.map {|i| i.to_s}.join(',')
      end
    end


  end
end
