# tioga_primitives.rb, direct inclusion of graphics primitives in the graphs.
# copyright (c) 2007, 2008 by Vincent Fourmond: 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details (in the COPYING file).

require 'Dobjects/Dvector'
require 'CTioga/debug'
require 'CTioga/log'
require 'MetaBuilder/parameters'
require 'shellwords'

module CTioga

  Version::register_svn_info('$Revision: 784 $', '$Date: 2008-03-14 15:45:20 +0100 (Fri, 14 Mar 2008) $')
  
  # A class to parse the format of graphics primitives, which is something
  # like:
  #  text: 12,34 "nice text" angle=35
  #
  class TiogaPrimitiveMaker

    extend Debug
    include Log

    # TODO: primitives should not be funcalls, but elements in their
    # own right.

    # TODO: provided this is true, we can implement a much more complex
    # mechanism for positioning, that would allow fine placement such
    # as 0+12pt,36-2mm (should be easy to do).

    # TODO: extend the position pick stuff for other things than tangents,
    # allowing something like @x=0+0,12pt -- @index=12 pr @index=0.45.
    # All these would be relative to the previous curve.

    # TODO: in the same spirit, allow 'absolute' positioning, such as
    # frame:0.5,0.2 for frame position.

    # TODO: allow non-clipping elements (that would come in the same context,
    # but after the axes have been drawn, and not clipped

    # Internal structure to represent the syntax of a primitive call

    # The symbol for the funcall
    attr_accessor :symbol
    # An array of [name, types] for compulsory arguments
    attr_accessor :compulsory  
    # A hash of the optional arguments
    attr_accessor :optional 

    # Creates a TiogaPrimitiveMaker object, from the specifications
    def initialize(symb, comp, opt)
      @symbol = symb
      @compulsory = comp
      @optional = opt
    end

    # This function is fed with an array of Strings. The example in
    # TiogaPrimitiveMaker would look like
    #  ["12,34", "nice text", "angle=35"]
    # It returns the hash that can be fed to the appropriate
    # Tioga primitive. It is fed the plot
    def parse_args(args)
      ret = {}
      for name, type_spec in @compulsory
        type_spec ||= {:type => :string} # Absent means string
        type = MetaBuilder::ParameterType.get_type(type_spec)
        val = type.string_to_type(args.shift)
        ret[name] = val
      end
      # We now parse the rest of the arguments:
      for opt in args
        if opt =~ /^([^=]+)=(.*)/
          name = $1
          arg = $2
          type_spec = @optional[name] || {:type => :string}
          type = MetaBuilder::ParameterType.get_type(type_spec)
          val = type.string_to_type(arg)
          ret[name] = val
        else
          warn "Malformed optional argument #{opt}"
        end
      end
      return ret
    end

    # Turns a graphics spefication into a Tioga Funcall.
    def make_funcall(args, plotmaker)
      dict = parse_args(args)
      return TiogaFuncall.new(@symbol, dict) 
    end

    # A few predefined types to make it all more readable -
    # and easier to maintain !!
    FloatArray = {:type => :array, :subtype => :float}
    Point = FloatArray
    Boolean = {:type => :boolean}
    Marker = {                  # Damn useful !!
      :type => :array, 
      :subtype => {:type => :integer, :namespace => Tioga::MarkerConstants},
      :namespace => Tioga::MarkerConstants,
      :shortcuts => {'No' => 'None', 'None' => 'None',
        'no' => 'None', 'none' => 'None',
      } # For no/none to work fine
    }
    Number = {:type => :float}
    Color = {
      :type => :array, :subtype => :float, 
      :namespace => Tioga::ColorConstants
    }
    # Very bad definition, but, well...
    LineStyle = {:type => :array, :subtype => :integer,
      :namespace => Styles,
    }

    Justification = {:type => :integer, 
      :shortcuts => {
        'Left' => Tioga::FigureConstants::LEFT_JUSTIFIED,
        'left' => Tioga::FigureConstants::LEFT_JUSTIFIED,
        'l' => Tioga::FigureConstants::LEFT_JUSTIFIED,
        'Center' => Tioga::FigureConstants::CENTERED,
        'center' => Tioga::FigureConstants::CENTERED,
        'c' => Tioga::FigureConstants::CENTERED,
        'Right' => Tioga::FigureConstants::RIGHT_JUSTIFIED,
        'right' => Tioga::FigureConstants::RIGHT_JUSTIFIED,
        'r' => Tioga::FigureConstants::RIGHT_JUSTIFIED,
      }
    }

    Alignment = {:type => :integer, 
      :shortcuts => {
        'Top' => Tioga::FigureConstants::ALIGNED_AT_TOP,
        'top' => Tioga::FigureConstants::ALIGNED_AT_TOP,
        't' => Tioga::FigureConstants::ALIGNED_AT_TOP,
        'Mid' => Tioga::FigureConstants::ALIGNED_AT_MIDHEIGHT,
        'mid' => Tioga::FigureConstants::ALIGNED_AT_MIDHEIGHT,
        'm' => Tioga::FigureConstants::ALIGNED_AT_MIDHEIGHT,
        'Base' => Tioga::FigureConstants::ALIGNED_AT_BASELINE,
        'base' => Tioga::FigureConstants::ALIGNED_AT_BASELINE,
        'B' => Tioga::FigureConstants::ALIGNED_AT_BASELINE,
        'Bottom' => Tioga::FigureConstants::ALIGNED_AT_BOTTOM,
        'bottom' => Tioga::FigureConstants::ALIGNED_AT_BOTTOM,
        'b' => Tioga::FigureConstants::ALIGNED_AT_BOTTOM,
      }
    }
    
    # Available primitives:
    PRIMITIVES = {
      "text" => self.new(:show_text, 
                         [ 
                          ['point', Point],
                          ['text']
                         ],
                         {
                           'angle' => Number,
                           'color' => Color,
                           'scale' => Number,
                           'justification' => Justification,
                           'alignment' => Alignment,
                         }),
      "arrow" => self.new(:show_arrow, 
                          [ 
                           ['tail', Point ],
                           ['head', Point ],
                          ],
                          {
                            'head_marker' => Marker,
                            'tail_marker' => Marker,
                            'head_scale'  => Number,
                            'tail_scale'  => Number,
                            'line_width'  => Number,
                            'line_style'  => LineStyle,
                            'color'       => Color,
                          }),
      "marker" => self.new(:show_marker, 
                           [ 
                            ['point', Point ],
                            ['marker', Marker ],
                           ],
                           {
                             'color'       => Color,
                             'angle'       => Number,
                             'scale'       => Number,
                           }),
    }
                       
    
    # Parse a specification such as
    #  text: 12,34 "nice text" angle=35
    # plotmaker is a pointer to the plotmaker instance.
    def self.parse_spec(spec, plotmaker)
      spec =~ /^([^:]+):(.*)/
      name = $1
      args = Shellwords.shellwords($2)
      if PRIMITIVES.key? name
        ret = PRIMITIVES[name].make_funcall(args, plotmaker)
        debug "draw: #{name} -> #{ret.inspect}"
      else
        error "Unkown graphic primitive: #{name}"
      end
      ret
    end

    # Returns a small descriptive text about the currently known
    # graphics primitives. 
    def self.introspect(details = true)
      str = ""
      for name, spec in PRIMITIVES
        str += "\t#{name}: " +
          spec.compulsory.map do |a|
          a[0]
        end. join(' ') +
          if details
            "\n\t\toptions: #{spec.optional.keys.join(',')}\n\t\t"
          else
            " [options]\t "
          end + "see Tioga function #{spec.symbol}\n"
      end
      return str
    end
  end

  class TangentSemiPrimitive < TiogaPrimitiveMaker
    TiogaPrimitiveMaker::PRIMITIVES["tangent"] = 
      self.new(:show_arrow,
               [
                ['spec', :string],
               ],
               {                
                 'xextent'  => FloatArray,
                 'yextent'  => FloatArray,
                 'xuntil'  => FloatArray,
                 'yuntil'  => FloatArray,
               }.merge(TiogaPrimitiveMaker::PRIMITIVES['arrow'].optional)
               # We automatically add stuff from the arrow specs,
               # as they share a lot of keys...
               )
    
    def make_funcall(args, plotmaker)
      dict = parse_args(args)
      curve = plotmaker.last_curve
      raise 'The tangent drawing command needs to be specified *after* the curve it applies to' unless curve
      # We now need to transform
      index = curve.parse_position(dict["spec"])
      dict.delete('spec')
      debug "Tangent point index is #{index}"
      tangent = plotmaker.last_curve.tangent(index)
      debug "Tangent slope is #{tangent}"
      point = curve.function.point(index)
      debug "Tangent point is #{point}"

      # We are now preparing the coordinates of the arrow:
      # * if markonly is on, it doesn't matter much
      # * if xextent, we take the array as X extents in either
      #   direction
      # * the same obviously applies for yextent.
      # Only one spec can be used at a time
      if dict['xextent']
        fact = dict['xextent'][0]/tangent[0]
        dict['head'] = point + (tangent * fact)
        fact = (dict['xextent'][1] || 0.0)/tangent[0]
        dict['tail'] = point - (tangent * fact)
      elsif dict['yextent']
        fact = dict['yextent'][0]/tangent[1]
        dict['head'] = point + (tangent * fact)
        fact = (dict['yextent'][1] || 0.0)/tangent[1]
        dict['tail'] = point - (tangent * fact)
        # We prolong the tangent until it intersects the given
        # X= or Y= positions
      elsif dict['xuntil']
        fact = (dict['xuntil'][0] - point[0])/tangent[0]
        dict['head'] = point + (tangent * fact)
        if dict['xuntil'][1]
          fact = (dict['xuntil'][1] - point[0])/tangent[0]
        else
          fact = 0
        end
        dict['tail'] = point + (tangent * fact)
      elsif dict['yuntil']
        fact = (dict['yuntil'][0] - point[1])/tangent[1]
        dict['head'] = point + (tangent * fact)
        if dict['yuntil'][1]
          fact = (dict['yuntil'][1] - point[1])/tangent[1]
        else
          fact = 0
        end
        dict['tail'] = point + (tangent * fact)
      else
        dict['line_width'] = 0
        dict['head'] = point
        dict['tail'] = point - tangent
      end
      # Remove unnecessary keys...
      %w(spec xextent yextent xuntil yuntil).map {|k| dict.delete(k)}

      # We setup other defaults than the usual ones, from
      # the current curve.
      dict['color'] ||= curve.style.color # color from curve
      dict['line_width'] ||= curve.style.linewidth
      # TODO: add linestyle when that is possible

      dict['tail_marker'] ||= 'None' # No tail by default.

      debug "Tangent coordinates: #{dict['head']} -> #{dict['tail']}"


      # And we return the Funcall...
      return TiogaFuncall.new(@symbol, dict) 
    end
  end

  # A ruler
  class RulerSemiPrimitive < TiogaPrimitiveMaker
    TiogaPrimitiveMaker::PRIMITIVES["vrule"] = 
      self.new(:vrule,
               [
                ['point', Point],
                ['size', :float],
               ],
               {                
                 'label'  => :string,
               }.merge(TiogaPrimitiveMaker::PRIMITIVES['arrow'].optional)
               )

    TiogaPrimitiveMaker::PRIMITIVES["hrule"] = 
      self.new(:hrule,
               [
                ['point', Point],
                ['size', :float],
               ],
               {                
                 'label'  => :string,
               }.merge(TiogaPrimitiveMaker::PRIMITIVES['arrow'].optional)
               )

    include Tioga::FigureConstants
    
    def make_funcall(args, plotmaker)
      dict = parse_args(args)

      ret = []

      line_dict = dict.dup
      line_dict['head_marker'] ||= Tioga::FigureConstants::BarThin
      line_dict['tail_marker'] ||= Tioga::FigureConstants::BarThin

      line_dict['tail_scale'] ||= 0.5
      line_dict['head_scale'] ||= 0.5

      
      line_dict['tail'] = dict['point']
      # We start from point and go on
      line_dict['head'] = dict['point'].dup
      if @symbol == :vrule
        line_dict['head'][1] += dict['size']
      else
        line_dict['head'][0] += dict['size']
      end

      for k in %w(label size point)
        line_dict.delete k
      end

      ret << TiogaFuncall.new(:show_arrow, line_dict)

      # For the label, we tweak the aligment
      if dict.key? 'label'
        label = {}
        label['at'] = (Dobjects::Dvector.new(line_dict['head']) + 
                       Dobjects::Dvector.new(line_dict['tail'])) * 0.5
        label['text'] = dict['label']
        # We place the label on the left/bottom
        if @symbol == :vrule
          label['justification'] = RIGHT_JUSTIFIED
          label['alignment'] = ALIGNED_AT_MIDHEIGHT
        else
          label['justification'] = CENTERED
          label['alignment'] = ALIGNED_AT_TOP
        end
        ret << TiogaFuncall.new(:show_text, label)
      end
      
                             
      # And we return the Funcall(s)...
      return ret
    end
  end

end
