/* cups-pdf.c -- CUPS Backend 
   08.02.2003, Volker C. Behr
   Exp. Physik V, Uni Wuerzburg 
   vrbehr@cip.physik.uni-wuerzburg.de
   http://cip.physik.uni-wuerzburg.de/~vrbehr/cups-pdf

   This code may be freely distributed as long as this header 
   is preserved. 

   This code is distributed under the GPL.
   (http://www.gnu.org/copyleft/gpl.html)

   ---------------------------------------------------------------------------

   Copyright (C) 2003  Volker C. Behr

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   ---------------------------------------------------------------------------
  
   If you want to redistribute modified sources/binaries this header
   has to be preserved and all modifications should be clearly 
   indicated.
   In case you want to include this code into your own programs 
   I would appreaciated some feedback via email.

  
   HISTORY: 
   11/29/2003 : 1.3.1	   - improved handling of domain prefix
   11/02/2003 : 1.3	   - new option to cut file name extensions
   10/25/2003 : 1.2        - untitled documents are labelled with job-id,
			     new option to add a domain prefix to user names,
                             removed a minor bug during character replacement
   08/02/2003 : 1.1	   - files not viewable until creation is finished,
   			     new qualifier "$HOME" for output directory in the header file,
			     "$HOME"-related new define in the header file
   07/17/2003 : 1.0.1	   - improved user name handling for multi-word names,
                             code now distributed under the GPL
   06/29/2003 : 1.0	   - enhanced debug output,
   			     replacement of special characters in title,
			     new option in the header file
   05/26/2003 : 1.0pre3    - several minor bugfixes and improvements 
   05/21/2003 : 1.0pre2    - optimized memory management
   05/16/2003 : 1.0pre1    - introduced cups-pdf.h, 
			     code now entirely written in C,
 			     cups-pdfgen is obsolete
   02/14/2003 : 0.2    - script re-located to /usr/lib/cups/filter/cups-pdfgen
   02/12/2003 : 0.1    - first beta release			                */

#define CPERROR 	1
#define CPSTATUS 	2
#define CPDEBUG 	4

#define BUFSIZE	1024
#define TBUFSIZE "1024"

#define CPGSCALL "%s -q -dCompatibilityLevel=%s -dNOPAUSE -dBATCH -sDEVICE=pdfwrite -sOutputFile=\"%s\" -c save pop -f %s"

#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <stddef.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "cups-pdf.h"

extern int errno;

void log_event(char type, char message[48], char *detail) {
  if (strlen(CPLOG)) { 
    FILE *fp;
    time_t *secs;
    char ctype[8], *timestring;

    secs=calloc(1, sizeof(time_t));
    if (secs != NULL)  {
      time(secs);
      timestring=ctime(secs);
      timestring[strlen(timestring)-1]='\0';

      if (type & CPLOGTYPE) {
        if (type == CPERROR) 
          sprintf(ctype, "ERROR");
        else if (type == CPSTATUS)
          sprintf(ctype, "STATUS");
        else
          sprintf(ctype, "DEBUG");
        fp=fopen(CPLOG"/cups-pdf_log", "a");
        if (detail != NULL)  {
          while (detail[strlen(detail)-1] == '\n') 
            detail[strlen(detail)-1]='\0';
          fprintf(fp,"%s  [%s] %s (%s)\n", timestring, ctype, message, detail);
	  if ((CPLOGTYPE & CPDEBUG) && (type & CPERROR)) 
	    fprintf(fp,"%s  [DEBUG] ERRNO: %d\n", timestring, errno);
        }
        else
          fprintf(fp,"%s  [%s] %s\n", timestring, ctype, message);
        fclose(fp);
      }
      free(secs);
    }
    else 
      fputs("CUPS-PDF: failed to allocate memory\n", stderr);
  }
  return;
}
    
int init() {
  struct stat *fstatus;

  fstatus=calloc(1, sizeof(struct stat));
  if (fstatus != NULL) {
    if (strlen(CPLOG)) {
      if (stat(CPLOG, fstatus) || !S_ISDIR(fstatus->st_mode)) {
        if (mkdir(CPLOG, 0700))
          return 1;
        log_event(CPSTATUS, "log directory created", CPLOG);
      }
    }
    if (stat(CPSPOOL, fstatus) || !S_ISDIR(fstatus->st_mode)) {
      if (mkdir(CPSPOOL, 0700)) {
        log_event(CPERROR, "failed to create spool directory", CPSPOOL);
        return 1;
      }
      log_event(CPSTATUS, "spool directory created", CPSPOOL);
    } 
    if (!strncmp(CPOUT, "$HOME", 5)) 
      log_event(CPDEBUG, "output redirected to home directory", "$HOME");
    else if (stat(CPOUT, fstatus) || !S_ISDIR(fstatus->st_mode)) {
      if (mkdir(CPOUT, 0755)) {
        log_event(CPERROR, "failed to create output directory", CPOUT);
        return 1;
      }
      log_event(CPSTATUS, "output directory created", CPOUT);
    }
    free(fstatus);
    return 0;
  }
  fputs("CUPS-PDF: failed to allocate memory\n", stderr);
  return 1;
}

int prepareuser(struct passwd *passwd, char *dirname) {
  struct stat *fstatus;

  fstatus=calloc(1, sizeof(struct stat));
  if (fstatus == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    return 1;
  }
  if (stat(dirname, fstatus) || !S_ISDIR(fstatus->st_mode)) {
    if (!strcmp(passwd->pw_name, CPANONUSER)) {
      if (mkdir(dirname, 0777)) {
        log_event(CPERROR, "failed to create anonymous output directory", dirname);
        return 1;
      }
      log_event(CPSTATUS, "anonymous output directory created", dirname);
    }
    else {
      if (mkdir(dirname, 0700)) {
        log_event(CPERROR, "failed to create user output directory", dirname);
        return 1;
      }
      log_event(CPSTATUS, "user output directory created", dirname);
    }
    if (chown(dirname, passwd->pw_uid, passwd->pw_gid)) {
      log_event(CPERROR, "failed to set owner for output directory", passwd->pw_name);
      return 1;
    }
    log_event(CPDEBUG, "owner set for output directory", passwd->pw_name);
  }
  free(fstatus);
  return 0;
}

void replace_string(char *string) {
 unsigned int i;

 log_event(CPDEBUG, "removing special characters from title", string);
 for (i=0;i<strlen(string);i++) 
   if ( ( (int) string[i] < '0' || (int) string[i] > '9' ) &&
        ( (int) string[i] < 'A' || (int) string[i] > 'Z' ) &&
        ( (int) string[i] < 'a' || (int) string[i] > 'z' ) &&
        (int) string[i] != '-' && (int) string[i] != '+' )
    string[i]='_';
  return;
}
  
int preparespoolfile(FILE *fpsrc, char *spoolfile, char *title, int job) {
  char buffer[BUFSIZE], *cut;
  FILE *fpdest;

  if (fpsrc == NULL) {
    log_event(CPERROR, "failed to open source stream", NULL);
    return 1;
  }
  log_event(CPDEBUG, "source stream ready", NULL);
  fpdest=fopen(spoolfile, "w");
  if (fpdest == NULL) {
    log_event(CPERROR, "failed to open spoolfile", spoolfile);
    fclose(fpsrc);
    return 1;
  }
  log_event(CPDEBUG, "destination stream ready", spoolfile);
  while (fgets(buffer, BUFSIZE, fpsrc) != NULL) {
    if (!strncmp(buffer, "%!PS", 4)) {
      log_event(CPDEBUG, "found beginning of postscript code", buffer);
      break;
    }
  }
  log_event(CPDEBUG, "now extracting postscript code", NULL);
  fputs(buffer, fpdest);
  while (fgets(buffer, BUFSIZE, fpsrc) != NULL) {
    sscanf(buffer, "%%%%Title: %"TBUFSIZE"c", title);
    fputs(buffer, fpdest);
    if (!strncmp(buffer, "%%EOF", 5)) {
      log_event(CPDEBUG, "found end of postscript code", buffer);
      break;
    }
  }
  fclose(fpdest);
  fclose(fpsrc);
  log_event(CPDEBUG, "all data written to spoolfile", spoolfile);
  cut=strrchr(title, '/');
  if (cut != NULL) {
    log_event(CPDEBUG, "removing slashes from full title", title);
    snprintf(title, BUFSIZE, cut+1);
  }  
  if (title != NULL) 
    while (title[strlen(title)-1] == '\n') 
      title[strlen(title)-1]='\0';
  cut=strrchr(title, '.');
  if ((cut != NULL) && (strlen(cut) <= CPCUT+1) && (cut != title)) {
    log_event(CPDEBUG, "removing file name extension", cut);
    cut[0]='\0';
  }
  replace_string(title);
  if (!strcmp(title, "")) { 
    log_event(CPDEBUG, "no title found - using default value", "untitled_document");
    snprintf(title, BUFSIZE, "job_%i-untitled_document", job);
  }
  else 
    log_event(CPDEBUG, "title successfully retrieved", title);
  return 0;
}

int main(int argc, char *argv[]) {
  char *user, *dirname, *spoolfile, *outfile, *gscall, title[BUFSIZE]="";
  int size;
  mode_t mode;
  struct passwd *passwd;

  umask(0000);

  if (init()) return -1;
  log_event(CPDEBUG, "initialization finished", NULL);

  if (argc==1) {
    printf("file cups-pdf:/ \"PDF Printer\" \"Virtual Printer\"\n"); 
    log_event(CPSTATUS, "identification string sent", NULL);
    return 0;
  }
  if (argc<6 || argc>7) {
    fputs("Usage: cups-pdf job-id user title copies options [file]\n", stderr);
    log_event(CPSTATUS, "call contained illegal number of arguments", NULL);
  return 0;
  }
  size=strlen(CPUSERPREFIX)+strlen(argv[2])+1;
  user=calloc(size, sizeof(char));
  if (user == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    return -2;
  }  
  snprintf(user, size, "%s%s", CPUSERPREFIX, argv[2]);
  passwd=getpwnam(user);
  if (passwd == NULL) {
    if (strlen(CPANONUSER)) {
      passwd=getpwnam(CPANONUSER);
      if (passwd == NULL) {
        log_event(CPERROR, "username for anonymous access unknown", CPANONUSER);
        free(user);
        return -2;
      }
      log_event(CPDEBUG, "unknown user", user);
      if (!strncmp(CPOUT, "$HOME", 5)) {
        size=strlen(passwd->pw_dir)+strlen(CPHOMESUB)+4;
        dirname=calloc(size, sizeof(char));
        if (dirname == NULL) {
          fputs("CUPS-PDF: failed to allocate memory\n", stderr);
          free(user);
          return -2;
        }  
        snprintf(dirname, size, "%s/%s", passwd->pw_dir, CPHOMESUB);
      }
      else {
        size=strlen(CPOUT)+strlen(CPANONDIRNAME)+4;
        dirname=calloc(size, sizeof(char));
        if (dirname == NULL) {
          fputs("CUPS-PDF: failed to allocate memory\n", stderr);
          free(user);
          return -2;
        }  
        snprintf(dirname, size, "%s/%s", CPOUT, CPANONDIRNAME);
      }
    }
    else {
      log_event(CPSTATUS, "anonymous access denied", NULL);
      free(user);
      return 0;
    }
    mode=0666;
  } 
  else {
    log_event(CPDEBUG, "user identified", passwd->pw_name);
    if (!strncmp(CPOUT, "$HOME", 5)) {
      size=strlen(passwd->pw_dir)+strlen(CPHOMESUB)+4;
      dirname=calloc(size, sizeof(char));
      if (dirname == NULL) {
        fputs("CUPS-PDF: failed to allocate memory\n", stderr);
        free(user);
        return -2;
      }  
      snprintf(dirname, size, "%s/%s", passwd->pw_dir, CPHOMESUB);
    }
    else {
      size=strlen(CPOUT)+strlen(passwd->pw_name)+4;
      dirname=calloc(size, sizeof(char));
      if (dirname == NULL) {
        fputs("CUPS-PDF: failed to allocate memory\n", stderr);
        free(user);
        return -2;
      }  
      if (!CPDIRPREFIX)
        snprintf(dirname, size, "%s/%s", CPOUT, argv[2]);
      else
        snprintf(dirname, size, "%s/%s", CPOUT, passwd->pw_name);
    }
    mode=0600;
  }
  free(user);
  if (prepareuser(passwd, dirname)) return -2;
  log_event(CPDEBUG, "user information prepared", NULL);

  size=strlen(CPSPOOL)+22;
  spoolfile=calloc(size, sizeof(char));
  if (spoolfile == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    free(dirname);
    return -3;
  }
  snprintf(spoolfile, size, "%s/cups2pdf-%i", CPSPOOL, (int) getpid());
  log_event(CPDEBUG, "spoolfile name created", spoolfile);

  if (argc == 6) {
    if (preparespoolfile(stdin, spoolfile, title, atoi(argv[1]))) {
      free(dirname);
      free(spoolfile);
      return -4;
    }
    log_event(CPDEBUG, "input data read from stdin", NULL);
  }
  else {
    if (preparespoolfile(fopen(argv[6], "r"), spoolfile, title, atoi(argv[1]))) {
      free(dirname);
      free(spoolfile);
      return -4;
    }
    log_event(CPDEBUG, "input data read from file", argv[6]);
  }  

  size=strlen(dirname)+strlen(title)+6;
  outfile=calloc(size, sizeof(char));
  if (outfile == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    free(dirname);
    free(spoolfile);
    return -5;
  }
  snprintf(outfile, size, "%s/%s.pdf", dirname, title);
  log_event(CPDEBUG, "output filename created", outfile);

  size=strlen(CPGSCALL)+strlen(CPGHOSTSCRIPT)+strlen(CPPDFVER)+strlen(outfile)+strlen(spoolfile)+6;
  gscall=calloc(size, sizeof(char));
  if (gscall == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    free(dirname);
    free(spoolfile);
    free(outfile);
    return -6;
  }
  snprintf(gscall, size, CPGSCALL, CPGHOSTSCRIPT, CPPDFVER, outfile, spoolfile);
  log_event(CPDEBUG, "ghostscript commandline built", gscall);

  unlink(outfile);
  log_event(CPDEBUG, "output file unlinked", outfile);

  umask(0777);  
  system(gscall);
  log_event(CPDEBUG, "ghostscript has finished", NULL);
  
  unlink(spoolfile);
  log_event(CPDEBUG, "spoolfile unlinked", spoolfile);

  if (chown(outfile, passwd->pw_uid, passwd->pw_gid)) {
    log_event(CPERROR, "failed to set owner for PDF file", outfile);
    free(dirname);
    free(spoolfile);
    free(outfile);
    free(gscall);
    return -7;
  }
  log_event(CPDEBUG, "owner set for user output", outfile);

  if (chmod(outfile, mode)) {
    log_event(CPERROR, "failed to set file mode for PDF file", outfile);
    free(dirname);
    free(spoolfile);
    free(outfile);
    free(gscall);
    return -8;
  }
  log_event(CPDEBUG, "file mode set for user output", outfile);
  
  free(dirname);
  free(spoolfile);
  free(outfile);
  free(gscall);
  
  log_event(CPDEBUG, "all memory has been freed", NULL);

  log_event(CPSTATUS, "PDF creation successfully finished", passwd->pw_name);

  return 0;
} 
