/* cups-pdf.c -- CUPS Backend 
   08.02.2003, Volker C. Behr
   Exp. Physik V, Uni Wuerzburg 
   vrbehr@cip.physik.uni-wuerzburg.de
   http://cip.physik.uni-wuerzburg.de/~vrbehr/cups-pdf

   This code may be freely distributed as long as this header 
   is preserved. 

   This code is distributed under the GPL.
   (http://www.gnu.org/copyleft/gpl.html)

   ---------------------------------------------------------------------------

   Copyright (C) 2003,2004,2005  Volker C. Behr

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   ---------------------------------------------------------------------------
  
   If you want to redistribute modified sources/binaries this header
   has to be preserved and all modifications should be clearly 
   indicated.
   In case you want to include this code into your own programs 
   I would appreaciated some feedback via email.

  
   HISTORY: 
   2005-06-16 : 1.7.1	   - PDFs are protected during creation by file mode 000
   2005-03-07 : 1.7.0a
   2005-03-01 : 1.7.0      - call of GhostScript from a non-privileged child process,
                             no more additional programs in the call of GhostScript,
                             several code optimizations
   2005-02-09 : 1.6.6      - updated PostScript PPD file supporting more DIN formats
   2005-01-04 : 1.6.5      - cups-pdf is running with gid of group defined in CPGRP
   2004-10-11 : 1.6.4      - improved configurability for different versions of 'su'
   2004-10-07 : 1.6.3      - corrected erroneous call of GhostScript
   2004-10-05 : 1.6.2      - added necessary quotes in the call of GhostScript
   2004-10-03 : 1.6.1      - allow users w/o valid shell in /etc/passwd,
                             several code optimizations
   2004-09-08 : 1.6.0      - new option to check against user names in lower case,
                             new option to label all created PDFs with a job-id
   2004-08-10 : 1.5.2      - fixed insecure creation of spoolfile,
     			     several smaller code optimizations
   2004-08-09 : 1.5.1	   - fixed possible access to element outside allocated area,
                             fixed possible conflict with overlapping memory areas
   2004-08-03 : 1.5.0      - call to GS without root privileges for enhanced security
   2004-08-01 : 1.4.3      - added -dSAFER to the call to GS to improve security
   2004-02-25 : 1.4.2a
   2004-02-24 : 1.4.2      - improved extraction of filenames,
                             handling of embedded (e)ps code (RVT)
   2004-02-09 : 1.4.1	   - logfile is created with mode 644 if it does not exist
   2004-01-27 : 1.4.0      - new option to set permissions for PDF output,
                             alternative options for improved image conversion
                             (moved CPGSCALL to cups-pdf.h)
   2004-01-06 : 1.3.2	   - exit status of ghostscript gets logged in debug mode
   2004-01-02 : 1.3.1a
   2003-11-29 : 1.3.1	   - improved handling of domain prefix
   2003-11-02 : 1.3	   - new option to cut file name extensions
   2003-10-25 : 1.2        - untitled documents are labelled with job-id,
			     new option to add a domain prefix to user names,
                             removed a minor bug during character replacement
   2003-08-02 : 1.1	   - files not viewable until creation is finished,
   			     new qualifier "$HOME" for output directory,
			     "$HOME"-related new define in the header file
   2003-07-17 : 1.0.1	   - improved user name handling for multi-word names,
                             code now distributed under the GPL
   2003-06-29 : 1.0	   - enhanced debug output,
   			     replacement of special characters in title,
			     new option in the header file
   2003-05-26 : 1.0pre3    - several minor bugfixes and improvements 
   2003-05-21 : 1.0pre2    - optimized memory management
   2003-05-16 : 1.0pre1    - introduced cups-pdf.h, 
			     code now entirely written in C,
 			     cups-pdfgen is obsolete
   2003-02-14 : 0.2    - script re-located to /usr/lib/cups/filter/cups-pdfgen
   2003-02-12 : 0.1    - first beta release			                */

#define CPERROR 	1
#define CPSTATUS 	2
#define CPDEBUG 	4

#define BUFSIZE	1024
#define TBUFSIZE "1024"

#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <stddef.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "cups-pdf.h"

extern int errno;

FILE *logfp;

void log_event(char type, char message[48], char *detail) {
  time_t secs;
  char ctype[8], *timestring;

  if (strlen(CPLOG) && (type & CPLOGTYPE)) { 
    time(&secs);
    timestring=ctime(&secs);
    timestring[strlen(timestring)-1]='\0';

    if (type == CPERROR) 
      sprintf(ctype, "ERROR");
    else if (type == CPSTATUS)
      sprintf(ctype, "STATUS");
    else
      sprintf(ctype, "DEBUG");
    if (detail != NULL)  {
      while (detail[strlen(detail)-1] == '\n') 
        detail[strlen(detail)-1]='\0';
      fprintf(logfp,"%s  [%s] %s (%s)\n", timestring, ctype, message, detail);
      if ((CPLOGTYPE & CPDEBUG) && (type & CPERROR)) 
        fprintf(logfp,"%s  [DEBUG] ERRNO: %d\n", timestring, errno);
    }
    else
      fprintf(logfp,"%s  [%s] %s\n", timestring, ctype, message);
  }
  return;
}
    
int init() {
  struct stat fstatus;
  struct group *group;

  umask(0077);

  group=getgrnam(CPGRP);
  if (group)
    setgid(group->gr_gid);

  if (strlen(CPLOG)) {
    if (stat(CPLOG, &fstatus) || !S_ISDIR(fstatus.st_mode)) {
      if (mkdir(CPLOG, 0700))
        return 1;
      log_event(CPSTATUS, "log directory created", CPLOG);
    }
    logfp=fopen(CPLOG"/cups-pdf_log", "a");
  }

  if (!group) {
    log_event(CPERROR, "CPGRP not found", CPGRP);
    return 1;
  }
  else 
    log_event(CPDEBUG, "switching to new gid", CPGRP);

  umask(0022);
  if (!strncmp(CPOUT, "$HOME", 5)) 
    log_event(CPDEBUG, "output redirected to home directory", "$HOME");
  else if (stat(CPOUT, &fstatus) || !S_ISDIR(fstatus.st_mode)) {
    if (mkdir(CPOUT, 0755)) {
      log_event(CPERROR, "failed to create output directory", CPOUT);
      return 1;
    }
    log_event(CPSTATUS, "output directory created", CPOUT);
  }

  if (stat(CPSPOOL, &fstatus) || !S_ISDIR(fstatus.st_mode)) {
    if (mkdir(CPSPOOL, 0755)) {
      log_event(CPERROR, "failed to create spool directory", CPSPOOL);
      return 1;
    }
    log_event(CPSTATUS, "spool directory created", CPSPOOL);
  }

  umask(0077);
  return 0;
}

int prepareuser(struct passwd *passwd, char *dirname) {
  struct stat fstatus;

  umask(0000);
  if (stat(dirname, &fstatus) || !S_ISDIR(fstatus.st_mode)) {
    if (!strcmp(passwd->pw_name, CPANONUSER)) {
      if (mkdir(dirname, (0777&~CPANONUMASK))) {
        log_event(CPERROR, "failed to create anonymous output directory", dirname);
        return 1;
      }
      log_event(CPSTATUS, "anonymous output directory created", dirname);
    }
    else {
      if (mkdir(dirname, (0777&~CPUSERUMASK))) {
        log_event(CPERROR, "failed to create user output directory", dirname);
        return 1;
      }
      log_event(CPSTATUS, "user output directory created", dirname);
    }
    if (chown(dirname, passwd->pw_uid, passwd->pw_gid)) {
      log_event(CPERROR, "failed to set owner for output directory", passwd->pw_name);
      return 1;
    }
    log_event(CPDEBUG, "owner set for output directory", passwd->pw_name);
  }
  umask(0077);
  return 0;
}

void replace_string(char *string) {
 unsigned int i;

 log_event(CPDEBUG, "removing special characters from title", string);
 for (i=0;i<strlen(string);i++) 
   if ( ( (int) string[i] < '0' || (int) string[i] > '9' ) &&
        ( (int) string[i] < 'A' || (int) string[i] > 'Z' ) &&
        ( (int) string[i] < 'a' || (int) string[i] > 'z' ) &&
        (int) string[i] != '-' && (int) string[i] != '+' )
    string[i]='_';
  return;
}
  
int preparespoolfile(FILE *fpsrc, char *spoolfile, char *title, int job, struct passwd *passwd) {
  char buffer[BUFSIZE], *cut;
  int rec_depth;
  FILE *fpdest;

  if (fpsrc == NULL) {
    log_event(CPERROR, "failed to open source stream", NULL);
    return 1;
  }
  log_event(CPDEBUG, "source stream ready", NULL);
  fpdest=fopen(spoolfile, "w");
  if (fpdest == NULL) {
    log_event(CPERROR, "failed to open spoolfile", spoolfile);
    fclose(fpsrc);
    return 1;
  }
  log_event(CPDEBUG, "destination stream ready", spoolfile);
  if (chown(spoolfile, passwd->pw_uid, -1)) {
    log_event(CPERROR, "failed to set owner for spoolfile", spoolfile);
    return 1;
  }
  log_event(CPDEBUG, "owner set for spoolfile", spoolfile);
  rec_depth=0;
  while (fgets(buffer, BUFSIZE, fpsrc) != NULL) {
    if (!strncmp(buffer, "%!PS", 4)) {
      log_event(CPDEBUG, "found beginning of postscript code", buffer);
      break;
    }
  }
  log_event(CPDEBUG, "now extracting postscript code", NULL);
  fputs(buffer, fpdest);
  while (fgets(buffer, BUFSIZE, fpsrc) != NULL) {
    fputs(buffer, fpdest);
    if (!rec_depth)
      sscanf(buffer, "%%%%Title: %"TBUFSIZE"c", title);
    if (!strncmp(buffer, "%!PS", 4)) {
      log_event(CPDEBUG, "found embedded (e)ps code", buffer);
      rec_depth++;
    }
    else if (!strncmp(buffer, "%%EOF", 5)) {
      if (!rec_depth) {
        log_event(CPDEBUG, "found end of postscript code", buffer);
        break;
      }
      else {
        log_event(CPDEBUG, "found end of embedded (e)ps code", buffer);
        rec_depth--;
      }
    }
  }
  fclose(fpdest);
  fclose(fpsrc);
  log_event(CPDEBUG, "all data written to spoolfile", spoolfile);
  cut=strrchr(title, '/');
  if (cut != NULL) {
    log_event(CPDEBUG, "removing slashes from full title", title);
    memmove(title, cut+1, strlen(cut+1)+1);
  }  
  if (title != NULL) 
    while (strlen(title) && ((title[strlen(title)-1] == '\n') || (title[strlen(title)-1] == '\r'))) 
      title[strlen(title)-1]='\0';
  cut=strrchr(title, '.');
  if ((cut != NULL) && (strlen(cut) <= CPCUT+1) && (cut != title)) {
    log_event(CPDEBUG, "removing file name extension", cut);
    cut[0]='\0';
  }
  replace_string(title);
  if (!strcmp(title, "")) { 
    snprintf(title, BUFSIZE, "job_%i-untitled_document", job);
    log_event(CPDEBUG, "no title found - using default value", title);
  }
  else {
    if (CPLABEL) {
      strcpy(buffer, title);
      snprintf(title, BUFSIZE, "job_%i-%s", job, buffer);
    }
    log_event(CPDEBUG, "title successfully retrieved", title);
  }
  return 0;
}

int main(int argc, char *argv[]) {
  char *user, *dirname, *spoolfile, *outfile, *gscall, title[BUFSIZE]="";
  int size;
  mode_t mode;
  struct passwd *passwd;
  pid_t pid;

  if (init()) return -1;
  log_event(CPDEBUG, "initialization finished", NULL);

  if (argc==1) {
    printf("file cups-pdf:/ \"PDF Printer\" \"Virtual Printer\"\n"); 
    log_event(CPSTATUS, "identification string sent", NULL);
    return 0;
  }
  if (argc<6 || argc>7) {
    fputs("Usage: cups-pdf job-id user title copies options [file]\n", stderr);
    log_event(CPSTATUS, "call contained illegal number of arguments", NULL);
  return 0;
  }
  size=strlen(CPUSERPREFIX)+strlen(argv[2])+1;
  user=calloc(size, sizeof(char));
  if (user == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    return -2;
  }  
  snprintf(user, size, "%s%s", CPUSERPREFIX, argv[2]);
  passwd=getpwnam(user);
  if (passwd == NULL && CPLOWERCASE) {
    log_event(CPDEBUG, "unknown user", user);
    for (size=0;size<(int) strlen(argv[2]);size++) 
      argv[2][size]=tolower(argv[2][size]);
    log_event(CPDEBUG, "trying lower case user name", argv[2]);
    size=strlen(CPUSERPREFIX)+strlen(argv[2])+1;
    snprintf(user, size, "%s%s", CPUSERPREFIX, argv[2]);
    passwd=getpwnam(user);
  }  
  if (passwd == NULL) {
    if (strlen(CPANONUSER)) {
      passwd=getpwnam(CPANONUSER);
      if (passwd == NULL) {
        log_event(CPERROR, "username for anonymous access unknown", CPANONUSER);
        free(user);
        fclose(logfp);
        return -2;
      }
      log_event(CPDEBUG, "unknown user", user);
      if (!strncmp(CPOUT, "$HOME", 5)) {
        size=strlen(passwd->pw_dir)+strlen(CPHOMESUB)+4;
        dirname=calloc(size, sizeof(char));
        if (dirname == NULL) {
          fputs("CUPS-PDF: failed to allocate memory\n", stderr);
          free(user);
          fclose(logfp);
          return -2;
        }  
        snprintf(dirname, size, "%s/%s", passwd->pw_dir, CPHOMESUB);
      }
      else {
        size=strlen(CPOUT)+strlen(CPANONDIRNAME)+4;
        dirname=calloc(size, sizeof(char));
        if (dirname == NULL) {
          fputs("CUPS-PDF: failed to allocate memory\n", stderr);
          free(user);
          fclose(logfp);
          return -2;
        }  
        snprintf(dirname, size, "%s/%s", CPOUT, CPANONDIRNAME);
      }
    }
    else {
      log_event(CPSTATUS, "anonymous access denied", user);
      free(user);
      fclose(logfp);
      return 0;
    }
    mode=(0666&~CPANONUMASK);
  } 
  else {
    log_event(CPDEBUG, "user identified", passwd->pw_name);
    if (!strncmp(CPOUT, "$HOME", 5)) {
      size=strlen(passwd->pw_dir)+strlen(CPHOMESUB)+4;
      dirname=calloc(size, sizeof(char));
      if (dirname == NULL) {
        fputs("CUPS-PDF: failed to allocate memory\n", stderr);
        free(user);
        fclose(logfp);
        return -2;
      }  
      snprintf(dirname, size, "%s/%s", passwd->pw_dir, CPHOMESUB);
    }
    else {
      size=strlen(CPOUT)+strlen(passwd->pw_name)+4;
      dirname=calloc(size, sizeof(char));
      if (dirname == NULL) {
        fputs("CUPS-PDF: failed to allocate memory\n", stderr);
        free(user);
        fclose(logfp);
        return -2;
      }  
      if (!CPDIRPREFIX)
        snprintf(dirname, size, "%s/%s", CPOUT, argv[2]);
      else
        snprintf(dirname, size, "%s/%s", CPOUT, passwd->pw_name);
    }
    mode=(0666&~CPUSERUMASK);
  }
  free(user);
  if (prepareuser(passwd, dirname)) return -2;
  log_event(CPDEBUG, "user information prepared", NULL);

  size=strlen(CPSPOOL)+22;
  spoolfile=calloc(size, sizeof(char));
  if (spoolfile == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    free(dirname);
    fclose(logfp);
    return -3;
  }
  snprintf(spoolfile, size, "%s/cups2pdf-%i", CPSPOOL, (int) getpid());
  log_event(CPDEBUG, "spoolfile name created", spoolfile);

  if (argc == 6) {
    if (preparespoolfile(stdin, spoolfile, title, atoi(argv[1]), passwd)) {
      free(dirname);
      free(spoolfile);
      fclose(logfp);
      return -4;
    }
    log_event(CPDEBUG, "input data read from stdin", NULL);
  }
  else {
    if (preparespoolfile(fopen(argv[6], "r"), spoolfile, title, atoi(argv[1]), passwd)) {
      free(dirname);
      free(spoolfile);
      fclose(logfp);
      return -4;
    }
    log_event(CPDEBUG, "input data read from file", argv[6]);
  }  

  size=strlen(dirname)+strlen(title)+6;
  outfile=calloc(size, sizeof(char));
  if (outfile == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    free(dirname);
    free(spoolfile);
    fclose(logfp);
    return -5;
  }
  snprintf(outfile, size, "%s/%s.pdf", dirname, title);
  log_event(CPDEBUG, "output filename created", outfile);

  size=strlen(CPGSCALL)+strlen(CPGHOSTSCRIPT)+strlen(CPPDFVER)+strlen(outfile)+strlen(spoolfile)+6;
  gscall=calloc(size, sizeof(char));
  if (gscall == NULL) {
    fputs("CUPS-PDF: failed to allocate memory\n", stderr);
    free(dirname);
    free(spoolfile);
    free(outfile);
    fclose(logfp);
    return -6;
  }
  snprintf(gscall, size, CPGSCALL, CPGHOSTSCRIPT, CPPDFVER, outfile, spoolfile);
  log_event(CPDEBUG, "ghostscript commandline built", gscall);

  unlink(outfile);
  log_event(CPDEBUG, "output file unlinked", outfile);

  if (setenv("TMPDIR", CPGSTMP, 1)) {
    log_event(CPERROR, "insufficient space in environment to set TMPDIR", CPGSTMP);
    free(dirname);
    free(spoolfile);
    free(outfile);
    free(gscall);
    fclose(logfp);
    return -7;
  }
  log_event(CPDEBUG, "TMPDIR set for GhostScript", getenv("TMPDIR"));

  fflush(logfp);
  pid=fork();

  if (!pid) {
    log_event(CPDEBUG, "entering child process", NULL);

    if (setgid(passwd->pw_gid))
      log_event(CPERROR, "failed to set GID for current user", NULL);
    else 
      log_event(CPDEBUG, "GID set for current user", NULL);
    if (setuid(passwd->pw_uid))
      log_event(CPERROR, "failed to set UID for current user", passwd->pw_name);
    else 
      log_event(CPDEBUG, "UID set for current user", passwd->pw_name);
     
    umask(0777);
    size=system(gscall);
    snprintf(title,BUFSIZE,"%d",size);
    log_event(CPDEBUG, "ghostscript has finished", title);
    return 0;
  }
  log_event(CPDEBUG, "waiting for child to exit", NULL);
  waitpid(pid,NULL,0);

  if (unlink(spoolfile)) 
    log_event(CPERROR, "failed to unlink spoolfile (non fatal)", spoolfile);
  else 
    log_event(CPDEBUG, "spoolfile unlinked", spoolfile);

  if (chmod(outfile, mode)) {
    log_event(CPERROR, "failed to set file mode for PDF file", outfile);
    free(dirname);
    free(spoolfile);
    free(outfile);
    free(gscall);
    fclose(logfp);
    return -9;
  }
  log_event(CPDEBUG, "file mode set for user output", outfile);
 
  free(dirname);
  free(spoolfile);
  free(outfile);
  free(gscall);
  
  log_event(CPDEBUG, "all memory has been freed", NULL);

  log_event(CPSTATUS, "PDF creation successfully finished", passwd->pw_name);

  fclose(logfp);
  return 0;
} 
