
/*
 * Copyright (c) 2000 David Stes.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: menufram.m,v 1.62 2002/06/15 19:19:31 stes Exp $
 */

#include <curses.h>
#include <menu.h>
#include <stdio.h>
#include <stdlib.h>

#include <unistd.h>

#include <Object.h>
#include <ocstring.h>
#include <ordcltn.h>
#include "slk.h"
#include "frame.h"
#include "menufram.h"
#include "item.h"
#include "var.h"
#include "cursel.h"
#include "assert.h"
#include "cmdmenu.h"
#include "process.h"

#define FMLIROWS 10

@implementation Menu

- items
{
  return items;
}

- items:i
{
  items=i;
  return self;
}

- (STR)name
{
   return (menu)?[menu str]:"Menu";
}

- curitem
{
  ITEM *x = current_item(eti_menu); return (id)item_userptr(x);
}

- enter
{
  if (ismultiselect) {
    id c = expandstr(done);
    if (c) evalcmd(c);
  } else {
    [[self curitem] eval];
  }
  if (self == activeframe) [self setcursor];
  return self;
}

- itemchanged
{
  id m,item;
  item = [self curitem];
  [self drawscrollbar];
  if (ismultiselect) [item setselected];
  [item setlininfo];
  m = [item itemmsg]; 
  if (m) showtransientmessage([m str]);
  return self;
}

void itemchanged(MENU *m)
{
  [(id)menu_userptr(m) itemchanged];
}

- destroy
{
  if (posted) [self unpost];
  if (eti_menu) free_menu(eti_menu);
  [items elementsPerform:_cmd];
  if (eti_items) free(eti_items);
  eti_menu = NULL;
  eti_items = NULL;
  return [super destroy];
}

- setframerect
{
  id r;
  int w,h;
  int needrows;
  int specifrows,specifcols;
  int probrows,probcols;
  int availrows,fitcols,maxcols;

  maxcols = ((DISPLAYW - 3) / 2);
  availrows = (DISPLAYH - [self fixedbegrow] - 3);
  fitcols = ((DISPLAYW - [self fixedbegcol] - 10) / (maxitemwidth+1));

  dbg("DISPLAYW: %i, DISPLAYH: %i\n",DISPLAYW,DISPLAYH);
  dbg("maxitemwidth: %i\n",maxitemwidth);
  dbg("maxcols: %i, fitcols: %i, availrows: %i\n",maxcols,fitcols,availrows);

  needrows = totalitems;

  if (hasdescription) {

    usedcols = 1; /* force one column, ignore "cols" descriptor */
    usedrows = (needrows>availrows)?availrows:needrows; 

    if (rows) {
      specifrows = atoi([rows str]);
      usedrows = (usedrows>specifrows)?specifrows:usedrows; 
    } else {
      usedrows = (usedrows>FMLIROWS)?FMLIROWS:usedrows; 
    }

  } else {

    if (columns==nil && rows==nil) {
      usedcols = 1;
      usedrows = (needrows>availrows)?availrows:needrows; 
      usedrows = (usedrows>FMLIROWS)?FMLIROWS:usedrows; 
    } 

    if (columns && rows==nil) {
      specifcols = atoi([columns str]);
      probcols = (specifcols > maxcols)?1:specifcols;
      probrows = ((totalitems - 1) / probcols) + 1;
      usedcols = (probrows > availrows)?1:probcols;
      usedrows = (probrows > availrows)?availrows:probrows;
    } 

    if (rows && columns==nil) {
      specifrows = atoi([rows str]);
      probrows = (specifrows>needrows)?needrows:specifrows;
      probrows = (probrows>availrows)?availrows:probrows;
      probcols = ((totalitems - 1) / probrows) + 1;
      usedcols = (probcols > fitcols)?fitcols:probcols;
      usedrows = probrows;
    }
 
    if (rows && columns) {
      specifrows = atoi([rows str]);
      specifcols = atoi([columns str]);
      probcols = (specifcols > maxcols)?maxcols:specifcols;
      probrows = ((totalitems - 1) / probcols) + 1;
      usedcols = (probrows > availrows)?1:probcols;
      specifrows = (specifrows > availrows)?availrows:specifrows;
      usedrows = (probrows > specifrows)?specifrows:probrows;
    }

  }

  dbg("usedrows: %i, usedcols: %i\n",usedrows,usedcols);
  set_menu_format(eti_menu,usedrows,usedcols);

  if (hasdescription) {
    /* maxitemwidth = width of name only */
    scale_menu(eti_menu,&h,&w);
  } else {
    h = usedrows;w = usedcols * (1 + maxitemwidth);
  }

  dbg("scalemenu rows : %i, scalemenu cols: %i\n",h,w);
  r = [self begframerect:w+3:h+2]; 
  if (r) {
   [self setframerect:r]; 
   return self;
  } else {
   return nil;
  }
}

- setmenuformat
{
  int availrows = [self height] - 2; 
  if (usedrows > availrows) usedrows = availrows;
  dbg("usedrows: %i, usedcols: %i\n",usedrows,usedcols);
  set_menu_format(eti_menu,usedrows,usedcols);
  return self;
}

- create
{
  id r;
  int ok;
  int i,w,n;
  WINDOW *t;

  if (eti_menu) [self destroy];
  if ((n = [items size]) == 0) return nil;

  eti_items = (ITEM**)malloc(sizeof(ITEM*) * (n + 1));
  maxitemwidth = strlen([self name]);
  if (maxitemwidth > DISPLAYW - 5) maxitemwidth = DISPLAYW - 5;

  for(i=0,totalitems=0;i<n;i++) {
    id item = [items at:i];
    if ([item showing]) {
       [item create];
       eti_items[totalitems++] = [item eti_item];
       if ([item hasdescription]) hasdescription=YES;
       w = [item width];
       if (w > maxitemwidth) maxitemwidth=w;
    }
  }

  if (totalitems == 0) return nil;

  eti_items[totalitems] = NULL;
  eti_menu = new_menu(eti_items);
  set_menu_userptr(eti_menu,(char*)self);
  set_item_init(eti_menu,itemchanged);

  set_menu_mark(eti_menu,">");

  if (hasdescription) {
    set_menu_pad(eti_menu,'-');
  }

#ifdef NCURSES_VERSION
  set_menu_spacing(eti_menu,3,1,1);
#endif

  if (cyclic) {
    menu_opts_off(eti_menu,O_NONCYCLIC);
  } else {
    menu_opts_on(eti_menu,O_NONCYCLIC);
  }

  menu_opts_off(eti_menu,O_ROWMAJOR);

  ismultiselect = expandbool(multiselect,NO);

  if (ismultiselect) {
    menu_opts_off(eti_menu,O_ONEVALUE);
  } else {
    menu_opts_on(eti_menu,O_ONEVALUE);
  }

  set_menu_grey(eti_menu,A_NORMAL);

  if (has_colors()) {
    set_menu_back(eti_menu,A_NORMAL|COLOR_PAIR(colorpairs[CP_TEXT]));
    set_menu_fore(eti_menu,A_BOLD|COLOR_PAIR(colorpairs[CP_MENU]));
  } else {
    set_menu_back(eti_menu,A_NORMAL);
    set_menu_fore(eti_menu,A_BOLD);
  }

  if (framerect) {
    r = [self setmenuformat];
  } else {
    r = [self setframerect];
  }

  if (!r) return [self frametoolarge];

  eti_win=subwin(stdscr,[self height],[self width],[self top],[self left]);
  assert(eti_win);

  set_menu_win(eti_menu,eti_win);
  eti_subwin=derwin(eti_win,[self height]-1,[self width]-1,1,1);
  set_menu_sub(eti_menu,eti_subwin);

  ok = keypad(eti_subwin,TRUE);
  assert(ERR != ok);

  if (has_colors()) {
    wbkgdset(eti_win,COLOR_PAIR(colorpairs[CP_TEXT])|' ');
    wbkgdset(eti_subwin,COLOR_PAIR(colorpairs[CP_TEXT])|' ');
  }

  [Var define:"NR" as:[String sprintf:"%i",totalitems] env:YES];
  return self;
}

- drawscrollbar
{
  int y1,y2;
  int h=[items size];
  int availrows = [self height] - 2; 
  if (availrows < 3) return self;
  y1 = top_row(eti_menu);
  y2 = y1+availrows;
  if (y2>h) y2=h;
  [self drawscrollbar:y1:y2:h];
  return self;
}

- drawframe
{
  return [self drawbox:[self name]];
}

- draw
{
  int ok;
  dbg("draw %s\n",[self name]);

  if (!eti_win) { if (![self create]) return nil; }

  if (has_colors()) {
    if (isactive) {
      set_menu_fore(eti_menu,A_BOLD|COLOR_PAIR(colorpairs[CP_TEXT]));
    } else {
      set_menu_fore(eti_menu,A_NORMAL|COLOR_PAIR(colorpairs[CP_TEXT]));
    }
  } else {
    if (isactive) {
      set_menu_fore(eti_menu,A_BOLD);
    } else {
      set_menu_fore(eti_menu,A_NORMAL);
    }
  }

  if (totalitems>0) {
    
    werase(eti_subwin);

    if (posted) {
      [self unpost];
    }
    ok = post_menu(eti_menu);
    if (multiselect) [items elementsPerform:@selector(setupselected)];
    dbg("post_menu: %i\n",ok);
    assert(ok == E_OK);
    posted++;
    [self drawframe];
    [self drawscrollbar];
  } 

  [self setcursor];

  needdisplay=NO;
  return self;
}

- setcursor
{
  int ok=pos_menu_cursor(eti_menu);
  assert(ok == E_OK);
  return self;
}

- activate
{
  if ([items size]==0) return nil; /* refuse to activate empty menu */
  [super activate];
  if (posted) [self setcursor];
  return self;
}

- unpost
{
  int ok;
  if (posted) {
    ok = unpost_menu(eti_menu);
    dbg("ok: %i\n",ok);
    assert(E_OK == ok);
    posted=0;
  }
  return self;
}

- home
{
  int ok = menu_driver(eti_menu,REQ_FIRST_ITEM);
  if (ok != E_OK) {
     beep();
  } else {
    if (![[self curitem] isselectable]) [self arrowdown];
  }
  return self;
}

- arrowleft
{
  if (usedcols==1) {
    return [self arrowup];
  } else {
    int ok = menu_driver(eti_menu,REQ_LEFT_ITEM);
    if (ok != E_OK) beep();
    return self;
  } 
}

- arrowright
{
  if (usedcols==1) {
    return [self arrowdown];
  } else {
    int ok = menu_driver(eti_menu,REQ_RIGHT_ITEM);
    if (ok != E_OK) beep();
    return self;
  }
}

- spacebar
{
  return [self arrowright];
}

- end
{
  int ok = menu_driver(eti_menu,REQ_LAST_ITEM);
  if (ok != E_OK) {
     beep();
  } else {
    if (![[self curitem] isselectable]) [self arrowup];
  }
  return self;
}

- downpage
{
  int ok = menu_driver(eti_menu,REQ_SCR_DPAGE);
  if (ok != E_OK) beep();
  return self;
}

- uppage
{
  int ok = menu_driver(eti_menu,REQ_SCR_UPAGE);
  if (ok != E_OK) beep();
  return self;
}

- arrowup
{
  int ok = menu_driver(eti_menu,REQ_UP_ITEM);
  if (ok != E_OK) {
     beep();
  } else {
    if (![[self curitem] isselectable]) [self arrowup];
  }
  return self;
}

- arrowdown
{
  int ok = menu_driver(eti_menu,REQ_DOWN_ITEM);
  if (ok != E_OK) {
     beep();
  } else {
    if (![[self curitem] isselectable]) [self arrowdown];
  }
  return self;
}

- controlt
{
  return [self arrowleft];
}

- controlb
{
  return [self home];
}

- controly
{
  return self;
}

- controlh
{
  return [self arrowleft];
}

- clearEol
{
  return self;
}

- homedown
{
  return [self end];
}

- mark
{
  if (ismultiselect) {
    [[self curitem] mark];
  } else {
    beep();
  }
  return self;
}

- scrolldown
{
  return [self arrowdown];
}

- scrollup
{
  return [self arrowup];
}

- reset
{
  return self;
}

- controlx
{
  return self;
}

- controlk
{
  return self;
}

- controld
{
  return [self arrowdown];
}

- controle
{
  return [self end];
}

- controla
{
  return self;
}

- controlo
{
  return self;
}

- controll
{
  return [self arrowleft];
}

- controln
{
  return [self arrowdown];
}

- controlw
{
  return [self pagedown];
}

- controlv
{
  return [self pageup];
}

- controlp
{
  return [self arrowup];
}

- controlr
{
  return [self arrowright];
}

- controli
{
  return [self arrowright];
}

- controlu
{
  return [self arrowup];
}

- getevent
{
  int c = [self getkey];
  dbg("menufram wgetch: %i\n",c);

  if (c == ' ') return [self spacebar];
  
  switch (menu_driver(eti_menu,c)) {
     case E_OK : break;
     case E_UNKNOWN_COMMAND : [self keydown:c];break;
     default : beep();break;
  }

  return self;
}

- setslk
{
  int n = MAXFUNKEYS;
  while (n--) slktab[n] = nil;
  if (ischoicemenu) {
   slktab[3] = k_enter;
   slktab[6] = k_cancel;
  } else if (iscmdmenu) {
   slktab[1] = k_help;
   slktab[6] = k_cancel;
  } else {
   slktab[1] = k_help;
   if (ismultiselect) slktab[2] = k_mark;
   slktab[3] = k_enter;
   slktab[4] = k_prev_frm;
   slktab[5] = k_next_frm;
   slktab[6] = k_cancel;
   slktab[7] = k_cmd_menu;
  }
  return [super setslk];
}

- writeOn:process delim:(char*)d
{
  int ok;
  int i,n;
  BOOL pdelim = NO;
  for(i=0,n=[items size];i<n;i++) {
    id item = [items at:i];
    if ([item isselected]) {
       if (pdelim) {
         [process dstwrite:d];
       }
       [item writeOn:process delim:d];
       pdelim=YES;
    }
  }
  return self;
}

@end
 
