#!/usr/bin/ruby -w
# -*- ruby -*-

# cvsdelta: summarizes CVS changes and executes the appropriate commands

require 'find'

require './progmet'
require './log'
require './file'
require './cvsignore'
require './cvsstats'
require './cvsdiff'
require './env'
require './cvsexec'
require './string'
require './cvsdeltaopt'


# A difference within a configuration management system.

class CVSDelta
  include Loggable
  
  attr_reader :added, :changed, :deleted, :total
  
  def initialize(options, args)
    @options = options

    # for showing that we're actually doing something
    @progress = if @options.quiet then nil else ProgressMeter.new(@options.verbose) end

    @ignored_patterns = IgnoredPatterns.new

    args = [ "." ] unless args.length > 0

    @unreadable = Hash.new
    @entries  = Array.new
    @entfiles = Array.new

    @args = args
  end

  def run
    if @options.changes
      read_changes(@args)
    else
      log "not processing changes"
    end

    # determine new files

    if @options.adds || @options.deletes
      read_ignored_directories(@args)
      
      read_added_deleted_files(@args)
    else
      log "not processing adds or deletes"
    end
  end

  def read_ignored_directories(dirs)
    dirs = @args.collect { |a| File.dirname(a) }.uniq
    dirs.each do |dir| 
      @ignored_patterns.read(dir)
    end
  end

  def read_ignored_directory(dirs)
    dirs = @args.collect { |a| File.dirname(a) }.uniq
    # dirs << cvsed_parent_directories(dirs)
    # dirs = @args.collect { |a| File.dirname(a) }.uniq
    # dirs << cvsed_parent_directories(dirs)

    dirs.each do |dir| 
      @ignored_patterns.read(dir)
    end

    exit
  end

  def read_changes(args)
    diff = CVSDiff.new(args)

    diff.fromdate = @options.fromdate
    diff.fromrev = @options.fromrev
    diff.todate = @options.todate
    diff.torev = @options.torev
    diff.tmpdir = @options.tmpdir

    diff.run

    @added    = FileHash.new
    @changed  = diff.changed
    @deleted  = FileHash.new
    @total    = diff.total

    # If it is only from, we can just get the line count from the local files.
    # If there is a to, we have to use that date as the update time.
    # And we have to deal with binary files.

    # Unfortunately, cvs spews out "update -p" without anything to indicate the
    # start or end of files. So we can get the total line count correctly, but
    # not for the individual files. Thus, they'll just get "???" for their line
    # counts, but the total will be right. And that's probably what most people
    # are interested in, so that's OK.

    if diff.missing.size > 0
      if @options.torev || @options.todate
        cmd = "cvs -fq -z" + @options.compression.to_s + " update -p "
        if @options.torev
          cmd += " -r #{@options.torev} "
        else
          cmd += " -D \"#{@options.todate}\" "
        end
        diff.missing.each do |mf|
          cmd +=  " \"#{mf}\""
        end

        lines = `#{cmd}`
        begin
          linecount = lines.select { |line| line.is_ascii? }.size
        rescue
          # this might have been a file that had not been properly
          # registered as a binary file.
          linecount = 1
        end
        diff.missing.each do |mf|
          fname = File.clean_name(mf)
          log "#{mf} => #{fname}"
          @added[fname] = AddedUncountedFile.new(fname)
        end

        log "incrementing total adds by #{linecount}"
        diff.total.adds += linecount
      else
        diff.missing.each do |mf|
          fname = File.clean_name(mf)
          log "#{mf} => #{fname}"
          add_file(fname)
        end
      end
    end

    if diff.deleted.size > 0
      diff.deleted.each do |mf|
        fname = File.clean_name(mf)
        log "#{mf} => #{fname}"
        @deleted[fname] = DeletedFile.new(fname)
      end
    end

  end

  def read_added_deleted_files(args)
    log "cvsdelta: read_added_deleted_files(" + args.join(", ") + ")"
    
    args.each do |arg|
      if File.directory?(arg)
        if @options.process_unknown_dirs
          read_each_subdirectory(arg)
        else
          read_each_cvsed_subdirectory(arg)
        end
      else
        # read the CVS/Entries file in the directory of the file
        dir = File.dirname(arg)
        read_entries_file(dir) unless dir.index(/\bCVS\b/)
        log "adding file #{arg}"
        consider_file_for_addition(arg)
      end
    end
  end

  def read_each_subdirectory(top)
    # log top

    File.find_directories(top).each do |dir|
      log "processing dir #{dir}"
      unless dir.index(/\bCVS\b/)
        read_entries_file(dir) 
      end
    end

    File.find_files(top).each do |f|
      # log "adding found file: #{f}"
      consider_file_for_addition(f)
    end
  end

  def read_each_cvsed_subdirectory(top)
    log top
    
    cvsdirs = File.find_where(top) do |fd| 
      File.is_directory?(fd) && File.exists?(fd + "/CVS/Entries")
    end
    
    cvsdirs.each do |dir|
      log "processing dir #{dir}"
      read_entries_file(dir) 
      
      File.local_files(dir).each do |f|
        log "adding local file: #{f}"
        consider_file_for_addition(f)
      end
    end
  end

  def read_entries_file(dir)
    entfile = dir + "/CVS/Entries"
    if @entfiles.include?(entfile)
      log "entries file " + entfile + " already read"
    elsif !File.exists?(entfile)
      log "no entries file: " + entfile
    else
      log "reading entries file: " + entfile
      IO.foreach(entfile) do |line|
        @progress.tick(dir) if @progress
        file, ver, date = line.split('/')[1 .. 3]
        log "file: #{file}; ver: #{ver}; date: #{date}"
        if file and ver and !file.empty? and !ver.empty?
          fullname = File.clean_name(dir + "/" + file)
          log "adding entry: " + fullname
          @entries.push(fullname)
          
          if date == "dummy timestamp"
            @changed[file] = NewFile.create(file)
          elsif not File.exists?(fullname)
            log "entry " + fullname + " is missing"
            add_deleted_file(fullname)
          end
        end
      end
      @entfiles.push(entfile)
      @ignored_patterns.read(dir)
    end
  end

  def add_file(fname)
    log "file name = #{fname}"
    
    if File.readable?(fname)
      unless @added.include?(fname)
        # don't add it twice
        @added[fname] = NewFile.create(fname)
        if @added[fname].is_counted?
          log "adding #{fname}: #adds: #{@added[fname].adds}"
          @total.adds += @added[fname].adds
        else
          log "adding #{fname}: not counted"
        end
      end
    elsif !@unreadable.has_key?(fname)
      puts "\rnot readable: " + fname 
      @unreadable[fname] = true
    end
  end

  def file_included?(fname)
    @entries.include?(fname)
  end

  def file_ignored?(fname)
    @ignored_patterns.is_ignored?(fname)
  end

  def consider_file_for_addition(file)
    if @options.adds
      fname = File.clean_name(file)
      unless file_included?(fname) || file_ignored?(fname)
        add_file(fname)
      end
    else
      log "not adding new file " + file
    end
  end

  def add_deleted_file(file)
    if @options.deletes
      @deleted[file] = DeletedFile.new(file)
    else
      log "not adding deleted file " + file
    end
  end
  
  def print_change_summary
    puts
    printf "%-7s  %-7s  %-7s  %-7s  %s\n", "total", "added", "changed", "deleted", "file"
    printf "=======  =======  =======  =======  ====================\n"

    files = Hash.new
    [ added, changed, deleted ].each do |ary|
      ary.each do |file, record| 
        files[file] = record
      end
    end

    files.sort.each { |file, record| record.print }
    printf "-------  -------  -------  -------  --------------------\n";
    total.print("Total")
  end
  
end

if __FILE__ == $0

  # use the defaults
  opts = CVSDeltaOptions.new

  Log.verbose = true
  
  delta = CVSDelta.new(opts, ARGV)
  delta.run

  puts "added:"
  puts delta.added.values.join(", ")

  puts "changed:"
  puts delta.changed.values.join(", ")

  puts "deleted:"
  puts delta.deleted.values.join(", ")
end
