#!/usr/bin/ruby -w
# -*- ruby -*-

require './file'
require './log'


# A file that has changed with respect to the configuration management system.
# This can be one that has been added (a new file), changed (a previously #
# existing file), or deleted (one that has been removed).

class DeltaFile
  include Loggable

  attr_accessor :adds, :changes, :deletes, :name

  UNKNOWN = -1
  BINARY = -2
  
  def initialize(name)
    # in Ruby, these are primitives, not Objects, so they are not
    # referencing the same primitive value (i.e., this is just like Java)
    @adds = @changes = @deletes = 0
    @name = File.clean_name(name)
  end

  def total
    @adds + @changes + @deletes
  end

  def is_counted?
    [ @adds, @changes, @deletes ].each do |v|
      if [ UNKNOWN, BINARY ].grep(v).length > 0
        return false
      end
    end
    return true
  end

  def print(nm = nil)
    nm = symbol + " " + name unless nm
    [total, adds, changes, deletes].each do |v|
      str = 
        case v
        when UNKNOWN
          "???"
        when BINARY
          "bin"
        else
          v.to_s
        end
      printf("%7s  ", str)
    end
    $stdout.print nm, "\n"
  end

  def to_s
    self.class.to_s + " \"#{@name}\" a:#{@adds} c:#{@changes} d:#{deletes}"
  end

end


class ExistingFile < DeltaFile

  def symbol; "*"; end
  
end


class DeletedFile < DeltaFile

  def initialize(name)
    super
    # it would be nice to know how long the file was, i.e., many lines were
    # deleted
    @deletes = UNKNOWN
  end

  def symbol; "-"; end
  
end


class NewFile < DeltaFile

  def NewFile.create(fname)
    File.is_text?(fname) ? NewFile.new(fname) : NewBinaryFile.new(fname)
  end

  def initialize(name)
    super
    @adds = IO.readlines(name).length
  end

  def symbol; "+"; end
  
end


class NewBinaryFile < DeltaFile

  def initialize(name)
    super
    f = File.new(name)
    # @adds = f.read.length
    # @adds = f.readlines.length
    @adds = BINARY
  end

  def symbol; "+"; end
  
#   def print(nm = nil)
#     nm = symbol + " " + name unless nm
#     [total, adds, changes, deletes].each do |v|
#       printf("%7sb ", v == @@UNKNOWN ? "???" : v.to_s)
#     end
#     $stdout.print nm, "\n"
#   end

end


class AddedUncountedFile < DeltaFile

  def initialize(name)
    super
    @adds = UNKNOWN
  end

  def symbol; "+"; end
  
end
