#!/usr/bin/ruby -w
# -*- ruby -*-

#     Attribute codes:
#         00=none 01=bold 04=underscore 05=blink 07=reverse 08=concealed
#     Text color codes:
#         30=black 31=red 32=green 33=yellow 34=blue 35=magenta 36=cyan 37=white
#     Background color codes:
#         40=black 41=red 42=green 43=yellow 44=blue 45=magenta 46=cyan 47=white

require './ansicolor'


# Very minimal logging output. If verbose is set, this displays the method and
# line number whence called. It can be a mixin to a class, which displays the
# class and method from where it called. If not in a class, it displays only the
# method.

class Log

  $LOGGING_LEVEL = nil
  
  module Severity
    DEBUG = 0
    INFO  = 1
    WARN  = 2
    ERROR = 3
    FATAL = 4
  end

  include Log::Severity

  def initialize
    $LOGGING_LEVEL = @level = FATAL
    @ignored_files = {}
    @ignored_methods = {}
    @ignored_classes = {}
    @width = 0
    @output = $stderr
    @fmt = "[%s:%04d] {%s}"
    @autoalign = false
    @colors = []
    @colorize_line = false
  end
    
  def verbose=(v)
    @level = v ? DEBUG : FATAL
  end

  def level=(lvl)
    @level = lvl
  end

  # Assigns output to the given stream.
  def output=(io)
    @output = io
  end

  # sets whether to colorize the entire line, or just the message.
  def colorize_line=(col)
    @colorize_line = col
  end

  # Assigns output to a file with the given name. Returns the file; client
  # is responsible for closing it.
  def outfile=(f)
    @output = if f.kind_of?(IO) then f else File.new(f, "w") end
  end

  # Creates a printf format for the given widths, for aligning output.
  def set_widths(file_width, line_width, func_width)
    @fmt = "[%#{file_width}s:%#{line_width}d] {%#{func_width}s}"
  end

  def ignore_file(fname)
    @ignored_files[fname] = true
  end
  
  def ignore_method(methname)
    @ignored_methods[methname] = true
  end
  
  def ignore_class(classname)
    @ignored_classes[classname] = true
  end

  def log_file(fname)
    @ignored_files.delete(fname)
  end
  
  def log_method(methname)
    @ignored_methods.delete(methname)
  end
  
  def log_class(classname)
    @ignored_classes.delete(classname)
  end

  # Sets auto-align of the {function} section.  set_widths will likely
  # result in nicer output.
  def autoalign
    @autoalign = true
  end

  def debug(msg = "", depth = 1, &blk)
    log(msg, DEBUG, depth + 1, &blk)
  end

  def info(msg = "", depth = 1, &blk)
    log(msg, INFO, depth + 1, &blk)
  end

  def warn(msg = "", depth = 1, &blk)
    log(msg, WARN, depth + 1, &blk)
  end

  def error(msg = "", depth = 1, &blk)
    log(msg, ERROR, depth + 1, &blk)
  end

  def fatal(msg = "", depth = 1, &blk)
    log(msg, FATAL, depth + 1, &blk)
  end

  # Logs the given message.
  def log(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    if level >= @level
      c = caller(depth)[0]
      c.index(/(.*):(\d+)(?::in \`(.*)\')?/)
      file, line, func = $1, $2, $3
      file.sub!(/.*\//, "")

      if cname
        func = cname + "#" + func
      end

      if @ignored_files[file] || (cname && @ignored_classes[cname]) || @ignored_methods[func]
        # skip this one.
      elsif @autoalign
        print_autoaligned(file, line, func, msg, level, &blk)
      else
        print_formatted(file, line, func, msg, level, &blk)
      end
    end
  end

  # Shows the current stack.
  def stack(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    if level >= @level
      stk = caller(depth)
      for c in stk
        # puts "c    : #{c}"
        c.index(/(.*):(\d+)(?::in \`(.*)\')?/)
        file, line, func = $1, $2, $3
        file.sub!(/.*\//, "")

        func ||= "???"

        # puts "func : #{func}"
        # puts "cname: #{cname}"
        
        if cname
          func = cname + "#" + func
        end
        
        if @ignored_files[file] || (cname && @ignored_classes[cname]) || @ignored_methods[func]
        # skip this one.
        elsif @autoalign
          print_autoaligned(file, line, func, msg, level, &blk)
        else
          print_formatted(file, line, func, msg, level, &blk)
        end
        msg = ""
      end
    end
  end

  def print_autoaligned(file, line, func, msg, level, &blk)
    @width = [ @width, func.length ].max
    hdr = sprintf "[%s:%04d] {%-*s} ", file, line, @width
    print(hdr, msg, level)
  end

  def print_formatted(file, line, func, msg, level, &blk)
    hdr = sprintf @fmt, file, line, func
    print(hdr, msg, level, &blk)
  end
  
  def print(hdr, msg, level, &blk)
    # puts "hdr: #{hdr}, msg #{msg}, level #{level}"
    if blk
      # puts "block is given"
      x = blk.call
      if x.kind_of?(String)
        # puts "got a string!"
        msg = x
      else
        # puts "did not get a string"
        return
      end
    else
      # puts "block not given"
    end

    if @colors[level]
      if @colorize_line
        @output.puts @colors[level] + hdr + " " + msg.to_s.chomp + ANSIColor.reset
      else
        @output.puts hdr + " " + @colors[level] + msg.to_s.chomp + ANSIColor.reset
      end
    else
      @output.puts hdr + " " + msg.to_s.chomp
    end      
  end

  def set_color(level, color)
    # log("#{level}, #{color}")
    @colors[level] = ANSIColor::code(color)
  end

  # by default, class methods delegate to a single app-wide log.

  @@log = Log.new

  def Log.verbose=(v)
    @@log.verbose = v
  end

  def Log.level=(lvl)
    @@log.level = lvl
  end

  # Assigns output to the given stream.
  def Log.output=(io)
    @@log.output = io
  end

  # sets whether to colorize the entire line, or just the message.
  def Log.colorize_line=(col)
    @@log.colorize_line = col
  end

  # Assigns output to a file with the given name. Returns the file; client
  # is responsible for closing it.
  def Log.outfile=(fname)
    @@log.outfile = fname
  end

  # Creates a printf format for the given widths, for aligning output.
  def Log.set_widths(file_width, line_width, func_width)
    @@log.set_widths(file_width, line_width, func_width)
  end

  def Log.ignore_file(fname)
    @@log.ignore_file(fname)
  end
  
  def Log.ignore_method(methname)
    @@ignored_methods[methname] = true
  end
  
  def Log.ignore_class(classname)
    @@ignored_classes[classname] = true
  end

  def Log.log_file(fname)
    @@log.log_file(fname)
  end
  
  def Log.log_method(methname)
    @@log.log_method(methname)
  end
  
  def Log.log_class(classname)
    @@log.log_class(classname)
  end

  # Sets auto-align of the {function} section.  set_widths will likely
  # result in nicer output.
  def Log.autoalign
    @@log.autoalign
  end

  def Log.debug(msg = "", depth = 1, &blk)
    @@log.log(msg, DEBUG, depth + 1, &blk)
  end

  def Log.info(msg = "", depth = 1, &blk)
    @@log.log(msg, INFO, depth + 1, &blk)
  end

  def Log.warn(msg = "", depth = 1, &blk)
    @@log.log(msg, WARN, depth + 1, &blk)
  end

  def Log.error(msg = "", depth = 1, &blk)
    @@log.log(msg, ERROR, depth + 1, &blk)
  end

  def Log.fatal(msg = "", depth = 1, &blk)
    @@log.log(msg, FATAL, depth + 1, &blk)
  end

  # Logs the given message.
  def Log.log(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    @@log.log(msg, level, depth + 1, cname, &blk)
  end

  def Log.set_color(level, color)
    @@log.set_color(level, color)
  end

  def Log.stack(msg = "", level = DEBUG, depth = 1, cname = nil, &blk)
    @@log.stack(msg, level, depth, cname, &blk)
  end

end


class AppLog < Log
  include Log::Severity

end


module Loggable

  # Logs the given message, including the class whence invoked.
  def log(msg = "", level = Log::DEBUG, depth = 1, &blk)
    AppLog.log(msg, level, depth + 1, self.class.to_s)
  end

  def debug(msg = "", depth = 1, &blk)
    AppLog.log(msg, Log::DEBUG, depth + 1, self.class.to_s, &blk)
  end

  def info(msg = "", depth = 1, &blk)
    AppLog.log(msg, Log::INFO, depth + 1, self.class.to_s, &blk)
  end

  def warn(msg = "", depth = 1, &blk)
    AppLog.log(msg, Log::WARN, depth + 1, self.class.to_s, &blk)
  end

  def error(msg = "", depth = 1, &blk)
    AppLog.log(msg, Log::ERROR, depth + 1, self.class.to_s, &blk)
  end

  def fatal(msg = "", depth = 1, &blk)
    AppLog.log(msg, Log::FATAL, depth + 1, self.class.to_s, &blk)
  end

  def stack(msg = "", level = Log::DEBUG, depth = 1, &blk)
    AppLog.stack(msg, level, depth + 1, self.class.to_s, &blk)
  end

end


if __FILE__ == $0
  Log.verbose = true
  Log.set_widths 15, -5, -35
  #Log.outfile = "/tmp/log." + $$.to_s

  class Demo
    include Loggable
    
    def initialize
      # log "hello"
      Log.set_color(Log::DEBUG, "cyan")
      Log.set_color(Log::INFO,  "bold cyan") 
      Log.set_color(Log::WARN,  "reverse")
      Log.set_color(Log::ERROR, "bold red")
      Log.set_color(Log::FATAL, "bold white on red")
    end

    def meth
      # log

      i = 4
      # info { "i: #{i}" }

      i /= 3
      debug { "i: #{i}" }

      i **= 3
      info "i: #{i}"

      i **= 2
      warn "i: #{i}"

      i <<= 4
      error "i: #{i}"

      i <<= 1
      fatal "i: #{i}"
    end

  end

  class Another
    include Loggable
    
    def Another.cmeth
      # /// "Log" only in instance methods
      # log "I'm sorry, Dave, I'm afraid I can't do that."

      # But this is legal.
      Log.log "happy, happy, joy, joy"
    end
  end

  demo = Demo.new
  demo.meth

  # Log.colorize_line = true

  # demo.meth
  # Another.cmeth

  # Log.info "we are done."
  
end
